/**
 * <h1>Amazon CloudWatch Synthetics Construct Library</h1>
 * <p>
 * Amazon CloudWatch Synthetics allow you to monitor your application by generating <strong>synthetic</strong> traffic. The traffic is produced by a <strong>canary</strong>: a configurable script that runs on a schedule. You configure the canary script to follow the same routes and perform the same actions as a user, which allows you to continually verify your user experience even when you don't have any traffic on your applications.
 * <p>
 * <h2>Canary</h2>
 * <p>
 * To illustrate how to use a canary, assume your application defines the following endpoint:
 * <p>
 * <blockquote><pre>
 * % curl "https://api.example.com/user/books/topbook/"
 * The Hitchhikers Guide to the Galaxy
 * </pre></blockquote>
 * <p>
 * The below code defines a canary that will hit the <code>books/topbook</code> endpoint every 5 minutes:
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .environmentVariables(Map.of(
 *                 "stage", "prod"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following is an example of an <code>index.js</code> file which exports the <code>handler</code> function:
 * <p>
 * <blockquote><pre>
 * const synthetics = require('Synthetics');
 * const log = require('SyntheticsLogger');
 * 
 * const pageLoadBlueprint = async function () {
 *   // Configure the stage of the API using environment variables
 *   const url = `https://api.example.com/${process.env.stage}/user/books/topbook/`;
 * 
 *   const page = await synthetics.getPage();
 *   const response = await page.goto(url, { waitUntil: 'domcontentloaded', timeout: 30000 });
 *   // Wait for page to render. Increase or decrease wait time based on endpoint being monitored.
 *   await page.waitFor(15000);
 *   // This will take a screenshot that will be included in test output artifacts.
 *   await synthetics.takeScreenshot('loaded', 'loaded');
 *   const pageTitle = await page.title();
 *   log.info('Page title: ' + pageTitle);
 *   if (response.status() !== 200) {
 *     throw 'Failed to load page!';
 *   }
 * };
 * 
 * exports.handler = async () =&gt; {
 *   return await pageLoadBlueprint();
 * };
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> The function <strong>must</strong> be called <code>handler</code>.
 * <p>
 * </blockquote>
 * <p>
 * The canary will automatically produce a CloudWatch Dashboard:
 * <p>
 * <img alt="UI Screenshot" src="images/ui-screenshot.png">
 * <p>
 * The Canary code will be executed in a lambda function created by Synthetics on your behalf. The Lambda function includes a custom <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html">runtime</a> provided by Synthetics. The provided runtime includes a variety of handy tools such as <a href="https://www.npmjs.com/package/puppeteer-core">Puppeteer</a> (for nodejs based one) and Chromium.
 * <p>
 * To learn more about Synthetics capabilities, check out the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries.html">docs</a>.
 * <p>
 * <h3>Canary Schedule</h3>
 * <p>
 * You can specify the schedule on which a canary runs by providing a
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-synthetics.Schedule.html"><code>Schedule</code></a>
 * object to the <code>schedule</code> property.
 * <p>
 * Configure a run rate of up to 60 minutes with <code>Schedule.rate</code>:
 * <p>
 * <blockquote><pre>
 * Schedule schedule = Schedule.rate(Duration.minutes(5));
 * </pre></blockquote>
 * <p>
 * You can also specify a <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_cron.html">cron expression</a> with <code>Schedule.cron</code>:
 * <p>
 * <blockquote><pre>
 * Schedule schedule = Schedule.cron(CronOptions.builder()
 *         .hour("0,8,16")
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you want the canary to run just once upon deployment, you can use <code>Schedule.once()</code>.
 * <p>
 * <h3>Automatic Retries</h3>
 * <p>
 * You can configure the canary to automatically retry failed runs by using the <code>maxRetries</code> property.
 * <p>
 * This is only supported on the following runtimes or newer: <code>Runtime.SYNTHETICS_NODEJS_PUPPETEER_10_0</code>, <code>Runtime.SYNTHETICS_NODEJS_PLAYWRIGHT_2_0</code>, <code>Runtime.SYNTHETICS_PYTHON_SELENIUM_5_1</code>.
 * <p>
 * Max retries can be set between 0 and 2. Canaries which time out after 10 minutes are automatically limited to one retry.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_autoretry.html">Configuring your canary to retry automatically</a>.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .handler("canary.handler")
 *                 .code(Code.fromAsset(join(__dirname, "canaries")))
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_PYTHON_SELENIUM_5_1)
 *         .maxRetries(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Active Tracing</h3>
 * <p>
 * You can choose to enable active AWS X-Ray tracing on canaries that use the <code>syn-nodejs-2.0</code> or later runtime by setting <code>activeTracing</code> to <code>true</code>.
 * <p>
 * With tracing enabled, traces are sent for all calls made by the canary that use the browser, the AWS SDK, or HTTP or HTTPS modules.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_tracing.html">Canaries and X-Ray tracing</a>.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .activeTracing(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Memory</h3>
 * <p>
 * You can set the maximum amount of memory the canary can use while running with the <code>memory</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .memory(Size.mebibytes(1024))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Timeout</h3>
 * <p>
 * You can set how long the canary is allowed to run before it must stop with the <code>timeout</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .timeout(Duration.seconds(60))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Browser Type Configuration</h3>
 * <p>
 * You can configure which browsers your canary uses for testing by specifying the <code>browserConfigs</code> property. This allows you to test your application across different browsers to ensure compatibility.
 * <p>
 * Available browser types:
 * <p>
 * <ul>
 * <li><code>BrowserType.CHROME</code> - Google Chrome browser</li>
 * <li><code>BrowserType.FIREFOX</code> - Mozilla Firefox browser</li>
 * </ul>
 * <p>
 * You can specify up to 2 browser configurations. When multiple browsers are configured, the canary will run tests on each browser sequentially.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_9_1)
 *         .browserConfigs(List.of(BrowserType.CHROME, BrowserType.FIREFOX))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> Firefox support is available for Node.js runtimes (Puppeteer and Playwright) but not for Python Selenium runtimes. When using Firefox, ensure your runtime version supports it.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Deleting underlying resources on canary deletion</h3>
 * <p>
 * When you delete a lambda, the following underlying resources are isolated in your AWS account:
 * <p>
 * <ul>
 * <li>Lambda Function that runs your canary script</li>
 * <li>S3 Bucket for artifact storage</li>
 * <li>IAM roles and policies</li>
 * <li>Log Groups in CloudWatch Logs.</li>
 * </ul>
 * <p>
 * To learn more about these underlying resources, see
 * <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/synthetics_canaries_deletion.html">Synthetics Canaries Deletion</a>.
 * <p>
 * In the CDK, you can configure your canary to delete the underlying lambda function when the canary is deleted.
 * This can be provisioned by setting <code>provisionedResourceCleanup</code> to <code>true</code>.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .handler("index.handler")
 *                 .code(Code.fromInline("/* Synthetics handler code"))
 *                 .build()))
 *         .provisionedResourceCleanup(true)
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: To properly clean up your canary on deletion, you still have to manually delete other resources
 * like S3 buckets and CloudWatch logs.
 * <p>
 * </blockquote><blockquote>
 * <p>
 * Note: The deletion of Lambda resources can also be performed by setting the <code>cleanup</code> argument to <code>Cleanup.LAMBDA</code>. However, this is an outdated argument that uses custom resources and is currently deprecated.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Configuring the Canary Script</h3>
 * <p>
 * To configure the script the canary executes, use the <code>test</code> property. The <code>test</code> property accepts a <code>Test</code> instance that can be initialized by the <code>Test</code> class static methods. Currently, the only implemented method is <code>Test.custom()</code>, which allows you to bring your own code. In the future, other methods will be added. <code>Test.custom()</code> accepts <code>code</code> and <code>handler</code> properties -- both are required by Synthetics to create a lambda function on your behalf.
 * <p>
 * The <code>synthetics.Code</code> class exposes static methods to bundle your code artifacts:
 * <p>
 * <ul>
 * <li><code>code.fromInline(code)</code> - specify an inline script.</li>
 * <li><code>code.fromAsset(path)</code> - specify a .zip file or a directory in the local filesystem which will be zipped and uploaded to S3 on deployment. See the below Note for directory structure.</li>
 * <li><code>code.fromBucket(bucket, key[, objectVersion])</code> - specify an S3 object that contains the .zip file of your runtime code. See the below Note for directory structure.</li>
 * </ul>
 * <p>
 * Using the <code>Code</code> class static initializers:
 * <p>
 * <blockquote><pre>
 * // To supply the code from a S3 bucket:
 * import software.amazon.awscdk.services.s3.*;
 * // To supply the code inline:
 * // To supply the code inline:
 * Canary.Builder.create(this, "Inline Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromInline("/* Synthetics handler code *&#47;"))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .build();
 * 
 * // To supply the code from your local filesystem:
 * // To supply the code from your local filesystem:
 * Canary.Builder.create(this, "Asset Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .build();
 * Bucket bucket = new Bucket(this, "Code Bucket");
 * Canary.Builder.create(this, "Bucket Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromBucket(bucket, "canary.zip"))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> Synthetics have a specified folder structure for canaries.
 * For Node with puppeteer scripts supplied via <code>code.fromAsset()</code> or <code>code.fromBucket()</code>, the canary resource requires the following folder structure for runtime versions older than <code>syn-nodejs-puppeteer-11.0</code>:
 * <p>
 * <blockquote><pre>
 * canary/
 * ├── nodejs/
 *    ├── node_modules/
 *         ├── &lt;filename&gt;.js
 * </pre></blockquote>
 * <p>
 * For puppeteer based runtime versions newer than or equal to <code>syn-nodejs-puppeteer-11.0</code>, <code>nodjs/node_modules</code> is not necessary but supported.
 * <p>
 * Both
 * <p>
 * <blockquote><pre>
 * canary/
 * ├── nodejs/
 *    ├── node_modules/
 *         ├── &lt;filename&gt;.js
 * </pre></blockquote>
 * <p>
 * And
 * <p>
 * <blockquote><pre>
 * canary/
 *  ├── &lt;filename&gt;.js
 * </pre></blockquote>
 * <p>
 * are supported.
 * <p>
 * For Node with playwright scripts supplied via <code>code.fromAsset()</code> or <code>code.fromBucket()</code>, the canary resource requires the following folder structure:
 * <p>
 * <blockquote><pre>
 * canary/
 * ├── &lt;filename&gt;.js,.mjs,.cjs
 * ├─some/dir/path
 *              ├── &lt;filename&gt;.js,.mjs,.cjs
 * </pre></blockquote>
 * <p>
 * If <code>&lt;filename&gt;.js</code> is placed in the canary directory, the handler should be specified as <code>filename.handler</code>.
 * However, if it is placed in the <code>some/dir/path</code> directory, the handler should be specified as <code>some/dir/path/filename.handler</code>.
 * For more information, see Synthetics <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Synthetics_WritingCanary_Nodejs_Playwright.html">docs</a>.
 * <p>
 * For Python scripts supplied via <code>code.fromAsset()</code> or <code>code.fromBucket()</code>, the canary resource requires the following folder structure:
 * <p>
 * <blockquote><pre>
 * canary/
 * ├── python/
 *     ├── &lt;filename&gt;.py
 * </pre></blockquote>
 * <p>
 * See Synthetics <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html">docs</a>.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Running a canary on a VPC</h3>
 * <p>
 * You can specify what <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_VPC.html">VPC a canary executes in</a>.
 * This can allow for monitoring services that may be internal to a specific VPC. To place a canary within a VPC, you can specify the <code>vpc</code> property with the desired <code>VPC</code> to place then canary in.
 * This will automatically attach the appropriate IAM permissions to attach to the VPC. This will also create a Security Group and attach to the default subnets for the VPC unless specified via <code>vpcSubnets</code> and <code>securityGroups</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * IVpc vpc;
 * 
 * Canary.Builder.create(this, "Vpc Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> By default, the Synthetics runtime needs access to the S3 and CloudWatch APIs, which will fail in a private subnet without internet access enabled (e.g. an isolated subnnet).
 * <p>
 * Ensure that the Canary is placed in a VPC either with internet connectivity or with VPC Endpoints for S3 and CloudWatch enabled and configured.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_VPC.html">Synthetics VPC docs</a>.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Alarms</h3>
 * <p>
 * You can configure a CloudWatch Alarm on a canary metric. Metrics are emitted by CloudWatch automatically and can be accessed by the following APIs:
 * <p>
 * <ul>
 * <li><code>canary.metricSuccessPercent()</code> - percentage of successful canary runs over a given time</li>
 * <li><code>canary.metricDuration()</code> - how much time each canary run takes, in seconds.</li>
 * <li><code>canary.metricFailed()</code> - number of failed canary runs over a given time</li>
 * </ul>
 * <p>
 * Create an alarm that tracks the canary metric:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * Canary canary;
 * 
 * Alarm.Builder.create(this, "CanaryAlarm")
 *         .metric(canary.metricSuccessPercent())
 *         .evaluationPeriods(2)
 *         .threshold(90)
 *         .comparisonOperator(ComparisonOperator.LESS_THAN_THRESHOLD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Performing safe canary updates</h3>
 * <p>
 * You can configure a canary to first perform a dry run before applying any updates. The <code>dryRunAndUpdate</code> property can be used to safely update canaries by validating the changes before they're applied.
 * This feature is supported for canary runtime versions <code>syn-nodejs-puppeteer-10.0+</code>, <code>syn-nodejs-playwright-2.0+</code>, and <code>syn-python-selenium-5.1+</code>.
 * <p>
 * When <code>dryRunAndUpdate</code> is set to <code>true</code>, CDK will execute a dry run to validate the changes before applying them to the canary.
 * If the dry run succeeds, the canary will be updated with the changes.
 * If the dry run fails, the CloudFormation deployment will fail with the dry run's failure reason.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_PYTHON_SELENIUM_5_1)
 *         .dryRunAndUpdate(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/performing-safe-canary-upgrades.html">Performing safe canary updates</a>.
 * <p>
 * <h3>Artifacts</h3>
 * <p>
 * You can pass an S3 bucket to store artifacts from canary runs. If you do not,
 * one will be auto-generated when the canary is created. You may add
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lifecycle-mgmt.html">lifecycle rules</a>
 * to the auto-generated bucket.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_6_2)
 *         .artifactsBucketLifecycleRules(List.of(LifecycleRule.builder()
 *                 .expiration(Duration.days(30))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Canary artifacts are encrypted at rest using an AWS-managed key by default.
 * <p>
 * You can choose the encryption options SSE-S3 or SSE-KMS by setting the <code>artifactS3EncryptionMode</code> property.
 * <p>
 * When you use SSE-KMS, you can also supply your own external KMS key by specifying the <code>kmsKey</code> property. If you don't, a KMS key will be automatically created and associated with the canary.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Key key = new Key(this, "myKey");
 * 
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_7_0)
 *         .artifactsBucketLifecycleRules(List.of(LifecycleRule.builder()
 *                 .expiration(Duration.days(30))
 *                 .build()))
 *         .artifactS3EncryptionMode(ArtifactsEncryptionMode.KMS)
 *         .artifactS3KmsKey(key)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Tag replication</h3>
 * <p>
 * You can configure a canary to replicate its tags to the underlying Lambda function. This is useful when you want the same tags that are applied to the canary to also be applied to the Lambda function that the canary uses.
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_7_0)
 *         .resourcesToReplicateTags(List.of(ResourceToReplicateTags.LAMBDA_FUNCTION))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you specify <code>ResourceToReplicateTags.LAMBDA_FUNCTION</code> in the <code>resourcesToReplicateTags</code> property, CloudWatch Synthetics will keep the tags of the canary and the Lambda function synchronized. Any future changes you make to the canary's tags will also be applied to the function.
 */
package software.amazon.awscdk.services.synthetics;
