package software.amazon.awscdk.services.ec2;

/**
 * The attributes for the instance types for a mixed instances policy.
 * <p>
 * When you specify multiple attributes, you get instance types that satisfy all of the specified attributes. If you specify multiple values for an attribute, you get instance types that satisfy any of the specified values.
 * <p>
 * To limit the list of instance types from which Amazon EC2 can identify matching instance types, you can use one of the following parameters, but not both in the same request:
 * <p>
 * <ul>
 * <li>AllowedInstanceTypes - The instance types to include in the list. All other instance types are ignored, even if they match your specified attributes.</li>
 * <li>ExcludedInstanceTypes - The instance types to exclude from the list, even if they match your specified attributes.</li>
 * </ul>
 * <p>
 * Note: You must specify VCpuCount and MemoryMiB. All other attributes are optional. Any unspecified optional attribute is set to its default.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Role infrastructureRole;
 * InstanceProfile instanceProfile;
 * Cluster cluster = Cluster.Builder.create(this, "Cluster").vpc(vpc).build();
 * // Create a Managed Instances Capacity Provider
 * ManagedInstancesCapacityProvider miCapacityProvider = ManagedInstancesCapacityProvider.Builder.create(this, "MICapacityProvider")
 *         .infrastructureRole(infrastructureRole)
 *         .ec2InstanceProfile(instanceProfile)
 *         .subnets(vpc.getPrivateSubnets())
 *         .securityGroups(List.of(SecurityGroup.Builder.create(this, "MISecurityGroup").vpc(vpc).build()))
 *         .instanceRequirements(InstanceRequirementsConfig.builder()
 *                 .vCpuCountMin(1)
 *                 .memoryMin(Size.gibibytes(2))
 *                 .cpuManufacturers(List.of(CpuManufacturer.INTEL))
 *                 .acceleratorManufacturers(List.of(AcceleratorManufacturer.NVIDIA))
 *                 .build())
 *         .propagateTags(PropagateManagedInstancesTags.CAPACITY_PROVIDER)
 *         .build();
 * // Optionally configure security group rules using IConnectable interface
 * miCapacityProvider.connections.allowFrom(Peer.ipv4(vpc.getVpcCidrBlock()), Port.tcp(80));
 * // Add the capacity provider to the cluster
 * cluster.addManagedInstancesCapacityProvider(miCapacityProvider);
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .networkMode(NetworkMode.AWS_VPC)
 *         .compatibility(Compatibility.MANAGED_INSTANCES)
 *         .build();
 * taskDefinition.addContainer("web", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryReservationMiB(256)
 *         .build());
 * FargateService.Builder.create(this, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider(miCapacityProvider.getCapacityProviderName())
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 */
@javax.annotation.Generated(value = "jsii-pacmak/1.119.0 (build 1634eac)", date = "2025-12-01T16:02:19.480Z")
@software.amazon.jsii.Jsii(module = software.amazon.awscdk.$Module.class, fqn = "aws-cdk-lib.aws_ec2.InstanceRequirementsConfig")
@software.amazon.jsii.Jsii.Proxy(InstanceRequirementsConfig.Jsii$Proxy.class)
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
public interface InstanceRequirementsConfig extends software.amazon.jsii.JsiiSerializable {

    /**
     * The minimum instance memory size for an instance type, in MiB.
     * <p>
     * Required: Yes
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    @org.jetbrains.annotations.NotNull software.amazon.awscdk.Size getMemoryMin();

    /**
     * The minimum number of vCPUs for an instance type.
     * <p>
     * Required: Yes
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    @org.jetbrains.annotations.NotNull java.lang.Number getVCpuCountMin();

    /**
     * The maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) for an instance type.
     * <p>
     * To exclude accelerator-enabled instance types, set Max to 0.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getAcceleratorCountMax() {
        return null;
    }

    /**
     * The minimum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) for an instance type.
     * <p>
     * To exclude accelerator-enabled instance types, set acceleratorCountMax to 0.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getAcceleratorCountMin() {
        return null;
    }

    /**
     * Indicates whether instance types must have accelerators by specific manufacturers.
     * <p>
     * <ul>
     * <li>For instance types with NVIDIA devices, specify nvidia.</li>
     * <li>For instance types with AMD devices, specify amd.</li>
     * <li>For instance types with AWS devices, specify amazon-web-services.</li>
     * <li>For instance types with Xilinx devices, specify xilinx.</li>
     * </ul>
     * <p>
     * Default: - Any manufacturer
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<software.amazon.awscdk.services.ec2.AcceleratorManufacturer> getAcceleratorManufacturers() {
        return null;
    }

    /**
     * Lists the accelerators that must be on an instance type.
     * <p>
     * <ul>
     * <li>For instance types with NVIDIA A100 GPUs, specify a100.</li>
     * <li>For instance types with NVIDIA V100 GPUs, specify v100.</li>
     * <li>For instance types with NVIDIA K80 GPUs, specify k80.</li>
     * <li>For instance types with NVIDIA T4 GPUs, specify t4.</li>
     * <li>For instance types with NVIDIA M60 GPUs, specify m60.</li>
     * <li>For instance types with AMD Radeon Pro V520 GPUs, specify radeon-pro-v520.</li>
     * <li>For instance types with Xilinx VU9P FPGAs, specify vu9p.</li>
     * </ul>
     * <p>
     * Default: - Any accelerator
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<software.amazon.awscdk.services.ec2.AcceleratorName> getAcceleratorNames() {
        return null;
    }

    /**
     * The maximum total memory size for the accelerators on an instance type, in MiB.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.Size getAcceleratorTotalMemoryMax() {
        return null;
    }

    /**
     * The minimum total memory size for the accelerators on an instance type, in MiB.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.Size getAcceleratorTotalMemoryMin() {
        return null;
    }

    /**
     * Lists the accelerator types that must be on an instance type.
     * <p>
     * <ul>
     * <li>For instance types with GPU accelerators, specify gpu.</li>
     * <li>For instance types with FPGA accelerators, specify fpga.</li>
     * <li>For instance types with inference accelerators, specify inference.</li>
     * </ul>
     * <p>
     * Default: - Any accelerator type
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<software.amazon.awscdk.services.ec2.AcceleratorType> getAcceleratorTypes() {
        return null;
    }

    /**
     * The instance types to apply your specified attributes against.
     * <p>
     * All other instance types are ignored, even if they match your specified attributes.
     * <p>
     * You can use strings with one or more wild cards, represented by an asterisk (<em>), to allow an instance type, size, or generation. The following are examples: m5.8xlarge, c5</em>.<em>, m5a.</em>, r*, <em>3</em>.
     * <p>
     * For example, if you specify c5*, Amazon EC2 Auto Scaling will allow the entire C5 instance family, which includes all C5a and C5n instance types. If you specify m5a.*, Amazon EC2 Auto Scaling will allow all the M5a instance types, but not the M5n instance types.
     * <p>
     * Note: If you specify AllowedInstanceTypes, you can't specify ExcludedInstanceTypes.
     * <p>
     * Default: - All instance types
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<java.lang.String> getAllowedInstanceTypes() {
        return null;
    }

    /**
     * Indicates whether bare metal instance types are included, excluded, or required.
     * <p>
     * Default: - excluded
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.services.ec2.BareMetal getBareMetal() {
        return null;
    }

    /**
     * The maximum baseline bandwidth performance for an instance type, in Mbps.
     * <p>
     * For more information, see Amazon EBS–optimized instances in the Amazon EC2 User Guide.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getBaselineEbsBandwidthMbpsMax() {
        return null;
    }

    /**
     * The minimum baseline bandwidth performance for an instance type, in Mbps.
     * <p>
     * For more information, see Amazon EBS–optimized instances in the Amazon EC2 User Guide.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getBaselineEbsBandwidthMbpsMin() {
        return null;
    }

    /**
     * Indicates whether burstable performance instance types are included, excluded, or required.
     * <p>
     * For more information, see Burstable performance instances in the Amazon EC2 User Guide.
     * <p>
     * Default: - excluded
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.services.ec2.BurstablePerformance getBurstablePerformance() {
        return null;
    }

    /**
     * Lists which specific CPU manufacturers to include.
     * <p>
     * <ul>
     * <li>For instance types with Intel CPUs, specify intel.</li>
     * <li>For instance types with AMD CPUs, specify amd.</li>
     * <li>For instance types with AWS CPUs, specify amazon-web-services.</li>
     * <li>For instance types with Apple CPUs, specify apple.</li>
     * </ul>
     * <p>
     * Note: Don't confuse the CPU hardware manufacturer with the CPU hardware architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
     * <p>
     * Default: - Any manufacturer
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<software.amazon.awscdk.services.ec2.CpuManufacturer> getCpuManufacturers() {
        return null;
    }

    /**
     * The instance types to exclude.
     * <p>
     * You can use strings with one or more wild cards, represented by an asterisk (<em>), to exclude an instance family, type, size, or generation. The following are examples: m5.8xlarge, c5</em>.<em>, m5a.</em>, r*, <em>3</em>.
     * <p>
     * For example, if you specify c5*, you are excluding the entire C5 instance family, which includes all C5a and C5n instance types. If you specify m5a.*, Amazon EC2 Auto Scaling will exclude all the M5a instance types, but not the M5n instance types.
     * <p>
     * Note: If you specify ExcludedInstanceTypes, you can't specify AllowedInstanceTypes.
     * <p>
     * Default: - No excluded instance types
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<java.lang.String> getExcludedInstanceTypes() {
        return null;
    }

    /**
     * Indicates whether current or previous generation instance types are included.
     * <p>
     * <ul>
     * <li>For current generation instance types, specify current. The current generation includes EC2 instance types currently recommended for use. This typically includes the latest two to three generations in each instance family. For more information, see Instance types in the Amazon EC2 User Guide.</li>
     * <li>For previous generation instance types, specify previous.</li>
     * </ul>
     * <p>
     * Default: - Any current or previous generation
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<software.amazon.awscdk.services.ec2.InstanceGeneration> getInstanceGenerations() {
        return null;
    }

    /**
     * Indicates whether instance types with instance store volumes are included, excluded, or required.
     * <p>
     * For more information, see Amazon EC2 instance store in the Amazon EC2 User Guide.
     * <p>
     * Default: - included
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.services.ec2.LocalStorage getLocalStorage() {
        return null;
    }

    /**
     * Indicates the type of local storage that is required.
     * <p>
     * <ul>
     * <li>For instance types with hard disk drive (HDD) storage, specify hdd.</li>
     * <li>For instance types with solid state drive (SSD) storage, specify ssd.</li>
     * </ul>
     * <p>
     * Default: - Any local storage type
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.util.List<software.amazon.awscdk.services.ec2.LocalStorageType> getLocalStorageTypes() {
        return null;
    }

    /**
     * [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified On-Demand price.
     * <p>
     * The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from either the lowest priced current generation instance types or, failing that, the lowest priced previous generation instance types that match your attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
     * <p>
     * The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage.
     * <p>
     * If you set DesiredCapacityType to vcpu or memory-mib, the price protection threshold is based on the per-vCPU or per-memory price instead of the per instance price.
     * <p>
     * Note: Only one of SpotMaxPricePercentageOverLowestPrice or MaxSpotPriceAsPercentageOfOptimalOnDemandPrice can be specified. If you don't specify either, Amazon EC2 Auto Scaling will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as 999999.
     * <p>
     * Default: - Automatic optimal price protection
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice() {
        return null;
    }

    /**
     * The maximum instance memory size for an instance type, in MiB.
     * <p>
     * Default: - No maximum limit
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.Size getMemoryMax() {
        return null;
    }

    /**
     * The maximum amount of memory per vCPU for an instance type, in GiB.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.Size getMemoryPerVCpuMax() {
        return null;
    }

    /**
     * The minimum amount of memory per vCPU for an instance type, in GiB.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable software.amazon.awscdk.Size getMemoryPerVCpuMin() {
        return null;
    }

    /**
     * The maximum amount of network bandwidth, in gigabits per second (Gbps).
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getNetworkBandwidthGbpsMax() {
        return null;
    }

    /**
     * The minimum amount of network bandwidth, in gigabits per second (Gbps).
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getNetworkBandwidthGbpsMin() {
        return null;
    }

    /**
     * The maximum number of network interfaces for an instance type.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getNetworkInterfaceCountMax() {
        return null;
    }

    /**
     * The minimum number of network interfaces for an instance type.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getNetworkInterfaceCountMin() {
        return null;
    }

    /**
     * [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an identified On-Demand price.
     * <p>
     * The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from either the lowest priced current generation instance types or, failing that, the lowest priced previous generation instance types that match your attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
     * <p>
     * The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage.
     * <p>
     * To turn off price protection, specify a high value, such as 999999.
     * <p>
     * If you set DesiredCapacityType to vcpu or memory-mib, the price protection threshold is applied based on the per-vCPU or per-memory price instead of the per instance price.
     * <p>
     * Default: - 20
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getOnDemandMaxPricePercentageOverLowestPrice() {
        return null;
    }

    /**
     * Indicates whether instance types must provide On-Demand Instance hibernation support.
     * <p>
     * Default: - false
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Boolean getRequireHibernateSupport() {
        return null;
    }

    /**
     * [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an identified Spot price.
     * <p>
     * The identified Spot price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from either the lowest priced current generation instance types or, failing that, the lowest priced previous generation instance types that match your attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
     * <p>
     * The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage.
     * <p>
     * If you set DesiredCapacityType to vcpu or memory-mib, the price protection threshold is based on the per-vCPU or per-memory price instead of the per instance price.
     * <p>
     * Note: Only one of SpotMaxPricePercentageOverLowestPrice or MaxSpotPriceAsPercentageOfOptimalOnDemandPrice can be specified. If you don't specify either, Amazon EC2 Auto Scaling will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as 999999.
     * <p>
     * Default: - Automatic optimal price protection
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getSpotMaxPricePercentageOverLowestPrice() {
        return null;
    }

    /**
     * The maximum total local storage size for an instance type, in GB.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getTotalLocalStorageGBMax() {
        return null;
    }

    /**
     * The minimum total local storage size for an instance type, in GB.
     * <p>
     * Default: - No minimum or maximum limits
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getTotalLocalStorageGBMin() {
        return null;
    }

    /**
     * The maximum number of vCPUs for an instance type.
     * <p>
     * Default: - No maximum limit
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    default @org.jetbrains.annotations.Nullable java.lang.Number getVCpuCountMax() {
        return null;
    }

    /**
     * @return a {@link Builder} of {@link InstanceRequirementsConfig}
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    static Builder builder() {
        return new Builder();
    }
    /**
     * A builder for {@link InstanceRequirementsConfig}
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static final class Builder implements software.amazon.jsii.Builder<InstanceRequirementsConfig> {
        software.amazon.awscdk.Size memoryMin;
        java.lang.Number vCpuCountMin;
        java.lang.Number acceleratorCountMax;
        java.lang.Number acceleratorCountMin;
        java.util.List<software.amazon.awscdk.services.ec2.AcceleratorManufacturer> acceleratorManufacturers;
        java.util.List<software.amazon.awscdk.services.ec2.AcceleratorName> acceleratorNames;
        software.amazon.awscdk.Size acceleratorTotalMemoryMax;
        software.amazon.awscdk.Size acceleratorTotalMemoryMin;
        java.util.List<software.amazon.awscdk.services.ec2.AcceleratorType> acceleratorTypes;
        java.util.List<java.lang.String> allowedInstanceTypes;
        software.amazon.awscdk.services.ec2.BareMetal bareMetal;
        java.lang.Number baselineEbsBandwidthMbpsMax;
        java.lang.Number baselineEbsBandwidthMbpsMin;
        software.amazon.awscdk.services.ec2.BurstablePerformance burstablePerformance;
        java.util.List<software.amazon.awscdk.services.ec2.CpuManufacturer> cpuManufacturers;
        java.util.List<java.lang.String> excludedInstanceTypes;
        java.util.List<software.amazon.awscdk.services.ec2.InstanceGeneration> instanceGenerations;
        software.amazon.awscdk.services.ec2.LocalStorage localStorage;
        java.util.List<software.amazon.awscdk.services.ec2.LocalStorageType> localStorageTypes;
        java.lang.Number maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        software.amazon.awscdk.Size memoryMax;
        software.amazon.awscdk.Size memoryPerVCpuMax;
        software.amazon.awscdk.Size memoryPerVCpuMin;
        java.lang.Number networkBandwidthGbpsMax;
        java.lang.Number networkBandwidthGbpsMin;
        java.lang.Number networkInterfaceCountMax;
        java.lang.Number networkInterfaceCountMin;
        java.lang.Number onDemandMaxPricePercentageOverLowestPrice;
        java.lang.Boolean requireHibernateSupport;
        java.lang.Number spotMaxPricePercentageOverLowestPrice;
        java.lang.Number totalLocalStorageGbMax;
        java.lang.Number totalLocalStorageGbMin;
        java.lang.Number vCpuCountMax;

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getMemoryMin}
         * @param memoryMin The minimum instance memory size for an instance type, in MiB. This parameter is required.
         *                  Required: Yes
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder memoryMin(software.amazon.awscdk.Size memoryMin) {
            this.memoryMin = memoryMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getVCpuCountMin}
         * @param vCpuCountMin The minimum number of vCPUs for an instance type. This parameter is required.
         *                     Required: Yes
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder vCpuCountMin(java.lang.Number vCpuCountMin) {
            this.vCpuCountMin = vCpuCountMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorCountMax}
         * @param acceleratorCountMax The maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) for an instance type.
         *                            To exclude accelerator-enabled instance types, set Max to 0.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder acceleratorCountMax(java.lang.Number acceleratorCountMax) {
            this.acceleratorCountMax = acceleratorCountMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorCountMin}
         * @param acceleratorCountMin The minimum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) for an instance type.
         *                            To exclude accelerator-enabled instance types, set acceleratorCountMax to 0.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder acceleratorCountMin(java.lang.Number acceleratorCountMin) {
            this.acceleratorCountMin = acceleratorCountMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorManufacturers}
         * @param acceleratorManufacturers Indicates whether instance types must have accelerators by specific manufacturers.
         *                                 <ul>
         *                                 <li>For instance types with NVIDIA devices, specify nvidia.</li>
         *                                 <li>For instance types with AMD devices, specify amd.</li>
         *                                 <li>For instance types with AWS devices, specify amazon-web-services.</li>
         *                                 <li>For instance types with Xilinx devices, specify xilinx.</li>
         *                                 </ul>
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @SuppressWarnings("unchecked")
        public Builder acceleratorManufacturers(java.util.List<? extends software.amazon.awscdk.services.ec2.AcceleratorManufacturer> acceleratorManufacturers) {
            this.acceleratorManufacturers = (java.util.List<software.amazon.awscdk.services.ec2.AcceleratorManufacturer>)acceleratorManufacturers;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorNames}
         * @param acceleratorNames Lists the accelerators that must be on an instance type.
         *                         <ul>
         *                         <li>For instance types with NVIDIA A100 GPUs, specify a100.</li>
         *                         <li>For instance types with NVIDIA V100 GPUs, specify v100.</li>
         *                         <li>For instance types with NVIDIA K80 GPUs, specify k80.</li>
         *                         <li>For instance types with NVIDIA T4 GPUs, specify t4.</li>
         *                         <li>For instance types with NVIDIA M60 GPUs, specify m60.</li>
         *                         <li>For instance types with AMD Radeon Pro V520 GPUs, specify radeon-pro-v520.</li>
         *                         <li>For instance types with Xilinx VU9P FPGAs, specify vu9p.</li>
         *                         </ul>
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @SuppressWarnings("unchecked")
        public Builder acceleratorNames(java.util.List<? extends software.amazon.awscdk.services.ec2.AcceleratorName> acceleratorNames) {
            this.acceleratorNames = (java.util.List<software.amazon.awscdk.services.ec2.AcceleratorName>)acceleratorNames;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorTotalMemoryMax}
         * @param acceleratorTotalMemoryMax The maximum total memory size for the accelerators on an instance type, in MiB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder acceleratorTotalMemoryMax(software.amazon.awscdk.Size acceleratorTotalMemoryMax) {
            this.acceleratorTotalMemoryMax = acceleratorTotalMemoryMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorTotalMemoryMin}
         * @param acceleratorTotalMemoryMin The minimum total memory size for the accelerators on an instance type, in MiB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder acceleratorTotalMemoryMin(software.amazon.awscdk.Size acceleratorTotalMemoryMin) {
            this.acceleratorTotalMemoryMin = acceleratorTotalMemoryMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAcceleratorTypes}
         * @param acceleratorTypes Lists the accelerator types that must be on an instance type.
         *                         <ul>
         *                         <li>For instance types with GPU accelerators, specify gpu.</li>
         *                         <li>For instance types with FPGA accelerators, specify fpga.</li>
         *                         <li>For instance types with inference accelerators, specify inference.</li>
         *                         </ul>
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @SuppressWarnings("unchecked")
        public Builder acceleratorTypes(java.util.List<? extends software.amazon.awscdk.services.ec2.AcceleratorType> acceleratorTypes) {
            this.acceleratorTypes = (java.util.List<software.amazon.awscdk.services.ec2.AcceleratorType>)acceleratorTypes;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getAllowedInstanceTypes}
         * @param allowedInstanceTypes The instance types to apply your specified attributes against.
         *                             All other instance types are ignored, even if they match your specified attributes.
         *                             <p>
         *                             You can use strings with one or more wild cards, represented by an asterisk (<em>), to allow an instance type, size, or generation. The following are examples: m5.8xlarge, c5</em>.<em>, m5a.</em>, r*, <em>3</em>.
         *                             <p>
         *                             For example, if you specify c5*, Amazon EC2 Auto Scaling will allow the entire C5 instance family, which includes all C5a and C5n instance types. If you specify m5a.*, Amazon EC2 Auto Scaling will allow all the M5a instance types, but not the M5n instance types.
         *                             <p>
         *                             Note: If you specify AllowedInstanceTypes, you can't specify ExcludedInstanceTypes.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder allowedInstanceTypes(java.util.List<java.lang.String> allowedInstanceTypes) {
            this.allowedInstanceTypes = allowedInstanceTypes;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getBareMetal}
         * @param bareMetal Indicates whether bare metal instance types are included, excluded, or required.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder bareMetal(software.amazon.awscdk.services.ec2.BareMetal bareMetal) {
            this.bareMetal = bareMetal;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getBaselineEbsBandwidthMbpsMax}
         * @param baselineEbsBandwidthMbpsMax The maximum baseline bandwidth performance for an instance type, in Mbps.
         *                                    For more information, see Amazon EBS–optimized instances in the Amazon EC2 User Guide.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder baselineEbsBandwidthMbpsMax(java.lang.Number baselineEbsBandwidthMbpsMax) {
            this.baselineEbsBandwidthMbpsMax = baselineEbsBandwidthMbpsMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getBaselineEbsBandwidthMbpsMin}
         * @param baselineEbsBandwidthMbpsMin The minimum baseline bandwidth performance for an instance type, in Mbps.
         *                                    For more information, see Amazon EBS–optimized instances in the Amazon EC2 User Guide.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder baselineEbsBandwidthMbpsMin(java.lang.Number baselineEbsBandwidthMbpsMin) {
            this.baselineEbsBandwidthMbpsMin = baselineEbsBandwidthMbpsMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getBurstablePerformance}
         * @param burstablePerformance Indicates whether burstable performance instance types are included, excluded, or required.
         *                             For more information, see Burstable performance instances in the Amazon EC2 User Guide.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder burstablePerformance(software.amazon.awscdk.services.ec2.BurstablePerformance burstablePerformance) {
            this.burstablePerformance = burstablePerformance;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getCpuManufacturers}
         * @param cpuManufacturers Lists which specific CPU manufacturers to include.
         *                         <ul>
         *                         <li>For instance types with Intel CPUs, specify intel.</li>
         *                         <li>For instance types with AMD CPUs, specify amd.</li>
         *                         <li>For instance types with AWS CPUs, specify amazon-web-services.</li>
         *                         <li>For instance types with Apple CPUs, specify apple.</li>
         *                         </ul>
         *                         <p>
         *                         Note: Don't confuse the CPU hardware manufacturer with the CPU hardware architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @SuppressWarnings("unchecked")
        public Builder cpuManufacturers(java.util.List<? extends software.amazon.awscdk.services.ec2.CpuManufacturer> cpuManufacturers) {
            this.cpuManufacturers = (java.util.List<software.amazon.awscdk.services.ec2.CpuManufacturer>)cpuManufacturers;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getExcludedInstanceTypes}
         * @param excludedInstanceTypes The instance types to exclude.
         *                              You can use strings with one or more wild cards, represented by an asterisk (<em>), to exclude an instance family, type, size, or generation. The following are examples: m5.8xlarge, c5</em>.<em>, m5a.</em>, r*, <em>3</em>.
         *                              <p>
         *                              For example, if you specify c5*, you are excluding the entire C5 instance family, which includes all C5a and C5n instance types. If you specify m5a.*, Amazon EC2 Auto Scaling will exclude all the M5a instance types, but not the M5n instance types.
         *                              <p>
         *                              Note: If you specify ExcludedInstanceTypes, you can't specify AllowedInstanceTypes.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder excludedInstanceTypes(java.util.List<java.lang.String> excludedInstanceTypes) {
            this.excludedInstanceTypes = excludedInstanceTypes;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getInstanceGenerations}
         * @param instanceGenerations Indicates whether current or previous generation instance types are included.
         *                            <ul>
         *                            <li>For current generation instance types, specify current. The current generation includes EC2 instance types currently recommended for use. This typically includes the latest two to three generations in each instance family. For more information, see Instance types in the Amazon EC2 User Guide.</li>
         *                            <li>For previous generation instance types, specify previous.</li>
         *                            </ul>
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @SuppressWarnings("unchecked")
        public Builder instanceGenerations(java.util.List<? extends software.amazon.awscdk.services.ec2.InstanceGeneration> instanceGenerations) {
            this.instanceGenerations = (java.util.List<software.amazon.awscdk.services.ec2.InstanceGeneration>)instanceGenerations;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getLocalStorage}
         * @param localStorage Indicates whether instance types with instance store volumes are included, excluded, or required.
         *                     For more information, see Amazon EC2 instance store in the Amazon EC2 User Guide.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder localStorage(software.amazon.awscdk.services.ec2.LocalStorage localStorage) {
            this.localStorage = localStorage;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getLocalStorageTypes}
         * @param localStorageTypes Indicates the type of local storage that is required.
         *                          <ul>
         *                          <li>For instance types with hard disk drive (HDD) storage, specify hdd.</li>
         *                          <li>For instance types with solid state drive (SSD) storage, specify ssd.</li>
         *                          </ul>
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @SuppressWarnings("unchecked")
        public Builder localStorageTypes(java.util.List<? extends software.amazon.awscdk.services.ec2.LocalStorageType> localStorageTypes) {
            this.localStorageTypes = (java.util.List<software.amazon.awscdk.services.ec2.LocalStorageType>)localStorageTypes;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice}
         * @param maxSpotPriceAsPercentageOfOptimalOnDemandPrice [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified On-Demand price.
         *                                                       The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from either the lowest priced current generation instance types or, failing that, the lowest priced previous generation instance types that match your attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
         *                                                       <p>
         *                                                       The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage.
         *                                                       <p>
         *                                                       If you set DesiredCapacityType to vcpu or memory-mib, the price protection threshold is based on the per-vCPU or per-memory price instead of the per instance price.
         *                                                       <p>
         *                                                       Note: Only one of SpotMaxPricePercentageOverLowestPrice or MaxSpotPriceAsPercentageOfOptimalOnDemandPrice can be specified. If you don't specify either, Amazon EC2 Auto Scaling will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as 999999.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder maxSpotPriceAsPercentageOfOptimalOnDemandPrice(java.lang.Number maxSpotPriceAsPercentageOfOptimalOnDemandPrice) {
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getMemoryMax}
         * @param memoryMax The maximum instance memory size for an instance type, in MiB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder memoryMax(software.amazon.awscdk.Size memoryMax) {
            this.memoryMax = memoryMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getMemoryPerVCpuMax}
         * @param memoryPerVCpuMax The maximum amount of memory per vCPU for an instance type, in GiB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder memoryPerVCpuMax(software.amazon.awscdk.Size memoryPerVCpuMax) {
            this.memoryPerVCpuMax = memoryPerVCpuMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getMemoryPerVCpuMin}
         * @param memoryPerVCpuMin The minimum amount of memory per vCPU for an instance type, in GiB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder memoryPerVCpuMin(software.amazon.awscdk.Size memoryPerVCpuMin) {
            this.memoryPerVCpuMin = memoryPerVCpuMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getNetworkBandwidthGbpsMax}
         * @param networkBandwidthGbpsMax The maximum amount of network bandwidth, in gigabits per second (Gbps).
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder networkBandwidthGbpsMax(java.lang.Number networkBandwidthGbpsMax) {
            this.networkBandwidthGbpsMax = networkBandwidthGbpsMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getNetworkBandwidthGbpsMin}
         * @param networkBandwidthGbpsMin The minimum amount of network bandwidth, in gigabits per second (Gbps).
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder networkBandwidthGbpsMin(java.lang.Number networkBandwidthGbpsMin) {
            this.networkBandwidthGbpsMin = networkBandwidthGbpsMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getNetworkInterfaceCountMax}
         * @param networkInterfaceCountMax The maximum number of network interfaces for an instance type.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder networkInterfaceCountMax(java.lang.Number networkInterfaceCountMax) {
            this.networkInterfaceCountMax = networkInterfaceCountMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getNetworkInterfaceCountMin}
         * @param networkInterfaceCountMin The minimum number of network interfaces for an instance type.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder networkInterfaceCountMin(java.lang.Number networkInterfaceCountMin) {
            this.networkInterfaceCountMin = networkInterfaceCountMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getOnDemandMaxPricePercentageOverLowestPrice}
         * @param onDemandMaxPricePercentageOverLowestPrice [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an identified On-Demand price.
         *                                                  The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from either the lowest priced current generation instance types or, failing that, the lowest priced previous generation instance types that match your attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
         *                                                  <p>
         *                                                  The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage.
         *                                                  <p>
         *                                                  To turn off price protection, specify a high value, such as 999999.
         *                                                  <p>
         *                                                  If you set DesiredCapacityType to vcpu or memory-mib, the price protection threshold is applied based on the per-vCPU or per-memory price instead of the per instance price.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder onDemandMaxPricePercentageOverLowestPrice(java.lang.Number onDemandMaxPricePercentageOverLowestPrice) {
            this.onDemandMaxPricePercentageOverLowestPrice = onDemandMaxPricePercentageOverLowestPrice;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getRequireHibernateSupport}
         * @param requireHibernateSupport Indicates whether instance types must provide On-Demand Instance hibernation support.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder requireHibernateSupport(java.lang.Boolean requireHibernateSupport) {
            this.requireHibernateSupport = requireHibernateSupport;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getSpotMaxPricePercentageOverLowestPrice}
         * @param spotMaxPricePercentageOverLowestPrice [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an identified Spot price.
         *                                              The identified Spot price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from either the lowest priced current generation instance types or, failing that, the lowest priced previous generation instance types that match your attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
         *                                              <p>
         *                                              The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage.
         *                                              <p>
         *                                              If you set DesiredCapacityType to vcpu or memory-mib, the price protection threshold is based on the per-vCPU or per-memory price instead of the per instance price.
         *                                              <p>
         *                                              Note: Only one of SpotMaxPricePercentageOverLowestPrice or MaxSpotPriceAsPercentageOfOptimalOnDemandPrice can be specified. If you don't specify either, Amazon EC2 Auto Scaling will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as 999999.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder spotMaxPricePercentageOverLowestPrice(java.lang.Number spotMaxPricePercentageOverLowestPrice) {
            this.spotMaxPricePercentageOverLowestPrice = spotMaxPricePercentageOverLowestPrice;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getTotalLocalStorageGbMax}
         * @param totalLocalStorageGbMax The maximum total local storage size for an instance type, in GB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder totalLocalStorageGbMax(java.lang.Number totalLocalStorageGbMax) {
            this.totalLocalStorageGbMax = totalLocalStorageGbMax;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getTotalLocalStorageGbMin}
         * @param totalLocalStorageGbMin The minimum total local storage size for an instance type, in GB.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder totalLocalStorageGbMin(java.lang.Number totalLocalStorageGbMin) {
            this.totalLocalStorageGbMin = totalLocalStorageGbMin;
            return this;
        }

        /**
         * Sets the value of {@link InstanceRequirementsConfig#getVCpuCountMax}
         * @param vCpuCountMax The maximum number of vCPUs for an instance type.
         * @return {@code this}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder vCpuCountMax(java.lang.Number vCpuCountMax) {
            this.vCpuCountMax = vCpuCountMax;
            return this;
        }

        /**
         * Builds the configured instance.
         * @return a new instance of {@link InstanceRequirementsConfig}
         * @throws NullPointerException if any required attribute was not provided
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @Override
        public InstanceRequirementsConfig build() {
            return new Jsii$Proxy(this);
        }
    }

    /**
     * An implementation for {@link InstanceRequirementsConfig}
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    @software.amazon.jsii.Internal
    final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements InstanceRequirementsConfig {
        private final software.amazon.awscdk.Size memoryMin;
        private final java.lang.Number vCpuCountMin;
        private final java.lang.Number acceleratorCountMax;
        private final java.lang.Number acceleratorCountMin;
        private final java.util.List<software.amazon.awscdk.services.ec2.AcceleratorManufacturer> acceleratorManufacturers;
        private final java.util.List<software.amazon.awscdk.services.ec2.AcceleratorName> acceleratorNames;
        private final software.amazon.awscdk.Size acceleratorTotalMemoryMax;
        private final software.amazon.awscdk.Size acceleratorTotalMemoryMin;
        private final java.util.List<software.amazon.awscdk.services.ec2.AcceleratorType> acceleratorTypes;
        private final java.util.List<java.lang.String> allowedInstanceTypes;
        private final software.amazon.awscdk.services.ec2.BareMetal bareMetal;
        private final java.lang.Number baselineEbsBandwidthMbpsMax;
        private final java.lang.Number baselineEbsBandwidthMbpsMin;
        private final software.amazon.awscdk.services.ec2.BurstablePerformance burstablePerformance;
        private final java.util.List<software.amazon.awscdk.services.ec2.CpuManufacturer> cpuManufacturers;
        private final java.util.List<java.lang.String> excludedInstanceTypes;
        private final java.util.List<software.amazon.awscdk.services.ec2.InstanceGeneration> instanceGenerations;
        private final software.amazon.awscdk.services.ec2.LocalStorage localStorage;
        private final java.util.List<software.amazon.awscdk.services.ec2.LocalStorageType> localStorageTypes;
        private final java.lang.Number maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        private final software.amazon.awscdk.Size memoryMax;
        private final software.amazon.awscdk.Size memoryPerVCpuMax;
        private final software.amazon.awscdk.Size memoryPerVCpuMin;
        private final java.lang.Number networkBandwidthGbpsMax;
        private final java.lang.Number networkBandwidthGbpsMin;
        private final java.lang.Number networkInterfaceCountMax;
        private final java.lang.Number networkInterfaceCountMin;
        private final java.lang.Number onDemandMaxPricePercentageOverLowestPrice;
        private final java.lang.Boolean requireHibernateSupport;
        private final java.lang.Number spotMaxPricePercentageOverLowestPrice;
        private final java.lang.Number totalLocalStorageGbMax;
        private final java.lang.Number totalLocalStorageGbMin;
        private final java.lang.Number vCpuCountMax;

        /**
         * Constructor that initializes the object based on values retrieved from the JsiiObject.
         * @param objRef Reference to the JSII managed object.
         */
        protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
            super(objRef);
            this.memoryMin = software.amazon.jsii.Kernel.get(this, "memoryMin", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.Size.class));
            this.vCpuCountMin = software.amazon.jsii.Kernel.get(this, "vCpuCountMin", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.acceleratorCountMax = software.amazon.jsii.Kernel.get(this, "acceleratorCountMax", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.acceleratorCountMin = software.amazon.jsii.Kernel.get(this, "acceleratorCountMin", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.acceleratorManufacturers = software.amazon.jsii.Kernel.get(this, "acceleratorManufacturers", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.AcceleratorManufacturer.class)));
            this.acceleratorNames = software.amazon.jsii.Kernel.get(this, "acceleratorNames", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.AcceleratorName.class)));
            this.acceleratorTotalMemoryMax = software.amazon.jsii.Kernel.get(this, "acceleratorTotalMemoryMax", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.Size.class));
            this.acceleratorTotalMemoryMin = software.amazon.jsii.Kernel.get(this, "acceleratorTotalMemoryMin", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.Size.class));
            this.acceleratorTypes = software.amazon.jsii.Kernel.get(this, "acceleratorTypes", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.AcceleratorType.class)));
            this.allowedInstanceTypes = software.amazon.jsii.Kernel.get(this, "allowedInstanceTypes", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(java.lang.String.class)));
            this.bareMetal = software.amazon.jsii.Kernel.get(this, "bareMetal", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.BareMetal.class));
            this.baselineEbsBandwidthMbpsMax = software.amazon.jsii.Kernel.get(this, "baselineEbsBandwidthMbpsMax", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.baselineEbsBandwidthMbpsMin = software.amazon.jsii.Kernel.get(this, "baselineEbsBandwidthMbpsMin", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.burstablePerformance = software.amazon.jsii.Kernel.get(this, "burstablePerformance", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.BurstablePerformance.class));
            this.cpuManufacturers = software.amazon.jsii.Kernel.get(this, "cpuManufacturers", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.CpuManufacturer.class)));
            this.excludedInstanceTypes = software.amazon.jsii.Kernel.get(this, "excludedInstanceTypes", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(java.lang.String.class)));
            this.instanceGenerations = software.amazon.jsii.Kernel.get(this, "instanceGenerations", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.InstanceGeneration.class)));
            this.localStorage = software.amazon.jsii.Kernel.get(this, "localStorage", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.LocalStorage.class));
            this.localStorageTypes = software.amazon.jsii.Kernel.get(this, "localStorageTypes", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.services.ec2.LocalStorageType.class)));
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = software.amazon.jsii.Kernel.get(this, "maxSpotPriceAsPercentageOfOptimalOnDemandPrice", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.memoryMax = software.amazon.jsii.Kernel.get(this, "memoryMax", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.Size.class));
            this.memoryPerVCpuMax = software.amazon.jsii.Kernel.get(this, "memoryPerVCpuMax", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.Size.class));
            this.memoryPerVCpuMin = software.amazon.jsii.Kernel.get(this, "memoryPerVCpuMin", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.Size.class));
            this.networkBandwidthGbpsMax = software.amazon.jsii.Kernel.get(this, "networkBandwidthGbpsMax", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.networkBandwidthGbpsMin = software.amazon.jsii.Kernel.get(this, "networkBandwidthGbpsMin", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.networkInterfaceCountMax = software.amazon.jsii.Kernel.get(this, "networkInterfaceCountMax", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.networkInterfaceCountMin = software.amazon.jsii.Kernel.get(this, "networkInterfaceCountMin", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.onDemandMaxPricePercentageOverLowestPrice = software.amazon.jsii.Kernel.get(this, "onDemandMaxPricePercentageOverLowestPrice", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.requireHibernateSupport = software.amazon.jsii.Kernel.get(this, "requireHibernateSupport", software.amazon.jsii.NativeType.forClass(java.lang.Boolean.class));
            this.spotMaxPricePercentageOverLowestPrice = software.amazon.jsii.Kernel.get(this, "spotMaxPricePercentageOverLowestPrice", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.totalLocalStorageGbMax = software.amazon.jsii.Kernel.get(this, "totalLocalStorageGBMax", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.totalLocalStorageGbMin = software.amazon.jsii.Kernel.get(this, "totalLocalStorageGBMin", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            this.vCpuCountMax = software.amazon.jsii.Kernel.get(this, "vCpuCountMax", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
        }

        /**
         * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
         */
        @SuppressWarnings("unchecked")
        protected Jsii$Proxy(final Builder builder) {
            super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
            this.memoryMin = java.util.Objects.requireNonNull(builder.memoryMin, "memoryMin is required");
            this.vCpuCountMin = java.util.Objects.requireNonNull(builder.vCpuCountMin, "vCpuCountMin is required");
            this.acceleratorCountMax = builder.acceleratorCountMax;
            this.acceleratorCountMin = builder.acceleratorCountMin;
            this.acceleratorManufacturers = (java.util.List<software.amazon.awscdk.services.ec2.AcceleratorManufacturer>)builder.acceleratorManufacturers;
            this.acceleratorNames = (java.util.List<software.amazon.awscdk.services.ec2.AcceleratorName>)builder.acceleratorNames;
            this.acceleratorTotalMemoryMax = builder.acceleratorTotalMemoryMax;
            this.acceleratorTotalMemoryMin = builder.acceleratorTotalMemoryMin;
            this.acceleratorTypes = (java.util.List<software.amazon.awscdk.services.ec2.AcceleratorType>)builder.acceleratorTypes;
            this.allowedInstanceTypes = builder.allowedInstanceTypes;
            this.bareMetal = builder.bareMetal;
            this.baselineEbsBandwidthMbpsMax = builder.baselineEbsBandwidthMbpsMax;
            this.baselineEbsBandwidthMbpsMin = builder.baselineEbsBandwidthMbpsMin;
            this.burstablePerformance = builder.burstablePerformance;
            this.cpuManufacturers = (java.util.List<software.amazon.awscdk.services.ec2.CpuManufacturer>)builder.cpuManufacturers;
            this.excludedInstanceTypes = builder.excludedInstanceTypes;
            this.instanceGenerations = (java.util.List<software.amazon.awscdk.services.ec2.InstanceGeneration>)builder.instanceGenerations;
            this.localStorage = builder.localStorage;
            this.localStorageTypes = (java.util.List<software.amazon.awscdk.services.ec2.LocalStorageType>)builder.localStorageTypes;
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = builder.maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
            this.memoryMax = builder.memoryMax;
            this.memoryPerVCpuMax = builder.memoryPerVCpuMax;
            this.memoryPerVCpuMin = builder.memoryPerVCpuMin;
            this.networkBandwidthGbpsMax = builder.networkBandwidthGbpsMax;
            this.networkBandwidthGbpsMin = builder.networkBandwidthGbpsMin;
            this.networkInterfaceCountMax = builder.networkInterfaceCountMax;
            this.networkInterfaceCountMin = builder.networkInterfaceCountMin;
            this.onDemandMaxPricePercentageOverLowestPrice = builder.onDemandMaxPricePercentageOverLowestPrice;
            this.requireHibernateSupport = builder.requireHibernateSupport;
            this.spotMaxPricePercentageOverLowestPrice = builder.spotMaxPricePercentageOverLowestPrice;
            this.totalLocalStorageGbMax = builder.totalLocalStorageGbMax;
            this.totalLocalStorageGbMin = builder.totalLocalStorageGbMin;
            this.vCpuCountMax = builder.vCpuCountMax;
        }

        @Override
        public final software.amazon.awscdk.Size getMemoryMin() {
            return this.memoryMin;
        }

        @Override
        public final java.lang.Number getVCpuCountMin() {
            return this.vCpuCountMin;
        }

        @Override
        public final java.lang.Number getAcceleratorCountMax() {
            return this.acceleratorCountMax;
        }

        @Override
        public final java.lang.Number getAcceleratorCountMin() {
            return this.acceleratorCountMin;
        }

        @Override
        public final java.util.List<software.amazon.awscdk.services.ec2.AcceleratorManufacturer> getAcceleratorManufacturers() {
            return this.acceleratorManufacturers;
        }

        @Override
        public final java.util.List<software.amazon.awscdk.services.ec2.AcceleratorName> getAcceleratorNames() {
            return this.acceleratorNames;
        }

        @Override
        public final software.amazon.awscdk.Size getAcceleratorTotalMemoryMax() {
            return this.acceleratorTotalMemoryMax;
        }

        @Override
        public final software.amazon.awscdk.Size getAcceleratorTotalMemoryMin() {
            return this.acceleratorTotalMemoryMin;
        }

        @Override
        public final java.util.List<software.amazon.awscdk.services.ec2.AcceleratorType> getAcceleratorTypes() {
            return this.acceleratorTypes;
        }

        @Override
        public final java.util.List<java.lang.String> getAllowedInstanceTypes() {
            return this.allowedInstanceTypes;
        }

        @Override
        public final software.amazon.awscdk.services.ec2.BareMetal getBareMetal() {
            return this.bareMetal;
        }

        @Override
        public final java.lang.Number getBaselineEbsBandwidthMbpsMax() {
            return this.baselineEbsBandwidthMbpsMax;
        }

        @Override
        public final java.lang.Number getBaselineEbsBandwidthMbpsMin() {
            return this.baselineEbsBandwidthMbpsMin;
        }

        @Override
        public final software.amazon.awscdk.services.ec2.BurstablePerformance getBurstablePerformance() {
            return this.burstablePerformance;
        }

        @Override
        public final java.util.List<software.amazon.awscdk.services.ec2.CpuManufacturer> getCpuManufacturers() {
            return this.cpuManufacturers;
        }

        @Override
        public final java.util.List<java.lang.String> getExcludedInstanceTypes() {
            return this.excludedInstanceTypes;
        }

        @Override
        public final java.util.List<software.amazon.awscdk.services.ec2.InstanceGeneration> getInstanceGenerations() {
            return this.instanceGenerations;
        }

        @Override
        public final software.amazon.awscdk.services.ec2.LocalStorage getLocalStorage() {
            return this.localStorage;
        }

        @Override
        public final java.util.List<software.amazon.awscdk.services.ec2.LocalStorageType> getLocalStorageTypes() {
            return this.localStorageTypes;
        }

        @Override
        public final java.lang.Number getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice() {
            return this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        }

        @Override
        public final software.amazon.awscdk.Size getMemoryMax() {
            return this.memoryMax;
        }

        @Override
        public final software.amazon.awscdk.Size getMemoryPerVCpuMax() {
            return this.memoryPerVCpuMax;
        }

        @Override
        public final software.amazon.awscdk.Size getMemoryPerVCpuMin() {
            return this.memoryPerVCpuMin;
        }

        @Override
        public final java.lang.Number getNetworkBandwidthGbpsMax() {
            return this.networkBandwidthGbpsMax;
        }

        @Override
        public final java.lang.Number getNetworkBandwidthGbpsMin() {
            return this.networkBandwidthGbpsMin;
        }

        @Override
        public final java.lang.Number getNetworkInterfaceCountMax() {
            return this.networkInterfaceCountMax;
        }

        @Override
        public final java.lang.Number getNetworkInterfaceCountMin() {
            return this.networkInterfaceCountMin;
        }

        @Override
        public final java.lang.Number getOnDemandMaxPricePercentageOverLowestPrice() {
            return this.onDemandMaxPricePercentageOverLowestPrice;
        }

        @Override
        public final java.lang.Boolean getRequireHibernateSupport() {
            return this.requireHibernateSupport;
        }

        @Override
        public final java.lang.Number getSpotMaxPricePercentageOverLowestPrice() {
            return this.spotMaxPricePercentageOverLowestPrice;
        }

        @Override
        public final java.lang.Number getTotalLocalStorageGBMax() {
            return this.totalLocalStorageGbMax;
        }

        @Override
        public final java.lang.Number getTotalLocalStorageGBMin() {
            return this.totalLocalStorageGbMin;
        }

        @Override
        public final java.lang.Number getVCpuCountMax() {
            return this.vCpuCountMax;
        }

        @Override
        @software.amazon.jsii.Internal
        public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
            final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
            final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

            data.set("memoryMin", om.valueToTree(this.getMemoryMin()));
            data.set("vCpuCountMin", om.valueToTree(this.getVCpuCountMin()));
            if (this.getAcceleratorCountMax() != null) {
                data.set("acceleratorCountMax", om.valueToTree(this.getAcceleratorCountMax()));
            }
            if (this.getAcceleratorCountMin() != null) {
                data.set("acceleratorCountMin", om.valueToTree(this.getAcceleratorCountMin()));
            }
            if (this.getAcceleratorManufacturers() != null) {
                data.set("acceleratorManufacturers", om.valueToTree(this.getAcceleratorManufacturers()));
            }
            if (this.getAcceleratorNames() != null) {
                data.set("acceleratorNames", om.valueToTree(this.getAcceleratorNames()));
            }
            if (this.getAcceleratorTotalMemoryMax() != null) {
                data.set("acceleratorTotalMemoryMax", om.valueToTree(this.getAcceleratorTotalMemoryMax()));
            }
            if (this.getAcceleratorTotalMemoryMin() != null) {
                data.set("acceleratorTotalMemoryMin", om.valueToTree(this.getAcceleratorTotalMemoryMin()));
            }
            if (this.getAcceleratorTypes() != null) {
                data.set("acceleratorTypes", om.valueToTree(this.getAcceleratorTypes()));
            }
            if (this.getAllowedInstanceTypes() != null) {
                data.set("allowedInstanceTypes", om.valueToTree(this.getAllowedInstanceTypes()));
            }
            if (this.getBareMetal() != null) {
                data.set("bareMetal", om.valueToTree(this.getBareMetal()));
            }
            if (this.getBaselineEbsBandwidthMbpsMax() != null) {
                data.set("baselineEbsBandwidthMbpsMax", om.valueToTree(this.getBaselineEbsBandwidthMbpsMax()));
            }
            if (this.getBaselineEbsBandwidthMbpsMin() != null) {
                data.set("baselineEbsBandwidthMbpsMin", om.valueToTree(this.getBaselineEbsBandwidthMbpsMin()));
            }
            if (this.getBurstablePerformance() != null) {
                data.set("burstablePerformance", om.valueToTree(this.getBurstablePerformance()));
            }
            if (this.getCpuManufacturers() != null) {
                data.set("cpuManufacturers", om.valueToTree(this.getCpuManufacturers()));
            }
            if (this.getExcludedInstanceTypes() != null) {
                data.set("excludedInstanceTypes", om.valueToTree(this.getExcludedInstanceTypes()));
            }
            if (this.getInstanceGenerations() != null) {
                data.set("instanceGenerations", om.valueToTree(this.getInstanceGenerations()));
            }
            if (this.getLocalStorage() != null) {
                data.set("localStorage", om.valueToTree(this.getLocalStorage()));
            }
            if (this.getLocalStorageTypes() != null) {
                data.set("localStorageTypes", om.valueToTree(this.getLocalStorageTypes()));
            }
            if (this.getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice() != null) {
                data.set("maxSpotPriceAsPercentageOfOptimalOnDemandPrice", om.valueToTree(this.getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice()));
            }
            if (this.getMemoryMax() != null) {
                data.set("memoryMax", om.valueToTree(this.getMemoryMax()));
            }
            if (this.getMemoryPerVCpuMax() != null) {
                data.set("memoryPerVCpuMax", om.valueToTree(this.getMemoryPerVCpuMax()));
            }
            if (this.getMemoryPerVCpuMin() != null) {
                data.set("memoryPerVCpuMin", om.valueToTree(this.getMemoryPerVCpuMin()));
            }
            if (this.getNetworkBandwidthGbpsMax() != null) {
                data.set("networkBandwidthGbpsMax", om.valueToTree(this.getNetworkBandwidthGbpsMax()));
            }
            if (this.getNetworkBandwidthGbpsMin() != null) {
                data.set("networkBandwidthGbpsMin", om.valueToTree(this.getNetworkBandwidthGbpsMin()));
            }
            if (this.getNetworkInterfaceCountMax() != null) {
                data.set("networkInterfaceCountMax", om.valueToTree(this.getNetworkInterfaceCountMax()));
            }
            if (this.getNetworkInterfaceCountMin() != null) {
                data.set("networkInterfaceCountMin", om.valueToTree(this.getNetworkInterfaceCountMin()));
            }
            if (this.getOnDemandMaxPricePercentageOverLowestPrice() != null) {
                data.set("onDemandMaxPricePercentageOverLowestPrice", om.valueToTree(this.getOnDemandMaxPricePercentageOverLowestPrice()));
            }
            if (this.getRequireHibernateSupport() != null) {
                data.set("requireHibernateSupport", om.valueToTree(this.getRequireHibernateSupport()));
            }
            if (this.getSpotMaxPricePercentageOverLowestPrice() != null) {
                data.set("spotMaxPricePercentageOverLowestPrice", om.valueToTree(this.getSpotMaxPricePercentageOverLowestPrice()));
            }
            if (this.getTotalLocalStorageGBMax() != null) {
                data.set("totalLocalStorageGBMax", om.valueToTree(this.getTotalLocalStorageGBMax()));
            }
            if (this.getTotalLocalStorageGBMin() != null) {
                data.set("totalLocalStorageGBMin", om.valueToTree(this.getTotalLocalStorageGBMin()));
            }
            if (this.getVCpuCountMax() != null) {
                data.set("vCpuCountMax", om.valueToTree(this.getVCpuCountMax()));
            }

            final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
            struct.set("fqn", om.valueToTree("aws-cdk-lib.aws_ec2.InstanceRequirementsConfig"));
            struct.set("data", data);

            final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
            obj.set("$jsii.struct", struct);

            return obj;
        }

        @Override
        public final boolean equals(final Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            InstanceRequirementsConfig.Jsii$Proxy that = (InstanceRequirementsConfig.Jsii$Proxy) o;

            if (!memoryMin.equals(that.memoryMin)) return false;
            if (!vCpuCountMin.equals(that.vCpuCountMin)) return false;
            if (this.acceleratorCountMax != null ? !this.acceleratorCountMax.equals(that.acceleratorCountMax) : that.acceleratorCountMax != null) return false;
            if (this.acceleratorCountMin != null ? !this.acceleratorCountMin.equals(that.acceleratorCountMin) : that.acceleratorCountMin != null) return false;
            if (this.acceleratorManufacturers != null ? !this.acceleratorManufacturers.equals(that.acceleratorManufacturers) : that.acceleratorManufacturers != null) return false;
            if (this.acceleratorNames != null ? !this.acceleratorNames.equals(that.acceleratorNames) : that.acceleratorNames != null) return false;
            if (this.acceleratorTotalMemoryMax != null ? !this.acceleratorTotalMemoryMax.equals(that.acceleratorTotalMemoryMax) : that.acceleratorTotalMemoryMax != null) return false;
            if (this.acceleratorTotalMemoryMin != null ? !this.acceleratorTotalMemoryMin.equals(that.acceleratorTotalMemoryMin) : that.acceleratorTotalMemoryMin != null) return false;
            if (this.acceleratorTypes != null ? !this.acceleratorTypes.equals(that.acceleratorTypes) : that.acceleratorTypes != null) return false;
            if (this.allowedInstanceTypes != null ? !this.allowedInstanceTypes.equals(that.allowedInstanceTypes) : that.allowedInstanceTypes != null) return false;
            if (this.bareMetal != null ? !this.bareMetal.equals(that.bareMetal) : that.bareMetal != null) return false;
            if (this.baselineEbsBandwidthMbpsMax != null ? !this.baselineEbsBandwidthMbpsMax.equals(that.baselineEbsBandwidthMbpsMax) : that.baselineEbsBandwidthMbpsMax != null) return false;
            if (this.baselineEbsBandwidthMbpsMin != null ? !this.baselineEbsBandwidthMbpsMin.equals(that.baselineEbsBandwidthMbpsMin) : that.baselineEbsBandwidthMbpsMin != null) return false;
            if (this.burstablePerformance != null ? !this.burstablePerformance.equals(that.burstablePerformance) : that.burstablePerformance != null) return false;
            if (this.cpuManufacturers != null ? !this.cpuManufacturers.equals(that.cpuManufacturers) : that.cpuManufacturers != null) return false;
            if (this.excludedInstanceTypes != null ? !this.excludedInstanceTypes.equals(that.excludedInstanceTypes) : that.excludedInstanceTypes != null) return false;
            if (this.instanceGenerations != null ? !this.instanceGenerations.equals(that.instanceGenerations) : that.instanceGenerations != null) return false;
            if (this.localStorage != null ? !this.localStorage.equals(that.localStorage) : that.localStorage != null) return false;
            if (this.localStorageTypes != null ? !this.localStorageTypes.equals(that.localStorageTypes) : that.localStorageTypes != null) return false;
            if (this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice != null ? !this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice.equals(that.maxSpotPriceAsPercentageOfOptimalOnDemandPrice) : that.maxSpotPriceAsPercentageOfOptimalOnDemandPrice != null) return false;
            if (this.memoryMax != null ? !this.memoryMax.equals(that.memoryMax) : that.memoryMax != null) return false;
            if (this.memoryPerVCpuMax != null ? !this.memoryPerVCpuMax.equals(that.memoryPerVCpuMax) : that.memoryPerVCpuMax != null) return false;
            if (this.memoryPerVCpuMin != null ? !this.memoryPerVCpuMin.equals(that.memoryPerVCpuMin) : that.memoryPerVCpuMin != null) return false;
            if (this.networkBandwidthGbpsMax != null ? !this.networkBandwidthGbpsMax.equals(that.networkBandwidthGbpsMax) : that.networkBandwidthGbpsMax != null) return false;
            if (this.networkBandwidthGbpsMin != null ? !this.networkBandwidthGbpsMin.equals(that.networkBandwidthGbpsMin) : that.networkBandwidthGbpsMin != null) return false;
            if (this.networkInterfaceCountMax != null ? !this.networkInterfaceCountMax.equals(that.networkInterfaceCountMax) : that.networkInterfaceCountMax != null) return false;
            if (this.networkInterfaceCountMin != null ? !this.networkInterfaceCountMin.equals(that.networkInterfaceCountMin) : that.networkInterfaceCountMin != null) return false;
            if (this.onDemandMaxPricePercentageOverLowestPrice != null ? !this.onDemandMaxPricePercentageOverLowestPrice.equals(that.onDemandMaxPricePercentageOverLowestPrice) : that.onDemandMaxPricePercentageOverLowestPrice != null) return false;
            if (this.requireHibernateSupport != null ? !this.requireHibernateSupport.equals(that.requireHibernateSupport) : that.requireHibernateSupport != null) return false;
            if (this.spotMaxPricePercentageOverLowestPrice != null ? !this.spotMaxPricePercentageOverLowestPrice.equals(that.spotMaxPricePercentageOverLowestPrice) : that.spotMaxPricePercentageOverLowestPrice != null) return false;
            if (this.totalLocalStorageGbMax != null ? !this.totalLocalStorageGbMax.equals(that.totalLocalStorageGbMax) : that.totalLocalStorageGbMax != null) return false;
            if (this.totalLocalStorageGbMin != null ? !this.totalLocalStorageGbMin.equals(that.totalLocalStorageGbMin) : that.totalLocalStorageGbMin != null) return false;
            return this.vCpuCountMax != null ? this.vCpuCountMax.equals(that.vCpuCountMax) : that.vCpuCountMax == null;
        }

        @Override
        public final int hashCode() {
            int result = this.memoryMin.hashCode();
            result = 31 * result + (this.vCpuCountMin.hashCode());
            result = 31 * result + (this.acceleratorCountMax != null ? this.acceleratorCountMax.hashCode() : 0);
            result = 31 * result + (this.acceleratorCountMin != null ? this.acceleratorCountMin.hashCode() : 0);
            result = 31 * result + (this.acceleratorManufacturers != null ? this.acceleratorManufacturers.hashCode() : 0);
            result = 31 * result + (this.acceleratorNames != null ? this.acceleratorNames.hashCode() : 0);
            result = 31 * result + (this.acceleratorTotalMemoryMax != null ? this.acceleratorTotalMemoryMax.hashCode() : 0);
            result = 31 * result + (this.acceleratorTotalMemoryMin != null ? this.acceleratorTotalMemoryMin.hashCode() : 0);
            result = 31 * result + (this.acceleratorTypes != null ? this.acceleratorTypes.hashCode() : 0);
            result = 31 * result + (this.allowedInstanceTypes != null ? this.allowedInstanceTypes.hashCode() : 0);
            result = 31 * result + (this.bareMetal != null ? this.bareMetal.hashCode() : 0);
            result = 31 * result + (this.baselineEbsBandwidthMbpsMax != null ? this.baselineEbsBandwidthMbpsMax.hashCode() : 0);
            result = 31 * result + (this.baselineEbsBandwidthMbpsMin != null ? this.baselineEbsBandwidthMbpsMin.hashCode() : 0);
            result = 31 * result + (this.burstablePerformance != null ? this.burstablePerformance.hashCode() : 0);
            result = 31 * result + (this.cpuManufacturers != null ? this.cpuManufacturers.hashCode() : 0);
            result = 31 * result + (this.excludedInstanceTypes != null ? this.excludedInstanceTypes.hashCode() : 0);
            result = 31 * result + (this.instanceGenerations != null ? this.instanceGenerations.hashCode() : 0);
            result = 31 * result + (this.localStorage != null ? this.localStorage.hashCode() : 0);
            result = 31 * result + (this.localStorageTypes != null ? this.localStorageTypes.hashCode() : 0);
            result = 31 * result + (this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice != null ? this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice.hashCode() : 0);
            result = 31 * result + (this.memoryMax != null ? this.memoryMax.hashCode() : 0);
            result = 31 * result + (this.memoryPerVCpuMax != null ? this.memoryPerVCpuMax.hashCode() : 0);
            result = 31 * result + (this.memoryPerVCpuMin != null ? this.memoryPerVCpuMin.hashCode() : 0);
            result = 31 * result + (this.networkBandwidthGbpsMax != null ? this.networkBandwidthGbpsMax.hashCode() : 0);
            result = 31 * result + (this.networkBandwidthGbpsMin != null ? this.networkBandwidthGbpsMin.hashCode() : 0);
            result = 31 * result + (this.networkInterfaceCountMax != null ? this.networkInterfaceCountMax.hashCode() : 0);
            result = 31 * result + (this.networkInterfaceCountMin != null ? this.networkInterfaceCountMin.hashCode() : 0);
            result = 31 * result + (this.onDemandMaxPricePercentageOverLowestPrice != null ? this.onDemandMaxPricePercentageOverLowestPrice.hashCode() : 0);
            result = 31 * result + (this.requireHibernateSupport != null ? this.requireHibernateSupport.hashCode() : 0);
            result = 31 * result + (this.spotMaxPricePercentageOverLowestPrice != null ? this.spotMaxPricePercentageOverLowestPrice.hashCode() : 0);
            result = 31 * result + (this.totalLocalStorageGbMax != null ? this.totalLocalStorageGbMax.hashCode() : 0);
            result = 31 * result + (this.totalLocalStorageGbMin != null ? this.totalLocalStorageGbMin.hashCode() : 0);
            result = 31 * result + (this.vCpuCountMax != null ? this.vCpuCountMax.hashCode() : 0);
            return result;
        }
    }
}
