/**
 * <h1>AWS S3 Deployment Construct Library</h1>
 * <p>
 * This library allows populating an S3 bucket with the contents of .zip files
 * from other S3 buckets or from local disk.
 * <p>
 * The following example defines a publicly accessible S3 bucket with web hosting
 * enabled and populates it from a local directory on disk.
 * <p>
 * <blockquote><pre>
 * Bucket websiteBucket = Bucket.Builder.create(this, "WebsiteBucket")
 *         .websiteIndexDocument("index.html")
 *         .publicReadAccess(true)
 *         .build();
 * 
 * BucketDeployment.Builder.create(this, "DeployWebsite")
 *         .sources(List.of(Source.asset("./website-dist")))
 *         .destinationBucket(websiteBucket)
 *         .destinationKeyPrefix("web/static")
 *         .build();
 * </pre></blockquote>
 * <p>
 * This is what happens under the hood:
 * <p>
 * <ol>
 * <li>When this stack is deployed (either via <code>cdk deploy</code> or via CI/CD), the
 * contents of the local <code>website-dist</code> directory will be archived and uploaded
 * to an intermediary assets bucket (the <code>StagingBucket</code> of the CDK bootstrap stack).
 * If there is more than one source, they will be individually uploaded.</li>
 * <li>The <code>BucketDeployment</code> construct synthesizes a Lambda-backed custom CloudFormation resource
 * of type <code>Custom::CDKBucketDeployment</code> into the template. The source bucket/key
 * is set to point to the assets bucket.</li>
 * <li>The custom resource invokes its associated Lambda function, which downloads the .zip archive,
 * extracts it and issues <code>aws s3 sync --delete</code> against the destination bucket (in this case
 * <code>websiteBucket</code>). If there is more than one source, the sources will be
 * downloaded and merged pre-deployment at this step.</li>
 * </ol>
 * <p>
 * If you are referencing the filled bucket in another construct that depends on
 * the files already be there, be sure to use <code>deployment.deployedBucket</code>. This
 * will ensure the bucket deployment has finished before the resource that uses
 * the bucket is created:
 * <p>
 * <blockquote><pre>
 * Bucket websiteBucket;
 * 
 * 
 * BucketDeployment deployment = BucketDeployment.Builder.create(this, "DeployWebsite")
 *         .sources(List.of(Source.asset(join(__dirname, "my-website"))))
 *         .destinationBucket(websiteBucket)
 *         .build();
 * 
 * new ConstructThatReadsFromTheBucket(this, "Consumer", Map.of(
 *         // Use 'deployment.deployedBucket' instead of 'websiteBucket' here
 *         "bucket", deployment.getDeployedBucket()));
 * </pre></blockquote>
 * <p>
 * It is also possible to add additional sources using the <code>addSource</code> method.
 * <p>
 * <blockquote><pre>
 * IBucket websiteBucket;
 * 
 * 
 * BucketDeployment deployment = BucketDeployment.Builder.create(this, "DeployWebsite")
 *         .sources(List.of(Source.asset("./website-dist")))
 *         .destinationBucket(websiteBucket)
 *         .destinationKeyPrefix("web/static")
 *         .build();
 * 
 * deployment.addSource(Source.asset("./another-asset"));
 * </pre></blockquote>
 * <p>
 * For the Lambda function to download object(s) from the source bucket, besides the obvious
 * <code>s3:GetObject*</code> permissions, the Lambda's execution role needs the <code>kms:Decrypt</code> and <code>kms:DescribeKey</code>
 * permissions on the KMS key that is used to encrypt the bucket. By default, when the source bucket is
 * encrypted with the S3 managed key of the account, these permissions are granted by the key's
 * resource-based policy, so they do not need to be on the Lambda's execution role policy explicitly.
 * However, if the encryption key is not the s3 managed one, its resource-based policy is quite likely
 * to NOT grant such KMS permissions. In this situation, the Lambda execution will fail with an error
 * message like below:
 * <p>
 * <blockquote><pre>
 * download failed: ...
 * An error occurred (AccessDenied) when calling the GetObject operation:
 * User: *** is not authorized to perform: kms:Decrypt on the resource associated with this ciphertext
 * because no identity-based policy allows the kms:Decrypt action
 * </pre></blockquote>
 * <p>
 * When this happens, users can use the public <code>handlerRole</code> property of <code>BucketDeployment</code> to manually
 * add the KMS permissions:
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket;
 * 
 * 
 * BucketDeployment deployment = BucketDeployment.Builder.create(this, "DeployFiles")
 *         .sources(List.of(Source.asset(join(__dirname, "source-files"))))
 *         .destinationBucket(destinationBucket)
 *         .build();
 * 
 * deployment.handlerRole.addToPolicy(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("kms:Decrypt", "kms:DescribeKey"))
 *         .effect(Effect.ALLOW)
 *         .resources(List.of("&lt;encryption key ARN&gt;"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The situation above could arise from the following scenarios:
 * <p>
 * <ul>
 * <li>User created a customer managed KMS key and passed its ID to the <code>cdk bootstrap</code> command via
 * the <code>--bootstrap-kms-key-id</code> CLI option.
 * The <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policy-default.html#key-policy-default-allow-root-enable-iam">default key policy</a>
 * alone is not sufficient to grant the Lambda KMS permissions.</li>
 * <li>A corporation uses its own custom CDK bootstrap process, which encrypts the CDK <code>StagingBucket</code>
 * by a KMS key from a management account of the corporation's AWS Organization. In this cross-account
 * access scenario, the KMS permissions must be explicitly present in the Lambda's execution role policy.</li>
 * <li>One of the sources for the <code>BucketDeployment</code> comes from the <code>Source.bucket</code> static method, which
 * points to a bucket whose encryption key is not the S3 managed one, and the resource-based policy
 * of the encryption key is not sufficient to grant the Lambda <code>kms:Decrypt</code> and <code>kms:DescribeKey</code>
 * permissions.</li>
 * </ul>
 * <p>
 * <h2>Supported sources</h2>
 * <p>
 * The following source types are supported for bucket deployments:
 * <p>
 * <ul>
 * <li>Local .zip file: <code>s3deploy.Source.asset('/path/to/local/file.zip')</code></li>
 * <li>Local directory: <code>s3deploy.Source.asset('/path/to/local/directory')</code></li>
 * <li>Another bucket: <code>s3deploy.Source.bucket(bucket, zipObjectKey)</code></li>
 * <li>String data: <code>s3deploy.Source.data('object-key.txt', 'hello, world!')</code>
 * (supports <a href="#data-with-deploy-time-values">deploy-time values</a>)</li>
 * <li>JSON data: <code>s3deploy.Source.jsonData('object-key.json', { json: 'object' })</code>
 * (supports <a href="#data-with-deploy-time-values">deploy-time values</a>)</li>
 * <li>YAML data: <code>s3deploy.Source.yamlData('object-key.yaml', { yaml: 'object' })</code>
 * (supports <a href="#data-with-deploy-time-values">deploy-time values</a>)</li>
 * </ul>
 * <p>
 * To create a source from a single file, you can pass <code>AssetOptions</code> to exclude
 * all but a single file:
 * <p>
 * <ul>
 * <li>Single file: <code>s3deploy.Source.asset('/path/to/local/directory', { exclude: ['**', '!onlyThisFile.txt'] })</code></li>
 * </ul>
 * <p>
 * <strong>IMPORTANT</strong> The <code>aws-s3-deployment</code> module is only intended to be used with
 * zip files from trusted sources. Directories bundled by the CDK CLI (by using
 * <code>Source.asset()</code> on a directory) are safe. If you are using <code>Source.asset()</code> or
 * <code>Source.bucket()</code> to reference an existing zip file, make sure you trust the
 * file you are referencing. Zips from untrusted sources might be able to execute
 * arbitrary code in the Lambda Function used by this module, and use its permissions
 * to read or write unexpected files in the S3 bucket.
 * <p>
 * <h2>Retain on Delete</h2>
 * <p>
 * By default, the contents of the destination bucket will <strong>not</strong> be deleted when the
 * <code>BucketDeployment</code> resource is removed from the stack or when the destination is
 * changed. You can use the option <code>retainOnDelete: false</code> to disable this behavior,
 * in which case the contents will be deleted.
 * <p>
 * Configuring this has a few implications you should be aware of:
 * <p>
 * <ul>
 * <li><strong>Logical ID Changes</strong>
 * <p>
 * Changing the logical ID of the <code>BucketDeployment</code> construct, without changing the destination
 * (for example due to refactoring, or intentional ID change) <strong>will result in the deletion of the objects</strong>.
 * This is because CloudFormation will first create the new resource, which will have no affect,
 * followed by a deletion of the old resource, which will cause a deletion of the objects,
 * since the destination hasn't changed, and <code>retainOnDelete</code> is <code>false</code>.</li>
 * <li><strong>Destination Changes</strong>
 * <p>
 * When the destination bucket or prefix is changed, all files in the previous destination will <strong>first</strong> be
 * deleted and then uploaded to the new destination location. This could have availability implications
 * on your users.</li>
 * </ul>
 * <p>
 * <h3>General Recommendations</h3>
 * <p>
 * <h4>Shared Bucket</h4>
 * <p>
 * If the destination bucket <strong>is not</strong> dedicated to the specific <code>BucketDeployment</code> construct (i.e shared by other entities),
 * we recommend to always configure the <code>destinationKeyPrefix</code> property. This will prevent the deployment from
 * accidentally deleting data that wasn't uploaded by it.
 * <p>
 * <h4>Dedicated Bucket</h4>
 * <p>
 * If the destination bucket <strong>is</strong> dedicated, it might be reasonable to skip the prefix configuration,
 * in which case, we recommend to remove <code>retainOnDelete: false</code>, and instead, configure the
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-s3-readme.html#bucket-deletion"><code>autoDeleteObjects</code></a>
 * property on the destination bucket. This will avoid the logical ID problem mentioned above.
 * <p>
 * <h2>Prune</h2>
 * <p>
 * By default, files in the destination bucket that don't exist in the source will be deleted
 * when the <code>BucketDeployment</code> resource is created or updated. You can use the option <code>prune: false</code> to disable
 * this behavior, in which case the files will not be deleted.
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket;
 * 
 * BucketDeployment.Builder.create(this, "DeployMeWithoutDeletingFilesOnDestination")
 *         .sources(List.of(Source.asset(join(__dirname, "my-website"))))
 *         .destinationBucket(destinationBucket)
 *         .prune(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * This option also enables you to
 * multiple bucket deployments for the same destination bucket &amp; prefix,
 * each with its own characteristics. For example, you can set different cache-control headers
 * based on file extensions:
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket;
 * 
 * BucketDeployment.Builder.create(this, "BucketDeployment")
 *         .sources(List.of(Source.asset("./website", AssetOptions.builder().exclude(List.of("index.html")).build())))
 *         .destinationBucket(destinationBucket)
 *         .cacheControl(List.of(CacheControl.maxAge(Duration.days(365)), CacheControl.immutable()))
 *         .prune(false)
 *         .build();
 * 
 * BucketDeployment.Builder.create(this, "HTMLBucketDeployment")
 *         .sources(List.of(Source.asset("./website", AssetOptions.builder().exclude(List.of("*", "!index.html")).build())))
 *         .destinationBucket(destinationBucket)
 *         .cacheControl(List.of(CacheControl.maxAge(Duration.seconds(0))))
 *         .prune(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Exclude and Include Filters</h2>
 * <p>
 * There are two points at which filters are evaluated in a deployment: asset bundling and the actual deployment. If you simply want to exclude files in the asset bundling process, you should leverage the <code>exclude</code> property of <code>AssetOptions</code> when defining your source:
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket;
 * 
 * BucketDeployment.Builder.create(this, "HTMLBucketDeployment")
 *         .sources(List.of(Source.asset("./website", AssetOptions.builder().exclude(List.of("*", "!index.html")).build())))
 *         .destinationBucket(destinationBucket)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to specify filters to be used in the deployment process, you can use the <code>exclude</code> and <code>include</code> filters on <code>BucketDeployment</code>.  If excluded, these files will not be deployed to the destination bucket. In addition, if the file already exists in the destination bucket, it will not be deleted if you are using the <code>prune</code> option:
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket;
 * 
 * BucketDeployment.Builder.create(this, "DeployButExcludeSpecificFiles")
 *         .sources(List.of(Source.asset(join(__dirname, "my-website"))))
 *         .destinationBucket(destinationBucket)
 *         .exclude(List.of("*.txt"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * These filters follow the same format that is used for the AWS CLI.  See the CLI documentation for information on <a href="https://docs.aws.amazon.com/cli/latest/reference/s3/index.html#use-of-exclude-and-include-filters">Using Include and Exclude Filters</a>.
 * <p>
 * <h2>Objects metadata</h2>
 * <p>
 * You can specify metadata to be set on all the objects in your deployment.
 * There are 2 types of metadata in S3: system-defined metadata and user-defined metadata.
 * System-defined metadata have a special purpose, for example cache-control defines how long to keep an object cached.
 * User-defined metadata are not used by S3 and keys always begin with <code>x-amz-meta-</code> (this prefix is added automatically).
 * <p>
 * System defined metadata keys include the following:
 * <p>
 * <ul>
 * <li>cache-control (<code>--cache-control</code> in <code>aws s3 sync</code>)</li>
 * <li>content-disposition (<code>--content-disposition</code> in <code>aws s3 sync</code>)</li>
 * <li>content-encoding (<code>--content-encoding</code> in <code>aws s3 sync</code>)</li>
 * <li>content-language (<code>--content-language</code> in <code>aws s3 sync</code>)</li>
 * <li>content-type (<code>--content-type</code> in <code>aws s3 sync</code>)</li>
 * <li>expires (<code>--expires</code> in <code>aws s3 sync</code>)</li>
 * <li>x-amz-storage-class (<code>--storage-class</code> in <code>aws s3 sync</code>)</li>
 * <li>x-amz-website-redirect-location (<code>--website-redirect</code> in <code>aws s3 sync</code>)</li>
 * <li>x-amz-server-side-encryption (<code>--sse</code> in <code>aws s3 sync</code>)</li>
 * <li>x-amz-server-side-encryption-aws-kms-key-id (<code>--sse-kms-key-id</code> in <code>aws s3 sync</code>)</li>
 * <li>x-amz-server-side-encryption-customer-algorithm (<code>--sse-c-copy-source</code> in <code>aws s3 sync</code>)</li>
 * <li>x-amz-acl (<code>--acl</code> in <code>aws s3 sync</code>)</li>
 * </ul>
 * <p>
 * You can find more information about system defined metadata keys in
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html">S3 PutObject documentation</a>
 * and <a href="https://docs.aws.amazon.com/cli/latest/reference/s3/sync.html"><code>aws s3 sync</code> documentation</a>.
 * <p>
 * <blockquote><pre>
 * Bucket websiteBucket = Bucket.Builder.create(this, "WebsiteBucket")
 *         .websiteIndexDocument("index.html")
 *         .publicReadAccess(true)
 *         .build();
 * 
 * BucketDeployment.Builder.create(this, "DeployWebsite")
 *         .sources(List.of(Source.asset("./website-dist")))
 *         .destinationBucket(websiteBucket)
 *         .destinationKeyPrefix("web/static") // optional prefix in destination bucket
 *         .metadata(Map.of("A", "1", "b", "2")) // user-defined metadata
 * 
 *         // system-defined metadata
 *         .contentType("text/html")
 *         .contentLanguage("en")
 *         .storageClass(StorageClass.INTELLIGENT_TIERING)
 *         .serverSideEncryption(ServerSideEncryption.AES_256)
 *         .cacheControl(List.of(CacheControl.setPublic(), CacheControl.maxAge(Duration.hours(1))))
 *         .accessControl(BucketAccessControl.BUCKET_OWNER_FULL_CONTROL)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>CloudFront Invalidation</h2>
 * <p>
 * You can provide a CloudFront distribution and optional paths to invalidate after the bucket deployment finishes.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudfront.*;
 * import software.amazon.awscdk.services.cloudfront.origins.*;
 * 
 * 
 * Bucket bucket = new Bucket(this, "Destination");
 * 
 * // Handles buckets whether or not they are configured for website hosting.
 * Distribution distribution = Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new S3Origin(bucket)).build())
 *         .build();
 * 
 * BucketDeployment.Builder.create(this, "DeployWithInvalidation")
 *         .sources(List.of(Source.asset("./website-dist")))
 *         .destinationBucket(bucket)
 *         .distribution(distribution)
 *         .distributionPaths(List.of("/images/*.png"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the deployment will wait for invalidation to succeed to complete. This will poll Cloudfront for a maximum of 13 minutes to check for a successful invalidation. The drawback to this is that the deployment will fail if invalidation fails or if it takes longer than 13 minutes. As a workaround, there is the option <code>waitForDistributionInvalidation</code>, which can be set to false to skip waiting for the invalidation, but this can be risky as invalidation errors will not be reported.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudfront.*;
 * 
 * IBucket bucket;
 * IDistribution distribution;
 * 
 * 
 * BucketDeployment.Builder.create(this, "DeployWithInvalidation")
 *         .sources(List.of(Source.asset("./website-dist")))
 *         .destinationBucket(bucket)
 *         .distribution(distribution)
 *         .distributionPaths(List.of("/images/*.png"))
 *         // Invalidate cache but don't wait or verify that invalidation has completed successfully.
 *         .waitForDistributionInvalidation(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Signed Content Payloads</h2>
 * <p>
 * By default, deployment uses streaming uploads which set the <code>x-amz-content-sha256</code>
 * request header to <code>UNSIGNED-PAYLOAD</code> (matching default behavior of the AWS CLI tool).
 * In cases where bucket policy restrictions require signed content payloads, you can enable
 * generation of a signed <code>x-amz-content-sha256</code> request header with <code>signContent: true</code>.
 * <p>
 * <blockquote><pre>
 * IBucket bucket;
 * 
 * 
 * BucketDeployment.Builder.create(this, "DeployWithSignedPayloads")
 *         .sources(List.of(Source.asset("./website-dist")))
 *         .destinationBucket(bucket)
 *         .signContent(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Size Limits</h2>
 * <p>
 * The default memory limit for the deployment resource is 128MiB. If you need to
 * copy larger files, you can use the <code>memoryLimit</code> configuration to increase the
 * size of the AWS Lambda resource handler.
 * <p>
 * The default ephemeral storage size for the deployment resource is 512MiB. If you
 * need to upload larger files, you may hit this limit. You can use the
 * <code>ephemeralStorageSize</code> configuration to increase the storage size of the AWS Lambda
 * resource handler.
 * <p>
 * <blockquote>
 * <p>
 * NOTE: a new AWS Lambda handler will be created in your stack for each combination
 * of memory and storage size.
 * <p>
 * </blockquote>
 * <p>
 * <h2>JSON-Aware Source Processing</h2>
 * <p>
 * When using <code>Source.jsonData</code> with CDK Tokens (references to construct properties), you may need to enable the escaping option. This is particularly important when the referenced properties might contain special characters that require proper JSON escaping (like double quotes, line breaks, etc.).
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * StringParameter param;
 * 
 * 
 * // Example with a secret value that contains double quotes
 * BucketDeployment deployment = BucketDeployment.Builder.create(this, "JsonDeployment")
 *         .sources(List.of(Source.jsonData("config.json", Map.of(
 *                 "api_endpoint", "https://api.example.com",
 *                 "secretValue", param.getStringValue(),  // value with double quotes
 *                 "config", Map.of(
 *                         "enabled", true,
 *                         "features", List.of("feature1", "feature2"))), JsonProcessingOptions.builder().escape(true).build())))
 *         .destinationBucket(bucket)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EFS Support</h2>
 * <p>
 * If your workflow needs more disk space than default (512 MB) disk space, you may attach an EFS storage to underlying
 * lambda function. To Enable EFS support set <code>efs</code> and <code>vpc</code> props for BucketDeployment.
 * <p>
 * Check sample usage below.
 * Please note that creating VPC inline may cause stack deletion failures. It is shown as below for simplicity.
 * To avoid such condition, keep your network infra (VPC) in a separate stack and pass as props.
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket;
 * Vpc vpc;
 * 
 * 
 * BucketDeployment.Builder.create(this, "DeployMeWithEfsStorage")
 *         .sources(List.of(Source.asset(join(__dirname, "my-website"))))
 *         .destinationBucket(destinationBucket)
 *         .destinationKeyPrefix("efs/")
 *         .useEfs(true)
 *         .vpc(vpc)
 *         .retainOnDelete(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Data with deploy-time values</h2>
 * <p>
 * The content passed to <code>Source.data()</code>, <code>Source.jsonData()</code>, or <code>Source.yamlData()</code> can include
 * references that will get resolved only during deployment. Only a subset of CloudFormation functions
 * are supported however, namely: Ref, Fn::GetAtt, Fn::Join, and Fn::Select (Fn::Split may be nested under Fn::Select).
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * Bucket destinationBucket;
 * Topic topic;
 * ApplicationTargetGroup tg;
 * 
 * 
 * Map&lt;String, Object&gt; appConfig = Map.of(
 *         "topic_arn", topic.getTopicArn(),
 *         "base_url", "https://my-endpoint",
 *         "lb_name", tg.getFirstLoadBalancerFullName());
 * 
 * BucketDeployment.Builder.create(this, "BucketDeployment")
 *         .sources(List.of(Source.jsonData("config.json", appConfig)))
 *         .destinationBucket(destinationBucket)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The value in <code>topic.topicArn</code> is a deploy-time value. It only gets resolved
 * during deployment by placing a marker in the generated source file and
 * substituting it when its deployed to the destination with the actual value.
 * <p>
 * <h3>Substitutions from Templated Files</h3>
 * <p>
 * The <code>DeployTimeSubstitutedFile</code> construct allows you to specify substitutions
 * to make from placeholders in a local file which will be resolved during deployment. This
 * is especially useful in situations like creating an API from a spec file, where users might
 * want to reference other CDK resources they have created.
 * <p>
 * The syntax for template variables is <code>{{ variableName }}</code> in your local file. Then, you would
 * specify the substitutions in CDK like this:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function myLambdaFunction;
 * Bucket destinationBucket;
 * //(Optional) if provided, the resulting processed file would be uploaded to the destinationBucket under the destinationKey name.
 * String destinationKey;
 * Role role;
 * 
 * 
 * DeployTimeSubstitutedFile.Builder.create(this, "MyFile")
 *         .source("my-file.yaml")
 *         .destinationKey(destinationKey)
 *         .destinationBucket(destinationBucket)
 *         .substitutions(Map.of(
 *                 "variableName", myLambdaFunction.getFunctionName()))
 *         .role(role)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Nested variables, like <code>{{ {{ foo }} }}</code> or <code>{{ foo {{ bar }} }}</code>, are not supported by this
 * construct. In the first case of a single variable being is double nested <code>{{ {{ foo }} }}</code>, only
 * the <code>{{ foo }}</code> would be replaced by the substitution, and the extra brackets would remain in the file.
 * In the second case of two nexted variables <code>{{ foo {{ bar }} }}</code>, only the <code>{{ bar }}</code> would be replaced
 * in the file.
 * <p>
 * <h2>Keep Files Zipped</h2>
 * <p>
 * By default, files are zipped, then extracted into the destination bucket.
 * <p>
 * You can use the option <code>extract: false</code> to disable this behavior, in which case, files will remain in a zip file when deployed to S3. To reference the object keys, or filenames, which will be deployed to the bucket, you can use the <code>objectKeys</code> getter on the bucket deployment.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * Bucket destinationBucket;
 * 
 * 
 * BucketDeployment myBucketDeployment = BucketDeployment.Builder.create(this, "DeployMeWithoutExtractingFilesOnDestination")
 *         .sources(List.of(Source.asset(join(__dirname, "my-website"))))
 *         .destinationBucket(destinationBucket)
 *         .extract(false)
 *         .build();
 * 
 * CfnOutput.Builder.create(this, "ObjectKey")
 *         .value(Fn.select(0, myBucketDeployment.getObjectKeys()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Controlling the Output of Source Object Keys</h2>
 * <p>
 * By default, the keys of the source objects copied to the destination bucket are returned in the Data property of the custom resource. However, you can disable this behavior by setting the outputObjectKeys property to false. This is particularly useful when the number of objects is too large and might exceed the size limit of the responseData property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * Bucket destinationBucket;
 * 
 * 
 * BucketDeployment myBucketDeployment = BucketDeployment.Builder.create(this, "DeployMeWithoutExtractingFilesOnDestination")
 *         .sources(List.of(Source.asset(join(__dirname, "my-website"))))
 *         .destinationBucket(destinationBucket)
 *         .outputObjectKeys(false)
 *         .build();
 * 
 * CfnOutput.Builder.create(this, "ObjectKey")
 *         .value(Fn.select(0, myBucketDeployment.getObjectKeys()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Specifying a Custom VPC, Subnets, and Security Groups in BucketDeployment</h2>
 * <p>
 * By default, the AWS CDK BucketDeployment construct runs in a publicly accessible environment. However, for enhanced security and compliance, you may need to deploy your assets from within a VPC while restricting network access through custom subnets and security groups.
 * <p>
 * <h3>Using a Custom VPC</h3>
 * <p>
 * To deploy assets within a private network, specify the vpc property in BucketDeploymentProps. This ensures that the deployment Lambda function executes within your specified VPC.
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "ExistingVPC", VpcLookupOptions.builder().vpcId("vpc-12345678").build());
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * BucketDeployment.Builder.create(this, "DeployToS3")
 *         .destinationBucket(bucket)
 *         .vpc(vpc)
 *         .sources(List.of(Source.asset("./website")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Specifying Subnets for Deployment</h3>
 * <p>
 * By default, when you specify a VPC, the BucketDeployment function is deployed in the private subnets of that VPC.
 * However, you can customize the subnet selection using the vpcSubnets property.
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "ExistingVPC", VpcLookupOptions.builder().vpcId("vpc-12345678").build());
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * BucketDeployment.Builder.create(this, "DeployToS3")
 *         .destinationBucket(bucket)
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .sources(List.of(Source.asset("./website")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Defining Custom Security Groups</h3>
 * <p>
 * For enhanced network security, you can now specify custom security groups in BucketDeploymentProps.
 * This allows fine-grained control over ingress and egress rules for the deployment Lambda function.
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "ExistingVPC", VpcLookupOptions.builder().vpcId("vpc-12345678").build());
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * SecurityGroup securityGroup = SecurityGroup.Builder.create(this, "CustomSG")
 *         .vpc(vpc)
 *         .description("Allow HTTPS outbound access")
 *         .allowAllOutbound(false)
 *         .build();
 * 
 * securityGroup.addEgressRule(Peer.anyIpv4(), Port.tcp(443), "Allow HTTPS traffic");
 * 
 * BucketDeployment.Builder.create(this, "DeployWithSecurityGroup")
 *         .destinationBucket(bucket)
 *         .vpc(vpc)
 *         .securityGroups(List.of(securityGroup))
 *         .sources(List.of(Source.asset("./website")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Notes</h2>
 * <p>
 * <ul>
 * <li>This library uses an AWS CloudFormation custom resource which is about 10MiB in
 * size. The code of this resource is bundled with this library.</li>
 * <li>AWS Lambda execution time is limited to 15min. This limits the amount of data
 * which can be deployed into the bucket by this timeout.</li>
 * <li>When the <code>BucketDeployment</code> is removed from the stack, the contents are retained
 * in the destination bucket (<a href="https://github.com/aws/aws-cdk/issues/952">#952</a>).</li>
 * <li>If you are using <code>s3deploy.Source.bucket()</code> to take the file source from
 * another bucket: the deployed files will only be updated if the key (file name)
 * of the file in the source  bucket changes. Mutating the file in place will not
 * be good enough: the custom resource will simply not run if the properties don't
 * change.
 * <p>
 * <ul>
 * <li>If you use assets (<code>s3deploy.Source.asset()</code>) you don't need to worry
 * about this: the asset system will make sure that if the files have changed,
 * the file name is unique and the deployment will run.</li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Development</h2>
 * <p>
 * The custom resource is implemented in Python 3.9 in order to be able to leverage
 * the AWS CLI for "aws s3 sync".
 * The code is now in the <code>&#64;aws-cdk/custom-resource-handlers</code> package under <a href="https://github.com/aws/aws-cdk/tree/main/packages/&#64;aws-cdk/custom-resource-handlers/lib/aws-s3-deployment/bucket-deployment-handler/"><code>lib/aws-s3-deployment/bucket-deployment-handler</code></a> and
 * unit tests are under <a href="https://github.com/aws/aws-cdk/tree/main/packages/&#64;aws-cdk/custom-resource-handlers/test/aws-s3-deployment/bucket-deployment-handler/"><code>test/aws-s3-deployment/bucket-deployment-handler</code></a>.
 * <p>
 * This package requires Python 3.9 during build time in order to create the custom
 * resource Lambda bundle and test it. It also relies on a few bash scripts, so
 * might be tricky to build on Windows.
 * <p>
 * <h2>Roadmap</h2>
 * <p>
 * <ul>
 * <li>[ ] Support "blue/green" deployments (<a href="https://github.com/aws/aws-cdk/issues/954">#954</a>)</li>
 * </ul>
 */
package software.amazon.awscdk.services.s3.deployment;
