/**
 * <h1>AWS CodeCommit Construct Library</h1>
 * <p>
 * AWS CodeCommit is a version control service that enables you to privately store and manage Git repositories in the AWS cloud.
 * <p>
 * For further information on CodeCommit,
 * see the <a href="https://docs.aws.amazon.com/codecommit">AWS CodeCommit documentation</a>.
 * <p>
 * To add a CodeCommit Repository to your stack:
 * <p>
 * <blockquote><pre>
 * Repository repo = Repository.Builder.create(this, "Repository")
 *         .repositoryName("MyRepositoryName")
 *         .description("Some description.")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use the <code>repositoryCloneUrlHttp</code>, <code>repositoryCloneUrlSsh</code> or <code>repositoryCloneUrlGrc</code>
 * property to clone your repository.
 * <p>
 * To add an Amazon SNS trigger to your repository:
 * <p>
 * <blockquote><pre>
 * Repository repo;
 * 
 * 
 * // trigger is established for all repository actions on all branches by default.
 * repo.notify("arn:aws:sns:*:123456789012:my_topic");
 * </pre></blockquote>
 * <p>
 * <h2>Add initial commit</h2>
 * <p>
 * It is possible to initialize the Repository via the <code>Code</code> class.
 * It provides methods for loading code from a directory, <code>.zip</code> file and from a pre-created CDK Asset.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Repository repo = Repository.Builder.create(this, "Repository")
 *         .repositoryName("MyRepositoryName")
 *         .code(Code.fromDirectory(join(__dirname, "directory/"), "develop"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Events</h2>
 * <p>
 * CodeCommit repositories emit Amazon CloudWatch events for certain activities.
 * Use the <code>repo.onXxx</code> methods to define rules that trigger on these events
 * and invoke targets as a result:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * Repository repo;
 * PipelineProject project;
 * Topic myTopic;
 * 
 * 
 * // starts a CodeBuild project when a commit is pushed to the "main" branch of the repo
 * repo.onCommit("CommitToMain", OnCommitOptions.builder()
 *         .target(new CodeBuildProject(project))
 *         .branches(List.of("main"))
 *         .build());
 * 
 * // publishes a message to an Amazon SNS topic when a comment is made on a pull request
 * Rule rule = repo.onCommentOnPullRequest("CommentOnPullRequest", OnEventOptions.builder()
 *         .target(new SnsTopic(myTopic))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>CodeStar Notifications</h2>
 * <p>
 * To define CodeStar Notification rules for Repositories, use one of the <code>notifyOnXxx()</code> methods.
 * They are very similar to <code>onXxx()</code> methods for CloudWatch events:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.chatbot.*;
 * 
 * Repository repository;
 * 
 * SlackChannelConfiguration target = SlackChannelConfiguration.Builder.create(this, "MySlackChannel")
 *         .slackChannelConfigurationName("YOUR_CHANNEL_NAME")
 *         .slackWorkspaceId("YOUR_SLACK_WORKSPACE_ID")
 *         .slackChannelId("YOUR_SLACK_CHANNEL_ID")
 *         .build();
 * INotificationRule rule = repository.notifyOnPullRequestCreated("NotifyOnPullRequestCreated", target);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.codecommit;
