/**
 * <h1>AWS CodeDeploy Construct Library</h1>
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#introduction">Introduction</a></li>
 * <li>Deploying to Amazon EC2 and on-premise instances
 * <p>
 * <ul>
 * <li><a href="#ec2on-premise-applications">EC2/on-premise Applications</a></li>
 * <li><a href="#ec2on-premise-deployment-groups">EC2/on-premise Deployment Groups</a></li>
 * <li><a href="#ec2on-premise-deployment-configurations">EC2/on-premise Deployment Configurations</a></li>
 * </ul></li>
 * <li>Deploying to AWS Lambda functions
 * <p>
 * <ul>
 * <li><a href="#lambda-applications">Lambda Applications</a></li>
 * <li><a href="#lambda-deployment-groups">Lambda Deployment Groups</a></li>
 * <li><a href="#lambda-deployment-configurations">Lambda Deployment Configurations</a></li>
 * </ul></li>
 * <li>Deploying to Amazon ECS services
 * <p>
 * <ul>
 * <li><a href="#ecs-applications">ECS Applications</a></li>
 * <li><a href="#ecs-deployment-groups">ECS Deployment Groups</a></li>
 * <li><a href="#ecs-deployment-configurations">ECS Deployment Configurations</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Introduction</h2>
 * <p>
 * AWS CodeDeploy is a deployment service that automates application deployments to
 * Amazon EC2 instances, on-premises instances, serverless Lambda functions, or
 * Amazon ECS services.
 * <p>
 * The CDK currently supports Amazon EC2, on-premise, AWS Lambda, and Amazon ECS applications.
 * <p>
 * <h2>EC2/on-premise Applications</h2>
 * <p>
 * To create a new CodeDeploy Application that deploys to EC2/on-premise instances:
 * <p>
 * <blockquote><pre>
 * ServerApplication application = ServerApplication.Builder.create(this, "CodeDeployApplication")
 *         .applicationName("MyApplication")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To import an already existing Application:
 * <p>
 * <blockquote><pre>
 * IServerApplication application = ServerApplication.fromServerApplicationName(this, "ExistingCodeDeployApplication", "MyExistingApplication");
 * </pre></blockquote>
 * <p>
 * <h2>EC2/on-premise Deployment Groups</h2>
 * <p>
 * To create a new CodeDeploy Deployment Group that deploys to EC2/on-premise instances:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.autoscaling.*;
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * ServerApplication application;
 * AutoScalingGroup asg;
 * Alarm alarm;
 * 
 * ServerDeploymentGroup deploymentGroup = ServerDeploymentGroup.Builder.create(this, "CodeDeployDeploymentGroup")
 *         .application(application)
 *         .deploymentGroupName("MyDeploymentGroup")
 *         .autoScalingGroups(List.of(asg))
 *         // adds User Data that installs the CodeDeploy agent on your auto-scaling groups hosts
 *         // default: true
 *         .installAgent(true)
 *         // adds EC2 instances matching tags
 *         .ec2InstanceTags(new InstanceTagSet(Map.of(
 *                 // any instance with tags satisfying
 *                 // key1=v1 or key1=v2 or key2 (any value) or value v3 (any key)
 *                 // will match this group
 *                 "key1", List.of("v1", "v2"),
 *                 "key2", List.of(),
 *                 "", List.of("v3"))))
 *         // adds on-premise instances matching tags
 *         .onPremiseInstanceTags(new InstanceTagSet(Map.of(
 *                 "key1", List.of("v1", "v2")), Map.of(
 *                 "key2", List.of("v3"))))
 *         // CloudWatch alarms
 *         .alarms(List.of(alarm))
 *         // whether to ignore failure to fetch the status of alarms from CloudWatch
 *         // default: false
 *         .ignorePollAlarmsFailure(false)
 *         // auto-rollback configuration
 *         .autoRollback(AutoRollbackConfig.builder()
 *                 .failedDeployment(true) // default: true
 *                 .stoppedDeployment(true) // default: false
 *                 .deploymentInAlarm(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * All properties are optional - if you don't provide an Application,
 * one will be automatically created.
 * <p>
 * To import an already existing Deployment Group:
 * <p>
 * <blockquote><pre>
 * ServerApplication application;
 * 
 * IServerDeploymentGroup deploymentGroup = ServerDeploymentGroup.fromServerDeploymentGroupAttributes(this, "ExistingCodeDeployDeploymentGroup", ServerDeploymentGroupAttributes.builder()
 *         .application(application)
 *         .deploymentGroupName("MyExistingDeploymentGroup")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Load balancers</h3>
 * <p>
 * You can <a href="https://docs.aws.amazon.com/codedeploy/latest/userguide/integrations-aws-elastic-load-balancing.html">specify a load balancer</a>
 * with the <code>loadBalancer</code> property when creating a Deployment Group.
 * <p>
 * <code>LoadBalancer</code> is an abstract class with static factory methods that allow you to create instances of it from various sources.
 * <p>
 * With Classic Elastic Load Balancer, you provide it directly:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancing.*;
 * 
 * LoadBalancer lb;
 * 
 * lb.addListener(LoadBalancerListener.builder()
 *         .externalPort(80)
 *         .build());
 * 
 * ServerDeploymentGroup deploymentGroup = ServerDeploymentGroup.Builder.create(this, "DeploymentGroup")
 *         .loadBalancer(LoadBalancer.classic(lb))
 *         .build();
 * </pre></blockquote>
 * <p>
 * With Application Load Balancer or Network Load Balancer,
 * you provide a Target Group as the load balancer:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * ApplicationLoadBalancer alb;
 * 
 * ApplicationListener listener = alb.addListener("Listener", BaseApplicationListenerProps.builder().port(80).build());
 * ApplicationTargetGroup targetGroup = listener.addTargets("Fleet", AddApplicationTargetsProps.builder().port(80).build());
 * 
 * ServerDeploymentGroup deploymentGroup = ServerDeploymentGroup.Builder.create(this, "DeploymentGroup")
 *         .loadBalancer(LoadBalancer.application(targetGroup))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EC2/on-premise Deployment Configurations</h2>
 * <p>
 * You can also pass a Deployment Configuration when creating the Deployment Group:
 * <p>
 * <blockquote><pre>
 * ServerDeploymentGroup deploymentGroup = ServerDeploymentGroup.Builder.create(this, "CodeDeployDeploymentGroup")
 *         .deploymentConfig(ServerDeploymentConfig.ALL_AT_ONCE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The default Deployment Configuration is <code>ServerDeploymentConfig.ONE_AT_A_TIME</code>.
 * <p>
 * You can also create a custom Deployment Configuration:
 * <p>
 * <blockquote><pre>
 * ServerDeploymentConfig deploymentConfig = ServerDeploymentConfig.Builder.create(this, "DeploymentConfiguration")
 *         .deploymentConfigName("MyDeploymentConfiguration") // optional property
 *         // one of these is required, but both cannot be specified at the same time
 *         .minimumHealthyHosts(MinimumHealthyHosts.count(2))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or import an existing one:
 * <p>
 * <blockquote><pre>
 * IServerDeploymentConfig deploymentConfig = ServerDeploymentConfig.fromServerDeploymentConfigName(this, "ExistingDeploymentConfiguration", "MyExistingDeploymentConfiguration");
 * </pre></blockquote>
 * <p>
 * <h2>Lambda Applications</h2>
 * <p>
 * To create a new CodeDeploy Application that deploys to a Lambda function:
 * <p>
 * <blockquote><pre>
 * LambdaApplication application = LambdaApplication.Builder.create(this, "CodeDeployApplication")
 *         .applicationName("MyApplication")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To import an already existing Application:
 * <p>
 * <blockquote><pre>
 * ILambdaApplication application = LambdaApplication.fromLambdaApplicationName(this, "ExistingCodeDeployApplication", "MyExistingApplication");
 * </pre></blockquote>
 * <p>
 * <h2>Lambda Deployment Groups</h2>
 * <p>
 * To enable traffic shifting deployments for Lambda functions, CodeDeploy uses Lambda Aliases, which can balance incoming traffic between two different versions of your function.
 * Before deployment, the alias sends 100% of invokes to the version used in production.
 * When you publish a new version of the function to your stack, CodeDeploy will send a small percentage of traffic to the new version, monitor, and validate before shifting 100% of traffic to the new version.
 * <p>
 * To create a new CodeDeploy Deployment Group that deploys to a Lambda function:
 * <p>
 * <blockquote><pre>
 * LambdaApplication myApplication;
 * Function func;
 * 
 * Version version = func.getCurrentVersion();
 * Alias version1Alias = Alias.Builder.create(this, "alias")
 *         .aliasName("prod")
 *         .version(version)
 *         .build();
 * 
 * LambdaDeploymentGroup deploymentGroup = LambdaDeploymentGroup.Builder.create(this, "BlueGreenDeployment")
 *         .application(myApplication) // optional property: one will be created for you if not provided
 *         .alias(version1Alias)
 *         .deploymentConfig(LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * In order to deploy a new version of this function:
 * <p>
 * <ol>
 * <li>Reference the version with the latest changes <code>const version = func.currentVersion</code>.</li>
 * <li>Re-deploy the stack (this will trigger a deployment).</li>
 * <li>Monitor the CodeDeploy deployment as traffic shifts between the versions.</li>
 * </ol>
 * <p>
 * <h3>Lambda Deployment Rollbacks and Alarms</h3>
 * <p>
 * CodeDeploy will roll back if the deployment fails. You can optionally trigger a rollback when one or more alarms are in a failed state:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * Alias alias;
 * 
 * // or add alarms to an existing group
 * Alias blueGreenAlias;
 * 
 * Alarm alarm = Alarm.Builder.create(this, "Errors")
 *         .comparisonOperator(ComparisonOperator.GREATER_THAN_THRESHOLD)
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .metric(alias.metricErrors())
 *         .build();
 * LambdaDeploymentGroup deploymentGroup = LambdaDeploymentGroup.Builder.create(this, "BlueGreenDeployment")
 *         .alias(alias)
 *         .deploymentConfig(LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE)
 *         .alarms(List.of(alarm))
 *         .build();
 * deploymentGroup.addAlarm(Alarm.Builder.create(this, "BlueGreenErrors")
 *         .comparisonOperator(ComparisonOperator.GREATER_THAN_THRESHOLD)
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .metric(blueGreenAlias.metricErrors())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Pre and Post Hooks</h3>
 * <p>
 * CodeDeploy allows you to run an arbitrary Lambda function before traffic shifting actually starts (PreTraffic Hook) and after it completes (PostTraffic Hook).
 * With either hook, you have the opportunity to run logic that determines whether the deployment must succeed or fail.
 * For example, with PreTraffic hook you could run integration tests against the newly created Lambda version (but not serving traffic). With PostTraffic hook, you could run end-to-end validation checks.
 * <p>
 * <blockquote><pre>
 * Function warmUpUserCache;
 * Function endToEndValidation;
 * Alias alias;
 * 
 * 
 * // pass a hook whe creating the deployment group
 * LambdaDeploymentGroup deploymentGroup = LambdaDeploymentGroup.Builder.create(this, "BlueGreenDeployment")
 *         .alias(alias)
 *         .deploymentConfig(LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE)
 *         .preHook(warmUpUserCache)
 *         .build();
 * 
 * // or configure one on an existing deployment group
 * deploymentGroup.addPostHook(endToEndValidation);
 * </pre></blockquote>
 * <p>
 * <h3>Import an existing Lambda Deployment Group</h3>
 * <p>
 * To import an already existing Deployment Group:
 * <p>
 * <blockquote><pre>
 * LambdaApplication application;
 * 
 * ILambdaDeploymentGroup deploymentGroup = LambdaDeploymentGroup.fromLambdaDeploymentGroupAttributes(this, "ExistingCodeDeployDeploymentGroup", LambdaDeploymentGroupAttributes.builder()
 *         .application(application)
 *         .deploymentGroupName("MyExistingDeploymentGroup")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Lambda Deployment Configurations</h2>
 * <p>
 * CodeDeploy for Lambda comes with predefined configurations for traffic shifting.
 * The predefined configurations are available as LambdaDeploymentConfig constants.
 * <p>
 * <blockquote><pre>
 * LambdaApplication application;
 * Alias alias;
 * ILambdaDeploymentConfig config = LambdaDeploymentConfig.CANARY_10PERCENT_30MINUTES;
 * LambdaDeploymentGroup deploymentGroup = LambdaDeploymentGroup.Builder.create(this, "BlueGreenDeployment")
 *         .application(application)
 *         .alias(alias)
 *         .deploymentConfig(config)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to specify your own strategy,
 * you can do so with the LambdaDeploymentConfig construct,
 * letting you specify precisely how fast a new function version is deployed.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * LambdaApplication application;
 * Alias alias;
 * LambdaDeploymentConfig config = LambdaDeploymentConfig.Builder.create(this, "CustomConfig")
 *         .trafficRoutingConfig(TimeBasedCanaryTrafficRoutingConfig.Builder.create()
 *                 .interval(cdk.Duration.minutes(15))
 *                 .percentage(5)
 *                 .build())
 *         .build();
 * LambdaDeploymentGroup deploymentGroup = LambdaDeploymentGroup.Builder.create(this, "BlueGreenDeployment")
 *         .application(application)
 *         .alias(alias)
 *         .deploymentConfig(config)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can specify a custom name for your deployment config, but if you do you will not be able to update the interval/percentage through CDK.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * LambdaDeploymentConfig config = LambdaDeploymentConfig.Builder.create(this, "CustomConfig")
 *         .trafficRoutingConfig(TimeBasedCanaryTrafficRoutingConfig.Builder.create()
 *                 .interval(cdk.Duration.minutes(15))
 *                 .percentage(5)
 *                 .build())
 *         .deploymentConfigName("MyDeploymentConfig")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To import an already existing Deployment Config:
 * <p>
 * <blockquote><pre>
 * ILambdaDeploymentConfig deploymentConfig = LambdaDeploymentConfig.fromLambdaDeploymentConfigName(this, "ExistingDeploymentConfiguration", "MyExistingDeploymentConfiguration");
 * </pre></blockquote>
 * <p>
 * <h2>ECS Applications</h2>
 * <p>
 * To create a new CodeDeploy Application that deploys an ECS service:
 * <p>
 * <blockquote><pre>
 * EcsApplication application = EcsApplication.Builder.create(this, "CodeDeployApplication")
 *         .applicationName("MyApplication")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To import an already existing Application:
 * <p>
 * <blockquote><pre>
 * IEcsApplication application = EcsApplication.fromEcsApplicationName(this, "ExistingCodeDeployApplication", "MyExistingApplication");
 * </pre></blockquote>
 * <p>
 * <h2>ECS Deployment Groups</h2>
 * <p>
 * CodeDeploy can be used to deploy to load-balanced ECS services.
 * CodeDeploy performs ECS blue-green deployments by managing ECS task sets and load balancer
 * target groups.  During a blue-green deployment, one task set and target group runs the
 * original version of your ECS task definition ('blue') and another task set and target group
 * runs the new version of your ECS task definition ('green').
 * <p>
 * CodeDeploy orchestrates traffic shifting during ECS blue-green deployments by using
 * a load balancer listener to balance incoming traffic between the 'blue' and 'green' task sets/target groups
 * running two different versions of your ECS task definition.
 * Before deployment, the load balancer listener sends 100% of requests to the 'blue' target group.
 * When you publish a new version of the task definition and start a CodeDeploy deployment,
 * CodeDeploy can send a small percentage of traffic to the new 'green' task set behind the 'green' target group,
 * monitor, and validate before shifting 100% of traffic to the new version.
 * <p>
 * To create a new CodeDeploy Deployment Group that deploys to an ECS service:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EcsApplication myApplication;
 * ecs.Cluster cluster;
 * ecs.FargateTaskDefinition taskDefinition;
 * elbv2.ITargetGroup blueTargetGroup;
 * elbv2.ITargetGroup greenTargetGroup;
 * elbv2.IApplicationListener listener;
 * 
 * 
 * Object service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .deploymentController(Map.of(
 *                 "type", ecs.getDeploymentControllerType().getCODE_DEPLOY()))
 *         .build();
 * 
 * EcsDeploymentGroup.Builder.create(stack, "BlueGreenDG")
 *         .service(service)
 *         .blueGreenDeploymentConfig(EcsBlueGreenDeploymentConfig.builder()
 *                 .blueTargetGroup(blueTargetGroup)
 *                 .greenTargetGroup(greenTargetGroup)
 *                 .listener(listener)
 *                 .build())
 *         .deploymentConfig(EcsDeploymentConfig.CANARY_10PERCENT_5MINUTES)
 *         .build();
 * </pre></blockquote>
 * <p>
 * In order to deploy a new task definition version to the ECS service,
 * deploy the changes directly through CodeDeploy using the CodeDeploy APIs or console.
 * When the <code>CODE_DEPLOY</code> deployment controller is used, the ECS service cannot be
 * deployed with a new task definition version through CloudFormation.
 * <p>
 * For more information on the behavior of CodeDeploy blue-green deployments for ECS, see
 * <a href="https://docs.aws.amazon.com/codedeploy/latest/userguide/deployment-steps-ecs.html#deployment-steps-what-happens">What happens during an Amazon ECS deployment</a>
 * in the CodeDeploy user guide.
 * <p>
 * Note: If you wish to deploy updates to your ECS service through CDK and CloudFormation instead of directly through CodeDeploy,
 * using the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.CfnCodeDeployBlueGreenHook.html"><code>CfnCodeDeployBlueGreenHook</code></a>
 * construct is the recommended approach instead of using the <code>EcsDeploymentGroup</code> construct.  For a comparison
 * of ECS blue-green deployments through CodeDeploy (using <code>EcsDeploymentGroup</code>) and through CloudFormation (using <code>CfnCodeDeployBlueGreenHook</code>),
 * see <a href="https://docs.aws.amazon.com/codedeploy/latest/userguide/deployments-create-ecs-cfn.html#differences-ecs-bg-cfn">Create an Amazon ECS blue/green deployment through AWS CloudFormation</a>
 * in the CloudFormation user guide.
 * <p>
 * <h3>ECS Deployment Rollbacks and Alarms</h3>
 * <p>
 * CodeDeploy will automatically roll back if a deployment fails.
 * You can optionally trigger an automatic rollback when one or more alarms are in a failed state during a deployment, or if the deployment stops.
 * <p>
 * In this example, CodeDeploy will monitor and roll back on alarms set for the
 * number of unhealthy ECS tasks in each of the blue and green target groups,
 * as well as alarms set for the number HTTP 5xx responses seen in each of the blue
 * and green target groups.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * // Alarm on the number of unhealthy ECS tasks in each target group
 * Alarm blueUnhealthyHosts = Alarm.Builder.create(stack, "BlueUnhealthyHosts")
 *         .alarmName(stack.getStackName() + "-Unhealthy-Hosts-Blue")
 *         .metric(blueTargetGroup.metricUnhealthyHostCount())
 *         .threshold(1)
 *         .evaluationPeriods(2)
 *         .build();
 * 
 * Alarm greenUnhealthyHosts = Alarm.Builder.create(stack, "GreenUnhealthyHosts")
 *         .alarmName(stack.getStackName() + "-Unhealthy-Hosts-Green")
 *         .metric(greenTargetGroup.metricUnhealthyHostCount())
 *         .threshold(1)
 *         .evaluationPeriods(2)
 *         .build();
 * 
 * // Alarm on the number of HTTP 5xx responses returned by each target group
 * Alarm blueApiFailure = Alarm.Builder.create(stack, "Blue5xx")
 *         .alarmName(stack.getStackName() + "-Http-5xx-Blue")
 *         .metric(blueTargetGroup.metricHttpCodeTarget(elbv2.getHttpCodeTarget().getTARGET_5XX_COUNT(), Map.of("period", cdk.Duration.minutes(1))))
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * 
 * Alarm greenApiFailure = Alarm.Builder.create(stack, "Green5xx")
 *         .alarmName(stack.getStackName() + "-Http-5xx-Green")
 *         .metric(greenTargetGroup.metricHttpCodeTarget(elbv2.getHttpCodeTarget().getTARGET_5XX_COUNT(), Map.of("period", cdk.Duration.minutes(1))))
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * 
 * EcsDeploymentGroup.Builder.create(stack, "BlueGreenDG")
 *         // CodeDeploy will monitor these alarms during a deployment and automatically roll back
 *         .alarms(List.of(blueUnhealthyHosts, greenUnhealthyHosts, blueApiFailure, greenApiFailure))
 *         .autoRollback(AutoRollbackConfig.builder()
 *                 // CodeDeploy will automatically roll back if a deployment is stopped
 *                 .stoppedDeployment(true)
 *                 .build())
 *         .service(service)
 *         .blueGreenDeploymentConfig(EcsBlueGreenDeploymentConfig.builder()
 *                 .blueTargetGroup(blueTargetGroup)
 *                 .greenTargetGroup(greenTargetGroup)
 *                 .listener(listener)
 *                 .build())
 *         .deploymentConfig(EcsDeploymentConfig.CANARY_10PERCENT_5MINUTES)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deployment validation and manual deployment approval</h3>
 * <p>
 * CodeDeploy blue-green deployments provide an opportunity to validate the new task definition version running on
 * the 'green' ECS task set prior to shifting any production traffic to the new version.  A second 'test' listener
 * serving traffic on a different port be added to the load balancer. For example, the test listener can serve
 * test traffic on port 9001 while the main listener serves production traffic on port 443.
 * During a blue-green deployment, CodeDeploy can then shift 100% of test traffic over to the 'green'
 * task set/target group prior to shifting any production traffic during the deployment.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EcsApplication myApplication;
 * ecs.FargateService service;
 * elbv2.ITargetGroup blueTargetGroup;
 * elbv2.ITargetGroup greenTargetGroup;
 * elbv2.IApplicationListener listener;
 * elbv2.IApplicationListener testListener;
 * 
 * 
 * EcsDeploymentGroup.Builder.create(stack, "BlueGreenDG")
 *         .service(service)
 *         .blueGreenDeploymentConfig(EcsBlueGreenDeploymentConfig.builder()
 *                 .blueTargetGroup(blueTargetGroup)
 *                 .greenTargetGroup(greenTargetGroup)
 *                 .listener(listener)
 *                 .testListener(testListener)
 *                 .build())
 *         .deploymentConfig(EcsDeploymentConfig.CANARY_10PERCENT_5MINUTES)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Automated validation steps can run during the CodeDeploy deployment after shifting test traffic and before
 * shifting production traffic.  CodeDeploy supports registering Lambda functions as lifecycle hooks for
 * an ECS deployment.  These Lambda functions can run automated validation steps against the test traffic
 * port, for example in response to the <code>AfterAllowTestTraffic</code> lifecycle hook.  For more information about
 * how to specify the Lambda functions to run for each CodeDeploy lifecycle hook in an ECS deployment, see the
 * <a href="https://docs.aws.amazon.com/codedeploy/latest/userguide/reference-appspec-file-structure-hooks.html#appspec-hooks-ecs">AppSpec 'hooks' for an Amazon ECS deployment</a>
 * section in the CodeDeploy user guide.
 * <p>
 * After provisioning the 'green' ECS task set and re-routing test traffic during a blue-green deployment,
 * CodeDeploy can wait for approval before continuing the deployment and re-routing production traffic.
 * During this approval wait time, you can complete additional validation steps prior to exposing the new
 * 'green' task set to production traffic, such as manual testing through the test listener port or
 * running automated integration test suites.
 * <p>
 * To approve the deployment, validation steps use the CodeDeploy
 * [ContinueDeployment API(https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ContinueDeployment.html).
 * If the ContinueDeployment API is not called within the approval wait time period, CodeDeploy will stop the
 * deployment and can automatically roll back the deployment.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EcsDeploymentGroup.Builder.create(stack, "BlueGreenDG")
 *         // The deployment will wait for approval for up to 8 hours before stopping the deployment
 *         .deploymentApprovalWaitTime(Duration.hours(8))
 *         .autoRollback(AutoRollbackConfig.builder()
 *                 // CodeDeploy will automatically roll back if the 8-hour approval period times out and the deployment stops
 *                 .stoppedDeployment(true)
 *                 .build())
 *         .service(service)
 *         .blueGreenDeploymentConfig(EcsBlueGreenDeploymentConfig.builder()
 *                 .blueTargetGroup(blueTargetGroup)
 *                 .greenTargetGroup(greenTargetGroup)
 *                 .listener(listener)
 *                 .testListener(testListener)
 *                 .build())
 *         .deploymentConfig(EcsDeploymentConfig.CANARY_10PERCENT_5MINUTES)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deployment bake time</h3>
 * <p>
 * You can specify how long CodeDeploy waits before it terminates the original 'blue' ECS task set when a blue-green deployment
 * is complete in order to let the deployment "bake" a while. During this bake time, CodeDeploy will continue to monitor any
 * CloudWatch alarms specified for the deployment group and will automatically roll back if those alarms go into a failed state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EcsDeploymentGroup.Builder.create(stack, "BlueGreenDG")
 *         .service(service)
 *         .blueGreenDeploymentConfig(EcsBlueGreenDeploymentConfig.builder()
 *                 .blueTargetGroup(blueTargetGroup)
 *                 .greenTargetGroup(greenTargetGroup)
 *                 .listener(listener)
 *                 // CodeDeploy will wait for 30 minutes after completing the blue-green deployment before it terminates the blue tasks
 *                 .terminationWaitTime(Duration.minutes(30))
 *                 .build())
 *         // CodeDeploy will continue to monitor these alarms during the 30-minute bake time and will automatically
 *         // roll back if they go into a failed state at any point during the deployment.
 *         .alarms(List.of(blueUnhealthyHosts, greenUnhealthyHosts, blueApiFailure, greenApiFailure))
 *         .deploymentConfig(EcsDeploymentConfig.CANARY_10PERCENT_5MINUTES)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Import an existing ECS Deployment Group</h3>
 * <p>
 * To import an already existing Deployment Group:
 * <p>
 * <blockquote><pre>
 * EcsApplication application;
 * 
 * IEcsDeploymentGroup deploymentGroup = EcsDeploymentGroup.fromEcsDeploymentGroupAttributes(this, "ExistingCodeDeployDeploymentGroup", EcsDeploymentGroupAttributes.builder()
 *         .application(application)
 *         .deploymentGroupName("MyExistingDeploymentGroup")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>ECS Deployment Configurations</h2>
 * <p>
 * CodeDeploy for ECS comes with predefined configurations for traffic shifting.
 * The predefined configurations are available as LambdaDeploymentConfig constants.
 * <p>
 * <blockquote><pre>
 * IEcsDeploymentConfig config = EcsDeploymentConfig.CANARY_10PERCENT_5MINUTES;
 * </pre></blockquote>
 * <p>
 * If you want to specify your own strategy,
 * you can do so with the EcsDeploymentConfig construct,
 * letting you specify precisely how fast an ECS service is deployed.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EcsDeploymentConfig.Builder.create(this, "CustomConfig")
 *         .trafficRoutingConfig(TimeBasedCanaryTrafficRoutingConfig.Builder.create()
 *                 .interval(cdk.Duration.minutes(15))
 *                 .percentage(5)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can specify a custom name for your deployment config, but if you do you will not be able to update the interval/percentage through CDK.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EcsDeploymentConfig config = EcsDeploymentConfig.Builder.create(this, "CustomConfig")
 *         .trafficRoutingConfig(TimeBasedCanaryTrafficRoutingConfig.Builder.create()
 *                 .interval(cdk.Duration.minutes(15))
 *                 .percentage(5)
 *                 .build())
 *         .deploymentConfigName("MyDeploymentConfig")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or import an existing one:
 * <p>
 * <blockquote><pre>
 * IEcsDeploymentConfig deploymentConfig = EcsDeploymentConfig.fromEcsDeploymentConfigName(this, "ExistingDeploymentConfiguration", "MyExistingDeploymentConfiguration");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.codedeploy;
