/**
 * <h1>AWS CodePipeline Construct Library</h1>
 * <p>
 * <h2>Pipeline</h2>
 * <p>
 * To construct an empty Pipeline:
 * <p>
 * <blockquote><pre>
 * // Construct an empty Pipeline
 * Pipeline pipeline = new Pipeline(this, "MyFirstPipeline");
 * </pre></blockquote>
 * <p>
 * To give the Pipeline a nice, human-readable name:
 * <p>
 * <blockquote><pre>
 * // Give the Pipeline a nice, human-readable name
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyFirstPipeline")
 *         .pipelineName("MyPipeline")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Be aware that in the default configuration, the <code>Pipeline</code> construct creates
 * an AWS Key Management Service (AWS KMS) Customer Master Key (CMK) for you to
 * encrypt the artifacts in the artifact bucket, which incurs a cost of
 * <strong>$1/month</strong>. This default configuration is necessary to allow cross-account
 * actions.
 * <p>
 * If you do not intend to perform cross-account deployments, you can disable
 * the creation of the Customer Master Keys by passing <code>crossAccountKeys: false</code>
 * when defining the Pipeline:
 * <p>
 * <blockquote><pre>
 * // Don't create Customer Master Keys
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyFirstPipeline")
 *         .crossAccountKeys(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to enable key rotation for the generated KMS keys,
 * you can configure it by passing <code>enableKeyRotation: true</code> when creating the pipeline.
 * Note that key rotation will incur an additional cost of <strong>$1/month</strong>.
 * <p>
 * <blockquote><pre>
 * // Enable key rotation for the generated KMS key
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyFirstPipeline")
 *         // ...
 *         .enableKeyRotation(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Stages</h2>
 * <p>
 * You can provide Stages when creating the Pipeline:
 * <p>
 * <blockquote><pre>
 * // Provide a Stage when creating a pipeline
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyFirstPipeline")
 *         .stages(List.of(StageProps.builder()
 *                 .stageName("Source")
 *                 .actions(List.of())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or append a Stage to an existing Pipeline:
 * <p>
 * <blockquote><pre>
 * // Append a Stage to an existing Pipeline
 * Pipeline pipeline;
 * 
 * IStage sourceStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of())
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can insert the new Stage at an arbitrary point in the Pipeline:
 * <p>
 * <blockquote><pre>
 * // Insert a new Stage at an arbitrary point
 * Pipeline pipeline;
 * IStage anotherStage;
 * IStage yetAnotherStage;
 * 
 * 
 * IStage someStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("SomeStage")
 *         .placement(StagePlacement.builder()
 *                 // note: you can only specify one of the below properties
 *                 .rightBefore(anotherStage)
 *                 .justAfter(yetAnotherStage)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can disable transition to a Stage:
 * <p>
 * <blockquote><pre>
 * // Disable transition to a stage
 * Pipeline pipeline;
 * 
 * 
 * IStage someStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("SomeStage")
 *         .transitionToEnabled(false)
 *         .transitionDisabledReason("Manual transition only")
 *         .build());
 * </pre></blockquote>
 * <p>
 * This is useful if you don't want every executions of the pipeline to flow into
 * this stage automatically. The transition can then be "manually" enabled later on.
 * <p>
 * <h2>Actions</h2>
 * <p>
 * Actions live in a separate package, <code>&#64;aws-cdk/aws-codepipeline-actions</code>.
 * <p>
 * To add an Action to a Stage, you can provide it when creating the Stage,
 * in the <code>actions</code> property,
 * or you can use the <code>IStage.addAction()</code> method to mutate an existing Stage:
 * <p>
 * <blockquote><pre>
 * // Use the `IStage.addAction()` method to mutate an existing Stage.
 * IStage sourceStage;
 * Action someAction;
 * 
 * sourceStage.addAction(someAction);
 * </pre></blockquote>
 * <p>
 * <h2>Custom Action Registration</h2>
 * <p>
 * To make your own custom CodePipeline Action requires registering the action provider. Look to the <code>JenkinsProvider</code> in <code>&#64;aws-cdk/aws-codepipeline-actions</code> for an implementation example.
 * <p>
 * <blockquote><pre>
 * // Make a custom CodePipeline Action
 * // Make a custom CodePipeline Action
 * CustomActionRegistration.Builder.create(this, "GenericGitSourceProviderResource")
 *         .category(ActionCategory.SOURCE)
 *         .artifactBounds(ActionArtifactBounds.builder().minInputs(0).maxInputs(0).minOutputs(1).maxOutputs(1).build())
 *         .provider("GenericGitSource")
 *         .version("1")
 *         .entityUrl("https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-create-custom-action.html")
 *         .executionUrl("https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-create-custom-action.html")
 *         .actionProperties(List.of(CustomActionProperty.builder()
 *                 .name("Branch")
 *                 .required(true)
 *                 .key(false)
 *                 .secret(false)
 *                 .queryable(false)
 *                 .description("Git branch to pull")
 *                 .type("String")
 *                 .build(), CustomActionProperty.builder()
 *                 .name("GitUrl")
 *                 .required(true)
 *                 .key(false)
 *                 .secret(false)
 *                 .queryable(false)
 *                 .description("SSH git clone URL")
 *                 .type("String")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Cross-account CodePipelines</h2>
 * <p>
 * <blockquote>
 * <p>
 * Cross-account Pipeline actions require that the Pipeline has <em>not</em> been
 * created with <code>crossAccountKeys: false</code>.
 * <p>
 * </blockquote>
 * <p>
 * Most pipeline Actions accept an AWS resource object to operate on. For example:
 * <p>
 * <ul>
 * <li><code>S3DeployAction</code> accepts an <code>s3.IBucket</code>.</li>
 * <li><code>CodeBuildAction</code> accepts a <code>codebuild.IProject</code>.</li>
 * <li>etc.</li>
 * </ul>
 * <p>
 * These resources can be either newly defined (<code>new s3.Bucket(...)</code>) or imported
 * (<code>s3.Bucket.fromBucketAttributes(...)</code>) and identify the resource that should
 * be changed.
 * <p>
 * These resources can be in different accounts than the pipeline itself. For
 * example, the following action deploys to an imported S3 bucket from a
 * different account:
 * <p>
 * <blockquote><pre>
 * // Deploy an imported S3 bucket from a different account
 * IStage stage;
 * Artifact input;
 * 
 * stage.addAction(S3DeployAction.Builder.create()
 *         .bucket(Bucket.fromBucketAttributes(this, "Bucket", BucketAttributes.builder()
 *                 .account("123456789012")
 *                 .build()))
 *         .input(input)
 *         .actionName("s3-deploy-action")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Actions that don't accept a resource object accept an explicit <code>account</code> parameter:
 * <p>
 * <blockquote><pre>
 * // Actions that don't accept a resource objet accept an explicit `account` parameter
 * IStage stage;
 * ArtifactPath templatePath;
 * 
 * stage.addAction(CloudFormationCreateUpdateStackAction.Builder.create()
 *         .account("123456789012")
 *         .templatePath(templatePath)
 *         .adminPermissions(false)
 *         .stackName(Stack.of(this).getStackName())
 *         .actionName("cloudformation-create-update")
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>Pipeline</code> construct automatically defines an <strong>IAM Role</strong> for you in the
 * target account which the pipeline will assume to perform that action. This
 * Role will be defined in a <strong>support stack</strong> named
 * <code>&lt;PipelineStackName&gt;-support-&lt;account&gt;</code>, that will automatically be deployed
 * before the stack containing the pipeline.
 * <p>
 * If you do not want to use the generated role, you can also explicitly pass a
 * <code>role</code> when creating the action. In that case, the action will operate in the
 * account the role belongs to:
 * <p>
 * <blockquote><pre>
 * // Explicitly pass in a `role` when creating an action.
 * IStage stage;
 * ArtifactPath templatePath;
 * 
 * stage.addAction(CloudFormationCreateUpdateStackAction.Builder.create()
 *         .templatePath(templatePath)
 *         .adminPermissions(false)
 *         .stackName(Stack.of(this).getStackName())
 *         .actionName("cloudformation-create-update")
 *         // ...
 *         .role(Role.fromRoleArn(this, "ActionRole", "..."))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Cross-region CodePipelines</h2>
 * <p>
 * Similar to how you set up a cross-account Action, the AWS resource object you
 * pass to actions can also be in different <em>Regions</em>. For example, the
 * following Action deploys to an imported S3 bucket from a different Region:
 * <p>
 * <blockquote><pre>
 * // Deploy to an imported S3 bucket from a different Region.
 * IStage stage;
 * Artifact input;
 * 
 * stage.addAction(S3DeployAction.Builder.create()
 *         .bucket(Bucket.fromBucketAttributes(this, "Bucket", BucketAttributes.builder()
 *                 .region("us-west-1")
 *                 .build()))
 *         .input(input)
 *         .actionName("s3-deploy-action")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Actions that don't take an AWS resource will accept an explicit <code>region</code>
 * parameter:
 * <p>
 * <blockquote><pre>
 * // Actions that don't take an AWS resource will accept an explicit `region` parameter.
 * IStage stage;
 * ArtifactPath templatePath;
 * 
 * stage.addAction(CloudFormationCreateUpdateStackAction.Builder.create()
 *         .templatePath(templatePath)
 *         .adminPermissions(false)
 *         .stackName(Stack.of(this).getStackName())
 *         .actionName("cloudformation-create-update")
 *         // ...
 *         .region("us-west-1")
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>Pipeline</code> construct automatically defines a <strong>replication bucket</strong> for
 * you in the target region, which the pipeline will replicate artifacts to and
 * from. This Bucket will be defined in a <strong>support stack</strong> named
 * <code>&lt;PipelineStackName&gt;-support-&lt;region&gt;</code>, that will automatically be deployed
 * before the stack containing the pipeline.
 * <p>
 * If you don't want to use these support stacks, and already have buckets in
 * place to serve as replication buckets, you can supply these at Pipeline definition
 * time using the <code>crossRegionReplicationBuckets</code> parameter. Example:
 * <p>
 * <blockquote><pre>
 * // Supply replication buckets for the Pipeline instead of using the generated support stack
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyFirstPipeline")
 *         // ...
 * 
 *         .crossRegionReplicationBuckets(Map.of(
 *                 // note that a physical name of the replication Bucket must be known at synthesis time
 *                 "us-west-1", Bucket.fromBucketAttributes(this, "UsWest1ReplicationBucket", BucketAttributes.builder()
 *                         .bucketName("my-us-west-1-replication-bucket")
 *                         // optional KMS key
 *                         .encryptionKey(Key.fromKeyArn(this, "UsWest1ReplicationKey", "arn:aws:kms:us-west-1:123456789012:key/1234-5678-9012"))
 *                         .build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-create-cross-region.html">the AWS docs here</a>
 * for more information on cross-region CodePipelines.
 * <p>
 * <h3>Creating an encrypted replication bucket</h3>
 * <p>
 * If you're passing a replication bucket created in a different stack,
 * like this:
 * <p>
 * <blockquote><pre>
 * // Passing a replication bucket created in a different stack.
 * App app = new App();
 * Stack replicationStack = Stack.Builder.create(app, "ReplicationStack")
 *         .env(Environment.builder()
 *                 .region("us-west-1")
 *                 .build())
 *         .build();
 * Key key = new Key(replicationStack, "ReplicationKey");
 * Bucket replicationBucket = Bucket.Builder.create(replicationStack, "ReplicationBucket")
 *         // like was said above - replication buckets need a set physical name
 *         .bucketName(PhysicalName.GENERATE_IF_NEEDED)
 *         .encryptionKey(key)
 *         .build();
 * 
 * // later...
 * // later...
 * Pipeline.Builder.create(replicationStack, "Pipeline")
 *         .crossRegionReplicationBuckets(Map.of(
 *                 "us-west-1", replicationBucket))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When trying to encrypt it
 * (and note that if any of the cross-region actions happen to be cross-account as well,
 * the bucket <em>has to</em> be encrypted - otherwise the pipeline will fail at runtime),
 * you cannot use a key directly - KMS keys don't have physical names,
 * and so you can't reference them across environments.
 * <p>
 * In this case, you need to use an alias in place of the key when creating the bucket:
 * <p>
 * <blockquote><pre>
 * // Passing an encrypted replication bucket created in a different stack.
 * App app = new App();
 * Stack replicationStack = Stack.Builder.create(app, "ReplicationStack")
 *         .env(Environment.builder()
 *                 .region("us-west-1")
 *                 .build())
 *         .build();
 * Key key = new Key(replicationStack, "ReplicationKey");
 * Alias alias = Alias.Builder.create(replicationStack, "ReplicationAlias")
 *         // aliasName is required
 *         .aliasName(PhysicalName.GENERATE_IF_NEEDED)
 *         .targetKey(key)
 *         .build();
 * Bucket replicationBucket = Bucket.Builder.create(replicationStack, "ReplicationBucket")
 *         .bucketName(PhysicalName.GENERATE_IF_NEEDED)
 *         .encryptionKey(alias)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Variables</h2>
 * <p>
 * The library supports the CodePipeline Variables feature.
 * Each action class that emits variables has a separate variables interface,
 * accessed as a property of the action instance called <code>variables</code>.
 * You instantiate the action class and assign it to a local variable;
 * when you want to use a variable in the configuration of a different action,
 * you access the appropriate property of the interface returned from <code>variables</code>,
 * which represents a single variable.
 * Example:
 * <p>
 * <blockquote><pre>
 * // MyAction is some action type that produces variables, like EcrSourceAction
 * MyAction myAction = new MyAction(new MyActionProps()
 *         // ...
 *         .actionName("myAction")
 *         );
 * new OtherAction(new OtherActionProps()
 *         // ...
 *         .config(myAction.getVariables().getMyVariable())
 *         .actionName("otherAction")
 *         );
 * </pre></blockquote>
 * <p>
 * The namespace name that will be used will be automatically generated by the pipeline construct,
 * based on the stage and action name;
 * you can pass a custom name when creating the action instance:
 * <p>
 * <blockquote><pre>
 * // MyAction is some action type that produces variables, like EcrSourceAction
 * MyAction myAction = new MyAction(new MyActionProps()
 *         // ...
 *         .variablesNamespace("MyNamespace")
 *         .actionName("myAction")
 *         );
 * </pre></blockquote>
 * <p>
 * There are also global variables available,
 * not tied to any action;
 * these are accessed through static properties of the <code>GlobalVariables</code> class:
 * <p>
 * <blockquote><pre>
 * // OtherAction is some action type that produces variables, like EcrSourceAction
 * // OtherAction is some action type that produces variables, like EcrSourceAction
 * new OtherAction(new OtherActionProps()
 *         // ...
 *         .config(GlobalVariables.executionId)
 *         .actionName("otherAction")
 *         );
 * </pre></blockquote>
 * <p>
 * Check the documentation of the <code>&#64;aws-cdk/aws-codepipeline-actions</code>
 * for details on how to use the variables for each action class.
 * <p>
 * See the <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-variables.html">CodePipeline documentation</a>
 * for more details on how to use the variables feature.
 * <p>
 * <h2>Events</h2>
 * <p>
 * <h3>Using a pipeline as an event target</h3>
 * <p>
 * A pipeline can be used as a target for a CloudWatch event rule:
 * <p>
 * <blockquote><pre>
 * // A pipeline being used as a target for a CloudWatch event rule.
 * import software.amazon.awscdk.services.events.targets.*;
 * import software.amazon.awscdk.services.events.*;
 * 
 * Pipeline pipeline;
 * 
 * 
 * // kick off the pipeline every day
 * Rule rule = Rule.Builder.create(this, "Daily")
 *         .schedule(Schedule.rate(Duration.days(1)))
 *         .build();
 * rule.addTarget(new CodePipeline(pipeline));
 * </pre></blockquote>
 * <p>
 * When a pipeline is used as an event target, the
 * "codepipeline:StartPipelineExecution" permission is granted to the AWS
 * CloudWatch Events service.
 * <p>
 * <h3>Event sources</h3>
 * <p>
 * Pipelines emit CloudWatch events. To define event rules for events emitted by
 * the pipeline, stages or action, use the <code>onXxx</code> methods on the respective
 * construct:
 * <p>
 * <blockquote><pre>
 * // Define event rules for events emitted by the pipeline
 * import software.amazon.awscdk.services.events.*;
 * 
 * Pipeline myPipeline;
 * IStage myStage;
 * Action myAction;
 * IRuleTarget target;
 * 
 * myPipeline.onStateChange("MyPipelineStateChange", OnEventOptions.builder().target(target).build());
 * myStage.onStateChange("MyStageStateChange", target);
 * myAction.onStateChange("MyActionStateChange", target);
 * </pre></blockquote>
 * <p>
 * <h2>CodeStar Notifications</h2>
 * <p>
 * To define CodeStar Notification rules for Pipelines, use one of the <code>notifyOnXxx()</code> methods.
 * They are very similar to <code>onXxx()</code> methods for CloudWatch events:
 * <p>
 * <blockquote><pre>
 * // Define CodeStar Notification rules for Pipelines
 * import software.amazon.awscdk.services.chatbot.*;
 * 
 * Pipeline pipeline;
 * 
 * SlackChannelConfiguration target = SlackChannelConfiguration.Builder.create(this, "MySlackChannel")
 *         .slackChannelConfigurationName("YOUR_CHANNEL_NAME")
 *         .slackWorkspaceId("YOUR_SLACK_WORKSPACE_ID")
 *         .slackChannelId("YOUR_SLACK_CHANNEL_ID")
 *         .build();
 * INotificationRule rule = pipeline.notifyOnExecutionStateChange("NotifyOnExecutionStateChange", target);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.codepipeline;
