/**
 * <h1>Amazon EKS Construct Library</h1>
 * <p>
 * This construct library allows you to define <a href="https://aws.amazon.com/eks/">Amazon Elastic Container Service for Kubernetes (EKS)</a> clusters.
 * In addition, the library also supports defining Kubernetes resource manifests within EKS clusters.
 * <p>
 * <h2>Table Of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#quick-start">Quick Start</a></li>
 * <li><a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-eks-readme.html">API Reference</a></li>
 * <li><a href="#architectural-overview">Architectural Overview</a></li>
 * <li><a href="#provisioning-clusters">Provisioning clusters</a>
 * <p>
 * <ul>
 * <li><a href="#managed-node-groups">Managed node groups</a></li>
 * <li><a href="#fargate-profiles">Fargate Profiles</a></li>
 * <li><a href="#self-managed-nodes">Self-managed nodes</a></li>
 * <li><a href="#endpoint-access">Endpoint Access</a></li>
 * <li><a href="#alb-controller">ALB Controller</a></li>
 * <li><a href="#vpc-support">VPC Support</a></li>
 * <li><a href="#kubectl-support">Kubectl Support</a></li>
 * <li><a href="#arm64-support">ARM64 Support</a></li>
 * <li><a href="#masters-role">Masters Role</a></li>
 * <li><a href="#encryption">Encryption</a></li>
 * </ul></li>
 * <li><a href="#permissions-and-security">Permissions and Security</a></li>
 * <li><a href="#applying-kubernetes-resources">Applying Kubernetes Resources</a>
 * <p>
 * <ul>
 * <li><a href="#kubernetes-manifests">Kubernetes Manifests</a></li>
 * <li><a href="#helm-charts">Helm Charts</a></li>
 * <li><a href="#cdk8s-charts">CDK8s Charts</a></li>
 * </ul></li>
 * <li><a href="#patching-kubernetes-resources">Patching Kubernetes Resources</a></li>
 * <li><a href="#querying-kubernetes-resources">Querying Kubernetes Resources</a></li>
 * <li><a href="#using-existing-clusters">Using existing clusters</a></li>
 * <li><a href="#known-issues-and-limitations">Known Issues and Limitations</a></li>
 * </ul>
 * <p>
 * <h2>Quick Start</h2>
 * <p>
 * This example defines an Amazon EKS cluster with the following configuration:
 * <p>
 * <ul>
 * <li>Dedicated VPC with default configuration (Implicitly created using <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#vpc">ec2.Vpc</a>)</li>
 * <li>A Kubernetes pod with a container based on the <a href="https://github.com/paulbouwer/hello-kubernetes">paulbouwer/hello-kubernetes</a> image.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // provisiong a cluster
 * Cluster cluster = Cluster.Builder.create(this, "hello-eks")
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * 
 * // apply a kubernetes manifest to the cluster
 * cluster.addManifest("mypod", Map.of(
 *         "apiVersion", "v1",
 *         "kind", "Pod",
 *         "metadata", Map.of("name", "mypod"),
 *         "spec", Map.of(
 *                 "containers", List.of(Map.of(
 *                         "name", "hello",
 *                         "image", "paulbouwer/hello-kubernetes:1.5",
 *                         "ports", List.of(Map.of("containerPort", 8080)))))));
 * </pre></blockquote>
 * <p>
 * In order to interact with your cluster through <code>kubectl</code>, you can use the <code>aws eks update-kubeconfig</code> <a href="https://docs.aws.amazon.com/cli/latest/reference/eks/update-kubeconfig.html">AWS CLI command</a>
 * to configure your local kubeconfig. The EKS module will define a CloudFormation output in your stack which contains the command to run. For example:
 * <p>
 * <blockquote><pre>
 * Outputs:
 * ClusterConfigCommand43AAE40F = aws eks update-kubeconfig --name cluster-xxxxx --role-arn arn:aws:iam::112233445566:role/yyyyy
 * </pre></blockquote>
 * <p>
 * Execute the <code>aws eks update-kubeconfig ...</code> command in your terminal to create or update a local kubeconfig context:
 * <p>
 * <blockquote><pre>
 * $ aws eks update-kubeconfig --name cluster-xxxxx --role-arn arn:aws:iam::112233445566:role/yyyyy
 * Added new context arn:aws:eks:rrrrr:112233445566:cluster/cluster-xxxxx to /home/boom/.kube/config
 * </pre></blockquote>
 * <p>
 * And now you can simply use <code>kubectl</code>:
 * <p>
 * <blockquote><pre>
 * $ kubectl get all -n kube-system
 * NAME                           READY   STATUS    RESTARTS   AGE
 * pod/aws-node-fpmwv             1/1     Running   0          21m
 * pod/aws-node-m9htf             1/1     Running   0          21m
 * pod/coredns-5cb4fb54c7-q222j   1/1     Running   0          23m
 * pod/coredns-5cb4fb54c7-v9nxx   1/1     Running   0          23m
 * ...
 * </pre></blockquote>
 * <p>
 * <h2>Architectural Overview</h2>
 * <p>
 * The following is a qualitative diagram of the various possible components involved in the cluster deployment.
 * <p>
 * <blockquote><pre>
 *  +-----------------------------------------------+               +-----------------+
 *  |                 EKS Cluster                   |    kubectl    |                 |
 *  |-----------------------------------------------|&lt;-------------+| Kubectl Handler |
 *  |                                               |               |                 |
 *  |                                               |               +-----------------+
 *  | +--------------------+    +-----------------+ |
 *  | |                    |    |                 | |
 *  | | Managed Node Group |    | Fargate Profile | |               +-----------------+
 *  | |                    |    |                 | |               |                 |
 *  | +--------------------+    +-----------------+ |               | Cluster Handler |
 *  |                                               |               |                 |
 *  +-----------------------------------------------+               +-----------------+
 *     ^                                   ^                          +
 *     |                                   |                          |
 *     | connect self managed capacity     |                          | aws-sdk
 *     |                                   | create/update/delete     |
 *     +                                   |                          v
 *  +--------------------+                 +              +-------------------+
 *  |                    |                 --------------+| eks.amazonaws.com |
 *  | Auto Scaling Group |                                +-------------------+
 *  |                    |
 *  +--------------------+
 * </pre></blockquote>
 * <p>
 * In a nutshell:
 * <p>
 * <ul>
 * <li><code>EKS Cluster</code> - The cluster endpoint created by EKS.</li>
 * <li><code>Managed Node Group</code> - EC2 worker nodes managed by EKS.</li>
 * <li><code>Fargate Profile</code> - Fargate worker nodes managed by EKS.</li>
 * <li><code>Auto Scaling Group</code> - EC2 worker nodes managed by the user.</li>
 * <li><code>KubectlHandler</code> - Lambda function for invoking <code>kubectl</code> commands on the cluster - created by CDK.</li>
 * <li><code>ClusterHandler</code> - Lambda function for interacting with EKS API to manage the cluster lifecycle - created by CDK.</li>
 * </ul>
 * <p>
 * A more detailed breakdown of each is provided further down this README.
 * <p>
 * <h2>Provisioning clusters</h2>
 * <p>
 * Creating a new cluster is done using the <code>Cluster</code> or <code>FargateCluster</code> constructs. The only required property is the kubernetes <code>version</code>.
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use <code>FargateCluster</code> to provision a cluster that uses only fargate workers.
 * <p>
 * <blockquote><pre>
 * FargateCluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>NOTE: Only 1 cluster per stack is supported.</strong> If you have a use-case for multiple clusters per stack, or would like to understand more about this limitation, see <a href="https://github.com/aws/aws-cdk/issues/10073">https://github.com/aws/aws-cdk/issues/10073</a>.
 * <p>
 * </blockquote>
 * <p>
 * Below you'll find a few important cluster configuration options. First of which is Capacity.
 * Capacity is the amount and the type of worker nodes that are available to the cluster for deploying resources. Amazon EKS offers 3 ways of configuring capacity, which you can combine as you like:
 * <p>
 * <h3>Managed node groups</h3>
 * <p>
 * Amazon EKS managed node groups automate the provisioning and lifecycle management of nodes (Amazon EC2 instances) for Amazon EKS Kubernetes clusters.
 * With Amazon EKS managed node groups, you don’t need to separately provision or register the Amazon EC2 instances that provide compute capacity to run your Kubernetes applications. You can create, update, or terminate nodes for your cluster with a single operation. Nodes run using the latest Amazon EKS optimized AMIs in your AWS account while node updates and terminations gracefully drain nodes to ensure that your applications stay available.
 * <p>
 * <blockquote>
 * <p>
 * For more details visit <a href="https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html">Amazon EKS Managed Node Groups</a>.
 * <p>
 * </blockquote>
 * <p>
 * <strong>Managed Node Groups are the recommended way to allocate cluster capacity.</strong>
 * <p>
 * By default, this library will allocate a managed node group with 2 <em>m5.large</em> instances (this instance type suits most common use-cases, and is good value for money).
 * <p>
 * At cluster instantiation time, you can customize the number of instances and their type:
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .defaultCapacity(5)
 *         .defaultCapacityInstance(InstanceType.of(InstanceClass.M5, InstanceSize.SMALL))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To access the node group that was created on your behalf, you can use <code>cluster.defaultNodegroup</code>.
 * <p>
 * Additional customizations are available post instantiation. To apply them, set the default capacity to 0, and use the <code>cluster.addNodegroupCapacity</code> method:
 * <p>
 * <blockquote><pre>
 * Cluster cluster = Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .defaultCapacity(0)
 *         .build();
 * 
 * cluster.addNodegroupCapacity("custom-node-group", NodegroupOptions.builder()
 *         .instanceTypes(List.of(new InstanceType("m5.large")))
 *         .minSize(4)
 *         .diskSize(100)
 *         .amiType(NodegroupAmiType.AL2_X86_64_GPU)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To set node taints, you can set <code>taints</code> option.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addNodegroupCapacity("custom-node-group", NodegroupOptions.builder()
 *         .instanceTypes(List.of(new InstanceType("m5.large")))
 *         .taints(List.of(TaintSpec.builder()
 *                 .effect(TaintEffect.NO_SCHEDULE)
 *                 .key("foo")
 *                 .value("bar")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Spot Instances Support</h4>
 * <p>
 * Use <code>capacityType</code> to create managed node groups comprised of spot instances. To maximize the availability of your applications while using
 * Spot Instances, we recommend that you configure a Spot managed node group to use multiple instance types with the <code>instanceTypes</code> property.
 * <p>
 * <blockquote>
 * <p>
 * For more details visit <a href="https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html#managed-node-group-capacity-types">Managed node group capacity types</a>.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addNodegroupCapacity("extra-ng-spot", NodegroupOptions.builder()
 *         .instanceTypes(List.of(
 *             new InstanceType("c5.large"),
 *             new InstanceType("c5a.large"),
 *             new InstanceType("c5d.large")))
 *         .minSize(3)
 *         .capacityType(CapacityType.SPOT)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Launch Template Support</h4>
 * <p>
 * You can specify a launch template that the node group will use. For example, this can be useful if you want to use
 * a custom AMI or add custom user data.
 * <p>
 * When supplying a custom user data script, it must be encoded in the MIME multi-part archive format, since Amazon EKS merges with its own user data. Visit the <a href="https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html#launch-template-user-data">Launch Template Docs</a>
 * for mode details.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * String userData = "MIME-Version: 1.0\nContent-Type: multipart/mixed; boundary=\"==MYBOUNDARY==\"\n\n--==MYBOUNDARY==\nContent-Type: text/x-shellscript; charset=\"us-ascii\"\n\n#!/bin/bash\necho \"Running custom user data script\"\n\n--==MYBOUNDARY==--\\\n";
 * CfnLaunchTemplate lt = CfnLaunchTemplate.Builder.create(this, "LaunchTemplate")
 *         .launchTemplateData(LaunchTemplateDataProperty.builder()
 *                 .instanceType("t3.small")
 *                 .userData(Fn.base64(userData))
 *                 .build())
 *         .build();
 * 
 * cluster.addNodegroupCapacity("extra-ng", NodegroupOptions.builder()
 *         .launchTemplateSpec(LaunchTemplateSpec.builder()
 *                 .id(lt.getRef())
 *                 .version(lt.getAttrLatestVersionNumber())
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * Note that when using a custom AMI, Amazon EKS doesn't merge any user data. Which means you do not need the multi-part encoding. and are responsible for supplying the required bootstrap commands for nodes to join the cluster.
 * In the following example, <code>/ect/eks/bootstrap.sh</code> from the AMI will be used to bootstrap the node.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * UserData userData = UserData.forLinux();
 * userData.addCommands("set -o xtrace", String.format("/etc/eks/bootstrap.sh %s", cluster.getClusterName()));
 * CfnLaunchTemplate lt = CfnLaunchTemplate.Builder.create(this, "LaunchTemplate")
 *         .launchTemplateData(LaunchTemplateDataProperty.builder()
 *                 .imageId("some-ami-id") // custom AMI
 *                 .instanceType("t3.small")
 *                 .userData(Fn.base64(userData.render()))
 *                 .build())
 *         .build();
 * cluster.addNodegroupCapacity("extra-ng", NodegroupOptions.builder()
 *         .launchTemplateSpec(LaunchTemplateSpec.builder()
 *                 .id(lt.getRef())
 *                 .version(lt.getAttrLatestVersionNumber())
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * You may specify one <code>instanceType</code> in the launch template or multiple <code>instanceTypes</code> in the node group, <strong>but not both</strong>.
 * <p>
 * <blockquote>
 * <p>
 * For more details visit <a href="https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html">Launch Template Support</a>.
 * <p>
 * </blockquote>
 * <p>
 * Graviton 2 instance types are supported including <code>c6g</code>, <code>m6g</code>, <code>r6g</code> and <code>t4g</code>.
 * Graviton 3 instance types are supported including <code>c7g</code>.
 * <p>
 * <h3>Fargate profiles</h3>
 * <p>
 * AWS Fargate is a technology that provides on-demand, right-sized compute
 * capacity for containers. With AWS Fargate, you no longer have to provision,
 * configure, or scale groups of virtual machines to run containers. This removes
 * the need to choose server types, decide when to scale your node groups, or
 * optimize cluster packing.
 * <p>
 * You can control which pods start on Fargate and how they run with Fargate
 * Profiles, which are defined as part of your Amazon EKS cluster.
 * <p>
 * See <a href="https://docs.aws.amazon.com/eks/latest/userguide/fargate.html#fargate-considerations">Fargate Considerations</a> in the AWS EKS User Guide.
 * <p>
 * You can add Fargate Profiles to any EKS cluster defined in your CDK app
 * through the <code>addFargateProfile()</code> method. The following example adds a profile
 * that will match all pods from the "default" namespace:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addFargateProfile("MyProfile", FargateProfileOptions.builder()
 *         .selectors(List.of(Selector.builder().namespace("default").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also directly use the <code>FargateProfile</code> construct to create profiles under different scopes:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * FargateProfile.Builder.create(this, "MyProfile")
 *         .cluster(cluster)
 *         .selectors(List.of(Selector.builder().namespace("default").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To create an EKS cluster that <strong>only</strong> uses Fargate capacity, you can use <code>FargateCluster</code>.
 * The following code defines an Amazon EKS cluster with a default Fargate Profile that matches all pods from the "kube-system" and "default" namespaces. It is also configured to <a href="https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns">run CoreDNS on Fargate</a>.
 * <p>
 * <blockquote><pre>
 * FargateCluster cluster = FargateCluster.Builder.create(this, "MyCluster")
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>FargateCluster</code> will create a default <code>FargateProfile</code> which can be accessed via the cluster's <code>defaultProfile</code> property. The created profile can also be customized by passing options as with <code>addFargateProfile</code>.
 * <p>
 * <strong>NOTE</strong>: Classic Load Balancers and Network Load Balancers are not supported on
 * pods running on Fargate. For ingress, we recommend that you use the <a href="https://docs.aws.amazon.com/eks/latest/userguide/alb-ingress.html">ALB Ingress
 * Controller</a>
 * on Amazon EKS (minimum version v1.1.4).
 * <p>
 * <h3>Self-managed nodes</h3>
 * <p>
 * Another way of allocating capacity to an EKS cluster is by using self-managed nodes.
 * EC2 instances that are part of the auto-scaling group will serve as worker nodes for the cluster.
 * This type of capacity is also commonly referred to as <em>EC2 Capacity</em>* or <em>EC2 Nodes</em>.
 * <p>
 * For a detailed overview please visit <a href="https://docs.aws.amazon.com/eks/latest/userguide/worker.html">Self Managed Nodes</a>.
 * <p>
 * Creating an auto-scaling group and connecting it to the cluster is done using the <code>cluster.addAutoScalingGroupCapacity</code> method:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addAutoScalingGroupCapacity("frontend-nodes", AutoScalingGroupCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.medium"))
 *         .minCapacity(3)
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * To connect an already initialized auto-scaling group, use the <code>cluster.connectAutoScalingGroupCapacity()</code> method:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * AutoScalingGroup asg;
 * 
 * cluster.connectAutoScalingGroupCapacity(asg, AutoScalingGroupOptions.builder().build());
 * </pre></blockquote>
 * <p>
 * To connect a self-managed node group to an imported cluster, use the <code>cluster.connectAutoScalingGroupCapacity()</code> method:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * AutoScalingGroup asg;
 * 
 * ICluster importedCluster = Cluster.fromClusterAttributes(this, "ImportedCluster", ClusterAttributes.builder()
 *         .clusterName(cluster.getClusterName())
 *         .clusterSecurityGroupId(cluster.getClusterSecurityGroupId())
 *         .build());
 * 
 * importedCluster.connectAutoScalingGroupCapacity(asg, AutoScalingGroupOptions.builder().build());
 * </pre></blockquote>
 * <p>
 * In both cases, the <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html#cluster-sg">cluster security group</a> will be automatically attached to
 * the auto-scaling group, allowing for traffic to flow freely between managed and self-managed nodes.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> The default <code>updateType</code> for auto-scaling groups does not replace existing nodes. Since security groups are determined at launch time, self-managed nodes that were provisioned with version <code>1.78.0</code> or lower, will not be updated.
 * To apply the new configuration on all your self-managed nodes, you'll need to replace the nodes using the <code>UpdateType.REPLACING_UPDATE</code> policy for the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-autoscaling.AutoScalingGroup.html#updatetypespan-classapi-icon-api-icon-deprecated-titlethis-api-element-is-deprecated-its-use-is-not-recommended%EF%B8%8Fspan"><code>updateType</code></a> property.
 * <p>
 * </blockquote>
 * <p>
 * You can customize the <a href="https://github.com/awslabs/amazon-eks-ami/blob/master/files/bootstrap.sh">/etc/eks/boostrap.sh</a> script, which is responsible
 * for bootstrapping the node to the EKS cluster. For example, you can use <code>kubeletExtraArgs</code> to add custom node labels or taints.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addAutoScalingGroupCapacity("spot", AutoScalingGroupCapacityOptions.builder()
 *         .instanceType(new InstanceType("t3.large"))
 *         .minCapacity(2)
 *         .bootstrapOptions(BootstrapOptions.builder()
 *                 .kubeletExtraArgs("--node-labels foo=bar,goo=far")
 *                 .awsApiRetryAttempts(5)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * To disable bootstrapping altogether (i.e. to fully customize user-data), set <code>bootstrapEnabled</code> to <code>false</code>.
 * You can also configure the cluster to use an auto-scaling group as the default capacity:
 * <p>
 * <blockquote><pre>
 * Cluster cluster = Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .defaultCapacityType(DefaultCapacityType.EC2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will allocate an auto-scaling group with 2 <em>m5.large</em> instances (this instance type suits most common use-cases, and is good value for money).
 * To access the <code>AutoScalingGroup</code> that was created on your behalf, you can use <code>cluster.defaultCapacity</code>.
 * You can also independently create an <code>AutoScalingGroup</code> and connect it to the cluster using the <code>cluster.connectAutoScalingGroupCapacity</code> method:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * AutoScalingGroup asg;
 * 
 * cluster.connectAutoScalingGroupCapacity(asg, AutoScalingGroupOptions.builder().build());
 * </pre></blockquote>
 * <p>
 * This will add the necessary user-data to access the apiserver and configure all connections, roles, and tags needed for the instances in the auto-scaling group to properly join the cluster.
 * <p>
 * <h4>Spot Instances</h4>
 * <p>
 * When using self-managed nodes, you can configure the capacity to use spot instances, greatly reducing capacity cost.
 * To enable spot capacity, use the <code>spotPrice</code> property:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addAutoScalingGroupCapacity("spot", AutoScalingGroupCapacityOptions.builder()
 *         .spotPrice("0.1094")
 *         .instanceType(new InstanceType("t3.large"))
 *         .maxCapacity(10)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Spot instance nodes will be labeled with <code>lifecycle=Ec2Spot</code> and tainted with <code>PreferNoSchedule</code>.
 * <p>
 * </blockquote>
 * <p>
 * The <a href="https://github.com/aws/aws-node-termination-handler">AWS Node Termination Handler</a> <code>DaemonSet</code> will be
 * installed from <a href="https://github.com/aws/eks-charts/tree/master/stable/aws-node-termination-handler">Amazon EKS Helm chart repository</a> on these nodes.
 * The termination handler ensures that the Kubernetes control plane responds appropriately to events that
 * can cause your EC2 instance to become unavailable, such as <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/monitoring-instances-status-check_sched.html">EC2 maintenance events</a>
 * and <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-interruptions.html">EC2 Spot interruptions</a> and helps gracefully stop all pods running on spot nodes that are about to be
 * terminated.
 * <p>
 * <blockquote>
 * <p>
 * Handler Version: <a href="https://github.com/aws/aws-node-termination-handler/releases/tag/v1.7.0">1.7.0</a>
 * <p>
 * Chart Version: <a href="https://github.com/aws/eks-charts/blob/v0.0.28/stable/aws-node-termination-handler/Chart.yaml">0.9.5</a>
 * <p>
 * </blockquote>
 * <p>
 * To disable the installation of the termination handler, set the <code>spotInterruptHandler</code> property to <code>false</code>. This applies both to <code>addAutoScalingGroupCapacity</code> and <code>connectAutoScalingGroupCapacity</code>.
 * <p>
 * <h4>Bottlerocket</h4>
 * <p>
 * <a href="https://aws.amazon.com/bottlerocket/">Bottlerocket</a> is a Linux-based open-source operating system that is purpose-built by Amazon Web Services for running containers on virtual machines or bare metal hosts.
 * <p>
 * <code>Bottlerocket</code> is supported when using managed nodegroups or self-managed auto-scaling groups.
 * <p>
 * To create a Bottlerocket managed nodegroup:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addNodegroupCapacity("BottlerocketNG", NodegroupOptions.builder()
 *         .amiType(NodegroupAmiType.BOTTLEROCKET_X86_64)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The following example will create an auto-scaling group of 2 <code>t3.small</code> Linux instances running with the <code>Bottlerocket</code> AMI.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addAutoScalingGroupCapacity("BottlerocketNodes", AutoScalingGroupCapacityOptions.builder()
 *         .instanceType(new InstanceType("t3.small"))
 *         .minCapacity(2)
 *         .machineImageType(MachineImageType.BOTTLEROCKET)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The specific Bottlerocket AMI variant will be auto selected according to the k8s version for the <code>x86_64</code> architecture.
 * For example, if the Amazon EKS cluster version is <code>1.17</code>, the Bottlerocket AMI variant will be auto selected as
 * <code>aws-k8s-1.17</code> behind the scene.
 * <p>
 * <blockquote>
 * <p>
 * See <a href="https://github.com/bottlerocket-os/bottlerocket/blob/develop/README.md#variants">Variants</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * Please note Bottlerocket does not allow to customize bootstrap options and <code>bootstrapOptions</code> properties is not supported when you create the <code>Bottlerocket</code> capacity.
 * <p>
 * For more details about Bottlerocket, see <a href="https://aws.amazon.com/bottlerocket/faqs/">Bottlerocket FAQs</a> and <a href="https://aws.amazon.com/blogs/opensource/announcing-the-general-availability-of-bottlerocket-an-open-source-linux-distribution-purpose-built-to-run-containers/">Bottlerocket Open Source Blog</a>.
 * <p>
 * <h3>Endpoint Access</h3>
 * <p>
 * When you create a new cluster, Amazon EKS creates an endpoint for the managed Kubernetes API server that you use to communicate with your cluster (using Kubernetes management tools such as <code>kubectl</code>)
 * <p>
 * By default, this API server endpoint is public to the internet, and access to the API server is secured using a combination of
 * AWS Identity and Access Management (IAM) and native Kubernetes <a href="https://kubernetes.io/docs/reference/access-authn-authz/rbac/">Role Based Access Control</a> (RBAC).
 * <p>
 * You can configure the <a href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">cluster endpoint access</a> by using the <code>endpointAccess</code> property:
 * <p>
 * <blockquote><pre>
 * Cluster cluster = Cluster.Builder.create(this, "hello-eks")
 *         .version(KubernetesVersion.V1_21)
 *         .endpointAccess(EndpointAccess.PRIVATE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The default value is <code>eks.EndpointAccess.PUBLIC_AND_PRIVATE</code>. Which means the cluster endpoint is accessible from outside of your VPC, but worker node traffic and <code>kubectl</code> commands issued by this library stay within your VPC.
 * <p>
 * <h3>Alb Controller</h3>
 * <p>
 * Some Kubernetes resources are commonly implemented on AWS with the help of the <a href="https://kubernetes-sigs.github.io/aws-load-balancer-controller/v2.3/">ALB Controller</a>.
 * <p>
 * From the docs:
 * <p>
 * <blockquote>
 * <p>
 * AWS Load Balancer Controller is a controller to help manage Elastic Load Balancers for a Kubernetes cluster.
 * <p>
 * <ul>
 * <li>It satisfies Kubernetes Ingress resources by provisioning Application Load Balancers.</li>
 * <li>It satisfies Kubernetes Service resources by provisioning Network Load Balancers.</li>
 * </ul>
 * <p>
 * </blockquote>
 * <p>
 * To deploy the controller on your EKS cluster, configure the <code>albController</code> property:
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .albController(AlbControllerOptions.builder()
 *                 .version(AlbControllerVersion.V2_4_1)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Querying the controller pods should look something like this:
 * <p>
 * <blockquote><pre>
 * ❯ kubectl get pods -n kube-system
 * NAME                                            READY   STATUS    RESTARTS   AGE
 * aws-load-balancer-controller-76bd6c7586-d929p   1/1     Running   0          109m
 * aws-load-balancer-controller-76bd6c7586-fqxph   1/1     Running   0          109m
 * ...
 * ...
 * </pre></blockquote>
 * <p>
 * Every Kubernetes manifest that utilizes the ALB Controller is effectively dependant on the controller.
 * If the controller is deleted before the manifest, it might result in dangling ELB/ALB resources.
 * Currently, the EKS construct library does not detect such dependencies, and they should be done explicitly.
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * KubernetesManifest manifest = cluster.addManifest("manifest", Map.of());
 * if (cluster.getAlbController()) {
 *     manifest.node.addDependency(cluster.getAlbController());
 * }
 * </pre></blockquote>
 * <p>
 * <h3>VPC Support</h3>
 * <p>
 * You can specify the VPC of the cluster using the <code>vpc</code> and <code>vpcSubnets</code> properties:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .vpc(vpc)
 *         .vpcSubnets(List.of(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_WITH_EGRESS).build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: Isolated VPCs (i.e with no internet access) are not currently supported. See https://github.com/aws/aws-cdk/issues/12171
 * <p>
 * </blockquote>
 * <p>
 * If you do not specify a VPC, one will be created on your behalf, which you can then access via <code>cluster.vpc</code>. The cluster VPC will be associated to any EKS managed capacity (i.e Managed Node Groups and Fargate Profiles).
 * <p>
 * Please note that the <code>vpcSubnets</code> property defines the subnets where EKS will place the <em>control plane</em> ENIs. To choose
 * the subnets where EKS will place the worker nodes, please refer to the <strong>Provisioning clusters</strong> section above.
 * <p>
 * If you allocate self managed capacity, you can specify which subnets should the auto-scaling group use:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Cluster cluster;
 * 
 * cluster.addAutoScalingGroupCapacity("nodes", AutoScalingGroupCapacityOptions.builder()
 *         .vpcSubnets(SubnetSelection.builder().subnets(vpc.getPrivateSubnets()).build())
 *         .instanceType(new InstanceType("t2.medium"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * There are two additional components you might want to provision within the VPC.
 * <p>
 * <h4>Kubectl Handler</h4>
 * <p>
 * The <code>KubectlHandler</code> is a Lambda function responsible to issuing <code>kubectl</code> and <code>helm</code> commands against the cluster when you add resource manifests to the cluster.
 * <p>
 * The handler association to the VPC is derived from the <code>endpointAccess</code> configuration. The rule of thumb is: <em>If the cluster VPC can be associated, it will be</em>.
 * <p>
 * Breaking this down, it means that if the endpoint exposes private access (via <code>EndpointAccess.PRIVATE</code> or <code>EndpointAccess.PUBLIC_AND_PRIVATE</code>), and the VPC contains <strong>private</strong> subnets, the Lambda function will be provisioned inside the VPC and use the private subnets to interact with the cluster. This is the common use-case.
 * <p>
 * If the endpoint does not expose private access (via <code>EndpointAccess.PUBLIC</code>) <strong>or</strong> the VPC does not contain private subnets, the function will not be provisioned within the VPC.
 * <p>
 * If your use-case requires control over the IAM role that the KubeCtl Handler assumes, a custom role can be passed through the ClusterProps (as <code>kubectlLambdaRole</code>) of the EKS Cluster construct.
 * <p>
 * <h4>Cluster Handler</h4>
 * <p>
 * The <code>ClusterHandler</code> is a set of Lambda functions (<code>onEventHandler</code>, <code>isCompleteHandler</code>) responsible for interacting with the EKS API in order to control the cluster lifecycle. To provision these functions inside the VPC, set the <code>placeClusterHandlerInVpc</code> property to <code>true</code>. This will place the functions inside the private subnets of the VPC based on the selection strategy specified in the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-eks.Cluster.html#vpcsubnetsspan-classapi-icon-api-icon-experimental-titlethis-api-element-is-experimental-it-may-change-without-noticespan"><code>vpcSubnets</code></a> property.
 * <p>
 * You can configure the environment of the Cluster Handler functions by specifying it at cluster instantiation. For example, this can be useful in order to configure an http proxy:
 * <p>
 * <blockquote><pre>
 * SecurityGroup proxyInstanceSecurityGroup;
 * 
 * Cluster cluster = Cluster.Builder.create(this, "hello-eks")
 *         .version(KubernetesVersion.V1_21)
 *         .clusterHandlerEnvironment(Map.of(
 *                 "https_proxy", "http://proxy.myproxy.com"))
 *         /**
 *            * If the proxy is not open publicly, you can pass a security group to the
 *            * Cluster Handler Lambdas so that it can reach the proxy.
 *            *&#47;
 *         .clusterHandlerSecurityGroup(proxyInstanceSecurityGroup)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Kubectl Support</h3>
 * <p>
 * The resources are created in the cluster by running <code>kubectl apply</code> from a python lambda function.
 * <p>
 * By default, CDK will create a new python lambda function to apply your k8s manifests. If you want to use an existing kubectl provider function, for example with tight trusted entities on your IAM Roles - you can import the existing provider and then use the imported provider when importing the cluster:
 * <p>
 * <blockquote><pre>
 * IRole handlerRole = Role.fromRoleArn(this, "HandlerRole", "arn:aws:iam::123456789012:role/lambda-role");
 * IKubectlProvider kubectlProvider = KubectlProvider.fromKubectlProviderAttributes(this, "KubectlProvider", KubectlProviderAttributes.builder()
 *         .functionArn("arn:aws:lambda:us-east-2:123456789012:function:my-function:1")
 *         .kubectlRoleArn("arn:aws:iam::123456789012:role/kubectl-role")
 *         .handlerRole(handlerRole)
 *         .build());
 * 
 * ICluster cluster = Cluster.fromClusterAttributes(this, "Cluster", ClusterAttributes.builder()
 *         .clusterName("cluster")
 *         .kubectlProvider(kubectlProvider)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Environment</h4>
 * <p>
 * You can configure the environment of this function by specifying it at cluster instantiation. For example, this can be useful in order to configure an http proxy:
 * <p>
 * <blockquote><pre>
 * Cluster cluster = Cluster.Builder.create(this, "hello-eks")
 *         .version(KubernetesVersion.V1_21)
 *         .kubectlEnvironment(Map.of(
 *                 "http_proxy", "http://proxy.myproxy.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Runtime</h4>
 * <p>
 * The kubectl handler uses <code>kubectl</code>, <code>helm</code> and the <code>aws</code> CLI in order to
 * interact with the cluster. These are bundled into AWS Lambda layers included in
 * the <code>&#64;aws-cdk/lambda-layer-awscli</code> and <code>&#64;aws-cdk/lambda-layer-kubectl</code> modules.
 * <p>
 * The version of kubectl used must be compatible with the Kubernetes version of the
 * cluster. kubectl is supported within one minor version (older or newer) of Kubernetes
 * (see <a href="https://kubernetes.io/releases/version-skew-policy/#kubectl">Kubernetes version skew policy</a>).
 * Only version 1.20 of kubectl is available in <code>aws-cdk-lib</code>. If you need a different
 * version, you will need to use one of the <code>&#64;aws-cdk/lambda-layer-kubectl-vXY</code> packages.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import aws.cdk.lib.lambda.layer.kubectl.v23.KubectlV23Layer;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "hello-eks")
 *         .version(KubernetesVersion.V1_23)
 *         .kubectlLayer(new KubectlV23Layer(this, "kubectl"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify a custom <code>lambda.LayerVersion</code> if you wish to use a
 * different version of these tools, or a version not available in any of the
 * <code>&#64;aws-cdk/lambda-layer-kubectl-vXY</code> packages. The handler expects the layer to
 * include the following two executables:
 * <p>
 * <blockquote><pre>
 * helm/helm
 * kubectl/kubectl
 * </pre></blockquote>
 * <p>
 * See more information in the
 * <a href="https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/lambda-layer-awscli/layer">Dockerfile</a> for &#64;aws-cdk/lambda-layer-awscli
 * and the
 * <a href="https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/lambda-layer-kubectl/layer">Dockerfile</a> for &#64;aws-cdk/lambda-layer-kubectl.
 * <p>
 * <blockquote><pre>
 * LayerVersion layer = LayerVersion.Builder.create(this, "KubectlLayer")
 *         .code(Code.fromAsset("layer.zip"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Now specify when the cluster is defined:
 * <p>
 * <blockquote><pre>
 * LayerVersion layer;
 * Vpc vpc;
 * 
 * 
 * Cluster cluster1 = Cluster.Builder.create(this, "MyCluster")
 *         .kubectlLayer(layer)
 *         .vpc(vpc)
 *         .clusterName("cluster-name")
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * 
 * // or
 * ICluster cluster2 = Cluster.fromClusterAttributes(this, "MyCluster", ClusterAttributes.builder()
 *         .kubectlLayer(layer)
 *         .vpc(vpc)
 *         .clusterName("cluster-name")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Memory</h4>
 * <p>
 * By default, the kubectl provider is configured with 1024MiB of memory. You can use the <code>kubectlMemory</code> option to specify the memory size for the AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * // or
 * Vpc vpc;
 * Cluster.Builder.create(this, "MyCluster")
 *         .kubectlMemory(Size.gibibytes(4))
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * Cluster.fromClusterAttributes(this, "MyCluster", ClusterAttributes.builder()
 *         .kubectlMemory(Size.gibibytes(4))
 *         .vpc(vpc)
 *         .clusterName("cluster-name")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>ARM64 Support</h3>
 * <p>
 * Instance types with <code>ARM64</code> architecture are supported in both managed nodegroup and self-managed capacity. Simply specify an ARM64 <code>instanceType</code> (such as <code>m6g.medium</code>), and the latest
 * Amazon Linux 2 AMI for ARM64 will be automatically selected.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // add a managed ARM64 nodegroup
 * cluster.addNodegroupCapacity("extra-ng-arm", NodegroupOptions.builder()
 *         .instanceTypes(List.of(new InstanceType("m6g.medium")))
 *         .minSize(2)
 *         .build());
 * 
 * // add a self-managed ARM64 nodegroup
 * cluster.addAutoScalingGroupCapacity("self-ng-arm", AutoScalingGroupCapacityOptions.builder()
 *         .instanceType(new InstanceType("m6g.medium"))
 *         .minCapacity(2)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Masters Role</h3>
 * <p>
 * When you create a cluster, you can specify a <code>mastersRole</code>. The <code>Cluster</code> construct will associate this role with the <code>system:masters</code> <a href="https://kubernetes.io/docs/reference/access-authn-authz/rbac/">RBAC</a> group, giving it super-user access to the cluster.
 * <p>
 * <blockquote><pre>
 * Role role;
 * 
 * Cluster.Builder.create(this, "HelloEKS")
 *         .version(KubernetesVersion.V1_21)
 *         .mastersRole(role)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you do not specify it, a default role will be created on your behalf, that can be assumed by anyone in the account with <code>sts:AssumeRole</code> permissions for this role.
 * <p>
 * This is the role you see as part of the stack outputs mentioned in the <a href="#quick-start">Quick Start</a>.
 * <p>
 * <blockquote><pre>
 * $ aws eks update-kubeconfig --name cluster-xxxxx --role-arn arn:aws:iam::112233445566:role/yyyyy
 * Added new context arn:aws:eks:rrrrr:112233445566:cluster/cluster-xxxxx to /home/boom/.kube/config
 * </pre></blockquote>
 * <p>
 * <h3>Encryption</h3>
 * <p>
 * When you create an Amazon EKS cluster, envelope encryption of Kubernetes secrets using the AWS Key Management Service (AWS KMS) can be enabled.
 * The documentation on <a href="https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html">creating a cluster</a>
 * can provide more details about the customer master key (CMK) that can be used for the encryption.
 * <p>
 * You can use the <code>secretsEncryptionKey</code> to configure which key the cluster will use to encrypt Kubernetes secrets. By default, an AWS Managed key will be used.
 * <p>
 * <blockquote>
 * <p>
 * This setting can only be specified when the cluster is created and cannot be updated.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * Key secretsKey = new Key(this, "SecretsKey");
 * Cluster cluster = Cluster.Builder.create(this, "MyCluster")
 *         .secretsEncryptionKey(secretsKey)
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use a similar configuration for running a cluster built using the FargateCluster construct.
 * <p>
 * <blockquote><pre>
 * Key secretsKey = new Key(this, "SecretsKey");
 * FargateCluster cluster = FargateCluster.Builder.create(this, "MyFargateCluster")
 *         .secretsEncryptionKey(secretsKey)
 *         .version(KubernetesVersion.V1_21)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The Amazon Resource Name (ARN) for that CMK can be retrieved.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * String clusterEncryptionConfigKeyArn = cluster.getClusterEncryptionConfigKeyArn();
 * </pre></blockquote>
 * <p>
 * <h2>Permissions and Security</h2>
 * <p>
 * Amazon EKS provides several mechanism of securing the cluster and granting permissions to specific IAM users and roles.
 * <p>
 * <h3>AWS IAM Mapping</h3>
 * <p>
 * As described in the <a href="https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html">Amazon EKS User Guide</a>, you can map AWS IAM users and roles to <a href="https://kubernetes.io/docs/reference/access-authn-authz/rbac">Kubernetes Role-based access control (RBAC)</a>.
 * <p>
 * The Amazon EKS construct manages the <em>aws-auth</em> <code>ConfigMap</code> Kubernetes resource on your behalf and exposes an API through the <code>cluster.awsAuth</code> for mapping
 * users, roles and accounts.
 * <p>
 * Furthermore, when auto-scaling group capacity is added to the cluster, the IAM instance role of the auto-scaling group will be automatically mapped to RBAC so nodes can connect to the cluster. No manual mapping is required.
 * <p>
 * For example, let's say you want to grant an IAM user administrative privileges on your cluster:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * User adminUser = new User(this, "Admin");
 * cluster.awsAuth.addUserMapping(adminUser, AwsAuthMapping.builder().groups(List.of("system:masters")).build());
 * </pre></blockquote>
 * <p>
 * A convenience method for mapping a role to the <code>system:masters</code> group is also available:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * Role role;
 * 
 * cluster.awsAuth.addMastersRole(role);
 * </pre></blockquote>
 * <p>
 * <h3>Cluster Security Group</h3>
 * <p>
 * When you create an Amazon EKS cluster, a <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">cluster security group</a>
 * is automatically created as well. This security group is designed to allow all traffic from the control plane and managed node groups to flow freely
 * between each other.
 * <p>
 * The ID for that security group can be retrieved after creating the cluster.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * String clusterSecurityGroupId = cluster.getClusterSecurityGroupId();
 * </pre></blockquote>
 * <p>
 * <h3>Node SSH Access</h3>
 * <p>
 * If you want to be able to SSH into your worker nodes, you must already have an SSH key in the region you're connecting to and pass it when
 * you add capacity to the cluster. You must also be able to connect to the hosts (meaning they must have a public IP and you
 * should be allowed to connect to them on port 22):
 * <p>
 * See <a href="test/example.ssh-into-nodes.lit.ts">SSH into nodes</a> for a code example.
 * <p>
 * If you want to SSH into nodes in a private subnet, you should set up a bastion host in a public subnet. That setup is recommended, but is
 * unfortunately beyond the scope of this documentation.
 * <p>
 * <h3>Service Accounts</h3>
 * <p>
 * With services account you can provide Kubernetes Pods access to AWS resources.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // add service account
 * ServiceAccount serviceAccount = cluster.addServiceAccount("MyServiceAccount");
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * bucket.grantReadWrite(serviceAccount);
 * 
 * KubernetesManifest mypod = cluster.addManifest("mypod", Map.of(
 *         "apiVersion", "v1",
 *         "kind", "Pod",
 *         "metadata", Map.of("name", "mypod"),
 *         "spec", Map.of(
 *                 "serviceAccountName", serviceAccount.getServiceAccountName(),
 *                 "containers", List.of(Map.of(
 *                         "name", "hello",
 *                         "image", "paulbouwer/hello-kubernetes:1.5",
 *                         "ports", List.of(Map.of("containerPort", 8080)))))));
 * 
 * // create the resource after the service account.
 * mypod.node.addDependency(serviceAccount);
 * 
 * // print the IAM role arn for this service account
 * // print the IAM role arn for this service account
 * CfnOutput.Builder.create(this, "ServiceAccountIamRole").value(serviceAccount.getRole().getRoleArn()).build();
 * </pre></blockquote>
 * <p>
 * Note that using <code>serviceAccount.serviceAccountName</code> above <strong>does not</strong> translate into a resource dependency.
 * This is why an explicit dependency is needed. See <a href="https://github.com/aws/aws-cdk/issues/9910">https://github.com/aws/aws-cdk/issues/9910</a> for more details.
 * <p>
 * It is possible to pass annotations and labels to the service account.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // add service account with annotations and labels
 * ServiceAccount serviceAccount = cluster.addServiceAccount("MyServiceAccount", ServiceAccountOptions.builder()
 *         .annotations(Map.of(
 *                 "eks.amazonaws.com/sts-regional-endpoints", "false"))
 *         .labels(Map.of(
 *                 "some-label", "with-some-value"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also add service accounts to existing clusters.
 * To do so, pass the <code>openIdConnectProvider</code> property when you import the cluster into the application.
 * <p>
 * <blockquote><pre>
 * // or create a new one using an existing issuer url
 * String issuerUrl;
 * // you can import an existing provider
 * IOpenIdConnectProvider provider = OpenIdConnectProvider.fromOpenIdConnectProviderArn(this, "Provider", "arn:aws:iam::123456:oidc-provider/oidc.eks.eu-west-1.amazonaws.com/id/AB123456ABC");
 * OpenIdConnectProvider provider2 = OpenIdConnectProvider.Builder.create(this, "Provider")
 *         .url(issuerUrl)
 *         .build();
 * 
 * ICluster cluster = Cluster.fromClusterAttributes(this, "MyCluster", ClusterAttributes.builder()
 *         .clusterName("Cluster")
 *         .openIdConnectProvider(provider)
 *         .kubectlRoleArn("arn:aws:iam::123456:role/service-role/k8sservicerole")
 *         .build());
 * 
 * ServiceAccount serviceAccount = cluster.addServiceAccount("MyServiceAccount");
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * bucket.grantReadWrite(serviceAccount);
 * </pre></blockquote>
 * <p>
 * Note that adding service accounts requires running <code>kubectl</code> commands against the cluster.
 * This means you must also pass the <code>kubectlRoleArn</code> when importing the cluster.
 * See <a href="https://github.com/aws/aws-cdk/tree/main/packages/&#64;aws-cdk/aws-eks#using-existing-clusters">Using existing Clusters</a>.
 * <p>
 * <h2>Applying Kubernetes Resources</h2>
 * <p>
 * The library supports several popular resource deployment mechanisms, among which are:
 * <p>
 * <h3>Kubernetes Manifests</h3>
 * <p>
 * The <code>KubernetesManifest</code> construct or <code>cluster.addManifest</code> method can be used
 * to apply Kubernetes resource manifests to this cluster.
 * <p>
 * <blockquote>
 * <p>
 * When using <code>cluster.addManifest</code>, the manifest construct is defined within the cluster's stack scope. If the manifest contains
 * attributes from a different stack which depend on the cluster stack, a circular dependency will be created and you will get a synth time error.
 * To avoid this, directly use <code>new KubernetesManifest</code> to create the manifest in the scope of the other stack.
 * <p>
 * </blockquote>
 * <p>
 * The following examples will deploy the <a href="https://github.com/paulbouwer/hello-kubernetes">paulbouwer/hello-kubernetes</a>
 * service on the cluster:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * Map&lt;String, String&gt; appLabel = Map.of("app", "hello-kubernetes");
 * 
 * Map&lt;String, Object&gt; deployment = Map.of(
 *         "apiVersion", "apps/v1",
 *         "kind", "Deployment",
 *         "metadata", Map.of("name", "hello-kubernetes"),
 *         "spec", Map.of(
 *                 "replicas", 3,
 *                 "selector", Map.of("matchLabels", appLabel),
 *                 "template", Map.of(
 *                         "metadata", Map.of("labels", appLabel),
 *                         "spec", Map.of(
 *                                 "containers", List.of(Map.of(
 *                                         "name", "hello-kubernetes",
 *                                         "image", "paulbouwer/hello-kubernetes:1.5",
 *                                         "ports", List.of(Map.of("containerPort", 8080))))))));
 * 
 * Map&lt;String, Object&gt; service = Map.of(
 *         "apiVersion", "v1",
 *         "kind", "Service",
 *         "metadata", Map.of("name", "hello-kubernetes"),
 *         "spec", Map.of(
 *                 "type", "LoadBalancer",
 *                 "ports", List.of(Map.of("port", 80, "targetPort", 8080)),
 *                 "selector", appLabel));
 * 
 * // option 1: use a construct
 * // option 1: use a construct
 * KubernetesManifest.Builder.create(this, "hello-kub")
 *         .cluster(cluster)
 *         .manifest(List.of(deployment, service))
 *         .build();
 * 
 * // or, option2: use `addManifest`
 * cluster.addManifest("hello-kub", service, deployment);
 * </pre></blockquote>
 * <p>
 * <h4>ALB Controller Integration</h4>
 * <p>
 * The <code>KubernetesManifest</code> construct can detect ingress resources inside your manifest and automatically add the necessary annotations
 * so they are picked up by the ALB Controller.
 * <p>
 * <blockquote>
 * <p>
 * See <a href="#alb-controller">Alb Controller</a>
 * <p>
 * </blockquote>
 * <p>
 * To that end, it offers the following properties:
 * <p>
 * <ul>
 * <li><code>ingressAlb</code> - Signal that the ingress detection should be done.</li>
 * <li><code>ingressAlbScheme</code> - Which ALB scheme should be applied. Defaults to <code>internal</code>.</li>
 * </ul>
 * <p>
 * <h4>Adding resources from a URL</h4>
 * <p>
 * The following example will deploy the resource manifest hosting on remote server:
 * <p>
 * <blockquote><pre>
 * // This example is only available in TypeScript
 * 
 * import * as yaml from 'js-yaml';
 * import * as request from 'sync-request';
 * 
 * declare const cluster: eks.Cluster;
 * const manifestUrl = 'https://url/of/manifest.yaml';
 * const manifest = yaml.safeLoadAll(request('GET', manifestUrl).getBody());
 * cluster.addManifest('my-resource', manifest);
 * </pre></blockquote>
 * <p>
 * <h4>Dependencies</h4>
 * <p>
 * There are cases where Kubernetes resources must be deployed in a specific order.
 * For example, you cannot define a resource in a Kubernetes namespace before the
 * namespace was created.
 * <p>
 * You can represent dependencies between <code>KubernetesManifest</code>s using
 * <code>resource.node.addDependency()</code>:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * KubernetesManifest namespace = cluster.addManifest("my-namespace", Map.of(
 *         "apiVersion", "v1",
 *         "kind", "Namespace",
 *         "metadata", Map.of("name", "my-app")));
 * 
 * KubernetesManifest service = cluster.addManifest("my-service", Map.of(
 *         "metadata", Map.of(
 *                 "name", "myservice",
 *                 "namespace", "my-app"),
 *         "spec", Map.of()));
 * 
 * service.node.addDependency(namespace);
 * </pre></blockquote>
 * <p>
 * <strong>NOTE:</strong> when a <code>KubernetesManifest</code> includes multiple resources (either directly
 * or through <code>cluster.addManifest()</code>) (e.g. <code>cluster.addManifest('foo', r1, r2, r3,...)</code>), these resources will be applied as a single manifest via <code>kubectl</code>
 * and will be applied sequentially (the standard behavior in <code>kubectl</code>).
 * <p>
 * <hr>
 * <p>
 * Since Kubernetes manifests are implemented as CloudFormation resources in the
 * CDK. This means that if the manifest is deleted from your code (or the stack is
 * deleted), the next <code>cdk deploy</code> will issue a <code>kubectl delete</code> command and the
 * Kubernetes resources in that manifest will be deleted.
 * <p>
 * <h4>Resource Pruning</h4>
 * <p>
 * When a resource is deleted from a Kubernetes manifest, the EKS module will
 * automatically delete these resources by injecting a <em>prune label</em> to all
 * manifest resources. This label is then passed to <a href="https://kubernetes.io/docs/tasks/manage-kubernetes-objects/declarative-config/#alternative-kubectl-apply-f-directory-prune-l-your-label"><code>kubectl apply --prune</code></a>.
 * <p>
 * Pruning is enabled by default but can be disabled through the <code>prune</code> option
 * when a cluster is defined:
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "MyCluster")
 *         .version(KubernetesVersion.V1_21)
 *         .prune(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Manifests Validation</h4>
 * <p>
 * The <code>kubectl</code> CLI supports applying a manifest by skipping the validation.
 * This can be accomplished by setting the <code>skipValidation</code> flag to <code>true</code> in the <code>KubernetesManifest</code> props.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * KubernetesManifest.Builder.create(this, "HelloAppWithoutValidation")
 *         .cluster(cluster)
 *         .manifest(List.of(Map.of("foo", "bar")))
 *         .skipValidation(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Helm Charts</h3>
 * <p>
 * The <code>HelmChart</code> construct or <code>cluster.addHelmChart</code> method can be used
 * to add Kubernetes resources to this cluster using Helm.
 * <p>
 * <blockquote>
 * <p>
 * When using <code>cluster.addHelmChart</code>, the manifest construct is defined within the cluster's stack scope. If the manifest contains
 * attributes from a different stack which depend on the cluster stack, a circular dependency will be created and you will get a synth time error.
 * To avoid this, directly use <code>new HelmChart</code> to create the chart in the scope of the other stack.
 * <p>
 * </blockquote>
 * <p>
 * The following example will install the <a href="https://kubernetes.github.io/ingress-nginx/">NGINX Ingress Controller</a> to your cluster using Helm.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // option 1: use a construct
 * // option 1: use a construct
 * HelmChart.Builder.create(this, "NginxIngress")
 *         .cluster(cluster)
 *         .chart("nginx-ingress")
 *         .repository("https://helm.nginx.com/stable")
 *         .namespace("kube-system")
 *         .build();
 * 
 * // or, option2: use `addHelmChart`
 * cluster.addHelmChart("NginxIngress", HelmChartOptions.builder()
 *         .chart("nginx-ingress")
 *         .repository("https://helm.nginx.com/stable")
 *         .namespace("kube-system")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Helm charts will be installed and updated using <code>helm upgrade --install</code>, where a few parameters
 * are being passed down (such as <code>repo</code>, <code>values</code>, <code>version</code>, <code>namespace</code>, <code>wait</code>, <code>timeout</code>, etc).
 * This means that if the chart is added to CDK with the same release name, it will try to update
 * the chart in the cluster.
 * <p>
 * Additionally, the <code>chartAsset</code> property can be an <code>aws-s3-assets.Asset</code>. This allows the use of local, private helm charts.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.assets.*;
 * 
 * Cluster cluster;
 * 
 * Asset chartAsset = Asset.Builder.create(this, "ChartAsset")
 *         .path("/path/to/asset")
 *         .build();
 * 
 * cluster.addHelmChart("test-chart", HelmChartOptions.builder()
 *         .chartAsset(chartAsset)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Nested values passed to the <code>values</code> parameter should be provided as a nested dictionary:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * cluster.addHelmChart("ExternalSecretsOperator", Map.of(
 *         "chart", "external-secrets",
 *         "release", "external-secrets",
 *         "repository", "https://charts.external-secrets.io",
 *         "namespace", "external-secrets",
 *         "values", Map.of(
 *                 "installCRDs", true,
 *                 "webhook", Map.of(
 *                         "port", 9443))));
 * </pre></blockquote>
 * <p>
 * <h3>OCI Charts</h3>
 * <p>
 * OCI charts are also supported.
 * Also replace the <code>${VARS}</code> with appropriate values.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // option 1: use a construct
 * // option 1: use a construct
 * HelmChart.Builder.create(this, "MyOCIChart")
 *         .cluster(cluster)
 *         .chart("some-chart")
 *         .repository("oci://${ACCOUNT_ID}.dkr.ecr.${ACCOUNT_REGION}.amazonaws.com/${REPO_NAME}")
 *         .namespace("oci")
 *         .version("0.0.1")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Helm charts are implemented as CloudFormation resources in CDK.
 * This means that if the chart is deleted from your code (or the stack is
 * deleted), the next <code>cdk deploy</code> will issue a <code>helm uninstall</code> command and the
 * Helm chart will be deleted.
 * <p>
 * When there is no <code>release</code> defined, a unique ID will be allocated for the release based
 * on the construct path.
 * <p>
 * By default, all Helm charts will be installed concurrently. In some cases, this
 * could cause race conditions where two Helm charts attempt to deploy the same
 * resource or if Helm charts depend on each other. You can use
 * <code>chart.node.addDependency()</code> in order to declare a dependency order between
 * charts:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * HelmChart chart1 = cluster.addHelmChart("MyChart", HelmChartOptions.builder()
 *         .chart("foo")
 *         .build());
 * HelmChart chart2 = cluster.addHelmChart("MyChart", HelmChartOptions.builder()
 *         .chart("bar")
 *         .build());
 * 
 * chart2.node.addDependency(chart1);
 * </pre></blockquote>
 * <p>
 * <h4>CDK8s Charts</h4>
 * <p>
 * <a href="https://cdk8s.io/">CDK8s</a> is an open-source library that enables Kubernetes manifest authoring using familiar programming languages. It is founded on the same technologies as the AWS CDK, such as <a href="https://github.com/aws/constructs"><code>constructs</code></a> and <a href="https://github.com/aws/jsii"><code>jsii</code></a>.
 * <p>
 * <blockquote>
 * <p>
 * To learn more about cdk8s, visit the <a href="https://cdk8s.io/docs/latest/getting-started/">Getting Started</a> tutorials.
 * <p>
 * </blockquote>
 * <p>
 * The EKS module natively integrates with cdk8s and allows you to apply cdk8s charts on AWS EKS clusters via the <code>cluster.addCdk8sChart</code> method.
 * <p>
 * In addition to <code>cdk8s</code>, you can also use <a href="https://cdk8s.io/docs/latest/plus/"><code>cdk8s+</code></a>, which provides higher level abstraction for the core kubernetes api objects.
 * You can think of it like the <code>L2</code> constructs for Kubernetes. Any other <code>cdk8s</code> based libraries are also supported, for example <a href="https://github.com/toricls/cdk8s-debore"><code>cdk8s-debore</code></a>.
 * <p>
 * To get started, add the following dependencies to your <code>package.json</code> file:
 * <p>
 * <blockquote><pre>
 * "dependencies": {
 *   "cdk8s": "^2.0.0",
 *   "cdk8s-plus-22": "^2.0.0-rc.30",
 *   "constructs": "^10.0.0"
 * }
 * </pre></blockquote>
 * <p>
 * Note that here we are using <code>cdk8s-plus-22</code> as we are targeting Kubernetes version 1.22.0. If you operate a different kubernetes version, you should
 * use the corresponding <code>cdk8s-plus-XX</code> library.
 * See <a href="https://cdk8s.io/docs/latest/plus/#i-operate-kubernetes-version-1xx-which-cdk8s-library-should-i-be-using">Select the appropriate cdk8s+ library</a>
 * for more details.
 * <p>
 * Similarly to how you would create a stack by extending <code>aws-cdk-lib.Stack</code>, we recommend you create a chart of your own that extends <code>cdk8s.Chart</code>,
 * and add your kubernetes resources to it. You can use <code>aws-cdk</code> construct attributes and properties inside your <code>cdk8s</code> construct freely.
 * <p>
 * In this example we create a chart that accepts an <code>s3.Bucket</code> and passes its name to a kubernetes pod as an environment variable.
 * <p>
 * <code>+ my-chart.ts</code>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.s3.*;
 * import software.constructs.*;
 * import org.cdk8s.*;
 * import cdk8s.plus.22.*;
 * 
 * public class MyChartProps {
 *     private Bucket bucket;
 *     public Bucket getBucket() {
 *         return this.bucket;
 *     }
 *     public MyChartProps bucket(Bucket bucket) {
 *         this.bucket = bucket;
 *         return this;
 *     }
 * }
 * 
 * public class MyChart extends Chart {
 *     public MyChart(Construct scope, String id, MyChartProps props) {
 *         super(scope, id);
 * 
 *         Pod.Builder.create(this, "Pod")
 *                 .containers(List.of(Map.of(
 *                         "image", "my-image",
 *                         "env", Map.of(
 *                                 "BUCKET_NAME", EnvValue.fromValue(props.getBucket().getBucketName())))))
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Then, in your AWS CDK app:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Cluster cluster;
 * 
 * 
 * // some bucket..
 * Bucket bucket = new Bucket(this, "Bucket");
 * 
 * // create a cdk8s chart and use `cdk8s.App` as the scope.
 * MyChart myChart = new MyChart(new App(), "MyChart", new MyChartProps().bucket(bucket));
 * 
 * // add the cdk8s chart to the cluster
 * cluster.addCdk8sChart("my-chart", myChart);
 * </pre></blockquote>
 * <p>
 * <h5>Custom CDK8s Constructs</h5>
 * <p>
 * You can also compose a few stock <code>cdk8s+</code> constructs into your own custom construct. However, since mixing scopes between <code>aws-cdk</code> and <code>cdk8s</code> is currently not supported, the <code>Construct</code> class
 * you'll need to use is the one from the <a href="https://github.com/aws/constructs"><code>constructs</code></a> module, and not from <code>&#64;aws-cdk/core</code> like you normally would.
 * This is why we used <code>new cdk8s.App()</code> as the scope of the chart above.
 * <p>
 * <blockquote><pre>
 * import software.constructs.*;
 * import org.cdk8s.*;
 * import org.cdk8s.plus21.*;
 * 
 * public class LoadBalancedWebService {
 *     private Number port;
 *     public Number getPort() {
 *         return this.port;
 *     }
 *     public LoadBalancedWebService port(Number port) {
 *         this.port = port;
 *         return this;
 *     }
 *     private String image;
 *     public String getImage() {
 *         return this.image;
 *     }
 *     public LoadBalancedWebService image(String image) {
 *         this.image = image;
 *         return this;
 *     }
 *     private Number replicas;
 *     public Number getReplicas() {
 *         return this.replicas;
 *     }
 *     public LoadBalancedWebService replicas(Number replicas) {
 *         this.replicas = replicas;
 *         return this;
 *     }
 * }
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "my-chart");
 * 
 * public class LoadBalancedWebService extends Construct {
 *     public LoadBalancedWebService(Construct scope, String id, LoadBalancedWebService props) {
 *         super(scope, id);
 * 
 *         Deployment deployment = Deployment.Builder.create(chart, "Deployment")
 *                 .replicas(props.getReplicas())
 *                 .containers(List.of(Container.Builder.create().image(props.getImage()).build()))
 *                 .build();
 * 
 *         deployment.exposeViaService(ExposeDeploymentViaServiceOptions.builder()
 *                 .port(props.getPort())
 *                 .serviceType(ServiceType.LOAD_BALANCER)
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h5>Manually importing k8s specs and CRD's</h5>
 * <p>
 * If you find yourself unable to use <code>cdk8s+</code>, or just like to directly use the <code>k8s</code> native objects or CRD's, you can do so by manually importing them using the <code>cdk8s-cli</code>.
 * <p>
 * See <a href="https://cdk8s.io/docs/latest/cli/import/">Importing kubernetes objects</a> for detailed instructions.
 * <p>
 * <h2>Patching Kubernetes Resources</h2>
 * <p>
 * The <code>KubernetesPatch</code> construct can be used to update existing kubernetes
 * resources. The following example can be used to patch the <code>hello-kubernetes</code>
 * deployment from the example above with 5 replicas.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * KubernetesPatch.Builder.create(this, "hello-kub-deployment-label")
 *         .cluster(cluster)
 *         .resourceName("deployment/hello-kubernetes")
 *         .applyPatch(Map.of("spec", Map.of("replicas", 5)))
 *         .restorePatch(Map.of("spec", Map.of("replicas", 3)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Querying Kubernetes Resources</h2>
 * <p>
 * The <code>KubernetesObjectValue</code> construct can be used to query for information about kubernetes objects,
 * and use that as part of your CDK application.
 * <p>
 * For example, you can fetch the address of a <a href="https://kubernetes.io/docs/concepts/services-networking/service/#loadbalancer"><code>LoadBalancer</code></a> type service:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // query the load balancer address
 * KubernetesObjectValue myServiceAddress = KubernetesObjectValue.Builder.create(this, "LoadBalancerAttribute")
 *         .cluster(cluster)
 *         .objectType("service")
 *         .objectName("my-service")
 *         .jsonPath(".status.loadBalancer.ingress[0].hostname")
 *         .build();
 * 
 * // pass the address to a lambda function
 * Function proxyFunction = Function.Builder.create(this, "ProxyFunction")
 *         .handler("index.handler")
 *         .code(Code.fromInline("my-code"))
 *         .runtime(Runtime.NODEJS_14_X)
 *         .environment(Map.of(
 *                 "myServiceAddress", myServiceAddress.getValue()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Specifically, since the above use-case is quite common, there is an easier way to access that information:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * String loadBalancerAddress = cluster.getServiceLoadBalancerAddress("my-service");
 * </pre></blockquote>
 * <p>
 * <h2>Using existing clusters</h2>
 * <p>
 * The Amazon EKS library allows defining Kubernetes resources such as <a href="#kubernetes-resources">Kubernetes
 * manifests</a> and <a href="#helm-charts">Helm charts</a> on clusters
 * that are not defined as part of your CDK app.
 * <p>
 * First, you'll need to "import" a cluster to your CDK app. To do that, use the
 * <code>eks.Cluster.fromClusterAttributes()</code> static method:
 * <p>
 * <blockquote><pre>
 * ICluster cluster = Cluster.fromClusterAttributes(this, "MyCluster", ClusterAttributes.builder()
 *         .clusterName("my-cluster-name")
 *         .kubectlRoleArn("arn:aws:iam::1111111:role/iam-role-that-has-masters-access")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Then, you can use <code>addManifest</code> or <code>addHelmChart</code> to define resources inside
 * your Kubernetes cluster. For example:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addManifest("Test", Map.of(
 *         "apiVersion", "v1",
 *         "kind", "ConfigMap",
 *         "metadata", Map.of(
 *                 "name", "myconfigmap"),
 *         "data", Map.of(
 *                 "Key", "value",
 *                 "Another", "123454")));
 * </pre></blockquote>
 * <p>
 * At the minimum, when importing clusters for <code>kubectl</code> management, you will need
 * to specify:
 * <p>
 * <ul>
 * <li><code>clusterName</code> - the name of the cluster.</li>
 * <li><code>kubectlRoleArn</code> - the ARN of an IAM role mapped to the <code>system:masters</code> RBAC
 * role. If the cluster you are importing was created using the AWS CDK, the
 * CloudFormation stack has an output that includes an IAM role that can be used.
 * Otherwise, you can create an IAM role and map it to <code>system:masters</code> manually.
 * The trust policy of this role should include the the
 * <code>arn:aws::iam::${accountId}:root</code> principal in order to allow the execution
 * role of the kubectl resource to assume it.</li>
 * </ul>
 * <p>
 * If the cluster is configured with private-only or private and restricted public
 * Kubernetes <a href="#endpoint-access">endpoint access</a>, you must also specify:
 * <p>
 * <ul>
 * <li><code>kubectlSecurityGroupId</code> - the ID of an EC2 security group that is allowed
 * connections to the cluster's control security group. For example, the EKS managed <a href="#cluster-security-group">cluster security group</a>.</li>
 * <li><code>kubectlPrivateSubnetIds</code> - a list of private VPC subnets IDs that will be used
 * to access the Kubernetes endpoint.</li>
 * </ul>
 * <p>
 * <h2>Logging</h2>
 * <p>
 * EKS supports cluster logging for 5 different types of events:
 * <p>
 * <ul>
 * <li>API requests to the cluster.</li>
 * <li>Cluster access via the Kubernetes API.</li>
 * <li>Authentication requests into the cluster.</li>
 * <li>State of cluster controllers.</li>
 * <li>Scheduling decisions.</li>
 * </ul>
 * <p>
 * You can enable logging for each one separately using the <code>clusterLogging</code>
 * property. For example:
 * <p>
 * <blockquote><pre>
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         // ...
 *         .version(KubernetesVersion.V1_21)
 *         .clusterLogging(List.of(ClusterLoggingTypes.API, ClusterLoggingTypes.AUTHENTICATOR, ClusterLoggingTypes.SCHEDULER))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Known Issues and Limitations</h2>
 * <p>
 * <ul>
 * <li><a href="https://github.com/aws/aws-cdk/issues/10073">One cluster per stack</a></li>
 * <li><a href="https://github.com/aws/aws-cdk/issues/9910">Service Account dependencies</a></li>
 * <li><a href="https://github.com/aws/aws-cdk/issues/12171">Support isolated VPCs</a></li>
 * </ul>
 */
package software.amazon.awscdk.services.eks;
