/**
 * <h2>AWS CodeDeploy Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>AWS CodeDeploy is a deployment service that automates application deployments to
 * Amazon EC2 instances, on-premises instances, serverless Lambda functions, or
 * Amazon ECS services.</p>
 * <p>The CDK currently supports Amazon EC2, on-premise and AWS Lambda applications.</p>
 * <h3>EC2/on-premise Applications</h3>
 * <p>To create a new CodeDeploy Application that deploys to EC2/on-premise instances:</p>
 * <pre><code class="language-ts">// This example is in TypeScript, examples in Java are coming soon.
 * import codedeploy = require('@aws-cdk/aws-codedeploy');
 * 
 * const application = new codedeploy.ServerApplication(this, 'CodeDeployApplication', {
 *     applicationName: 'MyApplication', // optional property
 * });
 * </code></pre>
 * <p>To import an already existing Application:</p>
 * <pre><code class="language-ts">const application = codedeploy.ServerApplication.fromServerApplicationName(
 *   this, 'ExistingCodeDeployApplication', 'MyExistingApplication'
 * );
 * </code></pre>
 * <h3>EC2/on-premise Deployment Groups</h3>
 * <p>To create a new CodeDeploy Deployment Group that deploys to EC2/on-premise instances:</p>
 * <pre><code class="language-ts">const deploymentGroup = new codedeploy.ServerDeploymentGroup(this, 'CodeDeployDeploymentGroup', {
 *     application,
 *     deploymentGroupName: 'MyDeploymentGroup',
 *     autoScalingGroups: [asg1, asg2],
 *     // adds User Data that installs the CodeDeploy agent on your auto-scaling groups hosts
 *     // default: true
 *     installAgent: true,
 *     // adds EC2 instances matching tags
 *     ec2InstanceTags: new codedeploy.InstanceTagSet(
 *         {
 *             // any instance with tags satisfying
 *             // key1=v1 or key1=v2 or key2 (any value) or value v3 (any key)
 *             // will match this group
 *             'key1': ['v1', 'v2'],
 *             'key2': [],
 *             '': ['v3'],
 *         },
 *     ),
 *     // adds on-premise instances matching tags
 *     onPremiseInstanceTags: new codedeploy.InstanceTagSet(
 *         // only instances with tags (key1=v1 or key1=v2) AND key2=v3 will match this set
 *         {
 *             'key1': ['v1', 'v2'],
 *         },
 *         {
 *             'key2': ['v3'],
 *         },
 *     ),
 *     // CloudWatch alarms
 *     alarms: [
 *         new cloudwatch.Alarm(/* ... *{@literal /}),
 *     ],
 *     // whether to ignore failure to fetch the status of alarms from CloudWatch
 *     // default: false
 *     ignorePollAlarmsFailure: false,
 *     // auto-rollback configuration
 *     autoRollback: {
 *         failedDeployment: true, // default: true
 *         stoppedDeployment: true, // default: false
 *         deploymentInAlarm: true, // default: true if you provided any alarms, false otherwise
 *     },
 * });
 * </code></pre>
 * <p>All properties are optional - if you don't provide an Application,
 * one will be automatically created.</p>
 * <p>To import an already existing Deployment Group:</p>
 * <pre><code class="language-ts">const deploymentGroup = codedeploy.ServerDeploymentGroup.fromLambdaDeploymentGroupAttributes(this, 'ExistingCodeDeployDeploymentGroup', {
 *     application,
 *     deploymentGroupName: 'MyExistingDeploymentGroup',
 * });
 * </code></pre>
 * <h4>Load balancers</h4>
 * <p>You can <a href="https://docs.aws.amazon.com/codedeploy/latest/userguide/integrations-aws-elastic-load-balancing.html">specify a load balancer</a>
 * with the <code>loadBalancer</code> property when creating a Deployment Group.</p>
 * <p><code>LoadBalancer</code> is an abstract class with static factory methods that allow you to create instances of it from various sources.</p>
 * <p>With Classic Elastic Load Balancer, you provide it directly:</p>
 * <pre><code class="language-ts">import lb = require('@aws-cdk/aws-elasticloadbalancing');
 * 
 * const elb = new lb.LoadBalancer(this, 'ELB', {
 *   // ...
 * });
 * elb.addTarget(/* ... *{@literal /});
 * elb.addListener({
 *   // ...
 * });
 * 
 * const deploymentGroup = new codedeploy.ServerDeploymentGroup(this, 'DeploymentGroup', {
 *   loadBalancer: codedeploy.LoadBalancer.classic(elb),
 * });
 * </code></pre>
 * <p>With Application Load Balancer or Network Load Balancer,
 * you provide a Target Group as the load balancer:</p>
 * <pre><code class="language-ts">import lbv2 = require('@aws-cdk/aws-elasticloadbalancingv2');
 * 
 * const alb = new lbv2.ApplicationLoadBalancer(this, 'ALB', {
 *   // ...
 * });
 * const listener = alb.addListener('Listener', {
 *   // ...
 * });
 * const targetGroup = listener.addTargets('Fleet', {
 *   // ...
 * });
 * 
 * const deploymentGroup = new codedeploy.ServerDeploymentGroup(this, 'DeploymentGroup', {
 *   loadBalancer: codedeploy.LoadBalancer.application(targetGroup),
 * });
 * </code></pre>
 * <h3>Deployment Configurations</h3>
 * <p>You can also pass a Deployment Configuration when creating the Deployment Group:</p>
 * <pre><code class="language-ts">const deploymentGroup = new codedeploy.ServerDeploymentGroup(this, 'CodeDeployDeploymentGroup', {
 *     deploymentConfig: codedeploy.ServerDeploymentConfig.ALL_AT_ONCE,
 * });
 * </code></pre>
 * <p>The default Deployment Configuration is <code>ServerDeploymentConfig.ONE_AT_A_TIME</code>.</p>
 * <p>You can also create a custom Deployment Configuration:</p>
 * <pre><code class="language-ts">const deploymentConfig = new codedeploy.ServerDeploymentConfig(this, 'DeploymentConfiguration', {
 *     deploymentConfigName: 'MyDeploymentConfiguration', // optional property
 *     // one of these is required, but both cannot be specified at the same time
 *     minHealthyHostCount: 2,
 *     minHealthyHostPercentage: 75,
 * });
 * </code></pre>
 * <p>Or import an existing one:</p>
 * <pre><code class="language-ts">const deploymentConfig = codedeploy.ServerDeploymentConfig.fromServerDeploymentConfigName(
 *   this, 'ExistingDeploymentConfiguration', 'MyExistingDeploymentConfiguration'
 * );
 * </code></pre>
 * <h3>Lambda Applications</h3>
 * <p>To create a new CodeDeploy Application that deploys to a Lambda function:</p>
 * <pre><code class="language-ts">import codedeploy = require('@aws-cdk/aws-codedeploy');
 * 
 * const application = new codedeploy.LambdaApplication(this, 'CodeDeployApplication', {
 *     applicationName: 'MyApplication', // optional property
 * });
 * </code></pre>
 * <p>To import an already existing Application:</p>
 * <pre><code class="language-ts">const application = codedeploy.LambdaApplication.fromLambdaApplicationName(
 *   this, 'ExistingCodeDeployApplication', 'MyExistingApplication'
 * );
 * </code></pre>
 * <h3>Lambda Deployment Groups</h3>
 * <p>To enable traffic shifting deployments for Lambda functions, CodeDeploy uses Lambda Aliases, which can balance incoming traffic between two different versions of your function.
 * Before deployment, the alias sends 100% of invokes to the version used in production.
 * When you publish a new version of the function to your stack, CodeDeploy will send a small percentage of traffic to the new version, monitor, and validate before shifting 100% of traffic to the new version.</p>
 * <p>To create a new CodeDeploy Deployment Group that deploys to a Lambda function:</p>
 * <pre><code class="language-ts">import codedeploy = require('@aws-cdk/aws-codedeploy');
 * import lambda = require('@aws-cdk/aws-lambda');
 * 
 * const myApplication = new codedeploy.LambdaApplication(..);
 * const func = new lambda.Function(..);
 * const version = func.addVersion('1');
 * const version1Alias = new lambda.Alias(this, 'alias', {
 *   aliasName: 'prod',
 *   version
 * });
 * 
 * const deploymentGroup = new codedeploy.LambdaDeploymentGroup(stack, 'BlueGreenDeployment', {
 *   application: myApplication, // optional property: one will be created for you if not provided
 *   alias: version1Alias,
 *   deploymentConfig: codedeploy.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE,
 * });
 * </code></pre>
 * <p>In order to deploy a new version of this function:</p>
 * <ol>
 * <li>Increment the version, e.g. <code>const version = func.addVersion('2')</code>.</li>
 * <li>Re-deploy the stack (this will trigger a deployment).</li>
 * <li>Monitor the CodeDeploy deployment as traffic shifts between the versions.</li>
 * </ol>
 * <h4>Rollbacks and Alarms</h4>
 * <p>CodeDeploy will roll back if the deployment fails. You can optionally trigger a rollback when one or more alarms are in a failed state:</p>
 * <pre><code class="language-ts">const deploymentGroup = new codedeploy.LambdaDeploymentGroup(stack, 'BlueGreenDeployment', {
 *   alias,
 *   deploymentConfig: codedeploy.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE,
 *   alarms: [
 *     // pass some alarms when constructing the deployment group
 *     new cloudwatch.Alarm(stack, 'Errors', {
 *       comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
 *       threshold: 1,
 *       evaluationPeriods: 1,
 *       metric: alias.metricErrors()
 *     })
 *   ]
 * });
 * 
 * // or add alarms to an existing group
 * deploymentGroup.addAlarm(new cloudwatch.Alarm(stack, 'BlueGreenErrors', {
 *   comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
 *   threshold: 1,
 *   evaluationPeriods: 1,
 *   metric: blueGreenAlias.metricErrors()
 * }));
 * </code></pre>
 * <h4>Pre and Post Hooks</h4>
 * <p>CodeDeploy allows you to run an arbitrary Lambda function before traffic shifting actually starts (PreTraffic Hook) and after it completes (PostTraffic Hook).
 * With either hook, you have the opportunity to run logic that determines whether the deployment must succeed or fail.
 * For example, with PreTraffic hook you could run integration tests against the newly created Lambda version (but not serving traffic). With PostTraffic hook, you could run end-to-end validation checks.</p>
 * <pre><code class="language-ts">const warmUpUserCache = new lambda.Function(..);
 * const endToEndValidation = new lambda.Function(..);
 * 
 * // pass a hook whe creating the deployment group
 * const deploymentGroup = new codedeploy.LambdaDeploymentGroup(stack, 'BlueGreenDeployment', {
 *   alias: alias,
 *   deploymentConfig: codedeploy.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE,
 *   preHook: warmUpUserCache,
 * });
 * 
 * // or configure one on an existing deployment group
 * deploymentGroup.onPostHook(endToEndValidation);
 * </code></pre>
 * <h4>Import an existing Deployment Group</h4>
 * <p>To import an already existing Deployment Group:</p>
 * <pre><code class="language-ts">const deploymentGroup = codedeploy.LambdaDeploymentGroup.import(this, 'ExistingCodeDeployDeploymentGroup', {
 *     application,
 *     deploymentGroupName: 'MyExistingDeploymentGroup',
 * });
 * </code></pre>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.codedeploy;
