/**
 * <h2>Amazon DynamoDB Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Here is a minimal deployable DynamoDB table definition:</p>
 * <pre><code class="language-ts">import dynamodb = require('@aws-cdk/aws-dynamodb');
 * 
 * const table = new dynamodb.Table(this, 'Table', {
 *   partitionKey: { name: 'id', type: dynamodb.AttributeType.STRING }
 * });
 * </code></pre>
 * <h3>Keys</h3>
 * <p>When a table is defined, you must define it's schema using the <code>partitionKey</code>
 * (required) and <code>sortKey</code> (optional) properties.</p>
 * <h3>Billing Mode</h3>
 * <p>DynamoDB supports two billing modes:</p>
 * <ul>
 * <li>PROVISIONED - the default mode where the table and global secondary indexes have configured read and write capacity.</li>
 * <li>PAY_PER_REQUEST - on-demand pricing and scaling. You only pay for what you use and there is no read and write capacity for the table or its global secondary indexes.</li>
 * </ul>
 * <pre><code class="language-ts">import dynamodb = require('@aws-cdk/aws-dynamodb');
 * 
 * const table = new dynamodb.Table(this, 'Table', {
 *   partitionKey: { name: 'id', type: dynamodb.AttributeType.STRING },
 *   billingMode: dynamodb.BillingMode.PAY_PER_REQUEST
 * });
 * </code></pre>
 * <p>Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.ReadWriteCapacityMode.</p>
 * <h3>Configure AutoScaling for your table</h3>
 * <p>You can have DynamoDB automatically raise and lower the read and write capacities
 * of your table by setting up autoscaling. You can use this to either keep your
 * tables at a desired utilization level, or by scaling up and down at preconfigured
 * times of the day:</p>
 * <p>Auto-scaling is only relevant for tables with the billing mode, PROVISIONED.</p>
 * <pre><code class="language-ts">const readScaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 50 });
 * 
 * readScaling.scaleOnUtilization({
 *   targetUtilizationPercent: 50
 * });
 * 
 * readScaling.scaleOnSchedule('ScaleUpInTheMorning', {
 *   schedule: appscaling.Schedule.cron({ hour: '8', minute: '0' }),
 *   minCapacity: 20,
 * });
 * 
 * readScaling.scaleOnSchedule('ScaleDownAtNight', {
 *   schedule: appscaling.Schedule.cron({ hour: '20', minute: '0' }),
 *   maxCapacity: 20
 * });
 * </code></pre>
 * <p>Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/AutoScaling.html
 * https://aws.amazon.com/blogs/database/how-to-use-aws-cloudformation-to-configure-auto-scaling-for-amazon-dynamodb-tables-and-indexes/</p>
 * <h3>Amazon DynamoDB Global Tables</h3>
 * <p>Please see the <code>@aws-cdk/aws-dynamodb-global</code> package.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.dynamodb;
