/**
 * <h1>Amazon EC2 Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * The <code>&#64;aws-cdk/aws-ec2</code> package contains primitives for setting up networking and
 * instances.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.ec2.*;
 * </pre></blockquote>
 * <p>
 * <h2>VPC</h2>
 * <p>
 * Most projects need a Virtual Private Cloud to provide security by means of
 * network partitioning. This is achieved by creating an instance of
 * <code>Vpc</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = new Vpc(this, "VPC");
 * </pre></blockquote>
 * <p>
 * All default constructs require EC2 instances to be launched inside a VPC, so
 * you should generally start by defining a VPC whenever you need to launch
 * instances for your project.
 * <p>
 * <h3>Subnet Types</h3>
 * <p>
 * A VPC consists of one or more subnets that instances can be placed into. CDK
 * distinguishes three different subnet types:
 * <p>
 * <ul>
 * <li><strong>Public (<code>SubnetType.PUBLIC</code>)</strong> - public subnets connect directly to the Internet using an
 * Internet Gateway. If you want your instances to have a public IP address
 * and be directly reachable from the Internet, you must place them in a
 * public subnet.</li>
 * <li><strong>Private with Internet Access (<code>SubnetType.PRIVATE_WITH_NAT</code>)</strong> - instances in private subnets are not directly routable from the
 * Internet, and connect out to the Internet via a NAT gateway. By default, a
 * NAT gateway is created in every public subnet for maximum availability. Be
 * aware that you will be charged for NAT gateways.</li>
 * <li><strong>Isolated (<code>SubnetType.PRIVATE_ISOLATED</code>)</strong> - isolated subnets do not route from or to the Internet, and
 * as such do not require NAT gateways. They can only connect to or be
 * connected to from other instances in the same VPC. A default VPC configuration
 * will not include isolated subnets,</li>
 * </ul>
 * <p>
 * A default VPC configuration will create public and <strong>private</strong> subnets. However, if
 * <code>natGateways:0</code> <strong>and</strong> <code>subnetConfiguration</code> is undefined, default VPC configuration
 * will create public and <strong>isolated</strong> subnets. See <a href="#advanced-subnet-configuration"><em>Advanced Subnet Configuration</em></a>
 * below for information on how to change the default subnet configuration.
 * <p>
 * Constructs using the VPC will "launch instances" (or more accurately, create
 * Elastic Network Interfaces) into one or more of the subnets. They all accept
 * a property called <code>subnetSelection</code> (sometimes called <code>vpcSubnets</code>) to allow
 * you to select in what subnet to place the ENIs, usually defaulting to
 * <em>private</em> subnets if the property is omitted.
 * <p>
 * If you would like to save on the cost of NAT gateways, you can use
 * <em>isolated</em> subnets instead of <em>private</em> subnets (as described in Advanced
 * <em>Subnet Configuration</em>). If you need private instances to have
 * internet connectivity, another option is to reduce the number of NAT gateways
 * created by setting the <code>natGateways</code> property to a lower value (the default
 * is one NAT gateway per availability zone). Be aware that this may have
 * availability implications for your application.
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html">Read more about
 * subnets</a>.
 * <p>
 * <h3>Control over availability zones</h3>
 * <p>
 * By default, a VPC will spread over at most 3 Availability Zones available to
 * it. To change the number of Availability Zones that the VPC will spread over,
 * specify the <code>maxAzs</code> property when defining it.
 * <p>
 * The number of Availability Zones that are available depends on the <em>region</em>
 * and <em>account</em> of the Stack containing the VPC. If the <a href="https://docs.aws.amazon.com/cdk/latest/guide/environments.html">region and account are
 * specified</a> on
 * the Stack, the CLI will <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#using-regions-availability-zones-describe">look up the existing Availability
 * Zones</a>
 * and get an accurate count. If region and account are not specified, the stack
 * could be deployed anywhere and it will have to make a safe choice, limiting
 * itself to 2 Availability Zones.
 * <p>
 * Therefore, to get the VPC to spread over 3 or more availability zones, you
 * must specify the environment where the stack will be deployed.
 * <p>
 * You can gain full control over the availability zones selection strategy by overriding the Stack's <a href="https://github.com/aws/aws-cdk/blob/master/packages/&#64;aws-cdk/core/lib/stack.ts"><code>get availabilityZones()</code></a> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * public class MyStack extends Stack {get availabilityZones(): string[] {
 *         return ['us-west-2a', 'us-west-2b'];
 *       }
 * 
 *     public MyStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Note that overriding the <code>get availabilityZones()</code> method will override the default behavior for all constructs defined within the Stack.
 * <p>
 * <h3>Choosing subnets for resources</h3>
 * <p>
 * When creating resources that create Elastic Network Interfaces (such as
 * databases or instances), there is an option to choose which subnets to place
 * them in. For example, a VPC endpoint by default is placed into a subnet in
 * every availability zone, but you can override which subnets to use. The property
 * is typically called one of <code>subnets</code>, <code>vpcSubnets</code> or <code>subnetSelection</code>.
 * <p>
 * The example below will place the endpoint into two AZs (<code>us-east-1a</code> and <code>us-east-1c</code>),
 * in Isolated subnets:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * InterfaceVpcEndpoint.Builder.create(stack, "VPC Endpoint")
 *         .vpc(vpc)
 *         .service(new InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443))
 *         .subnets(Map.of(
 *                 "subnetType", SubnetType.getISOLATED(),
 *                 "availabilityZones", asList("us-east-1a", "us-east-1c")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify specific subnet objects for granular control:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * InterfaceVpcEndpoint.Builder.create(stack, "VPC Endpoint")
 *         .vpc(vpc)
 *         .service(new InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443))
 *         .subnets(Map.of(
 *                 "subnets", asList(subnet1, subnet2)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Which subnets are selected is evaluated as follows:
 * <p>
 * <ul>
 * <li><code>subnets</code>: if specific subnet objects are supplied, these are selected, and no other
 * logic is used.</li>
 * <li><code>subnetType</code>/<code>subnetGroupName</code>: otherwise, a set of subnets is selected by
 * supplying either type or name:
 * <p>
 * <ul>
 * <li><code>subnetType</code> will select all subnets of the given type.</li>
 * <li><code>subnetGroupName</code> should be used to distinguish between multiple groups of subnets of
 * the same type (for example, you may want to separate your application instances and your
 * RDS instances into two distinct groups of Isolated subnets).</li>
 * <li>If neither are given, the first available subnet group of a given type that
 * exists in the VPC will be used, in this order: Private, then Isolated, then Public.
 * In short: by default ENIs will preferentially be placed in subnets not connected to
 * the Internet.</li>
 * </ul></li>
 * <li><code>availabilityZones</code>/<code>onePerAz</code>: finally, some availability-zone based filtering may be done.
 * This filtering by availability zones will only be possible if the VPC has been created or
 * looked up in a non-environment agnostic stack (so account and region have been set and
 * availability zones have been looked up).
 * <p>
 * <ul>
 * <li><code>availabilityZones</code>: only the specific subnets from the selected subnet groups that are
 * in the given availability zones will be returned.</li>
 * <li><code>onePerAz</code>: per availability zone, a maximum of one subnet will be returned (Useful for resource
 * types that do not allow creating two ENIs in the same availability zone).</li>
 * </ul></li>
 * <li><code>subnetFilters</code>: additional filtering on subnets using any number of user-provided filters which
 * extend <code>SubnetFilter</code>.  The following methods on the <code>SubnetFilter</code> class can be used to create
 * a filter:
 * <p>
 * <ul>
 * <li><code>byIds</code>: chooses subnets from a list of ids</li>
 * <li><code>availabilityZones</code>: chooses subnets in the provided list of availability zones</li>
 * <li><code>onePerAz</code>: chooses at most one subnet per availability zone</li>
 * <li><code>containsIpAddresses</code>: chooses a subnet which contains <em>any</em> of the listed ip addresses</li>
 * <li><code>byCidrMask</code>: chooses subnets that have the provided CIDR netmask</li>
 * </ul></li>
 * </ul>
 * <p>
 * <h3>Using NAT instances</h3>
 * <p>
 * By default, the <code>Vpc</code> construct will create NAT <em>gateways</em> for you, which
 * are managed by AWS. If you would prefer to use your own managed NAT
 * <em>instances</em> instead, specify a different value for the <code>natGatewayProvider</code>
 * property, as follows:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Configure the `natGatewayProvider` when defining a Vpc
 * NatInstanceProvider natGatewayProvider = ec2.NatProvider.instance(new NatInstanceProps()
 *         .instanceType(new InstanceType("t3.small")));
 * 
 * Vpc vpc = new Vpc(this, "MyVpc", new VpcProps()
 *         .natGatewayProvider(natGatewayProvider)
 * 
 *         // The 'natGateways' parameter now controls the number of NAT instances
 *         .natGateways(2));
 * </pre></blockquote>
 * <p>
 * The construct will automatically search for the most recent NAT gateway AMI.
 * If you prefer to use a custom AMI, use <code>machineImage: MachineImage.genericLinux({ ... })</code> and configure the right AMI ID for the
 * regions you want to deploy to.
 * <p>
 * By default, the NAT instances will route all traffic. To control what traffic
 * gets routed, pass <code>allowAllTraffic: false</code> and access the
 * <code>NatInstanceProvider.connections</code> member after having passed it to the VPC:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object provider = NatProvider.instance(Map.of(
 *         "instanceType", ,
 *         "allowAllTraffic", false));
 * Vpc.Builder.create(stack, "TheVPC")
 *         .natGatewayProvider(provider)
 *         .build();
 * provider.connections.allowFrom(Peer.ipv4("1.2.3.4/8"), Port.tcp(80));
 * </pre></blockquote>
 * <p>
 * <h3>Advanced Subnet Configuration</h3>
 * <p>
 * If the default VPC configuration (public and private subnets spanning the
 * size of the VPC) don't suffice for you, you can configure what subnets to
 * create by specifying the <code>subnetConfiguration</code> property. It allows you
 * to configure the number and size of all subnets. Specifying an advanced
 * subnet configuration could look like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = Vpc.Builder.create(this, "TheVPC")
 *         // 'cidr' configures the IP range and size of the entire VPC.
 *         // The IP space will be divided over the configured subnets.
 *         .cidr("10.0.0.0/21")
 * 
 *         // 'maxAzs' configures the maximum number of availability zones to use
 *         .maxAzs(3)
 * 
 *         // 'subnetConfiguration' specifies the "subnet groups" to create.
 *         // Every subnet group will have a subnet for each AZ, so this
 *         // configuration will create `3 groups × 3 AZs = 9` subnets.
 *         .subnetConfiguration(asList(Map.of(
 *                 // 'subnetType' controls Internet access, as described above.
 *                 "subnetType", ec2.SubnetType.getPUBLIC(),
 * 
 *                 // 'name' is used to name this particular subnet group. You will have to
 *                 // use the name for subnet selection if you have more than one subnet
 *                 // group of the same type.
 *                 "name", "Ingress",
 * 
 *                 // 'cidrMask' specifies the IP addresses in the range of of individual
 *                 // subnets in the group. Each of the subnets in this group will contain
 *                 // `2^(32 address bits - 24 subnet bits) - 2 reserved addresses = 254`
 *                 // usable IP addresses.
 *                 //
 *                 // If 'cidrMask' is left out the available address space is evenly
 *                 // divided across the remaining subnet groups.
 *                 "cidrMask", 24), Map.of(
 *                 "cidrMask", 24,
 *                 "name", "Application",
 *                 "subnetType", ec2.SubnetType.getPRIVATE_WITH_NAT()), Map.of(
 *                 "cidrMask", 28,
 *                 "name", "Database",
 *                 "subnetType", ec2.SubnetType.getPRIVATE_ISOLATED(),
 * 
 *                 // 'reserved' can be used to reserve IP address space. No resources will
 *                 // be created for this subnet, but the IP range will be kept available for
 *                 // future creation of this subnet, or even for future subdivision.
 *                 "reserved", true)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The example above is one possible configuration, but the user can use the
 * constructs above to implement many other network configurations.
 * <p>
 * The <code>Vpc</code> from the above configuration in a Region with three
 * availability zones will be the following:
 * <p>
 * Subnet Name       |Type      |IP Block      |AZ|Features
 * ------------------|----------|--------------|--|--------
 * IngressSubnet1    |<code>PUBLIC</code>  |<code>10.0.0.0/24</code> |#1|NAT Gateway
 * IngressSubnet2    |<code>PUBLIC</code>  |<code>10.0.1.0/24</code> |#2|NAT Gateway
 * IngressSubnet3    |<code>PUBLIC</code>  |<code>10.0.2.0/24</code> |#3|NAT Gateway
 * ApplicationSubnet1|<code>PRIVATE</code> |<code>10.0.3.0/24</code> |#1|Route to NAT in IngressSubnet1
 * ApplicationSubnet2|<code>PRIVATE</code> |<code>10.0.4.0/24</code> |#2|Route to NAT in IngressSubnet2
 * ApplicationSubnet3|<code>PRIVATE</code> |<code>10.0.5.0/24</code> |#3|Route to NAT in IngressSubnet3
 * DatabaseSubnet1   |<code>ISOLATED</code>|<code>10.0.6.0/28</code> |#1|Only routes within the VPC
 * DatabaseSubnet2   |<code>ISOLATED</code>|<code>10.0.6.16/28</code>|#2|Only routes within the VPC
 * DatabaseSubnet3   |<code>ISOLATED</code>|<code>10.0.6.32/28</code>|#3|Only routes within the VPC
 * <p>
 * <h3>Accessing the Internet Gateway</h3>
 * <p>
 * If you need access to the internet gateway, you can get its ID like so:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object igwId = vpc.getInternetGatewayId();
 * </pre></blockquote>
 * <p>
 * For a VPC with only <code>ISOLATED</code> subnets, this value will be undefined.
 * <p>
 * This is only supported for VPCs created in the stack - currently you're
 * unable to get the ID for imported VPCs. To do that you'd have to specifically
 * look up the Internet Gateway by name, which would require knowing the name
 * beforehand.
 * <p>
 * This can be useful for configuring routing using a combination of gateways:
 * for more information see <a href="#routing">Routing</a> below.
 * <p>
 * <h4>Routing</h4>
 * <p>
 * It's possible to add routes to any subnets using the <code>addRoute()</code> method. If for
 * example you want an isolated subnet to have a static route via the default
 * Internet Gateway created for the public subnet - perhaps for routing a VPN
 * connection - you can do so like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object vpc = ec2.Vpc(this, "VPC", {
 *   subnetConfiguration: [{
 *       subnetType: SubnetType.PUBLIC,
 *       name: 'Public',
 *     },{
 *       subnetType: SubnetType.ISOLATED,
 *       name: 'Isolated',
 *     }]
 * })
 * (vpc.isolatedSubnets[0] as Subnet).addRoute("StaticRoute", Map.of(
 *         "routerId", vpc.getInternetGatewayId(),
 *         "routerType", RouterType.getGATEWAY(),
 *         "destinationCidrBlock", "8.8.8.8/32"));
 * </pre></blockquote>
 * <p>
 * <em>Note that we cast to <code>Subnet</code> here because the list of subnets only returns an
 * <code>ISubnet</code>.</em>
 * <p>
 * <h3>Reserving subnet IP space</h3>
 * <p>
 * There are situations where the IP space for a subnet or number of subnets
 * will need to be reserved. This is useful in situations where subnets would
 * need to be added after the vpc is originally deployed, without causing IP
 * renumbering for existing subnets. The IP space for a subnet may be reserved
 * by setting the <code>reserved</code> subnetConfiguration property to true, as shown
 * below:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = Vpc.Builder.create(this, "TheVPC")
 *         .natGateways(1)
 *         .subnetConfiguration(asList(Map.of(
 *                 "cidrMask", 26,
 *                 "name", "Public",
 *                 "subnetType", ec2.SubnetType.getPUBLIC()), Map.of(
 *                 "cidrMask", 26,
 *                 "name", "Application1",
 *                 "subnetType", ec2.SubnetType.getPRIVATE_WITH_NAT()), Map.of(
 *                 "cidrMask", 26,
 *                 "name", "Application2",
 *                 "subnetType", ec2.SubnetType.getPRIVATE_WITH_NAT(),
 *                 "reserved", true), Map.of(
 *                 "cidrMask", 27,
 *                 "name", "Database",
 *                 "subnetType", ec2.SubnetType.getISOLATED())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * In the example above, the subnet for Application2 is not actually provisioned
 * but its IP space is still reserved. If in the future this subnet needs to be
 * provisioned, then the <code>reserved: true</code> property should be removed. Reserving
 * parts of the IP space prevents the other subnets from getting renumbered.
 * <p>
 * <h3>Sharing VPCs between stacks</h3>
 * <p>
 * If you are creating multiple <code>Stack</code>s inside the same CDK application, you
 * can reuse a VPC defined in one Stack in another by simply passing the VPC
 * instance around:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * /**
 *  * Stack1 creates the VPC
 *  *{@literal /}
 * public class Stack1 extends Stack {
 *     public final Vpc vpc;
 * 
 *     public Stack1(App scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public Stack1(App scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         this.vpc = new Vpc(this, "VPC");
 *     }
 * }
 * 
 * public class Stack2Props extends StackProps {
 *     private IVpc vpc;
 *     public IVpc getVpc() {
 *         return this.vpc;
 *     }
 *     public Stack2Props vpc(IVpc vpc) {
 *         this.vpc = vpc;
 *         return this;
 *     }
 * }
 * 
 * /**
 *  * Stack2 consumes the VPC
 *  *{@literal /}
 * public class Stack2 extends Stack {
 *     public Stack2(App scope, String id, Stack2Props props) {
 *         super(scope, id, props);
 * 
 *         // Pass the VPC to a construct that needs it
 *         // Pass the VPC to a construct that needs it
 *         new ConstructThatTakesAVpc(this, "Construct", new ConstructThatTakesAVpcProps()
 *                 .vpc(props.getVpc()));
 *     }
 * }
 * 
 * Stack1 stack1 = new Stack1(app, "Stack1");
 * Stack2 stack2 = new Stack2(app, "Stack2", new Stack2Props()
 *         .vpc(stack1.getVpc()));
 * </pre></blockquote>
 * <p>
 * <h3>Importing an existing VPC</h3>
 * <p>
 * If your VPC is created outside your CDK app, you can use <code>Vpc.fromLookup()</code>.
 * The CDK CLI will search for the specified VPC in the the stack's region and
 * account, and import the subnet configuration. Looking up can be done by VPC
 * ID, but more flexibly by searching for a specific tag on the VPC.
 * <p>
 * Subnet types will be determined from the <code>aws-cdk:subnet-type</code> tag on the
 * subnet if it exists, or the presence of a route to an Internet Gateway
 * otherwise. Subnet names will be determined from the <code>aws-cdk:subnet-name</code> tag
 * on the subnet if it exists, or will mirror the subnet type otherwise (i.e.
 * a public subnet will have the name <code>"Public"</code>).
 * <p>
 * The result of the <code>Vpc.fromLookup()</code> operation will be written to a file
 * called <code>cdk.context.json</code>. You must commit this file to source control so
 * that the lookup values are available in non-privileged environments such
 * as CI build steps, and to ensure your template builds are repeatable.
 * <p>
 * Here's how <code>Vpc.fromLookup()</code> can be used:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * IVpc vpc = ec2.Vpc.fromLookup(stack, "VPC", new VpcLookupOptions()
 *         // This imports the default VPC but you can also
 *         // specify a 'vpcName' or 'tags'.
 *         .isDefault(true));
 * </pre></blockquote>
 * <p>
 * <code>Vpc.fromLookup</code> is the recommended way to import VPCs. If for whatever
 * reason you do not want to use the context mechanism to look up a VPC at
 * synthesis time, you can also use <code>Vpc.fromVpcAttributes</code>. This has the
 * following limitations:
 * <p>
 * <ul>
 * <li>Every subnet group in the VPC must have a subnet in each availability zone
 * (for example, each AZ must have both a public and private subnet). Asymmetric
 * VPCs are not supported.</li>
 * <li>All VpcId, SubnetId, RouteTableId, ... parameters must either be known at
 * synthesis time, or they must come from deploy-time list parameters whose
 * deploy-time lengths are known at synthesis time.</li>
 * </ul>
 * <p>
 * Using <code>Vpc.fromVpcAttributes()</code> looks like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object vpc = ec2.Vpc.fromVpcAttributes(stack, "VPC", Map.of(
 *         "vpcId", "vpc-1234",
 *         "availabilityZones", asList("us-east-1a", "us-east-1b"),
 * 
 *         // Either pass literals for all IDs
 *         "publicSubnetIds", asList("s-12345", "s-67890"),
 * 
 *         // OR: import a list of known length
 *         "privateSubnetIds", Fn.importListValue("PrivateSubnetIds", 2),
 * 
 *         // OR: split an imported string to a list of known length
 *         "isolatedSubnetIds", Fn.split(",", ssm.StringParameter.valueForStringParameter(stack, "MyParameter"), 2)));
 * </pre></blockquote>
 * <p>
 * <h2>Allowing Connections</h2>
 * <p>
 * In AWS, all network traffic in and out of <strong>Elastic Network Interfaces</strong> (ENIs)
 * is controlled by <strong>Security Groups</strong>. You can think of Security Groups as a
 * firewall with a set of rules. By default, Security Groups allow no incoming
 * (ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
 * to them to allow incoming traffic streams. To exert fine-grained control over
 * egress traffic, set <code>allowAllOutbound: false</code> on the <code>SecurityGroup</code>, after
 * which you can add egress traffic rules.
 * <p>
 * You can manipulate Security Groups directly:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SecurityGroup mySecurityGroup = new SecurityGroup(this, "SecurityGroup", new SecurityGroupProps()
 *         .vpc(vpc)
 *         .description("Allow ssh access to ec2 instances")
 *         .allowAllOutbound(true));
 * mySecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), "allow ssh access from the world");
 * </pre></blockquote>
 * <p>
 * All constructs that create ENIs on your behalf (typically constructs that create
 * EC2 instances or other VPC-connected resources) will all have security groups
 * automatically assigned. Those constructs have an attribute called
 * <strong>connections</strong>, which is an object that makes it convenient to update the
 * security groups. If you want to allow connections between two constructs that
 * have security groups, you have to add an <strong>Egress</strong> rule to one Security Group,
 * and an <strong>Ingress</strong> rule to the other. The connections object will automatically
 * take care of this for you:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Allow connections from anywhere
 * loadBalancer.connections.allowFromAnyIpv4(ec2.Port.tcp(443), "Allow inbound HTTPS");
 * 
 * // The same, but an explicit IP address
 * loadBalancer.connections.allowFrom(ec2.Peer.ipv4("1.2.3.4/32"), ec2.Port.tcp(443), "Allow inbound HTTPS");
 * 
 * // Allow connection between AutoScalingGroups
 * appFleet.connections.allowTo(dbFleet, ec2.Port.tcp(443), "App can call database");
 * </pre></blockquote>
 * <p>
 * <h3>Connection Peers</h3>
 * <p>
 * There are various classes that implement the connection peer part:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Simple connection peers
 * IPeer peer = ec2.Peer.ipv4("10.0.0.0/16");
 * peer = ec2.Peer.anyIpv4();
 * peer = ec2.Peer.ipv6("::0/0");
 * peer = ec2.Peer.anyIpv6();
 * peer = ec2.Peer.prefixList("pl-12345");
 * appFleet.connections.allowTo(peer, ec2.Port.tcp(443), "Allow outbound HTTPS");
 * </pre></blockquote>
 * <p>
 * Any object that has a security group can itself be used as a connection peer:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // These automatically create appropriate ingress and egress rules in both security groups
 * fleet1.connections.allowTo(fleet2, ec2.Port.tcp(80), "Allow between fleets");
 * 
 * appFleet.connections.allowFromAnyIpv4(ec2.Port.tcp(80), "Allow from load balancer");
 * </pre></blockquote>
 * <p>
 * <h3>Port Ranges</h3>
 * <p>
 * The connections that are allowed are specified by port ranges. A number of classes provide
 * the connection specifier:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * ec2.Port.tcp(80);
 * ec2.Port.tcpRange(60000, 65535);
 * ec2.Port.allTcp();
 * ec2.Port.allTraffic();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: This set is not complete yet; for example, there is no library support for ICMP at the moment.
 * However, you can write your own classes to implement those.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Default Ports</h3>
 * <p>
 * Some Constructs have default ports associated with them. For example, the
 * listener of a load balancer does (it's the public port), or instances of an
 * RDS database (it's the port the database is accepting connections on).
 * <p>
 * If the object you're calling the peering method on has a default port associated with it, you can call
 * <code>allowDefaultPortFrom()</code> and omit the port specifier. If the argument has an associated default port, call
 * <code>allowDefaultPortTo()</code>.
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Port implicit in listener
 * listener.connections.allowDefaultPortFromAnyIpv4("Allow public");
 * 
 * // Port implicit in peer
 * appFleet.connections.allowDefaultPortTo(rdsDatabase, "Fleet can access database");
 * </pre></blockquote>
 * <p>
 * <h3>Security group rules</h3>
 * <p>
 * By default, security group wills be added inline to the security group in the output cloud formation
 * template, if applicable.  This includes any static rules by ip address and port range.  This
 * optimization helps to minimize the size of the template.
 * <p>
 * In some environments this is not desirable, for example if your security group access is controlled
 * via tags. You can disable inline rules per security group or globally via the context key
 * <code>&#64;aws-cdk/aws-ec2.securityGroupDisableInlineRules</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SecurityGroup mySecurityGroupWithoutInlineRules = new SecurityGroup(this, "SecurityGroup", new SecurityGroupProps()
 *         .vpc(vpc)
 *         .description("Allow ssh access to ec2 instances")
 *         .allowAllOutbound(true)
 *         .disableInlineRules(true));
 * //This will add the rule as an external cloud formation construct
 * mySecurityGroupWithoutInlineRules.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), "allow ssh access from the world");
 * </pre></blockquote>
 * <p>
 * <h2>Machine Images (AMIs)</h2>
 * <p>
 * AMIs control the OS that gets launched when you start your EC2 instance. The EC2
 * library contains constructs to select the AMI you want to use.
 * <p>
 * Depending on the type of AMI, you select it a different way. Here are some
 * examples of things you might want to use:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Pick the right Amazon Linux edition. All arguments shown are optional
 * // and will default to these values when omitted.
 * IMachineImage amznLinux = ec2.MachineImage.latestAmazonLinux(new AmazonLinuxImageProps()
 *         .generation(ec2.AmazonLinuxGeneration.getAMAZON_LINUX())
 *         .edition(ec2.AmazonLinuxEdition.getSTANDARD())
 *         .virtualization(ec2.AmazonLinuxVirt.getHVM())
 *         .storage(ec2.AmazonLinuxStorage.getGENERAL_PURPOSE())
 *         .cpuType(ec2.AmazonLinuxCpuType.getX86_64()));
 * 
 * // Pick a Windows edition to use
 * IMachineImage windows = ec2.MachineImage.latestWindows(ec2.WindowsVersion.getWINDOWS_SERVER_2019_ENGLISH_FULL_BASE());
 * 
 * // Read AMI id from SSM parameter store
 * IMachineImage ssm = ec2.MachineImage.fromSSMParameter("/my/ami", ec2.OperatingSystemType.getLINUX());
 * 
 * // Look up the most recent image matching a set of AMI filters.
 * // In this case, look up the NAT instance AMI, by using a wildcard
 * // in the 'name' field:
 * IMachineImage natAmi = ec2.MachineImage.lookup(new LookupMachineImageProps()
 *         .name("amzn-ami-vpc-nat-*")
 *         .owners(asList("amazon")));
 * 
 * // For other custom (Linux) images, instantiate a `GenericLinuxImage` with
 * // a map giving the AMI to in for each region:
 * IMachineImage linux = ec2.MachineImage.genericLinux(Map.of(
 *         "us-east-1", "ami-97785bed",
 *         "eu-west-1", "ami-12345678"));
 * 
 * // For other custom (Windows) images, instantiate a `GenericWindowsImage` with
 * // a map giving the AMI to in for each region:
 * IMachineImage genericWindows = ec2.MachineImage.genericWindows(Map.of(
 *         "us-east-1", "ami-97785bed",
 *         "eu-west-1", "ami-12345678"));
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: The AMIs selected by <code>MachineImage.lookup()</code> will be cached in
 * <code>cdk.context.json</code>, so that your AutoScalingGroup instances aren't replaced while
 * you are making unrelated changes to your CDK app.
 * <p>
 * To query for the latest AMI again, remove the relevant cache entry from
 * <code>cdk.context.json</code>, or use the <code>cdk context</code> command. For more information, see
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/context.html">Runtime Context</a> in the CDK
 * developer guide.
 * <p>
 * <code>MachineImage.genericLinux()</code>, <code>MachineImage.genericWindows()</code> will use <code>CfnMapping</code> in
 * an agnostic stack.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Special VPC configurations</h2>
 * <p>
 * <h3>VPN connections to a VPC</h3>
 * <p>
 * Create your VPC with VPN connections by specifying the <code>vpnConnections</code> props (keys are construct <code>id</code>s):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = Vpc.Builder.create(this, "MyVpc")
 *         .vpnConnections(Map.of(
 *                 "dynamic", Map.of(// Dynamic routing (BGP)
 *                         "ip", "1.2.3.4"),
 *                 "static", Map.of(// Static routing
 *                         "ip", "4.5.6.7",
 *                         "staticRoutes", asList("192.168.10.0/24", "192.168.20.0/24"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To create a VPC that can accept VPN connections, set <code>vpnGateway</code> to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = Vpc.Builder.create(this, "MyVpc")
 *         .vpnGateway(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * VPN connections can then be added:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * vpc.addVpnConnection("Dynamic", new VpnConnectionOptions()
 *         .ip("1.2.3.4"));
 * </pre></blockquote>
 * <p>
 * By default, routes will be propagated on the route tables associated with the private subnets. If no
 * private subnets exists, isolated subnets are used. If no isolated subnets exists, public subnets are
 * used. Use the <code>Vpc</code> property <code>vpnRoutePropagation</code> to customize this behavior.
 * <p>
 * VPN connections expose <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-cloudwatch/README.md">metrics (cloudwatch.Metric)</a> across all tunnels in the account/region and per connection:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Across all tunnels in the account/region
 * Metric allDataOut = ec2.VpnConnection.metricAllTunnelDataOut();
 * 
 * // For a specific vpn connection
 * VpnConnection vpnConnection = vpc.addVpnConnection("Dynamic", new VpnConnectionOptions()
 *         .ip("1.2.3.4"));
 * Metric state = vpnConnection.metricTunnelState();
 * </pre></blockquote>
 * <p>
 * <h3>VPC endpoints</h3>
 * <p>
 * A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.
 * <p>
 * Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Add gateway endpoints when creating the VPC
 * Vpc vpc = new Vpc(this, "MyVpc", new VpcProps()
 *         .gatewayEndpoints(Map.of(
 *                 "S3", new GatewayVpcEndpointOptions()
 *                         .service(ec2.GatewayVpcEndpointAwsService.getS3()))));
 * 
 * // Alternatively gateway endpoints can be added on the VPC
 * GatewayVpcEndpoint dynamoDbEndpoint = vpc.addGatewayEndpoint("DynamoDbEndpoint", new GatewayVpcEndpointOptions()
 *         .service(ec2.GatewayVpcEndpointAwsService.getDYNAMODB()));
 * 
 * // This allows to customize the endpoint policy
 * dynamoDbEndpoint.addToPolicy(
 * new PolicyStatement(new PolicyStatementProps()// Restrict to listing and describing tables
 *         .principals(asList(new AnyPrincipal()))
 *         .actions(asList("dynamodb:DescribeTable", "dynamodb:ListTables"))
 *         .resources(asList("*"))));
 * 
 * // Add an interface endpoint
 * vpc.addInterfaceEndpoint("EcrDockerEndpoint", Map.of(
 *         "service", ec2.InterfaceVpcEndpointAwsService.getECR_DOCKER()));
 * </pre></blockquote>
 * <p>
 * By default, CDK will place a VPC endpoint in one subnet per AZ. If you wish to override the AZs CDK places the VPC endpoint in,
 * use the <code>subnets</code> parameter as follows:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * InterfaceVpcEndpoint.Builder.create(stack, "VPC Endpoint")
 *         .vpc(vpc)
 *         .service(new InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443))
 *         // Choose which availability zones to place the VPC endpoint in, based on
 *         // available AZs
 *         .subnets(Map.of(
 *                 "availabilityZones", asList("us-east-1a", "us-east-1c")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Per the <a href="https://aws.amazon.com/premiumsupport/knowledge-center/interface-endpoint-availability-zone/">AWS documentation</a>, not all
 * VPC endpoint services are available in all AZs. If you specify the parameter <code>lookupSupportedAzs</code>, CDK attempts to discover which
 * AZs an endpoint service is available in, and will ensure the VPC endpoint is not placed in a subnet that doesn't match those AZs.
 * These AZs will be stored in cdk.context.json.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * InterfaceVpcEndpoint.Builder.create(stack, "VPC Endpoint")
 *         .vpc(vpc)
 *         .service(new InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443))
 *         // Choose which availability zones to place the VPC endpoint in, based on
 *         // available AZs
 *         .lookupSupportedAzs(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Pre-defined AWS services are defined in the <a href="lib/vpc-endpoint.ts">InterfaceVpcEndpointAwsService</a> class, and can be used to
 * create VPC endpoints without having to configure name, ports, etc. For example, a Keyspaces endpoint can be created for
 * use in your VPC:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * InterfaceVpcEndpoint.Builder.create(stack, "VPC Endpoint").vpc(vpc).service(InterfaceVpcEndpointAwsService.getKEYSPACES()).build();
 * </pre></blockquote>
 * <p>
 * <h4>Security groups for interface VPC endpoints</h4>
 * <p>
 * By default, interface VPC endpoints create a new security group and traffic is <strong>not</strong>
 * automatically allowed from the VPC CIDR.
 * <p>
 * Use the <code>connections</code> object to allow traffic to flow to the endpoint:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * myEndpoint.connections.allowDefaultPortFromAnyIpv4();
 * </pre></blockquote>
 * <p>
 * Alternatively, existing security groups can be used by specifying the <code>securityGroups</code> prop.
 * <p>
 * <h3>VPC endpoint services</h3>
 * <p>
 * A VPC endpoint service enables you to expose a Network Load Balancer(s) as a provider service to consumers, who connect to your service over a VPC endpoint. You can restrict access to your service via allowed principals (anything that extends ArnPrincipal), and require that new connections be manually accepted.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * VpcEndpointService.Builder.create(this, "EndpointService")
 *         .vpcEndpointServiceLoadBalancers(asList(networkLoadBalancer1, networkLoadBalancer2))
 *         .acceptanceRequired(true)
 *         .allowedPrincipals(asList(new ArnPrincipal("arn:aws:iam::123456789012:root")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Endpoint services support private DNS, which makes it easier for clients to connect to your service by automatically setting up DNS in their VPC.
 * You can enable private DNS on an endpoint service like so:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.route53.VpcEndpointServiceDomainName;
 * 
 * 
 * new VpcEndpointServiceDomainName(stack, "EndpointDomain", new VpcEndpointServiceDomainNameProps()
 *         .endpointService(vpces)
 *         .domainName("my-stuff.aws-cdk.dev")
 *         .publicHostedZone(zone));
 * </pre></blockquote>
 * <p>
 * Note: The domain name must be owned (registered through Route53) by the account the endpoint service is in, or delegated to the account.
 * The VpcEndpointServiceDomainName will handle the AWS side of domain verification, the process for which can be found
 * <a href="https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-dns-validation.html">here</a>
 * <p>
 * <h3>Client VPN endpoint</h3>
 * <p>
 * AWS Client VPN is a managed client-based VPN service that enables you to securely access your AWS
 * resources and resources in your on-premises network. With Client VPN, you can access your resources
 * from any location using an OpenVPN-based VPN client.
 * <p>
 * Use the <code>addClientVpnEndpoint()</code> method to add a client VPN endpoint to a VPC:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * vpc.addClientVpnEndpoint("Endpoint", new ClientVpnEndpointOptions()
 *         .cidr("10.100.0.0/16")
 *         .serverCertificateArn("arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id")
 *         // Mutual authentication
 *         .clientCertificateArn("arn:aws:acm:us-east-1:123456789012:certificate/client-certificate-id")
 *         // User-based authentication
 *         .userBasedAuthentication(ec2.ClientVpnUserBasedAuthentication.federated(samlProvider)));
 * </pre></blockquote>
 * <p>
 * The endpoint must use at least one <a href="https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/client-authentication.html">authentication method</a>:
 * <p>
 * <ul>
 * <li>Mutual authentication with a client certificate</li>
 * <li>User-based authentication (directory or federated)</li>
 * </ul>
 * <p>
 * If user-based authentication is used, the <a href="https://docs.aws.amazon.com/vpn/latest/clientvpn-user/self-service-portal.html">self-service portal URL</a>
 * is made available via a CloudFormation output.
 * <p>
 * By default, a new security group is created and logging is enabled. Moreover, a rule to
 * authorize all users to the VPC CIDR is created.
 * <p>
 * To customize authorization rules, set the <code>authorizeAllUsersToVpcCidr</code> prop to <code>false</code>
 * and use <code>addaddAuthorizationRule()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * ClientVpnEndpoint endpoint = vpc.addClientVpnEndpoint("Endpoint", new ClientVpnEndpointOptions()
 *         .cidr("10.100.0.0/16")
 *         .serverCertificateArn("arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id")
 *         .userBasedAuthentication(ec2.ClientVpnUserBasedAuthentication.federated(samlProvider))
 *         .authorizeAllUsersToVpcCidr(false));
 * 
 * endpoint.addAuthorizationRule("Rule", new ClientVpnAuthorizationRuleOptions()
 *         .cidr("10.0.10.0/32")
 *         .groupId("group-id"));
 * </pre></blockquote>
 * <p>
 * Use <code>addRoute()</code> to configure network routes:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * ClientVpnEndpoint endpoint = vpc.addClientVpnEndpoint("Endpoint", new ClientVpnEndpointOptions()
 *         .cidr("10.100.0.0/16")
 *         .serverCertificateArn("arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id")
 *         .userBasedAuthentication(ec2.ClientVpnUserBasedAuthentication.federated(samlProvider)));
 * 
 * // Client-to-client access
 * endpoint.addRoute("Route", new ClientVpnRouteOptions()
 *         .cidr("10.100.0.0/16")
 *         .target(ec2.ClientVpnRouteTarget.local()));
 * </pre></blockquote>
 * <p>
 * Use the <code>connections</code> object of the endpoint to allow traffic to other security groups.
 * <p>
 * <h2>Instances</h2>
 * <p>
 * You can use the <code>Instance</code> class to start up a single EC2 instance. For production setups, we recommend
 * you use an <code>AutoScalingGroup</code> from the <code>aws-autoscaling</code> module instead, as AutoScalingGroups will take
 * care of restarting your instance if it ever fails.
 * <p>
 * <h3>Configuring Instances using CloudFormation Init (cfn-init)</h3>
 * <p>
 * CloudFormation Init allows you to configure your instances by writing files to them, installing software
 * packages, starting services and running arbitrary commands. By default, if any of the instance setup
 * commands throw an error, the deployment will fail and roll back to the previously known good state.
 * The following documentation also applies to <code>AutoScalingGroup</code>s.
 * <p>
 * For the full set of capabilities of this system, see the documentation for
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html"><code>AWS::CloudFormation::Init</code></a>.
 * Here is an example of applying some configuration to an instance:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Instance.Builder.create(this, "Instance")
 *         // Showing the most complex setup, if you have simpler requirements
 *         // you can use `CloudFormationInit.fromElements()`.
 *         .init(ec2.CloudFormationInit.fromConfigSets(Map.of(
 *                 "configSets", Map.of(
 *                         // Applies the configs below in this order
 *                         "default", asList("yumPreinstall", "config")),
 *                 "configs", Map.of(
 *                         "yumPreinstall", new InitConfig(asList(ec2.InitPackage.yum("git"))),
 *                         "config", new InitConfig(asList(ec2.InitFile.fromObject("/etc/stack.json", Map.of(
 *                                 "stackId", stack.getStackId(),
 *                                 "stackName", stack.getStackName(),
 *                                 "region", stack.getRegion())), ec2.InitGroup.fromName("my-group"), ec2.InitUser.fromName("my-user"), ec2.InitPackage.rpm("http://mirrors.ukfast.co.uk/sites/dl.fedoraproject.org/pub/epel/8/Everything/x86_64/Packages/r/rubygem-git-1.5.0-2.el8.noarch.rpm")))))))
 *         .initOptions(Map.of(
 *                 // Optional, which configsets to activate (['default'] by default)
 *                 "configSets", asList("default"),
 * 
 *                 // Optional, how long the installation is expected to take (5 minutes by default)
 *                 "timeout", Duration.minutes(30),
 * 
 *                 // Optional, whether to include the --url argument when running cfn-init and cfn-signal commands (false by default)
 *                 "includeUrl", true,
 * 
 *                 // Optional, whether to include the --role argument when running cfn-init and cfn-signal commands (false by default)
 *                 "includeRole", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can have services restarted after the init process has made changes to the system.
 * To do that, instantiate an <code>InitServiceRestartHandle</code> and pass it to the config elements
 * that need to trigger the restart and the service itself. For example, the following
 * config writes a config file for nginx, extracts an archive to the root directory, and then
 * restarts nginx so that it picks up the new config and files:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object handle = new InitServiceRestartHandle();
 * 
 * ec2.CloudFormationInit.fromElements(ec2.InitFile.fromString("/etc/nginx/nginx.conf", "...", Map.of("serviceRestartHandles", asList(handle))), ec2.InitSource.fromBucket("/var/www/html", myBucket, "html.zip", Map.of("serviceRestartHandles", asList(handle))), ec2.InitService.enable("nginx", Map.of(
 *         "serviceRestartHandle", handle)));
 * </pre></blockquote>
 * <p>
 * <h3>Bastion Hosts</h3>
 * <p>
 * A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
 * You can use bastion hosts using a standard SSH connection targeting port 22 on the host. As an alternative, you can connect the SSH connection
 * feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)
 * <p>
 * A default bastion host for use via SSM can be configured like:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * BastionHostLinux host = new BastionHostLinux(this, "BastionHost", new BastionHostLinuxProps().vpc(vpc));
 * </pre></blockquote>
 * <p>
 * If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * BastionHostLinux host = new BastionHostLinux(this, "BastionHost", new BastionHostLinuxProps()
 *         .vpc(vpc)
 *         .subnetSelection(new SubnetSelection().subnetType(ec2.SubnetType.getPUBLIC())));
 * host.allowSshAccessFrom(ec2.Peer.ipv4("1.2.3.4/32"));
 * </pre></blockquote>
 * <p>
 * As there are no SSH public keys deployed on this machine, you need to use <a href="https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/">EC2 Instance Connect</a>
 * with the command <code>aws ec2-instance-connect send-ssh-public-key</code> to provide your SSH public key.
 * <p>
 * EBS volume for the bastion host can be encrypted like:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object host = BastionHostLinux.Builder.create(stack, "BastionHost")
 *         .vpc(vpc)
 *         .blockDevices(asList(Map.of(
 *                 "deviceName", "EBSBastionHost",
 *                 "volume", BlockDeviceVolume.ebs(10, Map.of(
 *                         "encrypted", true)))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Block Devices</h3>
 * <p>
 * To add EBS block device mappings, specify the <code>blockDevices</code> property. The following example sets the EBS-backed
 * root device (<code>/dev/sda1</code>) size to 50 GiB, and adds another EBS-backed device mapped to <code>/dev/sdm</code> that is 100 GiB in
 * size:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Instance.Builder.create(this, "Instance")
 *         // ...
 *         .blockDevices(asList(Map.of(
 *                 "deviceName", "/dev/sda1",
 *                 "volume", ec2.BlockDeviceVolume.ebs(50)), Map.of(
 *                 "deviceName", "/dev/sdm",
 *                 "volume", ec2.BlockDeviceVolume.ebs(100))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Volumes</h3>
 * <p>
 * Whereas a <code>BlockDeviceVolume</code> is an EBS volume that is created and destroyed as part of the creation and destruction of a specific instance. A <code>Volume</code> is for when you want an EBS volume separate from any particular instance. A <code>Volume</code> is an EBS block device that can be attached to, or detached from, any instance at any time. Some types of <code>Volume</code>s can also be attached to multiple instances at the same time to allow you to have shared storage between those instances.
 * <p>
 * A notable restriction is that a Volume can only be attached to instances in the same availability zone as the Volume itself.
 * <p>
 * The following demonstrates how to create a 500 GiB encrypted Volume in the <code>us-west-2a</code> availability zone, and give a role the ability to attach that Volume to a specific instance:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object instance = Instance.Builder.create(this, "Instance").build();
 * Role role = new Role(stack, "SomeRole", new RoleProps()
 *         .assumedBy(new AccountRootPrincipal()));
 * Object volume = Volume.Builder.create(this, "Volume")
 *         .availabilityZone("us-west-2a")
 *         .size(cdk.Size.gibibytes(500))
 *         .encrypted(true)
 *         .build();
 * 
 * volume.grantAttachVolume(role, asList(instance));
 * </pre></blockquote>
 * <p>
 * <h4>Instances Attaching Volumes to Themselves</h4>
 * <p>
 * If you need to grant an instance the ability to attach/detach an EBS volume to/from itself, then using <code>grantAttachVolume</code> and <code>grantDetachVolume</code> as outlined above
 * will lead to an unresolvable circular reference between the instance role and the instance. In this case, use <code>grantAttachVolumeByResourceTag</code> and <code>grantDetachVolumeByResourceTag</code> as follows:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object instance = Instance.Builder.create(this, "Instance").build();
 * Object volume = Volume.Builder.create(this, "Volume").build();
 * 
 * Object attachGrant = volume.grantAttachVolumeByResourceTag(instance.getGrantPrincipal(), asList(instance));
 * Object detachGrant = volume.grantDetachVolumeByResourceTag(instance.getGrantPrincipal(), asList(instance));
 * </pre></blockquote>
 * <p>
 * <h4>Attaching Volumes</h4>
 * <p>
 * The Amazon EC2 documentation for
 * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AmazonEBS.html">Linux Instances</a> and
 * <a href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ebs-volumes.html">Windows Instances</a> contains information on how
 * to attach and detach your Volumes to/from instances, and how to format them for use.
 * <p>
 * The following is a sample skeleton of EC2 UserData that can be used to attach a Volume to the Linux instance that it is running on:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object volume = Volume.Builder.create(this, "Volume").build();
 * Object instance = Instance.Builder.create(this, "Instance").build();
 * volume.grantAttachVolumeByResourceTag(instance.getGrantPrincipal(), asList(instance));
 * String targetDevice = "/dev/xvdz";
 * instance.userData.addCommands("TOKEN=$(curl -SsfX PUT \"http://169.254.169.254/latest/api/token\" -H \"X-aws-ec2-metadata-token-ttl-seconds: 21600\")", "INSTANCE_ID=$(curl -SsfH \"X-aws-ec2-metadata-token: $TOKEN\" http://169.254.169.254/latest/meta-data/instance-id)", String.format("aws --region %s ec2 attach-volume --volume-id %s --instance-id $INSTANCE_ID --device %s", Stack.of(this).getRegion(), volume.getVolumeId(), targetDevice), String.format("while ! test -e %s; do sleep 1; done", targetDevice));
 * </pre></blockquote>
 * <p>
 * <h2>VPC Flow Logs</h2>
 * <p>
 * VPC Flow Logs is a feature that enables you to capture information about the IP traffic going to and from network interfaces in your VPC. Flow log data can be published to Amazon CloudWatch Logs and Amazon S3. After you've created a flow log, you can retrieve and view its data in the chosen destination. (<a href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html">https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html</a>).
 * <p>
 * By default a flow log will be created with CloudWatch Logs as the destination.
 * <p>
 * You can create a flow log like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * FlowLog.Builder.create(this, "FlowLog")
 *         .resourceType(ec2.FlowLogResourceType.fromVpc(vpc))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or you can add a Flow Log to a VPC by using the addFlowLog method like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = new Vpc(this, "Vpc");
 * 
 * vpc.addFlowLog("FlowLog");
 * </pre></blockquote>
 * <p>
 * You can also add multiple flow logs with different destinations.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object vpc = new Vpc(this, "Vpc");
 * 
 * vpc.addFlowLog("FlowLogS3", Map.of(
 *         "destination", ec2.FlowLogDestination.toS3()));
 * 
 * vpc.addFlowLog("FlowLogCloudWatch", Map.of(
 *         "trafficType", ec2.FlowLogTrafficType.getREJECT()));
 * </pre></blockquote>
 * <p>
 * By default the CDK will create the necessary resources for the destination. For the CloudWatch Logs destination
 * it will create a CloudWatch Logs Log Group as well as the IAM role with the necessary permissions to publish to
 * the log group. In the case of an S3 destination, it will create the S3 bucket.
 * <p>
 * If you want to customize any of the destination resources you can provide your own as part of the <code>destination</code>.
 * <p>
 * <em>CloudWatch Logs</em>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object logGroup = new LogGroup(this, "MyCustomLogGroup");
 * 
 * Role role = new Role(this, "MyCustomRole", new RoleProps()
 *         .assumedBy(new ServicePrincipal("vpc-flow-logs.amazonaws.com")));
 * 
 * FlowLog.Builder.create(this, "FlowLog")
 *         .resourceType(ec2.FlowLogResourceType.fromVpc(vpc))
 *         .destination(ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <em>S3</em>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Bucket bucket = new Bucket(this, "MyCustomBucket");
 * 
 * FlowLog.Builder.create(this, "FlowLog")
 *         .resourceType(ec2.FlowLogResourceType.fromVpc(vpc))
 *         .destination(ec2.FlowLogDestination.toS3(bucket))
 *         .build();
 * 
 * FlowLog.Builder.create(this, "FlowLogWithKeyPrefix")
 *         .resourceType(ec2.FlowLogResourceType.fromVpc(vpc))
 *         .destination(ec2.FlowLogDestination.toS3(bucket, "prefix/"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>User Data</h2>
 * <p>
 * User data enables you to run a script when your instances start up.  In order to configure these scripts you can add commands directly to the script
 * or you can use the UserData's convenience functions to aid in the creation of your script.
 * <p>
 * A user data could be configured to run a script found in an asset through the following:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object asset = Asset.Builder.create(this, "Asset").path(path.join(__dirname, "configure.sh")).build();
 * Object instance = Instance.Builder.create(this, "Instance").build();
 * Object localPath = instance.userData.addS3DownloadCommand(Map.of(
 *         "bucket", asset.getBucket(),
 *         "bucketKey", asset.getS3ObjectKey()));
 * instance.userData.addExecuteFileCommand(Map.of(
 *         "filePath", localPath,
 *         "arguments", "--verbose -y"));
 * asset.grantRead(instance.getRole());
 * </pre></blockquote>
 * <p>
 * <h3>Multipart user data</h3>
 * <p>
 * In addition, to above the <code>MultipartUserData</code> can be used to change instance startup behavior. Multipart user data are composed
 * from separate parts forming archive. The most common parts are scripts executed during instance set-up. However, there are other
 * kinds, too.
 * <p>
 * The advantage of multipart archive is in flexibility when it's needed to add additional parts or to use specialized parts to
 * fine tune instance startup. Some services (like AWS Batch) supports only <code>MultipartUserData</code>.
 * <p>
 * The parts can be executed at different moment of instance start-up and can serve a different purposes. This is controlled by <code>contentType</code> property.
 * For common scripts, <code>text/x-shellscript; charset="utf-8"</code> can be used as content type.
 * <p>
 * In order to create archive the <code>MultipartUserData</code> has to be instantiated. Than, user can add parts to multipart archive using <code>addPart</code>. The <code>MultipartBody</code> contains methods supporting creation of body parts.
 * <p>
 * If the very custom part is required, it can be created using <code>MultipartUserData.fromRawBody</code>, in this case full control over content type,
 * transfer encoding, and body properties is given to the user.
 * <p>
 * Below is an example for creating multipart user data with single body part responsible for installing <code>awscli</code> and configuring maximum size
 * of storage used by Docker containers:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bootHookConf = ec2.UserData.forLinux();
 * bootHookConf.addCommands("cloud-init-per once docker_options echo 'OPTIONS=\"${OPTIONS} --storage-opt dm.basesize=40G\"' &gt;&gt; /etc/sysconfig/docker");
 * 
 * Object setupCommands = ec2.UserData.forLinux();
 * setupCommands.addCommands("sudo yum install awscli &amp;&amp; echo Packages installed らと &gt; /var/tmp/setup");
 * 
 * Object multipartUserData = new MultipartUserData();
 * // The docker has to be configured at early stage, so content type is overridden to boothook
 * multipartUserData.addPart(ec2.MultipartBody.fromUserData(bootHookConf, "text/cloud-boothook; charset=\"us-ascii\""));
 * // Execute the rest of setup
 * multipartUserData.addPart(ec2.MultipartBody.fromUserData(setupCommands));
 * 
 * LaunchTemplate.Builder.create(stack, "")
 *         .userData(multipartUserData)
 *         .blockDevices(asList())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information see
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data">Specifying Multiple User Data Blocks Using a MIME Multi Part Archive</a>
 * <p>
 * <h4>Using add*Command on MultipartUserData</h4>
 * <p>
 * To use the <code>add*Command</code> methods, that are inherited from the <code>UserData</code> interface, on <code>MultipartUserData</code> you must add a part
 * to the <code>MultipartUserData</code> and designate it as the reciever for these methods. This is accomplished by using the <code>addUserDataPart()</code>
 * method on <code>MultipartUserData</code> with the <code>makeDefault</code> argument set to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object multipartUserData = new MultipartUserData();
 * Object commandsUserData = ec2.UserData.forLinux();
 * multipartUserData.addUserDataPart(commandsUserData, MultipartBody.getSHELL_SCRIPT(), true);
 * 
 * // Adding commands to the multipartUserData adds them to commandsUserData, and vice-versa.
 * multipartUserData.addCommands("touch /root/multi.txt");
 * commandsUserData.addCommands("touch /root/userdata.txt");
 * </pre></blockquote>
 * <p>
 * When used on an EC2 instance, the above <code>multipartUserData</code> will create both <code>multi.txt</code> and <code>userdata.txt</code> in <code>/root</code>.
 * <p>
 * <h2>Importing existing subnet</h2>
 * <p>
 * To import an existing Subnet, call <code>Subnet.fromSubnetAttributes()</code> or
 * <code>Subnet.fromSubnetId()</code>. Only if you supply the subnet's Availability Zone
 * and Route Table Ids when calling <code>Subnet.fromSubnetAttributes()</code> will you be
 * able to use the CDK features that use these values (such as selecting one
 * subnet per AZ).
 * <p>
 * Importing an existing subnet looks like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Supply all properties
 * Object subnet = Subnet.fromSubnetAttributes(this, "SubnetFromAttributes", Map.of(
 *         "subnetId", "s-1234",
 *         "availabilityZone", "pub-az-4465",
 *         "routeTableId", "rt-145"));
 * 
 * // Supply only subnet id
 * Object subnet = Subnet.fromSubnetId(this, "SubnetFromId", "s-1234");
 * </pre></blockquote>
 * <p>
 * <h2>Launch Templates</h2>
 * <p>
 * A Launch Template is a standardized template that contains the configuration information to launch an instance.
 * They can be used when launching instances on their own, through Amazon EC2 Auto Scaling, EC2 Fleet, and Spot Fleet.
 * Launch templates enable you to store launch parameters so that you do not have to specify them every time you launch
 * an instance. For information on Launch Templates please see the
 * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">official documentation</a>.
 * <p>
 * The following demonstrates how to create a launch template with an Amazon Machine Image, and security group.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object vpc = new Vpc(...);
 * // ...
 * Object template = LaunchTemplate.Builder.create(this, "LaunchTemplate")
 *         .machineImage(new AmazonMachineImage())
 *         .securityGroup(SecurityGroup.Builder.create(this, "LaunchTemplateSG")
 *                 .vpc(vpc)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.ec2;
