/**
 * <h2>Amazon EC2 Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>The <code>@aws-cdk/aws-ec2</code> package contains primitives for setting up networking and
 * instances.</p>
 * <h2>VPC</h2>
 * <p>Most projects need a Virtual Private Cloud to provide security by means of
 * network partitioning. This is achieved by creating an instance of
 * <code>Vpc</code>:</p>
 * <pre><code class="language-ts">// This example is in TypeScript, examples in Java are coming soon.
 * import ec2 = require('@aws-cdk/aws-ec2');
 * 
 * const vpc = new ec2.Vpc(this, 'VPC');
 * </code></pre>
 * <p>All default constructs require EC2 instances to be launched inside a VPC, so
 * you should generally start by defining a VPC whenever you need to launch
 * instances for your project.</p>
 * <h3>Subnet Types</h3>
 * <p>A VPC consists of one or more subnets that instances can be placed into. CDK
 * distinguishes three different subnet types:</p>
 * <ul>
 * <li><strong>Public</strong> - public subnets connect directly to the Internet using an
 * Internet Gateway. If you want your instances to have a public IP address
 * and be directly reachable from the Internet, you must place them in a
 * public subnet.</li>
 * <li><strong>Private</strong> - instances in private subnets are not directly routable from the
 * Internet, and connect out to the Internet via a NAT gateway. By default, a
 * NAT gateway is created in every public subnet for maximum availability. Be
 * aware that you will be charged for NAT gateways.</li>
 * <li><strong>Isolated</strong> - isolated subnets do not route from or to the Internet, and
 * as such do not require NAT gateways. They can only connect to or be
 * connected to from other instances in the same VPC. A default VPC configuration
 * will not include isolated subnets,</li>
 * </ul>
 * <p>A default VPC configuration will create public and private subnets, but not
 * isolated subnets. See <em>Advanced Subnet Configuration</em> below for information
 * on how to change the default subnet configuration.</p>
 * <p>Constructs using the VPC will &quot;launch instances&quot; (or more accurately, create
 * Elastic Network Interfaces) into one or more of the subnets. They all accept
 * a property called <code>subnetSelection</code> (sometimes called <code>vpcSubnets</code>) to allow
 * you to select in what subnet to place the ENIs, usually defaulting to
 * <em>private</em> subnets if the property is omitted.</p>
 * <p>If you would like to save on the cost of NAT gateways, you can use
 * <em>isolated</em> subnets instead of <em>private</em> subnets (as described in Advanced
 * <em>Subnet Configuration</em>). If you need private instances to have
 * internet connectivity, another option is to reduce the number of NAT gateways
 * created by setting the <code>natGateways</code> property to a lower value (the default
 * is one NAT gateway per availability zone). Be aware that this may have
 * availability implications for your application.</p>
 * <p><a href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html">Read more about
 * subnets</a>.</p>
 * <h3>Control over availability zones</h3>
 * <p>By default, a VPC will spread over at most 3 Availability Zones available to
 * it. To change the number of Availability Zones that the VPC will spread over,
 * specify the <code>maxAzs</code> property when defining it.</p>
 * <p>The number of Availability Zones that are available depends on the <em>region</em>
 * and <em>account</em> of the Stack containing the VPC. If the <a href="https://docs.aws.amazon.com/cdk/latest/guide/environments.html">region and account are
 * specified</a> on
 * the Stack, the CLI will <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#using-regions-availability-zones-describe">look up the existing Availability
 * Zones</a>
 * and get an accurate count. If region and account are not specified, the stack
 * could be deployed anywhere and it will have to make a safe choice, limiting
 * itself to 2 Availability Zones.</p>
 * <p>Therefore, to get the VPC to spread over 3 or more availability zones, you
 * must specify the environment where the stack will be deployed.</p>
 * <h3>Using NAT instances</h3>
 * <p>By default, the <code>Vpc</code> construct will create NAT <em>gateways</em> for you, which
 * are managed by AWS. If you would prefer to use your own managed NAT
 * <em>instances</em> instead, specify a different value for the <code>natGatewayProvider</code>
 * property, as follows:</p>
 * <pre><code class="language-ts">    // Configure the `natGatewayProvider` when defining a Vpc
 *     const vpc = new ec2.Vpc(this, 'MyVpc', {
 *       natGatewayProvider: ec2.NatProvider.instance({
 *         instanceType: new ec2.InstanceType('t3.small')
 *       }),
 * 
 *       // The 'natGateways' parameter now controls the number of NAT instances
 *       natGateways: 2,
 *     });
 * </code></pre>
 * <p>The construct will automatically search for the most recent NAT gateway AMI.
 * If you prefer to use a custom AMI, pass a <code>GenericLinuxImage</code> instance
 * for the instance's <code>machineImage</code> parameter and configure the right AMI ID
 * for the regions you want to deploy to.</p>
 * <h3>Advanced Subnet Configuration</h3>
 * <p>If the default VPC configuration (public and private subnets spanning the
 * size of the VPC) don't suffice for you, you can configure what subnets to
 * create by specifying the <code>subnetConfiguration</code> property. It allows you
 * to configure the number and size of all subnets. Specifying an advanced
 * subnet configuration could look like this:</p>
 * <pre><code class="language-ts">const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   // 'cidr' configures the IP range and size of the entire VPC.
 *   // The IP space will be divided over the configured subnets.
 *   cidr: '10.0.0.0/21',
 * 
 *   // 'maxAzs' configures the maximum number of availability zones to use
 *   maxAzs: 3,
 * 
 *   // 'subnetConfiguration' specifies the &quot;subnet groups&quot; to create.
 *   // Every subnet group will have a subnet for each AZ, so this
 *   // configuration will create `3 groups × 3 AZs = 9` subnets.
 *   subnetConfiguration: [
 *     {
 *       // 'subnetType' controls Internet access, as described above.
 *       subnetType: ec2.SubnetType.PUBLIC,
 * 
 *       // 'name' is used to name this particular subnet group. You will have to
 *       // use the name for subnet selection if you have more than one subnet
 *       // group of the same type.
 *       name: 'Ingress',
 * 
 *       // 'cidrMask' specifies the IP addresses in the range of of individual
 *       // subnets in the group. Each of the subnets in this group will contain
 *       // `2^(32 address bits - 24 subnet bits) - 2 reserved addresses = 254`
 *       // usable IP addresses.
 *       //
 *       // If 'cidrMask' is left out the available address space is evenly
 *       // divided across the remaining subnet groups.
 *       cidrMask: 24,
 *     },
 *     {
 *       cidrMask: 24,
 *       name: 'Application',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *     },
 *     {
 *       cidrMask: 28,
 *       name: 'Database',
 *       subnetType: ec2.SubnetType.ISOLATED,
 * 
 *       // 'reserved' can be used to reserve IP address space. No resources will
 *       // be created for this subnet, but the IP range will be kept available for
 *       // future creation of this subnet, or even for future subdivision.
 *       reserved: true
 *     }
 *   ],
 * });
 * </code></pre>
 * <p>The example above is one possible configuration, but the user can use the
 * constructs above to implement many other network configurations.</p>
 * <p>The <code>Vpc</code> from the above configuration in a Region with three
 * availability zones will be the following:</p>
 * <p>Subnet Name       |Type      |IP Block      |AZ|Features
 * ------------------|----------|--------------|--|--------
 * IngressSubnet1    |<code>PUBLIC</code>  |<code>10.0.0.0/24</code> |#1|NAT Gateway
 * IngressSubnet2    |<code>PUBLIC</code>  |<code>10.0.1.0/24</code> |#2|NAT Gateway
 * IngressSubnet3    |<code>PUBLIC</code>  |<code>10.0.2.0/24</code> |#3|NAT Gateway
 * ApplicationSubnet1|<code>PRIVATE</code> |<code>10.0.3.0/24</code> |#1|Route to NAT in IngressSubnet1
 * ApplicationSubnet2|<code>PRIVATE</code> |<code>10.0.4.0/24</code> |#2|Route to NAT in IngressSubnet2
 * ApplicationSubnet3|<code>PRIVATE</code> |<code>10.0.5.0/24</code> |#3|Route to NAT in IngressSubnet3
 * DatabaseSubnet1   |<code>ISOLATED</code>|<code>10.0.6.0/28</code> |#1|Only routes within the VPC
 * DatabaseSubnet2   |<code>ISOLATED</code>|<code>10.0.6.16/28</code>|#2|Only routes within the VPC
 * DatabaseSubnet3   |<code>ISOLATED</code>|<code>10.0.6.32/28</code>|#3|Only routes within the VPC</p>
 * <h3>Reserving subnet IP space</h3>
 * <p>There are situations where the IP space for a subnet or number of subnets
 * will need to be reserved. This is useful in situations where subnets would
 * need to be added after the vpc is originally deployed, without causing IP
 * renumbering for existing subnets. The IP space for a subnet may be reserved
 * by setting the <code>reserved</code> subnetConfiguration property to true, as shown
 * below:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   natGateways: 1,
 *   subnetConfiguration: [
 *     {
 *       cidrMask: 26,
 *       name: 'Public',
 *       subnetType: ec2.SubnetType.PUBLIC,
 *     },
 *     {
 *       cidrMask: 26,
 *       name: 'Application1',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *     },
 *     {
 *       cidrMask: 26,
 *       name: 'Application2',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *       reserved: true,   // &lt;---- This subnet group is reserved
 *     },
 *     {
 *       cidrMask: 27,
 *       name: 'Database',
 *       subnetType: ec2.SubnetType.ISOLATED,
 *     }
 *   ],
 * });
 * </code></pre>
 * <p>In the example above, the subnet for Application2 is not actually provisioned
 * but its IP space is still reserved. If in the future this subnet needs to be
 * provisioned, then the <code>reserved: true</code> property should be removed. Reserving
 * parts of the IP space prevents the other subnets from getting renumbered.</p>
 * <h3>Sharing VPCs between stacks</h3>
 * <p>If you are creating multiple <code>Stack</code>s inside the same CDK application, you
 * can reuse a VPC defined in one Stack in another by simply passing the VPC
 * instance around:</p>
 * <pre><code class="language-ts">/**
 *  * Stack1 creates the VPC
 *  *{@literal /}
 * class Stack1 extends cdk.Stack {
 *   public readonly vpc: ec2.Vpc;
 * 
 *   constructor(scope: cdk.App, id: string, props?: cdk.StackProps) {
 *     super(scope, id, props);
 * 
 *     this.vpc = new ec2.Vpc(this, 'VPC');
 *   }
 * }
 * 
 * interface Stack2Props extends cdk.StackProps {
 *   vpc: ec2.IVpc;
 * }
 * 
 * /**
 *  * Stack2 consumes the VPC
 *  *{@literal /}
 * class Stack2 extends cdk.Stack {
 *   constructor(scope: cdk.App, id: string, props: Stack2Props) {
 *     super(scope, id, props);
 * 
 *     // Pass the VPC to a construct that needs it
 *     new ConstructThatTakesAVpc(this, 'Construct', {
 *       vpc: props.vpc
 *     });
 *   }
 * }
 * 
 * const stack1 = new Stack1(app, 'Stack1');
 * const stack2 = new Stack2(app, 'Stack2', {
 *   vpc: stack1.vpc,
 * });
 * </code></pre>
 * <h3>Importing an existing VPC</h3>
 * <p>If your VPC is created outside your CDK app, you can use <code>Vpc.fromLookup()</code>.
 * The CDK CLI will search for the specified VPC in the the stack's region and
 * account, and import the subnet configuration. Looking up can be done by VPC
 * ID, but more flexibly by searching for a specific tag on the VPC.</p>
 * <p>The import does assume that the VPC will be <em>symmetric</em>, i.e. that there are
 * subnet groups that have a subnet in every Availability Zone that the VPC
 * spreads over. VPCs with other layouts cannot currently be imported, and will
 * either lead to an error on import, or when another construct tries to access
 * the subnets.</p>
 * <p>Subnet types will be determined from the <code>aws-cdk:subnet-type</code> tag on the
 * subnet if it exists, or the presence of a route to an Internet Gateway
 * otherwise. Subnet names will be determined from the <code>aws-cdk:subnet-name</code> tag
 * on the subnet if it exists, or will mirror the subnet type otherwise (i.e.
 * a public subnet will have the name <code>&quot;Public&quot;</code>).</p>
 * <p>Here's how <code>Vpc.fromLookup()</code> can be used:</p>
 * <pre><code class="language-ts">const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
 *   // This imports the default VPC but you can also
 *   // specify a 'vpcName' or 'tags'.
 *   isDefault: true
 * });
 * </code></pre>
 * <h2>Allowing Connections</h2>
 * <p>In AWS, all network traffic in and out of <strong>Elastic Network Interfaces</strong> (ENIs)
 * is controlled by <strong>Security Groups</strong>. You can think of Security Groups as a
 * firewall with a set of rules. By default, Security Groups allow no incoming
 * (ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
 * to them to allow incoming traffic streams. To exert fine-grained control over
 * egress traffic, set <code>allowAllOutbound: false</code> on the <code>SecurityGroup</code>, after
 * which you can add egress traffic rules.</p>
 * <p>You can manipulate Security Groups directly:</p>
 * <pre><code class="language-ts">const mySecurityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
 *   vpc,
 *   description: 'Allow ssh access to ec2 instances',
 *   allowAllOutbound: true   // Can be set to false
 * });
 * mySecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
 * </code></pre>
 * <p>All constructs that create ENIs on your behalf (typically constructs that create
 * EC2 instances or other VPC-connected resources) will all have security groups
 * automatically assigned. Those constructs have an attribute called
 * <strong>connections</strong>, which is an object that makes it convenient to update the
 * security groups. If you want to allow connections between two constructs that
 * have security groups, you have to add an <strong>Egress</strong> rule to one Security Group,
 * and an <strong>Ingress</strong> rule to the other. The connections object will automatically
 * take care of this for you:</p>
 * <pre><code class="language-ts">// Allow connections from anywhere
 * loadBalancer.connections.allowFromAnyIpv4(ec2.Port.tcp(443), 'Allow inbound HTTPS');
 * 
 * // The same, but an explicit IP address
 * loadBalancer.connections.allowFrom(ec2.Peer.ipv4('1.2.3.4/32'), ec2.Port.tcp(443), 'Allow inbound HTTPS');
 * 
 * // Allow connection between AutoScalingGroups
 * appFleet.connections.allowTo(dbFleet, ec2.Port.tcp(443), 'App can call database');
 * </code></pre>
 * <h3>Connection Peers</h3>
 * <p>There are various classes that implement the connection peer part:</p>
 * <pre><code class="language-ts">// Simple connection peers
 * let peer = ec2.Peer.ipv4(&quot;10.0.0.0/16&quot;);
 * let peer = ec2.Peer.anyIpv4();
 * let peer = ec2.Peer.ipv6(&quot;::0/0&quot;);
 * let peer = ec2.Peer.anyIpv6();
 * let peer = ec2.Peer.prefixList(&quot;pl-12345&quot;);
 * fleet.connections.allowTo(peer, ec2.Port.tcp(443), 'Allow outbound HTTPS');
 * </code></pre>
 * <p>Any object that has a security group can itself be used as a connection peer:</p>
 * <pre><code class="language-ts">// These automatically create appropriate ingress and egress rules in both security groups
 * fleet1.connections.allowTo(fleet2, ec2.Port.tcp(80), 'Allow between fleets');
 * 
 * fleet.connections.allowFromAnyIpv4(ec2.Port.tcp(80), 'Allow from load balancer');
 * </code></pre>
 * <h3>Port Ranges</h3>
 * <p>The connections that are allowed are specified by port ranges. A number of classes provide
 * the connection specifier:</p>
 * <pre><code class="language-ts">ec2.Port.tcp(80)
 * ec2.Port.tcpRange(60000, 65535)
 * ec2.Port.allTcp()
 * ec2.Port.allTraffic()
 * </code></pre>
 * <blockquote>
 * <p>NOTE: This set is not complete yet; for example, there is no library support for ICMP at the moment.
 * However, you can write your own classes to implement those.</p>
 * </blockquote>
 * <h3>Default Ports</h3>
 * <p>Some Constructs have default ports associated with them. For example, the
 * listener of a load balancer does (it's the public port), or instances of an
 * RDS database (it's the port the database is accepting connections on).</p>
 * <p>If the object you're calling the peering method on has a default port associated with it, you can call
 * <code>allowDefaultPortFrom()</code> and omit the port specifier. If the argument has an associated default port, call
 * <code>allowDefaultPortTo()</code>.</p>
 * <p>For example:</p>
 * <pre><code class="language-ts">// Port implicit in listener
 * listener.connections.allowDefaultPortFromAnyIpv4('Allow public');
 * 
 * // Port implicit in peer
 * fleet.connections.allowDefaultPortTo(rdsDatabase, 'Fleet can access database');
 * </code></pre>
 * <h2>Machine Images (AMIs)</h2>
 * <p>AMIs control the OS that gets launched when you start your EC2 instance. The EC2
 * library contains constructs to select the AMI you want to use.</p>
 * <p>Depending on the type of AMI, you select it a different way. Here are some
 * examples of things you might want to use:</p>
 * <pre><code class="language-ts">// Pick the right Amazon Linux edition. All arguments shown are optional
 * // and will default to these values when omitted.
 * const amznLinux = new ec2.AmazonLinuxImage({
 *   generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX,
 *   edition: ec2.AmazonLinuxEdition.STANDARD,
 *   virtualization: ec2.AmazonLinuxVirt.HVM,
 *   storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
 * });
 * 
 * // Pick a Windows edition to use
 * const windows = new ec2.WindowsImage(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE);
 * 
 * // Look up the most recent image matching a set of AMI filters.
 * // In this case, look up the NAT instance AMI, by using a wildcard
 * // in the 'name' field:
 * const natAmi = new LookupMachineImage({
 *   name: 'amzn-ami-vpc-nat-*',
 *   owners: ['amazon'],
 * });
 * 
 * // For other custom (Linux) images, instantiate a `GenericLinuxImage` with
 * // a map giving the AMI to in for each region:
 * 
 * const linux = new ec2.GenericLinuxImage({
 *     'us-east-1': 'ami-97785bed',
 *     'eu-west-1': 'ami-12345678',
 *     // ...
 * });
 * 
 * // For other custom (Windows) images, instantiate a `GenericWindowsImage` with
 * // a map giving the AMI to in for each region:
 * 
 * const genericWindows = new ec2.GenericWindowsImage({
 *   'us-east-1': 'ami-97785bed',
 *   'eu-west-1': 'ami-12345678',
 *   // ...
 * });
 * </code></pre>
 * <blockquote>
 * <p>NOTE: The AMIs selected by <code>AmazonLinuxImage</code> or <code>LookupImage</code> will be cached in
 * <code>cdk.context.json</code>, so that your AutoScalingGroup instances aren't replaced while
 * you are making unrelated changes to your CDK app.</p>
 * <p>To query for the latest AMI again, remove the relevant cache entry from
 * <code>cdk.context.json</code>, or use the <code>cdk context</code> command. For more information, see
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/context.html">Runtime Context</a> in the CDK
 * developer guide.</p>
 * </blockquote>
 * <h2>VPN connections to a VPC</h2>
 * <p>Create your VPC with VPN connections by specifying the <code>vpnConnections</code> props (keys are construct <code>id</code>s):</p>
 * <pre><code class="language-ts">const vpc = new ec2.Vpc(stack, 'MyVpc', {
 *   vpnConnections: {
 *     dynamic: { // Dynamic routing (BGP)
 *       ip: '1.2.3.4'
 *     },
 *     static: { // Static routing
 *       ip: '4.5.6.7',
 *       staticRoutes: [
 *         '192.168.10.0/24',
 *         '192.168.20.0/24'
 *       ]
 *     }
 *   }
 * });
 * </code></pre>
 * <p>To create a VPC that can accept VPN connections, set <code>vpnGateway</code> to <code>true</code>:</p>
 * <pre><code class="language-ts">const vpc = new ec2.Vpc(stack, 'MyVpc', {
 *   vpnGateway: true
 * });
 * </code></pre>
 * <p>VPN connections can then be added:</p>
 * <pre><code class="language-ts">vpc.addVpnConnection('Dynamic', {
 *   ip: '1.2.3.4'
 * });
 * </code></pre>
 * <p>Routes will be propagated on the route tables associated with the private subnets.</p>
 * <p>VPN connections expose <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-cloudwatch/README.md">metrics (cloudwatch.Metric)</a> across all tunnels in the account/region and per connection:</p>
 * <pre><code class="language-ts">// Across all tunnels in the account/region
 * const allDataOut = VpnConnection.metricAllTunnelDataOut();
 * 
 * // For a specific vpn connection
 * const vpnConnection = vpc.addVpnConnection('Dynamic', {
 *   ip: '1.2.3.4'
 * });
 * const state = vpnConnection.metricTunnelState();
 * </code></pre>
 * <h2>VPC endpoints</h2>
 * <p>A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.</p>
 * <p>Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.</p>
 * <pre><code class="language-ts">    // Add gateway endpoints when creating the VPC
 *     const vpc = new ec2.Vpc(this, 'MyVpc', {
 *       gatewayEndpoints: {
 *         S3: {
 *           service: ec2.GatewayVpcEndpointAwsService.S3
 *         }
 *       }
 *     });
 * 
 *     // Alternatively gateway endpoints can be added on the VPC
 *     const dynamoDbEndpoint = vpc.addGatewayEndpoint('DynamoDbEndpoint', {
 *       service: ec2.GatewayVpcEndpointAwsService.DYNAMODB
 *     });
 * 
 *     // This allows to customize the endpoint policy
 *     dynamoDbEndpoint.addToPolicy(
 *       new iam.PolicyStatement({ // Restrict to listing and describing tables
 *         principals: [new iam.AnyPrincipal()],
 *         actions: ['dynamodb:DescribeTable', 'dynamodb:ListTables'],
 *         resources: ['*'],
 *     }));
 * 
 *     // Add an interface endpoint
 *     const ecrDockerEndpoint = vpc.addInterfaceEndpoint('EcrDockerEndpoint', {
 *       service: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
 *     });
 * 
 *     // When working with an interface endpoint, use the connections object to
 *     // allow traffic to flow to the endpoint.
 *     ecrDockerEndpoint.connections.allowDefaultPortFromAnyIpv4();
 * </code></pre>
 * <h3>Security groups for interface VPC endpoints</h3>
 * <p>By default, interface VPC endpoints create a new security group and traffic is <strong>not</strong>
 * automatically allowed from the VPC CIDR.</p>
 * <p>Use the <code>connections</code> object to allow traffic to flow to the endpoint:</p>
 * <pre><code class="language-ts">myEndpoint.connections.allowDefaultPortFrom(...);
 * </code></pre>
 * <p>Alternatively, existing security groups can be used by specifying the <code>securityGroups</code> prop.</p>
 * <h2>Bastion Hosts</h2>
 * <p>A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
 * You can use bastion hosts using a standard SSH connection targetting port 22 on the host. As an alternative, you can connect the SSH connection
 * feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)</p>
 * <p>A default bastion host for use via SSM can be configured like:</p>
 * <pre><code class="language-ts">const host = new ec2.BastionHostLinux(this, 'BastionHost', { vpc });
 * </code></pre>
 * <p>If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.</p>
 * <pre><code class="language-ts">const host = new ec2.BastionHostLinux(this, 'BastionHost', {
 *   vpc,
 *   subnetSelection: { subnetType: SubnetType.PUBLIC },
 * });
 * host.allowSshAccessFrom(Peer.ipv4('1.2.3.4/32'));
 * </code></pre>
 * <p>As there are no SSH public keys deployed on this machine, you need to use <a href="https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/">EC2 Instance Connect</a>
 * with the command <code>aws ec2-instance-connect send-ssh-public-key</code> to provide your SSH public key.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.ec2;
