/**
 * <h2>Amazon EC2 Construct Library</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="Stability: Stable" src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * The <code>&#64;aws-cdk/aws-ec2</code> package contains primitives for setting up networking and
 * instances.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.ec2.*;
 * </pre></blockquote>
 * <p>
 * <h2>VPC</h2>
 * <p>
 * Most projects need a Virtual Private Cloud to provide security by means of
 * network partitioning. This is achieved by creating an instance of
 * <code>Vpc</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "VPC");
 * </pre></blockquote>
 * <p>
 * All default constructs require EC2 instances to be launched inside a VPC, so
 * you should generally start by defining a VPC whenever you need to launch
 * instances for your project.
 * <p>
 * <h3>Subnet Types</h3>
 * <p>
 * A VPC consists of one or more subnets that instances can be placed into. CDK
 * distinguishes three different subnet types:
 * <p>
 * <ul>
 * <li><strong>Public</strong> - public subnets connect directly to the Internet using an
 * Internet Gateway. If you want your instances to have a public IP address
 * and be directly reachable from the Internet, you must place them in a
 * public subnet.</li>
 * <li><strong>Private</strong> - instances in private subnets are not directly routable from the
 * Internet, and connect out to the Internet via a NAT gateway. By default, a
 * NAT gateway is created in every public subnet for maximum availability. Be
 * aware that you will be charged for NAT gateways.</li>
 * <li><strong>Isolated</strong> - isolated subnets do not route from or to the Internet, and
 * as such do not require NAT gateways. They can only connect to or be
 * connected to from other instances in the same VPC. A default VPC configuration
 * will not include isolated subnets,</li>
 * </ul>
 * <p>
 * A default VPC configuration will create public and private subnets, but not
 * isolated subnets. See <em>Advanced Subnet Configuration</em> below for information
 * on how to change the default subnet configuration.
 * <p>
 * Constructs using the VPC will "launch instances" (or more accurately, create
 * Elastic Network Interfaces) into one or more of the subnets. They all accept
 * a property called <code>subnetSelection</code> (sometimes called <code>vpcSubnets</code>) to allow
 * you to select in what subnet to place the ENIs, usually defaulting to
 * <em>private</em> subnets if the property is omitted.
 * <p>
 * If you would like to save on the cost of NAT gateways, you can use
 * <em>isolated</em> subnets instead of <em>private</em> subnets (as described in Advanced
 * <em>Subnet Configuration</em>). If you need private instances to have
 * internet connectivity, another option is to reduce the number of NAT gateways
 * created by setting the <code>natGateways</code> property to a lower value (the default
 * is one NAT gateway per availability zone). Be aware that this may have
 * availability implications for your application.
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html">Read more about
 * subnets</a>.
 * <p>
 * <h3>Control over availability zones</h3>
 * <p>
 * By default, a VPC will spread over at most 3 Availability Zones available to
 * it. To change the number of Availability Zones that the VPC will spread over,
 * specify the <code>maxAzs</code> property when defining it.
 * <p>
 * The number of Availability Zones that are available depends on the <em>region</em>
 * and <em>account</em> of the Stack containing the VPC. If the <a href="https://docs.aws.amazon.com/cdk/latest/guide/environments.html">region and account are
 * specified</a> on
 * the Stack, the CLI will <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#using-regions-availability-zones-describe">look up the existing Availability
 * Zones</a>
 * and get an accurate count. If region and account are not specified, the stack
 * could be deployed anywhere and it will have to make a safe choice, limiting
 * itself to 2 Availability Zones.
 * <p>
 * Therefore, to get the VPC to spread over 3 or more availability zones, you
 * must specify the environment where the stack will be deployed.
 * <p>
 * <h3>Using NAT instances</h3>
 * <p>
 * By default, the <code>Vpc</code> construct will create NAT <em>gateways</em> for you, which
 * are managed by AWS. If you would prefer to use your own managed NAT
 * <em>instances</em> instead, specify a different value for the <code>natGatewayProvider</code>
 * property, as follows:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Configure the `natGatewayProvider` when defining a Vpc
 * NatProvider natGatewayProvider = ec2.NatProvider.instance(new NatInstanceProps()
 *         .instanceType(new InstanceType("t3.small")));
 * 
 * Vpc vpc = new Vpc(this, "MyVpc", new VpcProps()
 *         .natGatewayProvider(natGatewayProvider)
 * 
 *         // The 'natGateways' parameter now controls the number of NAT instances
 *         .natGateways(2));
 * </pre></blockquote>
 * <p>
 * The construct will automatically search for the most recent NAT gateway AMI.
 * If you prefer to use a custom AMI, use <code>machineImage: MachineImage.genericLinux({ ... })</code> and configure the right AMI ID for the
 * regions you want to deploy to.
 * <p>
 * <h3>Advanced Subnet Configuration</h3>
 * <p>
 * If the default VPC configuration (public and private subnets spanning the
 * size of the VPC) don't suffice for you, you can configure what subnets to
 * create by specifying the <code>subnetConfiguration</code> property. It allows you
 * to configure the number and size of all subnets. Specifying an advanced
 * subnet configuration could look like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "TheVPC", new VpcProps()
 *         // 'cidr' configures the IP range and size of the entire VPC.
 *         // The IP space will be divided over the configured subnets.
 *         .cidr("10.0.0.0/21")
 * 
 *         // 'maxAzs' configures the maximum number of availability zones to use
 *         .maxAzs(3)
 * 
 *         // 'subnetConfiguration' specifies the "subnet groups" to create.
 *         // Every subnet group will have a subnet for each AZ, so this
 *         // configuration will create `3 groups × 3 AZs = 9` subnets.
 *         .subnetConfiguration(asList(new SubnetConfiguration()
 *                 // 'subnetType' controls Internet access, as described above.
 *                 .subnetType(ec2.SubnetType.getPUBLIC())
 * 
 *                 // 'name' is used to name this particular subnet group. You will have to
 *                 // use the name for subnet selection if you have more than one subnet
 *                 // group of the same type.
 *                 .name("Ingress")
 * 
 *                 // 'cidrMask' specifies the IP addresses in the range of of individual
 *                 // subnets in the group. Each of the subnets in this group will contain
 *                 // `2^(32 address bits - 24 subnet bits) - 2 reserved addresses = 254`
 *                 // usable IP addresses.
 *                 //
 *                 // If 'cidrMask' is left out the available address space is evenly
 *                 // divided across the remaining subnet groups.
 *                 .cidrMask(24), new SubnetConfiguration()
 *                 .cidrMask(24)
 *                 .name("Application")
 *                 .subnetType(ec2.SubnetType.getPRIVATE()), new SubnetConfiguration()
 *                 .cidrMask(28)
 *                 .name("Database")
 *                 .subnetType(ec2.SubnetType.getISOLATED())
 * 
 *                 // 'reserved' can be used to reserve IP address space. No resources will
 *                 // be created for this subnet, but the IP range will be kept available for
 *                 // future creation of this subnet, or even for future subdivision.
 *                 .reserved(true))));
 * </pre></blockquote>
 * <p>
 * The example above is one possible configuration, but the user can use the
 * constructs above to implement many other network configurations.
 * <p>
 * The <code>Vpc</code> from the above configuration in a Region with three
 * availability zones will be the following:
 * <p>
 * Subnet Name       |Type      |IP Block      |AZ|Features
 * ------------------|----------|--------------|--|--------
 * IngressSubnet1    |<code>PUBLIC</code>  |<code>10.0.0.0/24</code> |#1|NAT Gateway
 * IngressSubnet2    |<code>PUBLIC</code>  |<code>10.0.1.0/24</code> |#2|NAT Gateway
 * IngressSubnet3    |<code>PUBLIC</code>  |<code>10.0.2.0/24</code> |#3|NAT Gateway
 * ApplicationSubnet1|<code>PRIVATE</code> |<code>10.0.3.0/24</code> |#1|Route to NAT in IngressSubnet1
 * ApplicationSubnet2|<code>PRIVATE</code> |<code>10.0.4.0/24</code> |#2|Route to NAT in IngressSubnet2
 * ApplicationSubnet3|<code>PRIVATE</code> |<code>10.0.5.0/24</code> |#3|Route to NAT in IngressSubnet3
 * DatabaseSubnet1   |<code>ISOLATED</code>|<code>10.0.6.0/28</code> |#1|Only routes within the VPC
 * DatabaseSubnet2   |<code>ISOLATED</code>|<code>10.0.6.16/28</code>|#2|Only routes within the VPC
 * DatabaseSubnet3   |<code>ISOLATED</code>|<code>10.0.6.32/28</code>|#3|Only routes within the VPC
 * <p>
 * <h3>Reserving subnet IP space</h3>
 * <p>
 * There are situations where the IP space for a subnet or number of subnets
 * will need to be reserved. This is useful in situations where subnets would
 * need to be added after the vpc is originally deployed, without causing IP
 * renumbering for existing subnets. The IP space for a subnet may be reserved
 * by setting the <code>reserved</code> subnetConfiguration property to true, as shown
 * below:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "TheVPC", new VpcProps()
 *         .natGateways(1)
 *         .subnetConfiguration(asList(new SubnetConfiguration()
 *                 .cidrMask(26)
 *                 .name("Public")
 *                 .subnetType(ec2.SubnetType.getPUBLIC()), new SubnetConfiguration()
 *                 .cidrMask(26)
 *                 .name("Application1")
 *                 .subnetType(ec2.SubnetType.getPRIVATE()), new SubnetConfiguration()
 *                 .cidrMask(26)
 *                 .name("Application2")
 *                 .subnetType(ec2.SubnetType.getPRIVATE())
 *                 .reserved(true), new SubnetConfiguration()
 *                 .cidrMask(27)
 *                 .name("Database")
 *                 .subnetType(ec2.SubnetType.getISOLATED()))));
 * </pre></blockquote>
 * <p>
 * In the example above, the subnet for Application2 is not actually provisioned
 * but its IP space is still reserved. If in the future this subnet needs to be
 * provisioned, then the <code>reserved: true</code> property should be removed. Reserving
 * parts of the IP space prevents the other subnets from getting renumbered.
 * <p>
 * <h3>Sharing VPCs between stacks</h3>
 * <p>
 * If you are creating multiple <code>Stack</code>s inside the same CDK application, you
 * can reuse a VPC defined in one Stack in another by simply passing the VPC
 * instance around:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * /**
 *  * Stack1 creates the VPC
 *  *{@literal /}
 * public class Stack1 extends Stack {
 *     public final Vpc vpc;
 * 
 *     public Stack1(App scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public Stack1(App scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         this.vpc = new Vpc(this, "VPC");
 *     }
 * }
 * 
 * public class Stack2Props extends StackProps {
 *     private IVpc vpc;
 *     public IVpc getVpc() {
 *         return this.vpc;
 *     }
 *     public Stack2Props vpc(IVpc vpc) {
 *         this.vpc = vpc;
 *         return this;
 *     }
 * }
 * 
 * /**
 *  * Stack2 consumes the VPC
 *  *{@literal /}
 * public class Stack2 extends Stack {
 *     public Stack2(App scope, String id, Stack2Props props) {
 *         super(scope, id, props);
 * 
 *         // Pass the VPC to a construct that needs it
 *         // Pass the VPC to a construct that needs it
 *         new ConstructThatTakesAVpc(this, "Construct", new ConstructThatTakesAVpcProps()
 *                 .vpc(props.getVpc()));
 *     }
 * }
 * 
 * Stack1 stack1 = new Stack1(app, "Stack1");
 * Stack2 stack2 = new Stack2(app, "Stack2", new Stack2Props()
 *         .vpc(stack1.getVpc()));
 * </pre></blockquote>
 * <p>
 * <h3>Importing an existing VPC</h3>
 * <p>
 * If your VPC is created outside your CDK app, you can use <code>Vpc.fromLookup()</code>.
 * The CDK CLI will search for the specified VPC in the the stack's region and
 * account, and import the subnet configuration. Looking up can be done by VPC
 * ID, but more flexibly by searching for a specific tag on the VPC.
 * <p>
 * Subnet types will be determined from the <code>aws-cdk:subnet-type</code> tag on the
 * subnet if it exists, or the presence of a route to an Internet Gateway
 * otherwise. Subnet names will be determined from the <code>aws-cdk:subnet-name</code> tag
 * on the subnet if it exists, or will mirror the subnet type otherwise (i.e.
 * a public subnet will have the name <code>"Public"</code>).
 * <p>
 * Here's how <code>Vpc.fromLookup()</code> can be used:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * IVpc vpc = ec2.Vpc.fromLookup(stack, "VPC", new VpcLookupOptions()
 *         // This imports the default VPC but you can also
 *         // specify a 'vpcName' or 'tags'.
 *         .isDefault(true));
 * </pre></blockquote>
 * <p>
 * <h2>Allowing Connections</h2>
 * <p>
 * In AWS, all network traffic in and out of <strong>Elastic Network Interfaces</strong> (ENIs)
 * is controlled by <strong>Security Groups</strong>. You can think of Security Groups as a
 * firewall with a set of rules. By default, Security Groups allow no incoming
 * (ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
 * to them to allow incoming traffic streams. To exert fine-grained control over
 * egress traffic, set <code>allowAllOutbound: false</code> on the <code>SecurityGroup</code>, after
 * which you can add egress traffic rules.
 * <p>
 * You can manipulate Security Groups directly:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SecurityGroup mySecurityGroup = new SecurityGroup(this, "SecurityGroup", new SecurityGroupProps()
 *         .vpc(vpc)
 *         .description("Allow ssh access to ec2 instances")
 *         .allowAllOutbound(true));
 * mySecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), "allow ssh access from the world");
 * </pre></blockquote>
 * <p>
 * All constructs that create ENIs on your behalf (typically constructs that create
 * EC2 instances or other VPC-connected resources) will all have security groups
 * automatically assigned. Those constructs have an attribute called
 * <strong>connections</strong>, which is an object that makes it convenient to update the
 * security groups. If you want to allow connections between two constructs that
 * have security groups, you have to add an <strong>Egress</strong> rule to one Security Group,
 * and an <strong>Ingress</strong> rule to the other. The connections object will automatically
 * take care of this for you:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Allow connections from anywhere
 * loadBalancer.connections.allowFromAnyIpv4(ec2.Port.tcp(443), "Allow inbound HTTPS");
 * 
 * // The same, but an explicit IP address
 * loadBalancer.connections.allowFrom(ec2.Peer.ipv4("1.2.3.4/32"), ec2.Port.tcp(443), "Allow inbound HTTPS");
 * 
 * // Allow connection between AutoScalingGroups
 * appFleet.connections.allowTo(dbFleet, ec2.Port.tcp(443), "App can call database");
 * </pre></blockquote>
 * <p>
 * <h3>Connection Peers</h3>
 * <p>
 * There are various classes that implement the connection peer part:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Simple connection peers
 * IPeer peer = ec2.Peer.ipv4("10.0.0.0/16");
 * peer = ec2.Peer.anyIpv4();
 * peer = ec2.Peer.ipv6("::0/0");
 * peer = ec2.Peer.anyIpv6();
 * peer = ec2.Peer.prefixList("pl-12345");
 * appFleet.connections.allowTo(peer, ec2.Port.tcp(443), "Allow outbound HTTPS");
 * </pre></blockquote>
 * <p>
 * Any object that has a security group can itself be used as a connection peer:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // These automatically create appropriate ingress and egress rules in both security groups
 * fleet1.connections.allowTo(fleet2, ec2.Port.tcp(80), "Allow between fleets");
 * 
 * appFleet.connections.allowFromAnyIpv4(ec2.Port.tcp(80), "Allow from load balancer");
 * </pre></blockquote>
 * <p>
 * <h3>Port Ranges</h3>
 * <p>
 * The connections that are allowed are specified by port ranges. A number of classes provide
 * the connection specifier:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * ec2.Port.tcp(80);
 * ec2.Port.tcpRange(60000, 65535);
 * ec2.Port.allTcp();
 * ec2.Port.allTraffic();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: This set is not complete yet; for example, there is no library support for ICMP at the moment.
 * However, you can write your own classes to implement those.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Default Ports</h3>
 * <p>
 * Some Constructs have default ports associated with them. For example, the
 * listener of a load balancer does (it's the public port), or instances of an
 * RDS database (it's the port the database is accepting connections on).
 * <p>
 * If the object you're calling the peering method on has a default port associated with it, you can call
 * <code>allowDefaultPortFrom()</code> and omit the port specifier. If the argument has an associated default port, call
 * <code>allowDefaultPortTo()</code>.
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Port implicit in listener
 * listener.connections.allowDefaultPortFromAnyIpv4("Allow public");
 * 
 * // Port implicit in peer
 * appFleet.connections.allowDefaultPortTo(rdsDatabase, "Fleet can access database");
 * </pre></blockquote>
 * <p>
 * <h2>Machine Images (AMIs)</h2>
 * <p>
 * AMIs control the OS that gets launched when you start your EC2 instance. The EC2
 * library contains constructs to select the AMI you want to use.
 * <p>
 * Depending on the type of AMI, you select it a different way. Here are some
 * examples of things you might want to use:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Pick the right Amazon Linux edition. All arguments shown are optional
 * // and will default to these values when omitted.
 * IMachineImage amznLinux = ec2.MachineImage.latestAmazonLinux(new AmazonLinuxImageProps()
 *         .generation(ec2.AmazonLinuxGeneration.getAMAZON_LINUX())
 *         .edition(ec2.AmazonLinuxEdition.getSTANDARD())
 *         .virtualization(ec2.AmazonLinuxVirt.getHVM())
 *         .storage(ec2.AmazonLinuxStorage.getGENERAL_PURPOSE()));
 * 
 * // Pick a Windows edition to use
 * IMachineImage windows = ec2.MachineImage.latestWindows(ec2.WindowsVersion.getWINDOWS_SERVER_2019_ENGLISH_FULL_BASE());
 * 
 * // Look up the most recent image matching a set of AMI filters.
 * // In this case, look up the NAT instance AMI, by using a wildcard
 * // in the 'name' field:
 * IMachineImage natAmi = ec2.MachineImage.lookup(new LookupMachineImageProps()
 *         .name("amzn-ami-vpc-nat-*")
 *         .owners(asList("amazon")));
 * 
 * // For other custom (Linux) images, instantiate a `GenericLinuxImage` with
 * // a map giving the AMI to in for each region:
 * IMachineImage linux = ec2.MachineImage.genericLinux(Map.of(
 *         "us-east-1", "ami-97785bed",
 *         "eu-west-1", "ami-12345678"));
 * 
 * // For other custom (Windows) images, instantiate a `GenericWindowsImage` with
 * // a map giving the AMI to in for each region:
 * IMachineImage genericWindows = ec2.MachineImage.genericWindows(Map.of(
 *         "us-east-1", "ami-97785bed",
 *         "eu-west-1", "ami-12345678"));
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: The AMIs selected by <code>MachineImage.lookup()</code> will be cached in
 * <code>cdk.context.json</code>, so that your AutoScalingGroup instances aren't replaced while
 * you are making unrelated changes to your CDK app.
 * <p>
 * To query for the latest AMI again, remove the relevant cache entry from
 * <code>cdk.context.json</code>, or use the <code>cdk context</code> command. For more information, see
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/context.html">Runtime Context</a> in the CDK
 * developer guide.
 * <p>
 * </blockquote>
 * <p>
 * <h2>VPN connections to a VPC</h2>
 * <p>
 * Create your VPC with VPN connections by specifying the <code>vpnConnections</code> props (keys are construct <code>id</code>s):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "MyVpc", new VpcProps()
 *         .vpnConnections(Map.of(
 *                 "dynamic", new VpnConnectionOptions()// Dynamic routing (BGP)
 *                         .ip("1.2.3.4"),
 *                 "static", new VpnConnectionOptions()// Static routing
 *                         .ip("4.5.6.7")
 *                         .staticRoutes(asList("192.168.10.0/24", "192.168.20.0/24")))));
 * </pre></blockquote>
 * <p>
 * To create a VPC that can accept VPN connections, set <code>vpnGateway</code> to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "MyVpc", new VpcProps()
 *         .vpnGateway(true));
 * </pre></blockquote>
 * <p>
 * VPN connections can then be added:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * vpc.addVpnConnection("Dynamic", new VpnConnectionOptions()
 *         .ip("1.2.3.4"));
 * </pre></blockquote>
 * <p>
 * By default, routes will be propagated on the route tables associated with the private subnets. If no
 * private subnets exists, isolated subnets are used. If no isolated subnets exists, public subnets are
 * used. Use the <code>Vpc</code> property <code>vpnRoutePropagation</code> to customize this behavior.
 * <p>
 * VPN connections expose <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-cloudwatch/README.md">metrics (cloudwatch.Metric)</a> across all tunnels in the account/region and per connection:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Across all tunnels in the account/region
 * Metric allDataOut = ec2.VpnConnection.metricAllTunnelDataOut();
 * 
 * // For a specific vpn connection
 * VpnConnection vpnConnection = vpc.addVpnConnection("Dynamic", new VpnConnectionOptions()
 *         .ip("1.2.3.4"));
 * Metric state = vpnConnection.metricTunnelState();
 * </pre></blockquote>
 * <p>
 * <h2>VPC endpoints</h2>
 * <p>
 * A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.
 * <p>
 * Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Add gateway endpoints when creating the VPC
 * Vpc vpc = new Vpc(this, "MyVpc", new VpcProps()
 *         .gatewayEndpoints(Map.of(
 *                 "S3", new GatewayVpcEndpointOptions()
 *                         .service(ec2.GatewayVpcEndpointAwsService.getS3()))));
 * 
 * // Alternatively gateway endpoints can be added on the VPC
 * GatewayVpcEndpoint dynamoDbEndpoint = vpc.addGatewayEndpoint("DynamoDbEndpoint", new GatewayVpcEndpointOptions()
 *         .service(ec2.GatewayVpcEndpointAwsService.getDYNAMODB()));
 * 
 * // This allows to customize the endpoint policy
 * dynamoDbEndpoint.addToPolicy(
 * new PolicyStatement(new PolicyStatementProps()// Restrict to listing and describing tables
 *         .principals(asList(new AnyPrincipal()))
 *         .actions(asList("dynamodb:DescribeTable", "dynamodb:ListTables"))
 *         .resources(asList("*"))));
 * 
 * // Add an interface endpoint
 * vpc.addInterfaceEndpoint("EcrDockerEndpoint", Map.of(
 *         "service", ec2.InterfaceVpcEndpointAwsService.getECR_DOCKER()));
 * </pre></blockquote>
 * <p>
 * <h3>Security groups for interface VPC endpoints</h3>
 * <p>
 * By default, interface VPC endpoints create a new security group and traffic is <strong>not</strong>
 * automatically allowed from the VPC CIDR.
 * <p>
 * Use the <code>connections</code> object to allow traffic to flow to the endpoint:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * myEndpoint.connections.allowDefaultPortFromAnyIpv4();
 * </pre></blockquote>
 * <p>
 * Alternatively, existing security groups can be used by specifying the <code>securityGroups</code> prop.
 * <p>
 * <h2>VPC endpoint services</h2>
 * <p>
 * A VPC endpoint service enables you to expose a Network Load Balancer(s) as a provider service to consumers, who connect to your service over a VPC endpoint. You can restrict access to your service via whitelisted principals (anything that extends ArnPrincipal), and require that new connections be manually accepted.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * VpcEndpointService.Builder.create(this, "EndpointService")
 *         .vpcEndpointServiceLoadBalancers(asList(networkLoadBalancer1, networkLoadBalancer2))
 *         .acceptanceRequired(true)
 *         .whitelistedPrincipals(asList(new ArnPrincipal("arn:aws:iam::123456789012:root")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Bastion Hosts</h2>
 * <p>
 * A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
 * You can use bastion hosts using a standard SSH connection targetting port 22 on the host. As an alternative, you can connect the SSH connection
 * feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)
 * <p>
 * A default bastion host for use via SSM can be configured like:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * BastionHostLinux host = new BastionHostLinux(this, "BastionHost", new BastionHostLinuxProps().vpc(vpc));
 * </pre></blockquote>
 * <p>
 * If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * BastionHostLinux host = new BastionHostLinux(this, "BastionHost", new BastionHostLinuxProps()
 *         .vpc(vpc)
 *         .subnetSelection(new SubnetSelection().subnetType(ec2.SubnetType.getPUBLIC())));
 * host.allowSshAccessFrom(ec2.Peer.ipv4("1.2.3.4/32"));
 * </pre></blockquote>
 * <p>
 * As there are no SSH public keys deployed on this machine, you need to use <a href="https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/">EC2 Instance Connect</a>
 * with the command <code>aws ec2-instance-connect send-ssh-public-key</code> to provide your SSH public key.
 * <p>
 * <h2>Block Devices</h2>
 * <p>
 * To add EBS block device mappings, specify the <code>blockDeviceMappings</code> property. The follow example sets the EBS-backed
 * root device (<code>/dev/sda1</code>) size to 50 GiB, and adds another EBS-backed device mapped to <code>/dev/sdm</code> that is 100 GiB in
 * size:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Instance.Builder.create(this, "Instance")Map.of(
 *         // ...
 *         "blockDeviceMappings", asList(
 *                 .deviceName("/dev/sda1")
 *                 .volume(ec2.BlockDeviceVolume.ebs(50))
 *                 .build(),
 *                 .deviceName("/dev/sdm")
 *                 .volume(ec2.BlockDeviceVolume.ebs(100))
 *                 .build()));
 * </pre></blockquote>
 * <p>
 * <h2>VPC Flow Logs</h2>
 * <p>
 * VPC Flow Logs is a feature that enables you to capture information about the IP traffic going to and from network interfaces in your VPC. Flow log data can be published to Amazon CloudWatch Logs and Amazon S3. After you've created a flow log, you can retrieve and view its data in the chosen destination. (https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html).
 * <p>
 * By default a flow log will be created with CloudWatch Logs as the destination.
 * <p>
 * You can create a flow log like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * new FlowLog(this, "FlowLog", new FlowLogProps()
 *         .resourceType(ec2.FlowLogResourceType.fromVpc(vpc)));
 * </pre></blockquote>
 * <p>
 * Or you can add a Flow Log to a VPC by using the addFlowLog method like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "Vpc");
 * 
 * vpc.addFlowLog("FlowLog");
 * </pre></blockquote>
 * <p>
 * You can also add multiple flow logs with different destinations.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Vpc vpc = new Vpc(this, "Vpc");
 * 
 * vpc.addFlowLog("FlowLogS3", new FlowLogOptions()
 *         .destination(ec2.FlowLogDestination.toS3()));
 * 
 * vpc.addFlowLog("FlowLogCloudWatch", new FlowLogOptions()
 *         .trafficType(ec2.FlowLogTrafficType.getREJECT()));
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.ec2;
