/**
 * <h2>Amazon EC2 Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>The <code>@aws-cdk/aws-ec2</code> package contains primitives for setting up networking and
 * instances.</p>
 * <h3>VPC</h3>
 * <p>Most projects need a Virtual Private Cloud to provide security by means of
 * network partitioning. This is easily achieved by creating an instance of
 * <code>Vpc</code>:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * 
 * const vpc = new ec2.Vpc(this, 'VPC');
 * </code></pre>
 * <p>All default Constructs requires EC2 instances to be launched inside a VPC, so
 * you should generally start by defining a VPC whenever you need to launch
 * instances for your project.</p>
 * <p>Our default <code>Vpc</code> class creates a private and public subnet for every
 * availability zone. Classes that use the VPC will generally launch instances
 * into all private subnets, and provide a parameter called <code>vpcSubnets</code> to
 * allow you to override the placement. <a href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html">Read more about
 * subnets</a>.</p>
 * <h4>Advanced Subnet Configuration</h4>
 * <p>If you require the ability to configure subnets the <code>Vpc</code> can be
 * customized with <code>SubnetConfiguration</code> array. This is best explained by an
 * example:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * 
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: '10.0.0.0/21',
 *   subnetConfiguration: [
 *     {
 *       cidrMask: 24,
 *       name: 'Ingress',
 *       subnetType: ec2.SubnetType.PUBLIC,
 *     },
 *     {
 *       cidrMask: 24,
 *       name: 'Application',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *     },
 *     {
 *       cidrMask: 28,
 *       name: 'Database',
 *       subnetType: ec2.SubnetType.ISOLATED,
 *     }
 *   ],
 * });
 * </code></pre>
 * <p>The example above is one possible configuration, but the user can use the
 * constructs above to implement many other network configurations.</p>
 * <p>The <code>Vpc</code> from the above configuration in a Region with three
 * availability zones will be the following:</p>
 * <p>Subnet Name       |Type      |IP Block      |AZ|Features
 * ------------------|----------|--------------|--|--------
 * IngressSubnet1    |<code>PUBLIC</code>  |<code>10.0.0.0/24</code> |#1|NAT Gateway
 * IngressSubnet2    |<code>PUBLIC</code>  |<code>10.0.1.0/24</code> |#2|NAT Gateway
 * IngressSubnet3    |<code>PUBLIC</code>  |<code>10.0.2.0/24</code> |#3|NAT Gateway
 * ApplicationSubnet1|<code>PRIVATE</code> |<code>10.0.3.0/24</code> |#1|Route to NAT in IngressSubnet1
 * ApplicationSubnet2|<code>PRIVATE</code> |<code>10.0.4.0/24</code> |#2|Route to NAT in IngressSubnet2
 * ApplicationSubnet3|<code>PRIVATE</code> |<code>10.0.5.0/24</code> |#3|Route to NAT in IngressSubnet3
 * DatabaseSubnet1   |<code>ISOLATED</code>|<code>10.0.6.0/28</code> |#1|Only routes within the VPC
 * DatabaseSubnet2   |<code>ISOLATED</code>|<code>10.0.6.16/28</code>|#2|Only routes within the VPC
 * DatabaseSubnet3   |<code>ISOLATED</code>|<code>10.0.6.32/28</code>|#3|Only routes within the VPC</p>
 * <p>Each <code>Public</code> Subnet will have a NAT Gateway. Each <code>Private</code> Subnet will have a
 * route to the NAT Gateway in the same availability zone. <code>Isolated</code> subnets
 * will not have a route to the internet, but are routeable within the VPC. The
 * numbers [1-3] will consistently map to availability zones (e.g. <em>IngressSubnet1</em>
 * and <em>ApplicationSubnet1</em> will be in the same avialbility zone).</p>
 * <p><code>Isolated</code> Subnets provide simplified secure networking principles, but come at
 * an operational complexity. The lack of an internet route means that if you deploy
 * instances in this subnet you will not be able to patch from the internet, this is
 * commonly reffered to as
 * <a href="https://aws.amazon.com/answers/configuration-management/aws-ami-design/">fully baked images</a>.
 * Features such as
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-signal.html">cfn-signal</a>
 * are also unavailable. Using these subnets for managed services (RDS,
 * Elasticache, Redshift) is a very practical use because the managed services do
 * not incur additional operational overhead.</p>
 * <p>Many times when you plan to build an application you don't know how many
 * instances of the application you will need and therefore you don't know how much
 * IP space to allocate. For example, you know the application will only have
 * Elastic Loadbalancers in the public subnets and you know you will have 1-3 RDS
 * databases for your data tier, and the rest of the IP space should just be evenly
 * distributed for the application.</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * 
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: '10.0.0.0/16',
 *   subnetConfiguration: [
 *     {
 *       cidrMask: 26,
 *       name: 'Public',
 *       subnetType: ec2.SubnetType.PUBLIC,
 *     },
 *     {
 *       name: 'Application',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *     },
 *     {
 *       cidrMask: 27,
 *       name: 'Database',
 *       subnetType: ec2.SubnetType.ISOLATED,
 *     }
 *   ],
 * });
 * </code></pre>
 * <p>The <code>Vpc</code> from the above configuration in a Region with three
 * availability zones will be the following:</p>
 * <p>Subnet Name       |Type      | IP Block
 * ------------------|----------|----------------
 * PublicSubnet1     |<code>PUBLIC</code>  |<code>10.0.0.0/26</code>
 * PublicSubnet2     |<code>PUBLIC</code>  |<code>10.0.0.64/26</code>
 * PublicSubnet3     |<code>PUBLIC</code>  |<code>10.0.2.128/26</code>
 * DatabaseSubnet1   |<code>PRIVATE</code> |<code>10.0.0.192/27</code>
 * DatabaseSubnet2   |<code>PRIVATE</code> |<code>10.0.0.224/27</code>
 * DatabaseSubnet3   |<code>PRIVATE</code> |<code>10.0.1.0/27</code>
 * ApplicationSubnet1|<code>ISOLATED</code>|<code>10.0.64.0/18</code>
 * ApplicationSubnet2|<code>ISOLATED</code>|<code>10.0.128.0/18</code>
 * ApplicationSubnet3|<code>ISOLATED</code>|<code>10.0.192.0/18</code></p>
 * <p>Any subnet configuration without a <code>cidrMask</code> will be counted up and allocated
 * evenly across the remaining IP space.</p>
 * <p>Teams may also become cost conscious and be willing to trade availability for
 * cost. For example, in your test environments perhaps you would like the same VPC
 * as production, but instead of 3 NAT Gateways you would like only 1. This will
 * save on the cost, but trade the 3 availability zone to a 1 for all egress
 * traffic. This can be accomplished with a single parameter configuration:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * 
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: '10.0.0.0/16',
 *   natGateways: 1,
 *   natGatewayPlacement: {subnetName: 'Public'},
 *   subnetConfiguration: [
 *     {
 *       cidrMask: 26,
 *       name: 'Public',
 *       subnetType: ec2.SubnetType.PUBLIC,
 *       natGateway: true,
 *     },
 *     {
 *       name: 'Application',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *     },
 *     {
 *       cidrMask: 27,
 *       name: 'Database',
 *       subnetType: ec2.SubnetType.ISOLATED,
 *     }
 *   ],
 * });
 * </code></pre>
 * <p>The <code>Vpc</code> above will have the exact same subnet definitions as listed
 * above. However, this time the VPC will have only 1 NAT Gateway and all
 * Application subnets will route to the NAT Gateway.</p>
 * <h4>Reserving subnet IP space</h4>
 * <p>There are situations where the IP space for a subnet or number of subnets
 * will need to be reserved. This is useful in situations where subnets
 * would need to be added after the vpc is originally deployed, without causing
 * IP renumbering for existing subnets. The IP space for a subnet may be reserved
 * by setting the <code>reserved</code> subnetConfiguration property to true, as shown below:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: '10.0.0.0/16',
 *   natGateways: 1,
 *   subnetConfiguration: [
 *     {
 *       cidrMask: 26,
 *       name: 'Public',
 *       subnetType: ec2.SubnetType.PUBLIC,
 *     },
 *     {
 *       cidrMask: 26,
 *       name: 'Application1',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *     },
 *     {
 *       cidrMask: 26,
 *       name: 'Application2',
 *       subnetType: ec2.SubnetType.PRIVATE,
 *       reserved: true,
 *     },
 *     {
 *       cidrMask: 27,
 *       name: 'Database',
 *       subnetType: ec2.SubnetType.ISOLATED,
 *     }
 *   ],
 * });
 * </code></pre>
 * <p>In the example above, the subnet for Application2 is not actually provisioned
 * but its IP space is still reserved. If in the future this subnet needs to be
 * provisioned, then the <code>reserved: true</code> property should be removed. Most
 * importantly, this action would not cause the Database subnet to get renumbered,
 * but rather the IP space that was previously reserved will be used for the
 * subnet provisioned for Application2. The <code>reserved</code> property also takes into
 * consideration the number of availability zones when reserving IP space.</p>
 * <h4>Sharing VPCs between stacks</h4>
 * <p>If you are creating multiple <code>Stack</code>s inside the same CDK application, you
 * can reuse a VPC defined in one Stack in another by simply passing the VPC
 * instance around:</p>
 * <pre><code class="language-ts">/**
 *  * Stack1 creates the VPC
 *  *{@literal /}
 * class Stack1 extends cdk.Stack {
 *   public readonly vpc: ec2.Vpc;
 * 
 *   constructor(scope: cdk.App, id: string, props?: cdk.StackProps) {
 *     super(scope, id, props);
 * 
 *     this.vpc = new ec2.Vpc(this, 'VPC');
 *   }
 * }
 * 
 * interface Stack2Props extends cdk.StackProps {
 *   vpc: ec2.IVpc;
 * }
 * 
 * /**
 *  * Stack2 consumes the VPC
 *  *{@literal /}
 * class Stack2 extends cdk.Stack {
 *   constructor(scope: cdk.App, id: string, props: Stack2Props) {
 *     super(scope, id, props);
 * 
 *     // Pass the VPC to a construct that needs it
 *     new ConstructThatTakesAVpc(this, 'Construct', {
 *       vpc: props.vpc
 *     });
 *   }
 * }
 * 
 * const stack1 = new Stack1(app, 'Stack1');
 * const stack2 = new Stack2(app, 'Stack2', {
 *   vpc: stack1.vpc,
 * });
 * </code></pre>
 * <h4>Importing an existing VPC</h4>
 * <p>If your VPC is created outside your CDK app, you can use <code>Vpc.fromLookup()</code>:</p>
 * <pre><code class="language-ts">const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
 *   // This imports the default VPC but you can also
 *   // specify a 'vpcName' or 'tags'.
 *   isDefault: true
 * });
 * </code></pre>
 * <h3>Allowing Connections</h3>
 * <p>In AWS, all network traffic in and out of <strong>Elastic Network Interfaces</strong> (ENIs)
 * is controlled by <strong>Security Groups</strong>. You can think of Security Groups as a
 * firewall with a set of rules. By default, Security Groups allow no incoming
 * (ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
 * to them to allow incoming traffic streams. To exert fine-grained control over
 * egress traffic, set <code>allowAllOutbound: false</code> on the <code>SecurityGroup</code>, after
 * which you can add egress traffic rules.</p>
 * <p>You can manipulate Security Groups directly:</p>
 * <pre><code class="language-ts">const mySecurityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
 *   vpc,
 *   description: 'Allow ssh access to ec2 instances',
 *   allowAllOutbound: true   // Can be set to false
 * });
 * mySecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
 * </code></pre>
 * <p>All constructs that create ENIs on your behalf (typically constructs that create
 * EC2 instances or other VPC-connected resources) will all have security groups
 * automatically assigned. Those constructs have an attribute called
 * <strong>connections</strong>, which is an object that makes it convenient to update the
 * security groups. If you want to allow connections between two constructs that
 * have security groups, you have to add an <strong>Egress</strong> rule to one Security Group,
 * and an <strong>Ingress</strong> rule to the other. The connections object will automatically
 * take care of this for you:</p>
 * <pre><code class="language-ts">// Allow connections from anywhere
 * loadBalancer.connections.allowFromAnyIpv4(ec2.Port.tcp(443), 'Allow inbound HTTPS');
 * 
 * // The same, but an explicit IP address
 * loadBalancer.connections.allowFrom(ec2.Peer.ipv4('1.2.3.4/32'), ec2.Port.tcp(443), 'Allow inbound HTTPS');
 * 
 * // Allow connection between AutoScalingGroups
 * appFleet.connections.allowTo(dbFleet, ec2.Port.tcp(443), 'App can call database');
 * </code></pre>
 * <h3>Connection Peers</h3>
 * <p>There are various classes that implement the connection peer part:</p>
 * <pre><code class="language-ts">// Simple connection peers
 * let peer = ec2.Peer.ipv4(&quot;10.0.0.0/16&quot;);
 * let peer = ec2.Peer.anyIpv4();
 * let peer = ec2.ipv6(&quot;::0/0&quot;);
 * let peer = ec2.anyIpv6();
 * let peer = ec2.prefixList(&quot;pl-12345&quot;);
 * fleet.connections.allowTo(peer, ec2.Port.tcp(443), 'Allow outbound HTTPS');
 * </code></pre>
 * <p>Any object that has a security group can itself be used as a connection peer:</p>
 * <pre><code class="language-ts">// These automatically create appropriate ingress and egress rules in both security groups
 * fleet1.connections.allowTo(fleet2, ec2.Port.tcp(80), 'Allow between fleets');
 * 
 * fleet.connections.allowFromAnyIpv4(ec2.Port.tcp(80), 'Allow from load balancer');
 * </code></pre>
 * <h3>Port Ranges</h3>
 * <p>The connections that are allowed are specified by port ranges. A number of classes provide
 * the connection specifier:</p>
 * <pre><code class="language-ts">ec2.Port.tcp(80)
 * ec2.tcpRange(60000, 65535)
 * ec2.allTcp()
 * ec2.allTraffic()
 * </code></pre>
 * <blockquote>
 * <p>NOTE: This set is not complete yet; for example, there is no library support for ICMP at the moment.
 * However, you can write your own classes to implement those.</p>
 * </blockquote>
 * <h3>Default Ports</h3>
 * <p>Some Constructs have default ports associated with them. For example, the
 * listener of a load balancer does (it's the public port), or instances of an
 * RDS database (it's the port the database is accepting connections on).</p>
 * <p>If the object you're calling the peering method on has a default port associated with it, you can call
 * <code>allowDefaultPortFrom()</code> and omit the port specifier. If the argument has an associated default port, call
 * <code>allowDefaultPortTo()</code>.</p>
 * <p>For example:</p>
 * <pre><code class="language-ts">// Port implicit in listener
 * listener.connections.allowDefaultPortFromAnyIpv4('Allow public');
 * 
 * // Port implicit in peer
 * fleet.connections.allowDefaultPortTo(rdsDatabase, 'Fleet can access database');
 * </code></pre>
 * <h3>Machine Images (AMIs)</h3>
 * <p>AMIs control the OS that gets launched when you start your EC2 instance. The EC2
 * library contains constructs to select the AMI you want to use.</p>
 * <p>Depending on the type of AMI, you select it a different way.</p>
 * <p>The latest version of Amazon Linux and Microsoft Windows images are
 * selectable by instantiating one of these classes:</p>
 * <pre><code class="language-ts">// Pick a Windows edition to use
 * const windows = new ec2.WindowsImage(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE);
 * 
 * // Pick the right Amazon Linux edition. All arguments shown are optional
 * // and will default to these values when omitted.
 * const amznLinux = new ec2.AmazonLinuxImage({
 *   generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX,
 *   edition: ec2.AmazonLinuxEdition.STANDARD,
 *   virtualization: ec2.AmazonLinuxVirt.HVM,
 *   storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
 * });
 * 
 * // For other custom (Linux) images, instantiate a `GenericLinuxImage` with
 * // a map giving the AMI to in for each region:
 * 
 * const linux = new ec2.GenericLinuxImage({
 *     'us-east-1': 'ami-97785bed',
 *     'eu-west-1': 'ami-12345678',
 *     // ...
 * });
 * 
 * // For other custom (Windows) images, instantiate a `GenericWindowsImage` with
 * // a map giving the AMI to in for each region:
 * 
 * const genericWindows = new ec2.GenericWindowsImage({
 *   'us-east-1': 'ami-97785bed',
 *   'eu-west-1': 'ami-12345678',
 *   // ...
 * });
 * </code></pre>
 * <blockquote>
 * <p>NOTE: The Amazon Linux images selected will be cached in your <code>cdk.json</code>, so that your
 * AutoScalingGroups don't automatically change out from under you when you're making unrelated
 * changes. To update to the latest version of Amazon Linux, remove the cache entry from the <code>context</code>
 * section of your <code>cdk.json</code>.</p>
 * <p>We will add command-line options to make this step easier in the future.</p>
 * </blockquote>
 * <h3>VPN connections to a VPC</h3>
 * <p>Create your VPC with VPN connections by specifying the <code>vpnConnections</code> props (keys are construct <code>id</code>s):</p>
 * <pre><code class="language-ts">const vpc = new ec2.Vpc(stack, 'MyVpc', {
 *   vpnConnections: {
 *     dynamic: { // Dynamic routing (BGP)
 *       ip: '1.2.3.4'
 *     },
 *     static: { // Static routing
 *       ip: '4.5.6.7',
 *       staticRoutes: [
 *         '192.168.10.0/24',
 *         '192.168.20.0/24'
 *       ]
 *     }
 *   }
 * });
 * </code></pre>
 * <p>To create a VPC that can accept VPN connections, set <code>vpnGateway</code> to <code>true</code>:</p>
 * <pre><code class="language-ts">const vpc = new ec2.Vpc(stack, 'MyVpc', {
 *   vpnGateway: true
 * });
 * </code></pre>
 * <p>VPN connections can then be added:</p>
 * <pre><code class="language-ts">vpc.addVpnConnection('Dynamic', {
 *   ip: '1.2.3.4'
 * });
 * </code></pre>
 * <p>Routes will be propagated on the route tables associated with the private subnets.</p>
 * <p>VPN connections expose <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-cloudwatch/README.md">metrics (cloudwatch.Metric)</a> across all tunnels in the account/region and per connection:</p>
 * <pre><code class="language-ts">// Across all tunnels in the account/region
 * const allDataOut = VpnConnection.metricAllTunnelDataOut();
 * 
 * // For a specific vpn connection
 * const vpnConnection = vpc.addVpnConnection('Dynamic', {
 *   ip: '1.2.3.4'
 * });
 * const state = vpnConnection.metricTunnelState();
 * </code></pre>
 * <h3>VPC endpoints</h3>
 * <p>A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.</p>
 * <p>Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.</p>
 * <pre><code class="language-ts">    // Add gateway endpoints when creating the VPC
 *     const vpc = new ec2.Vpc(this, 'MyVpc', {
 *       gatewayEndpoints: {
 *         S3: {
 *           service: ec2.GatewayVpcEndpointAwsService.S3
 *         }
 *       }
 *     });
 * 
 *     // Alternatively gateway endpoints can be added on the VPC
 *     const dynamoDbEndpoint = vpc.addGatewayEndpoint('DynamoDbEndpoint', {
 *       service: ec2.GatewayVpcEndpointAwsService.DYNAMODB
 *     });
 * 
 *     // This allows to customize the endpoint policy
 *     dynamoDbEndpoint.addToPolicy(
 *       new iam.PolicyStatement({ // Restrict to listing and describing tables
 *         principals: [new iam.AnyPrincipal()],
 *         actions: ['dynamodb:DescribeTable', 'dynamodb:ListTables'],
 *         resources: ['*'],
 *     }));
 * 
 *     // Add an interface endpoint
 *     const ecrDockerEndpoint = vpc.addInterfaceEndpoint('EcrDockerEndpoint', {
 *       service: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
 *     });
 * 
 *     // When working with an interface endpoint, use the connections object to
 *     // allow traffic to flow to the endpoint.
 *     ecrDockerEndpoint.connections.allowDefaultPortFromAnyIpv4();
 * </code></pre>
 * <h3>Bastion Hosts</h3>
 * <p>A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
 * You can use bastion hosts using a standard SSH connection targetting port 22 on the host. As an alternative, you can connect the SSH connection
 * feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)</p>
 * <p>A default bastion host for use via SSM can be configured like:</p>
 * <pre><code class="language-ts">const host = new ec2.BastionHostLinux(this, 'BastionHost', { vpc });
 * </code></pre>
 * <p>If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.</p>
 * <pre><code class="language-ts">const host = new ec2.BastionHostLinux(this, 'BastionHost', {
 *   vpc,
 *   subnetSelection: { subnetType: SubnetType.PUBLIC },
 * });
 * host.allowSshAccessFrom(Peer.ipv4('1.2.3.4/32'));
 * </code></pre>
 * <p>As there are no SSH public keys deployed on this machine, you need to use <a href="https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/">EC2 Instance Connect</a>
 * with the command <code>aws ec2-instance-connect send-ssh-public-key</code> to provide your SSH public key.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.ec2;
