/**
 * <h1>Amazon Elastic File System Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <a href="https://docs.aws.amazon.com/efs/latest/ug/whatisefs.html">Amazon Elastic File System</a> (Amazon EFS) provides a simple, scalable,
 * fully managed elastic NFS file system for use with AWS Cloud services and on-premises resources.
 * Amazon EFS provides file storage in the AWS Cloud. With Amazon EFS, you can create a file system,
 * mount the file system on an Amazon EC2 instance, and then read and write data to and from your file system.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>File Systems</h2>
 * <p>
 * Amazon EFS provides elastic, shared file storage that is POSIX-compliant. The file system you create
 * supports concurrent read and write access from multiple Amazon EC2 instances and is accessible from
 * all of the Availability Zones in the AWS Region where it is created. Learn more about <a href="https://docs.aws.amazon.com/efs/latest/ug/creating-using.html">EFS file systems</a>
 * <p>
 * <h3>Create an Amazon EFS file system</h3>
 * <p>
 * A Virtual Private Cloud (VPC) is required to create an Amazon EFS file system.
 * The following example creates a file system that is encrypted at rest, running in <code>General Purpose</code>
 * performance mode, and <code>Bursting</code> throughput mode and does not transition files to the Infrequent
 * Access (IA) storage class.
 * <p>
 * <blockquote><pre>
 * FileSystem fileSystem = FileSystem.Builder.create(this, "MyEfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .lifecyclePolicy(LifecyclePolicy.AFTER_14_DAYS) // files are not transitioned to infrequent access (IA) storage by default
 *         .performanceMode(PerformanceMode.GENERAL_PURPOSE) // default
 *         .outOfInfrequentAccessPolicy(OutOfInfrequentAccessPolicy.AFTER_1_ACCESS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ An Amazon EFS file system's performance mode can't be changed after the file system has been created.
 * Updating this property will replace the file system.
 * <p>
 * Any file system that has been created outside the stack can be imported into your CDK app.
 * <p>
 * Use the <code>fromFileSystemAttributes()</code> API to import an existing file system.
 * Here is an example of giving a role write permissions on a file system.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * 
 * IFileSystem importedFileSystem = FileSystem.fromFileSystemAttributes(this, "existingFS", FileSystemAttributes.builder()
 *         .fileSystemId("fs-12345678") // You can also use fileSystemArn instead of fileSystemId.
 *         .securityGroup(SecurityGroup.fromSecurityGroupId(this, "SG", "sg-123456789", SecurityGroupImportOptions.builder()
 *                 .allowAllOutbound(false)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Permissions</h3>
 * <p>
 * If you need to grant file system permissions to another resource, you can use the <code>.grant()</code> API.
 * As an example, the following code gives <code>elasticfilesystem:ClientWrite</code> permissions to an IAM role.
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new AnyPrincipal())
 *         .build();
 * 
 * fileSystem.grant(role, "elasticfilesystem:ClientWrite");
 * </pre></blockquote>
 * <p>
 * <h3>Access Point</h3>
 * <p>
 * An access point is an application-specific view into an EFS file system that applies an operating
 * system user and group, and a file system path, to any file system request made through the access
 * point. The operating system user and group override any identity information provided by the NFS
 * client. The file system path is exposed as the access point's root directory. Applications using
 * the access point can only access data in its own directory and below. To learn more, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html">Mounting a File System Using EFS Access Points</a>.
 * <p>
 * Use the <code>addAccessPoint</code> API to create an access point from a fileSystem.
 * <p>
 * <blockquote><pre>
 * fileSystem.addAccessPoint("AccessPoint");
 * </pre></blockquote>
 * <p>
 * By default, when you create an access point, the root(<code>/</code>) directory is exposed to the client
 * connecting to the access point. You can specify a custom path with the <code>path</code> property.
 * <p>
 * If <code>path</code> does not exist, it will be created with the settings defined in the <code>creationInfo</code>.
 * See <a href="https://docs.aws.amazon.com/efs/latest/ug/create-access-point.html">Creating Access Points</a> for more details.
 * <p>
 * Any access point that has been created outside the stack can be imported into your CDK app.
 * <p>
 * Use the <code>fromAccessPointAttributes()</code> API to import an existing access point.
 * <p>
 * <blockquote><pre>
 * AccessPoint.fromAccessPointAttributes(this, "ap", AccessPointAttributes.builder()
 *         .accessPointId("fsap-1293c4d9832fo0912")
 *         .fileSystem(FileSystem.fromFileSystemAttributes(this, "efs", FileSystemAttributes.builder()
 *                 .fileSystemId("fs-099d3e2f")
 *                 .securityGroup(SecurityGroup.fromSecurityGroupId(this, "sg", "sg-51530134"))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * ⚠️ Notice: When importing an Access Point using <code>fromAccessPointAttributes()</code>, you must make sure
 * the mount targets are deployed and their lifecycle state is <code>available</code>. Otherwise, you may encounter
 * the following error when deploying:
 * <p>
 * <blockquote>
 * <p>
 * EFS file system <ARN of efs> referenced by access point <ARN of access point of EFS> has
 * mount targets created in all availability zones the function will execute in, but not all
 * are in the available life cycle state yet. Please wait for them to become available and
 * try the request again.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Connecting</h3>
 * <p>
 * To control who can access the EFS, use the <code>.connections</code> attribute. EFS has
 * a fixed default port, so you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * fileSystem.connections.allowDefaultPortFrom(instance);
 * </pre></blockquote>
 * <p>
 * Learn more about <a href="https://docs.aws.amazon.com/efs/latest/ug/manage-fs-access.html">managing file system network accessibility</a>
 * <p>
 * <h3>Mounting the file system using User Data</h3>
 * <p>
 * After you create a file system, you can create mount targets. Then you can mount the file system on
 * EC2 instances, containers, and Lambda functions in your virtual private cloud (VPC).
 * <p>
 * The following example automatically mounts a file system during instance launch.
 * <p>
 * <blockquote><pre>
 * fileSystem.connections.allowDefaultPortFrom(instance);
 * 
 * instance.userData.addCommands("yum check-update -y", "yum upgrade -y", "yum install -y amazon-efs-utils", "yum install -y nfs-utils", "file_system_id_1=" + fileSystem.getFileSystemId(), "efs_mount_point_1=/mnt/efs/fs1", "mkdir -p \"${efs_mount_point_1}\"", "test -f \"/sbin/mount.efs\" &amp;&amp; echo \"${file_system_id_1}:/ ${efs_mount_point_1} efs defaults,_netdev\" &gt;&gt; /etc/fstab || " + "echo \"${file_system_id_1}.efs." + Stack.of(this).getRegion() + ".amazonaws.com:/ ${efs_mount_point_1} nfs4 nfsvers=4.1,rsize=1048576,wsize=1048576,hard,timeo=600,retrans=2,noresvport,_netdev 0 0\" &gt;&gt; /etc/fstab", "mount -a -t efs,nfs4 defaults");
 * </pre></blockquote>
 * <p>
 * Learn more about <a href="https://docs.aws.amazon.com/efs/latest/ug/mounting-fs.html">mounting EFS file systems</a>
 * <p>
 * <h3>Deleting</h3>
 * <p>
 * Since file systems are stateful resources, by default the file system will not be deleted when your
 * stack is deleted.
 * <p>
 * You can configure the file system to be destroyed on stack deletion by setting a <code>removalPolicy</code>
 * <p>
 * <blockquote><pre>
 * FileSystem fileSystem = FileSystem.Builder.create(this, "EfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.efs;
