/**
 * <h2>Amazon Elastic Load Balancing V2 Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge" alt="Stability: Experimental" /></p>
 * <blockquote>
 * <p>This API is still under active development and subject to non-backward
 * compatible changes or removal in any future version. Use of the API is not recommended in production
 * environments. Experimental APIs are not subject to the Semantic Versioning model.</p>
 * </blockquote>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>The <code>@aws-cdk/aws-elasticloadbalancingv2</code> package provides constructs for
 * configuring application and network load balancers.</p>
 * <p>For more information, see the AWS documentation for
 * <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/application/introduction.html">Application Load Balancers</a>
 * and <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/introduction.html">Network Load Balancers</a>.</p>
 * <h3>Defining an Application Load Balancer</h3>
 * <p>You define an application load balancer by creating an instance of
 * <code>ApplicationLoadBalancer</code>, adding a Listener to the load balancer
 * and adding Targets to the Listener:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * import elbv2 = require('@aws-cdk/aws-elasticloadbalancingv2');
 * import autoscaling = require('@aws-cdk/aws-autoscaling');
 * 
 * // ...
 * 
 * const vpc = new ec2.VpcNetwork(...);
 * 
 * // Create the load balancer in a VPC. 'internetFacing' is 'false'
 * // by default, which creates an internal load balancer.
 * const lb = new elbv2.ApplicationLoadBalancer(this, 'LB', {
 *     vpc,
 *     internetFacing: true
 * });
 * 
 * // Add a listener and open up the load balancer's security group
 * // to the world. 'open' is the default, set this to 'false'
 * // and use `listener.connections` if you want to be selective
 * // about who can access the listener.
 * const listener = lb.addListener('Listener', {
 *     port: 80,
 *     open: true,
 * });
 * 
 * // Create an AutoScaling group and add it as a load balancing
 * // target to the listener.
 * const asg = new autoscaling.AutoScalingGroup(...);
 * listener.addTargets('ApplicationFleet', {
 *     port: 8080,
 *     targets: [asg]
 * });
 * </code></pre>
 * <p>The security groups of the load balancer and the target are automatically
 * updated to allow the network traffic.</p>
 * <p>Use the <code>addFixedResponse()</code> method to add fixed response rules on the listener:</p>
 * <pre><code class="language-ts">listener.addFixedResponse('Fixed', {
 *     pathPattern: '/ok',
 *     contentType: elbv2.ContentType.TEXT_PLAIN,
 *     messageBody: 'OK',
 *     statusCode: '200'
 * });
 * </code></pre>
 * <h4>Conditions</h4>
 * <p>It's possible to route traffic to targets based on conditions in the incoming
 * HTTP request. Path- and host-based conditions are supported. For example,
 * the following will route requests to the indicated AutoScalingGroup
 * only if the requested host in the request is <code>example.com</code>:</p>
 * <pre><code class="language-ts">listener.addTargets('Example.Com Fleet', {
 *     priority: 10,
 *     hostHeader: 'example.com',
 *     port: 8080,
 *     targets: [asg]
 * });
 * </code></pre>
 * <p><code>priority</code> is a required field when you add targets with conditions. The lowest
 * number wins.</p>
 * <p>Every listener must have at least one target without conditions.</p>
 * <h3>Defining a Network Load Balancer</h3>
 * <p>Network Load Balancers are defined in a similar way to Application Load
 * Balancers:</p>
 * <pre><code class="language-ts">import ec2 = require('@aws-cdk/aws-ec2');
 * import elbv2 = require('@aws-cdk/aws-elasticloadbalancingv2');
 * import autoscaling = require('@aws-cdk/aws-autoscaling');
 * 
 * // Create the load balancer in a VPC. 'internetFacing' is 'false'
 * // by default, which creates an internal load balancer.
 * const lb = new elbv2.NetworkLoadBalancer(this, 'LB', {
 *     vpc,
 *     internetFacing: true
 * });
 * 
 * // Add a listener on a particular port.
 * const listener = lb.addListener('Listener', {
 *     port: 443,
 * });
 * 
 * // Add targets on a particular port.
 * listener.addTargets('AppFleet', {
 *     port: 443,
 *     targets: [asg]
 * });
 * </code></pre>
 * <p>One thing to keep in mind is that network load balancers do not have security
 * groups, and no automatic security group configuration is done for you. You will
 * have to configure the security groups of the target yourself to allow traffic by
 * clients and/or load balancer instances, depending on your target types.  See
 * <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-target-groups.html">Target Groups for your Network Load
 * Balancers</a>
 * and <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/target-group-register-targets.html">Register targets with your Target
 * Group</a>
 * for more information.</p>
 * <h3>Targets and Target Groups</h3>
 * <p>Application and Network Load Balancers organize load balancing targets in Target
 * Groups. If you add your balancing targets (such as AutoScalingGroups, ECS
 * services or individual instances) to your listener directly, the appropriate
 * <code>TargetGroup</code> will be automatically created for you.</p>
 * <p>If you need more control over the Target Groups created, create an instance of
 * <code>ApplicationTargetGroup</code> or <code>NetworkTargetGroup</code>, add the members you desire,
 * and add it to the listener by calling <code>addTargetGroups</code> instead of <code>addTargets</code>.</p>
 * <p><code>addTargets()</code> will always return the Target Group it just created for you:</p>
 * <pre><code class="language-ts">const group = listener.addTargets('AppFleet', {
 *     port: 443,
 *     targets: [asg1],
 * });
 * 
 * group.addTarget(asg2);
 * </code></pre>
 * <h3>Configuring Health Checks</h3>
 * <p>Health checks are configured upon creation of a target group:</p>
 * <pre><code class="language-ts">listener.addTargets('AppFleet', {
 *     port: 8080,
 *     targets: [asg],
 *     healthCheck: {
 *         path: '/ping',
 *         intervalSecs: 60,
 *     }
 * });
 * </code></pre>
 * <p>The health check can also be configured after creation by calling
 * <code>configureHealthCheck()</code> on the created object.</p>
 * <p>No attempts are made to configure security groups for the port you're
 * configuring a health check for, but if the health check is on the same port
 * you're routing traffic to, the security group already allows the traffic.
 * If not, you will have to configure the security groups appropriately:</p>
 * <pre><code class="language-ts">listener.addTargets('AppFleet', {
 *     port: 8080,
 *     targets: [asg],
 *     healthCheck: {
 *         port: 8088,
 *     }
 * });
 * 
 * listener.connections.allowFrom(lb, new TcpPort(8088));
 * </code></pre>
 * <h3>Protocol for Load Balancer Targets</h3>
 * <p>Constructs that want to be a load balancer target should implement
 * <code>IApplicationLoadBalancerTarget</code> and/or <code>INetworkLoadBalancerTarget</code>, and
 * provide an implementation for the function <code>attachToXxxTargetGroup()</code>, which can
 * call functions on the load balancer and should return metadata about the
 * load balancing target:</p>
 * <pre><code class="language-ts">public attachToApplicationTargetGroup(targetGroup: ApplicationTargetGroup): LoadBalancerTargetProps {
 *     targetGroup.registerConnectable(...);
 *     return {
 *         targetType: TargetType.Instance | TargetType.Ip
 *         targetJson: { id: ..., port: ... },
 *     };
 * }
 * </code></pre>
 * <p><code>targetType</code> should be one of <code>Instance</code> or <code>Ip</code>. If the target can be
 * directly added to the target group, <code>targetJson</code> should contain the <code>id</code> of
 * the target (either instance ID or IP address depending on the type) and
 * optionally a <code>port</code> or <code>availabilityZone</code> override.</p>
 * <p>Application load balancer targets can call <code>registerConnectable()</code> on the
 * target group to register themselves for addition to the load balancer's security
 * group rules.</p>
 * <p>If your load balancer target requires that the TargetGroup has been
 * associated with a LoadBalancer before registration can happen (such as is the
 * case for ECS Services for example), take a resource dependency on
 * <code>targetGroup.loadBalancerDependency()</code> as follows:</p>
 * <pre><code class="language-ts">// Make sure that the listener has been created, and so the TargetGroup
 * // has been associated with the LoadBalancer, before 'resource' is created.
 * resourced.addDependency(targetGroup.loadBalancerDependency());
 * </code></pre>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.elasticloadbalancingv2;
