/**
 * <h1>Amazon Elastic Load Balancing V2 Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * The <code>&#64;aws-cdk/aws-elasticloadbalancingv2</code> package provides constructs for
 * configuring application and network load balancers.
 * <p>
 * For more information, see the AWS documentation for
 * <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/application/introduction.html">Application Load Balancers</a>
 * and <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/introduction.html">Network Load Balancers</a>.
 * <p>
 * <h2>Defining an Application Load Balancer</h2>
 * <p>
 * You define an application load balancer by creating an instance of
 * <code>ApplicationLoadBalancer</code>, adding a Listener to the load balancer
 * and adding Targets to the Listener:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.autoscaling.AutoScalingGroup;
 * AutoScalingGroup asg;
 * 
 * Vpc vpc;
 * 
 * 
 * // Create the load balancer in a VPC. 'internetFacing' is 'false'
 * // by default, which creates an internal load balancer.
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .build();
 * 
 * // Add a listener and open up the load balancer's security group
 * // to the world.
 * ApplicationListener listener = lb.addListener("Listener", BaseApplicationListenerProps.builder()
 *         .port(80)
 * 
 *         // 'open: true' is the default, you can leave it out if you want. Set it
 *         // to 'false' and use `listener.connections` if you want to be selective
 *         // about who can access the load balancer.
 *         .open(true)
 *         .build());
 * 
 * // Create an AutoScaling group and add it as a load balancing
 * // target to the listener.
 * listener.addTargets("ApplicationFleet", AddApplicationTargetsProps.builder()
 *         .port(8080)
 *         .targets(List.of(asg))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The security groups of the load balancer and the target are automatically
 * updated to allow the network traffic.
 * <p>
 * One (or more) security groups can be associated with the load balancer;
 * if a security group isn't provided, one will be automatically created.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * SecurityGroup securityGroup1 = SecurityGroup.Builder.create(this, "SecurityGroup1").vpc(vpc).build();
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .securityGroup(securityGroup1)
 *         .build();
 * 
 * SecurityGroup securityGroup2 = SecurityGroup.Builder.create(this, "SecurityGroup2").vpc(vpc).build();
 * lb.addSecurityGroup(securityGroup2);
 * </pre></blockquote>
 * <p>
 * <h3>Conditions</h3>
 * <p>
 * It's possible to route traffic to targets based on conditions in the incoming
 * HTTP request. For example, the following will route requests to the indicated
 * AutoScalingGroup only if the requested host in the request is either for
 * <code>example.com/ok</code> or <code>example.com/path</code>:
 * <p>
 * <blockquote><pre>
 * ApplicationListener listener;
 * AutoScalingGroup asg;
 * 
 * 
 * listener.addTargets("Example.Com Fleet", AddApplicationTargetsProps.builder()
 *         .priority(10)
 *         .conditions(List.of(ListenerCondition.hostHeaders(List.of("example.com")), ListenerCondition.pathPatterns(List.of("/ok", "/path"))))
 *         .port(8080)
 *         .targets(List.of(asg))
 *         .build());
 * </pre></blockquote>
 * <p>
 * A target with a condition contains either <code>pathPatterns</code> or <code>hostHeader</code>, or
 * both. If both are specified, both conditions must be met for the requests to
 * be routed to the given target. <code>priority</code> is a required field when you add
 * targets with conditions. The lowest number wins.
 * <p>
 * Every listener must have at least one target without conditions, which is
 * where all requests that didn't match any of the conditions will be sent.
 * <p>
 * <h3>Convenience methods and more complex Actions</h3>
 * <p>
 * Routing traffic from a Load Balancer to a Target involves the following steps:
 * <p>
 * <ul>
 * <li>Create a Target Group, register the Target into the Target Group</li>
 * <li>Add an Action to the Listener which forwards traffic to the Target Group.</li>
 * </ul>
 * <p>
 * A new listener can be added to the Load Balancer by calling <code>addListener()</code>.
 * Listeners that have been added to the load balancer can be listed using the
 * <code>listeners</code> property.  Note that the <code>listeners</code> property will throw an Error
 * for imported or looked up Load Balancers.
 * <p>
 * Various methods on the <code>Listener</code> take care of this work for you to a greater
 * or lesser extent:
 * <p>
 * <ul>
 * <li><code>addTargets()</code> performs both steps: automatically creates a Target Group and the
 * required Action.</li>
 * <li><code>addTargetGroups()</code> gives you more control: you create the Target Group (or
 * Target Groups) yourself and the method creates Action that routes traffic to
 * the Target Groups.</li>
 * <li><code>addAction()</code> gives you full control: you supply the Action and wire it up
 * to the Target Groups yourself (or access one of the other ELB routing features).</li>
 * </ul>
 * <p>
 * Using <code>addAction()</code> gives you access to some of the features of an Elastic Load
 * Balancer that the other two convenience methods don't:
 * <p>
 * <ul>
 * <li><strong>Routing stickiness</strong>: use <code>ListenerAction.forward()</code> and supply a
 * <code>stickinessDuration</code> to make sure requests are routed to the same target group
 * for a given duration.</li>
 * <li><strong>Weighted Target Groups</strong>: use <code>ListenerAction.weightedForward()</code>
 * to give different weights to different target groups.</li>
 * <li><strong>Fixed Responses</strong>: use <code>ListenerAction.fixedResponse()</code> to serve
 * a static response (ALB only).</li>
 * <li><strong>Redirects</strong>: use <code>ListenerAction.redirect()</code> to serve an HTTP
 * redirect response (ALB only).</li>
 * <li><strong>Authentication</strong>: use <code>ListenerAction.authenticateOidc()</code> to
 * perform OpenID authentication before serving a request (see the
 * <code>&#64;aws-cdk/aws-elasticloadbalancingv2-actions</code> package for direct authentication
 * integration with Cognito) (ALB only).</li>
 * </ul>
 * <p>
 * Here's an example of serving a fixed response at the <code>/ok</code> URL:
 * <p>
 * <blockquote><pre>
 * ApplicationListener listener;
 * 
 * 
 * listener.addAction("Fixed", AddApplicationActionProps.builder()
 *         .priority(10)
 *         .conditions(List.of(ListenerCondition.pathPatterns(List.of("/ok"))))
 *         .action(ListenerAction.fixedResponse(200, FixedResponseOptions.builder()
 *                 .contentType(ContentType.TEXT_PLAIN)
 *                 .messageBody("OK")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Here's an example of using OIDC authentication before forwarding to a TargetGroup:
 * <p>
 * <blockquote><pre>
 * ApplicationListener listener;
 * ApplicationTargetGroup myTargetGroup;
 * 
 * 
 * listener.addAction("DefaultAction", AddApplicationActionProps.builder()
 *         .action(ListenerAction.authenticateOidc(AuthenticateOidcOptions.builder()
 *                 .authorizationEndpoint("https://example.com/openid")
 *                 // Other OIDC properties here
 *                 .clientId("...")
 *                 .clientSecret(SecretValue.secretsManager("..."))
 *                 .issuer("...")
 *                 .tokenEndpoint("...")
 *                 .userInfoEndpoint("...")
 * 
 *                 // Next
 *                 .next(ListenerAction.forward(List.of(myTargetGroup)))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you just want to redirect all incoming traffic on one port to another port, you can use the following code:
 * <p>
 * <blockquote><pre>
 * ApplicationLoadBalancer lb;
 * 
 * 
 * lb.addRedirect(ApplicationLoadBalancerRedirectConfig.builder()
 *         .sourceProtocol(ApplicationProtocol.HTTPS)
 *         .sourcePort(8443)
 *         .targetProtocol(ApplicationProtocol.HTTP)
 *         .targetPort(8080)
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you do not provide any options for this method, it redirects HTTP port 80 to HTTPS port 443.
 * <p>
 * By default all ingress traffic will be allowed on the source port. If you want to be more selective with your
 * ingress rules then set <code>open: false</code> and use the listener's <code>connections</code> object to selectively grant access to the listener.
 * <p>
 * <h2>Defining a Network Load Balancer</h2>
 * <p>
 * Network Load Balancers are defined in a similar way to Application Load
 * Balancers:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * AutoScalingGroup asg;
 * 
 * 
 * // Create the load balancer in a VPC. 'internetFacing' is 'false'
 * // by default, which creates an internal load balancer.
 * NetworkLoadBalancer lb = NetworkLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .build();
 * 
 * // Add a listener on a particular port.
 * NetworkListener listener = lb.addListener("Listener", BaseNetworkListenerProps.builder()
 *         .port(443)
 *         .build());
 * 
 * // Add targets on a particular port.
 * listener.addTargets("AppFleet", AddNetworkTargetsProps.builder()
 *         .port(443)
 *         .targets(List.of(asg))
 *         .build());
 * </pre></blockquote>
 * <p>
 * One thing to keep in mind is that network load balancers do not have security
 * groups, and no automatic security group configuration is done for you. You will
 * have to configure the security groups of the target yourself to allow traffic by
 * clients and/or load balancer instances, depending on your target types.  See
 * <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-target-groups.html">Target Groups for your Network Load
 * Balancers</a>
 * and <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/target-group-register-targets.html">Register targets with your Target
 * Group</a>
 * for more information.
 * <p>
 * <h2>Targets and Target Groups</h2>
 * <p>
 * Application and Network Load Balancers organize load balancing targets in Target
 * Groups. If you add your balancing targets (such as AutoScalingGroups, ECS
 * services or individual instances) to your listener directly, the appropriate
 * <code>TargetGroup</code> will be automatically created for you.
 * <p>
 * If you need more control over the Target Groups created, create an instance of
 * <code>ApplicationTargetGroup</code> or <code>NetworkTargetGroup</code>, add the members you desire,
 * and add it to the listener by calling <code>addTargetGroups</code> instead of <code>addTargets</code>.
 * <p>
 * <code>addTargets()</code> will always return the Target Group it just created for you:
 * <p>
 * <blockquote><pre>
 * NetworkListener listener;
 * AutoScalingGroup asg1;
 * AutoScalingGroup asg2;
 * 
 * 
 * NetworkTargetGroup group = listener.addTargets("AppFleet", AddNetworkTargetsProps.builder()
 *         .port(443)
 *         .targets(List.of(asg1))
 *         .build());
 * 
 * group.addTarget(asg2);
 * </pre></blockquote>
 * <p>
 * <h3>Sticky sessions for your Application Load Balancer</h3>
 * <p>
 * By default, an Application Load Balancer routes each request independently to a registered target based on the chosen load-balancing algorithm. However, you can use the sticky session feature (also known as session affinity) to enable the load balancer to bind a user's session to a specific target. This ensures that all requests from the user during the session are sent to the same target. This feature is useful for servers that maintain state information in order to provide a continuous experience to clients. To use sticky sessions, the client must support cookies.
 * <p>
 * Application Load Balancers support both duration-based cookies (<code>lb_cookie</code>) and application-based cookies (<code>app_cookie</code>). The key to managing sticky sessions is determining how long your load balancer should consistently route the user's request to the same target. Sticky sessions are enabled at the target group level. You can use a combination of duration-based stickiness, application-based stickiness, and no stickiness across all of your target groups.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * // Target group with duration-based stickiness with load-balancer generated cookie
 * ApplicationTargetGroup tg1 = ApplicationTargetGroup.Builder.create(this, "TG1")
 *         .targetType(TargetType.INSTANCE)
 *         .port(80)
 *         .stickinessCookieDuration(Duration.minutes(5))
 *         .vpc(vpc)
 *         .build();
 * 
 * // Target group with application-based stickiness
 * ApplicationTargetGroup tg2 = ApplicationTargetGroup.Builder.create(this, "TG2")
 *         .targetType(TargetType.INSTANCE)
 *         .port(80)
 *         .stickinessCookieDuration(Duration.minutes(5))
 *         .stickinessCookieName("MyDeliciousCookie")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information see: https://docs.aws.amazon.com/elasticloadbalancing/latest/application/sticky-sessions.html#application-based-stickiness
 * <p>
 * <h3>Setting the target group protocol version</h3>
 * <p>
 * By default, Application Load Balancers send requests to targets using HTTP/1.1. You can use the <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-protocol-version">protocol version</a> to send requests to targets using HTTP/2 or gRPC.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * ApplicationTargetGroup tg = ApplicationTargetGroup.Builder.create(this, "TG")
 *         .targetType(TargetType.IP)
 *         .port(50051)
 *         .protocol(ApplicationProtocol.HTTP)
 *         .protocolVersion(ApplicationProtocolVersion.GRPC)
 *         .healthCheck(HealthCheck.builder()
 *                 .enabled(true)
 *                 .healthyGrpcCodes("0-99")
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Using Lambda Targets</h2>
 * <p>
 * To use a Lambda Function as a target, use the integration class in the
 * <code>&#64;aws-cdk/aws-elasticloadbalancingv2-targets</code> package:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.targets.*;
 * 
 * Function lambdaFunction;
 * ApplicationLoadBalancer lb;
 * 
 * 
 * ApplicationListener listener = lb.addListener("Listener", BaseApplicationListenerProps.builder().port(80).build());
 * listener.addTargets("Targets", AddApplicationTargetsProps.builder()
 *         .targets(List.of(new LambdaTarget(lambdaFunction)))
 * 
 *         // For Lambda Targets, you need to explicitly enable health checks if you
 *         // want them.
 *         .healthCheck(HealthCheck.builder()
 *                 .enabled(true)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * Only a single Lambda function can be added to a single listener rule.
 * <p>
 * <h2>Using Application Load Balancer Targets</h2>
 * <p>
 * To use a single application load balancer as a target for the network load balancer, use the integration class in the
 * <code>&#64;aws-cdk/aws-elasticloadbalancingv2-targets</code> package:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.targets.*;
 * import software.amazon.awscdk.services.ecs.*;
 * import software.amazon.awscdk.services.ecs.patterns.*;
 * 
 * Vpc vpc;
 * 
 * 
 * FargateTaskDefinition task = FargateTaskDefinition.Builder.create(this, "Task").cpu(256).memoryLimitMiB(512).build();
 * task.addContainer("nginx", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("public.ecr.aws/nginx/nginx:latest"))
 *         .portMappings(List.of(PortMapping.builder().containerPort(80).build()))
 *         .build());
 * 
 * ApplicationLoadBalancedFargateService svc = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .taskDefinition(task)
 *         .publicLoadBalancer(false)
 *         .build();
 * 
 * NetworkLoadBalancer nlb = NetworkLoadBalancer.Builder.create(this, "Nlb")
 *         .vpc(vpc)
 *         .crossZoneEnabled(true)
 *         .internetFacing(true)
 *         .build();
 * 
 * NetworkListener listener = nlb.addListener("listener", BaseNetworkListenerProps.builder().port(80).build());
 * 
 * listener.addTargets("Targets", AddNetworkTargetsProps.builder()
 *         .targets(List.of(new AlbTarget(svc.getLoadBalancer(), 80)))
 *         .port(80)
 *         .build());
 * 
 * CfnOutput.Builder.create(this, "NlbEndpoint").value(String.format("http://%s", nlb.getLoadBalancerDnsName())).build();
 * </pre></blockquote>
 * <p>
 * Only the network load balancer is allowed to add the application load balancer as the target.
 * <p>
 * <h2>Configuring Health Checks</h2>
 * <p>
 * Health checks are configured upon creation of a target group:
 * <p>
 * <blockquote><pre>
 * ApplicationListener listener;
 * AutoScalingGroup asg;
 * 
 * 
 * listener.addTargets("AppFleet", AddApplicationTargetsProps.builder()
 *         .port(8080)
 *         .targets(List.of(asg))
 *         .healthCheck(HealthCheck.builder()
 *                 .path("/ping")
 *                 .interval(Duration.minutes(1))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * The health check can also be configured after creation by calling
 * <code>configureHealthCheck()</code> on the created object.
 * <p>
 * No attempts are made to configure security groups for the port you're
 * configuring a health check for, but if the health check is on the same port
 * you're routing traffic to, the security group already allows the traffic.
 * If not, you will have to configure the security groups appropriately:
 * <p>
 * <blockquote><pre>
 * ApplicationLoadBalancer lb;
 * ApplicationListener listener;
 * AutoScalingGroup asg;
 * 
 * 
 * listener.addTargets("AppFleet", AddApplicationTargetsProps.builder()
 *         .port(8080)
 *         .targets(List.of(asg))
 *         .healthCheck(HealthCheck.builder()
 *                 .port("8088")
 *                 .build())
 *         .build());
 * 
 * asg.connections.allowFrom(lb, Port.tcp(8088));
 * </pre></blockquote>
 * <p>
 * <h2>Using a Load Balancer from a different Stack</h2>
 * <p>
 * If you want to put your Load Balancer and the Targets it is load balancing to in
 * different stacks, you may not be able to use the convenience methods
 * <code>loadBalancer.addListener()</code> and <code>listener.addTargets()</code>.
 * <p>
 * The reason is that these methods will create resources in the same Stack as the
 * object they're called on, which may lead to cyclic references between stacks.
 * Instead, you will have to create an <code>ApplicationListener</code> in the target stack,
 * or an empty <code>TargetGroup</code> in the load balancer stack that you attach your
 * service to.
 * <p>
 * For an example of the alternatives while load balancing to an ECS service, see the
 * <a href="https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/">ecs/cross-stack-load-balancer
 * example</a>.
 * <p>
 * <h2>Protocol for Load Balancer Targets</h2>
 * <p>
 * Constructs that want to be a load balancer target should implement
 * <code>IApplicationLoadBalancerTarget</code> and/or <code>INetworkLoadBalancerTarget</code>, and
 * provide an implementation for the function <code>attachToXxxTargetGroup()</code>, which can
 * call functions on the load balancer and should return metadata about the
 * load balancing target:
 * <p>
 * <blockquote><pre>
 * public class MyTarget implements IApplicationLoadBalancerTarget {
 *     public LoadBalancerTargetProps attachToApplicationTargetGroup(ApplicationTargetGroup targetGroup) {
 *         // If we need to add security group rules
 *         // targetGroup.registerConnectable(...);
 *         return LoadBalancerTargetProps.builder()
 *                 .targetType(TargetType.IP)
 *                 .targetJson(Map.of("id", "1.2.3.4", "port", 8080))
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <code>targetType</code> should be one of <code>Instance</code> or <code>Ip</code>. If the target can be
 * directly added to the target group, <code>targetJson</code> should contain the <code>id</code> of
 * the target (either instance ID or IP address depending on the type) and
 * optionally a <code>port</code> or <code>availabilityZone</code> override.
 * <p>
 * Application load balancer targets can call <code>registerConnectable()</code> on the
 * target group to register themselves for addition to the load balancer's security
 * group rules.
 * <p>
 * If your load balancer target requires that the TargetGroup has been
 * associated with a LoadBalancer before registration can happen (such as is the
 * case for ECS Services for example), take a resource dependency on
 * <code>targetGroup.loadBalancerAttached</code> as follows:
 * <p>
 * <blockquote><pre>
 * Resource resource;
 * ApplicationTargetGroup targetGroup;
 * 
 * 
 * // Make sure that the listener has been created, and so the TargetGroup
 * // has been associated with the LoadBalancer, before 'resource' is created.
 * 
 * Node.of(resource).addDependency(targetGroup.getLoadBalancerAttached());
 * </pre></blockquote>
 * <p>
 * <h2>Looking up Load Balancers and Listeners</h2>
 * <p>
 * You may look up load balancers and load balancer listeners by using one of the
 * following lookup methods:
 * <p>
 * <ul>
 * <li><code>ApplicationLoadBalancer.fromlookup(options)</code> - Look up an application load
 * balancer.</li>
 * <li><code>ApplicationListener.fromLookup(options)</code> - Look up an application load
 * balancer listener.</li>
 * <li><code>NetworkLoadBalancer.fromLookup(options)</code> - Look up a network load balancer.</li>
 * <li><code>NetworkListener.fromLookup(options)</code> - Look up a network load balancer
 * listener.</li>
 * </ul>
 * <p>
 * <h3>Load Balancer lookup options</h3>
 * <p>
 * You may look up a load balancer by ARN or by associated tags. When you look a
 * load balancer up by ARN, that load balancer will be returned unless CDK detects
 * that the load balancer is of the wrong type. When you look up a load balancer by
 * tags, CDK will return the load balancer matching all specified tags. If more
 * than one load balancer matches, CDK will throw an error requesting that you
 * provide more specific criteria.
 * <p>
 * <strong>Look up a Application Load Balancer by ARN</strong>
 * <p>
 * <blockquote><pre>
 * IApplicationLoadBalancer loadBalancer = ApplicationLoadBalancer.fromLookup(this, "ALB", ApplicationLoadBalancerLookupOptions.builder()
 *         .loadBalancerArn("arn:aws:elasticloadbalancing:us-east-2:123456789012:loadbalancer/app/my-load-balancer/1234567890123456")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <strong>Look up an Application Load Balancer by tags</strong>
 * <p>
 * <blockquote><pre>
 * IApplicationLoadBalancer loadBalancer = ApplicationLoadBalancer.fromLookup(this, "ALB", ApplicationLoadBalancerLookupOptions.builder()
 *         .loadBalancerTags(Map.of(
 *                 // Finds a load balancer matching all tags.
 *                 "some", "tag",
 *                 "someother", "tag"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Load Balancer Listener lookup options</h2>
 * <p>
 * You may look up a load balancer listener by the following criteria:
 * <p>
 * <ul>
 * <li>Associated load balancer ARN</li>
 * <li>Associated load balancer tags</li>
 * <li>Listener ARN</li>
 * <li>Listener port</li>
 * <li>Listener protocol</li>
 * </ul>
 * <p>
 * The lookup method will return the matching listener. If more than one listener
 * matches, CDK will throw an error requesting that you specify additional
 * criteria.
 * <p>
 * <strong>Look up a Listener by associated Load Balancer, Port, and Protocol</strong>
 * <p>
 * <blockquote><pre>
 * IApplicationListener listener = ApplicationListener.fromLookup(this, "ALBListener", ApplicationListenerLookupOptions.builder()
 *         .loadBalancerArn("arn:aws:elasticloadbalancing:us-east-2:123456789012:loadbalancer/app/my-load-balancer/1234567890123456")
 *         .listenerProtocol(ApplicationProtocol.HTTPS)
 *         .listenerPort(443)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <strong>Look up a Listener by associated Load Balancer Tag, Port, and Protocol</strong>
 * <p>
 * <blockquote><pre>
 * IApplicationListener listener = ApplicationListener.fromLookup(this, "ALBListener", ApplicationListenerLookupOptions.builder()
 *         .loadBalancerTags(Map.of(
 *                 "Cluster", "MyClusterName"))
 *         .listenerProtocol(ApplicationProtocol.HTTPS)
 *         .listenerPort(443)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <strong>Look up a Network Listener by associated Load Balancer Tag, Port, and Protocol</strong>
 * <p>
 * <blockquote><pre>
 * INetworkListener listener = NetworkListener.fromLookup(this, "ALBListener", NetworkListenerLookupOptions.builder()
 *         .loadBalancerTags(Map.of(
 *                 "Cluster", "MyClusterName"))
 *         .listenerProtocol(Protocol.TCP)
 *         .listenerPort(12345)
 *         .build());
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.elasticloadbalancingv2;
