/**
 * <h1>AWS Glue Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * All classes with the <code>Cfn</code> prefix in this module (<a href="https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib">CFN Resources</a>) are always stable and safe to use.
 * <p>
 * </blockquote>
 * <p>
 * <img alt="cdk-constructs: Experimental" src="https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * The APIs of higher level constructs in this module are experimental and under active development.
 * They are subject to non-backward compatible changes or removal in any future version. These are
 * not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be
 * announced in the release notes. This means that while you may use them, you may need to update
 * your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Job</h2>
 * <p>
 * A <code>Job</code> encapsulates a script that connects to data sources, processes them, and then writes output to a data target.
 * <p>
 * There are 3 types of jobs supported by AWS Glue: Spark ETL, Spark Streaming, and Python Shell jobs.
 * <p>
 * The <code>glue.JobExecutable</code> allows you to specify the type of job, the language to use and the code assets required by the job.
 * <p>
 * <code>glue.Code</code> allows you to refer to the different code assets required by the job, either from an existing S3 location or from a local file path.
 * <p>
 * <h3>Spark Jobs</h3>
 * <p>
 * These jobs run in an Apache Spark environment managed by AWS Glue.
 * <p>
 * <h4>ETL Jobs</h4>
 * <p>
 * An ETL job processes data in batches using Apache Spark.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * Job.Builder.create(this, "ScalaSparkEtlJob")
 *         .executable(JobExecutable.scalaEtl(ScalaJobExecutableProps.builder()
 *                 .glueVersion(GlueVersion.V2_0)
 *                 .script(Code.fromBucket(bucket, "src/com/example/HelloWorld.scala"))
 *                 .className("com.example.HelloWorld")
 *                 .extraJars(List.of(Code.fromBucket(bucket, "jars/HelloWorld.jar")))
 *                 .build()))
 *         .description("an example Scala ETL job")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Streaming Jobs</h4>
 * <p>
 * A Streaming job is similar to an ETL job, except that it performs ETL on data streams. It uses the Apache Spark Structured Streaming framework. Some Spark job features are not available to streaming ETL jobs.
 * <p>
 * <blockquote><pre>
 * Job.Builder.create(this, "PythonSparkStreamingJob")
 *         .executable(JobExecutable.pythonStreaming(PythonSparkJobExecutableProps.builder()
 *                 .glueVersion(GlueVersion.V2_0)
 *                 .pythonVersion(PythonVersion.THREE)
 *                 .script(Code.fromAsset(join(__dirname, "job-script/hello_world.py")))
 *                 .build()))
 *         .description("an example Python Streaming job")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Python Shell Jobs</h3>
 * <p>
 * A Python shell job runs Python scripts as a shell and supports a Python version that depends on the AWS Glue version you are using.
 * This can be used to schedule and run tasks that don't require an Apache Spark environment.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * Job.Builder.create(this, "PythonShellJob")
 *         .executable(JobExecutable.pythonShell(PythonShellExecutableProps.builder()
 *                 .glueVersion(GlueVersion.V1_0)
 *                 .pythonVersion(PythonVersion.THREE)
 *                 .script(Code.fromBucket(bucket, "script.py"))
 *                 .build()))
 *         .description("an example Python Shell job")
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">documentation</a> for more information on adding jobs in Glue.
 * <p>
 * <h2>Connection</h2>
 * <p>
 * A <code>Connection</code> allows Glue jobs, crawlers and development endpoints to access certain types of data stores. For example, to create a network connection to connect to a data source within a VPC:
 * <p>
 * <blockquote><pre>
 * SecurityGroup securityGroup;
 * Subnet subnet;
 * 
 * Connection.Builder.create(this, "MyConnection")
 *         .type(ConnectionType.NETWORK)
 *         // The security groups granting AWS Glue inbound access to the data source within the VPC
 *         .securityGroups(List.of(securityGroup))
 *         // The VPC subnet which contains the data source
 *         .subnet(subnet)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you need to use a connection type that doesn't exist as a static member on <code>ConnectionType</code>, you can instantiate a <code>ConnectionType</code> object, e.g: <code>new glue.ConnectionType('NEW_TYPE')</code>.
 * <p>
 * See <a href="https://docs.aws.amazon.com/glue/latest/dg/populate-add-connection.html">Adding a Connection to Your Data Store</a> and <a href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-Connection">Connection Structure</a> documentation for more information on the supported data stores and their configurations.
 * <p>
 * <h2>SecurityConfiguration</h2>
 * <p>
 * A <code>SecurityConfiguration</code> is a set of security properties that can be used by AWS Glue to encrypt data at rest.
 * <p>
 * <blockquote><pre>
 * SecurityConfiguration.Builder.create(this, "MySecurityConfiguration")
 *         .securityConfigurationName("name")
 *         .cloudWatchEncryption(CloudWatchEncryption.builder()
 *                 .mode(CloudWatchEncryptionMode.KMS)
 *                 .build())
 *         .jobBookmarksEncryption(JobBookmarksEncryption.builder()
 *                 .mode(JobBookmarksEncryptionMode.CLIENT_SIDE_KMS)
 *                 .build())
 *         .s3Encryption(S3Encryption.builder()
 *                 .mode(S3EncryptionMode.KMS)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, a shared KMS key is created for use with the encryption configurations that require one. You can also supply your own key for each encryption config, for example, for CloudWatch encryption:
 * <p>
 * <blockquote><pre>
 * Key key;
 * 
 * SecurityConfiguration.Builder.create(this, "MySecurityConfiguration")
 *         .securityConfigurationName("name")
 *         .cloudWatchEncryption(CloudWatchEncryption.builder()
 *                 .mode(CloudWatchEncryptionMode.KMS)
 *                 .kmsKey(key)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/glue/latest/dg/encryption-security-configuration.html">documentation</a> for more info for Glue encrypting data written by Crawlers, Jobs, and Development Endpoints.
 * <p>
 * <h2>Database</h2>
 * <p>
 * A <code>Database</code> is a logical grouping of <code>Tables</code> in the Glue Catalog.
 * <p>
 * <blockquote><pre>
 * Database.Builder.create(this, "MyDatabase")
 *         .databaseName("my_database")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Table</h2>
 * <p>
 * A Glue table describes a table of data in S3: its structure (column names and types), location of data (S3 objects with a common prefix in a S3 bucket), and format for the files (Json, Avro, Parquet, etc.):
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build(), Column.builder()
 *                 .name("col2")
 *                 .type(Schema.array(Schema.STRING))
 *                 .comment("col2 is an array of strings")
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, a S3 bucket will be created to store the table's data but you can manually pass the <code>bucket</code> and <code>s3Prefix</code>:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket;
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .bucket(myBucket)
 *         .s3Prefix("my-table/")
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, an S3 bucket will be created to store the table's data and stored in the bucket root. You can also manually pass the <code>bucket</code> and <code>s3Prefix</code>:
 * <p>
 * <h3>Partition Keys</h3>
 * <p>
 * To improve query performance, a table can specify <code>partitionKeys</code> on which data is stored and queried separately. For example, you might partition a table by <code>year</code> and <code>month</code> to optimize queries based on a time window:
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .partitionKeys(List.of(Column.builder()
 *                 .name("year")
 *                 .type(Schema.SMALL_INT)
 *                 .build(), Column.builder()
 *                 .name("month")
 *                 .type(Schema.SMALL_INT)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Partition Indexes</h3>
 * <p>
 * Another way to improve query performance is to specify partition indexes. If no partition indexes are
 * present on the table, AWS Glue loads all partitions of the table and filters the loaded partitions using
 * the query expression. The query takes more time to run as the number of partitions increase. With an
 * index, the query will try to fetch a subset of the partitions instead of loading all partitions of the
 * table.
 * <p>
 * The keys of a partition index must be a subset of the partition keys of the table. You can have a
 * maximum of 3 partition indexes per table. To specify a partition index, you can use the <code>partitionIndexes</code>
 * property:
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .partitionKeys(List.of(Column.builder()
 *                 .name("year")
 *                 .type(Schema.SMALL_INT)
 *                 .build(), Column.builder()
 *                 .name("month")
 *                 .type(Schema.SMALL_INT)
 *                 .build()))
 *         .partitionIndexes(List.of(PartitionIndex.builder()
 *                 .indexName("my-index") // optional
 *                 .keyNames(List.of("year"))
 *                 .build())) // supply up to 3 indexes
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can call the <code>addPartitionIndex()</code> function on a table:
 * <p>
 * <blockquote><pre>
 * Table myTable;
 * 
 * myTable.addPartitionIndex(PartitionIndex.builder()
 *         .indexName("my-index")
 *         .keyNames(List.of("year"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2><a href="https://docs.aws.amazon.com/athena/latest/ug/encryption.html">Encryption</a></h2>
 * <p>
 * You can enable encryption on a Table's data:
 * <p>
 * <ul>
 * <li><code>Unencrypted</code> - files are not encrypted. The default encryption setting.</li>
 * <li><a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html">S3Managed</a> - Server side encryption (<code>SSE-S3</code>) with an Amazon S3-managed key.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .encryption(TableEncryption.S3_MANAGED)
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">Kms</a> - Server-side encryption (<code>SSE-KMS</code>) with an AWS KMS Key managed by the account owner.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * // KMS key is created automatically
 * // KMS key is created automatically
 * Table.Builder.create(this, "MyTable")
 *         .encryption(TableEncryption.KMS)
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * 
 * // with an explicit KMS key
 * // with an explicit KMS key
 * Table.Builder.create(this, "MyTable")
 *         .encryption(TableEncryption.KMS)
 *         .encryptionKey(new Key(this, "MyKey"))
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">KmsManaged</a> - Server-side encryption (<code>SSE-KMS</code>), like <code>Kms</code>, except with an AWS KMS Key managed by the AWS Key Management Service.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .encryption(TableEncryption.KMS_MANAGED)
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html#client-side-encryption-kms-managed-master-key-intro">ClientSideKms</a> - Client-side encryption (<code>CSE-KMS</code>) with an AWS KMS Key managed by the account owner.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * // KMS key is created automatically
 * // KMS key is created automatically
 * Table.Builder.create(this, "MyTable")
 *         .encryption(TableEncryption.CLIENT_SIDE_KMS)
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * 
 * // with an explicit KMS key
 * // with an explicit KMS key
 * Table.Builder.create(this, "MyTable")
 *         .encryption(TableEncryption.CLIENT_SIDE_KMS)
 *         .encryptionKey(new Key(this, "MyKey"))
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .columns(List.of(Column.builder()
 *                 .name("col1")
 *                 .type(Schema.STRING)
 *                 .build()))
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <em>Note: you cannot provide a <code>Bucket</code> when creating the <code>Table</code> if you wish to use server-side encryption (<code>KMS</code>, <code>KMS_MANAGED</code> or <code>S3_MANAGED</code>)</em>.
 * <p>
 * <h2>Types</h2>
 * <p>
 * A table's schema is a collection of columns, each of which have a <code>name</code> and a <code>type</code>. Types are recursive structures, consisting of primitive and complex types:
 * <p>
 * <blockquote><pre>
 * Database myDatabase;
 * 
 * Table.Builder.create(this, "MyTable")
 *         .columns(List.of(Column.builder()
 *                 .name("primitive_column")
 *                 .type(Schema.STRING)
 *                 .build(), Column.builder()
 *                 .name("array_column")
 *                 .type(Schema.array(Schema.INTEGER))
 *                 .comment("array&lt;integer&gt;")
 *                 .build(), Column.builder()
 *                 .name("map_column")
 *                 .type(Schema.map(Schema.STRING, Schema.TIMESTAMP))
 *                 .comment("map&lt;string,string&gt;")
 *                 .build(), Column.builder()
 *                 .name("struct_column")
 *                 .type(Schema.struct(List.of(Column.builder()
 *                         .name("nested_column")
 *                         .type(Schema.DATE)
 *                         .comment("nested comment")
 *                         .build())))
 *                 .comment("struct&lt;nested_column:date COMMENT 'nested comment'&gt;")
 *                 .build()))
 *         // ...
 *         .database(myDatabase)
 *         .tableName("my_table")
 *         .dataFormat(DataFormat.JSON)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Primitives</h3>
 * <p>
 * <h4>Numeric</h4>
 * <p>
 * | Name      	| Type     	| Comments                                                                                                          |
 * |-----------	|----------	|------------------------------------------------------------------------------------------------------------------	|
 * | FLOAT     	| Constant 	| A 32-bit single-precision floating point number                                                                   |
 * | INTEGER   	| Constant 	| A 32-bit signed value in two's complement format, with a minimum value of -2^31 and a maximum value of 2^31-1 	|
 * | DOUBLE    	| Constant 	| A 64-bit double-precision floating point number                                                                   |
 * | BIG_INT   	| Constant 	| A 64-bit signed INTEGER in two’s complement format, with a minimum value of -2^63 and a maximum value of 2^63 -1  |
 * | SMALL_INT 	| Constant 	| A 16-bit signed INTEGER in two’s complement format, with a minimum value of -2^15 and a maximum value of 2^15-1   |
 * | TINY_INT  	| Constant 	| A 8-bit signed INTEGER in two’s complement format, with a minimum value of -2^7 and a maximum value of 2^7-1      |
 * <p>
 * <h4>Date and time</h4>
 * <p>
 * | Name      	| Type     	| Comments                                                                                                                                                                	|
 * |-----------	|----------	|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------	|
 * | DATE      	| Constant 	| A date in UNIX format, such as YYYY-MM-DD.                                                                                                                              	|
 * | TIMESTAMP 	| Constant 	| Date and time instant in the UNiX format, such as yyyy-mm-dd hh:mm:ss[.f...]. For example, TIMESTAMP '2008-09-15 03:04:05.324'. This format uses the session time zone. 	|
 * <p>
 * <h4>String</h4>
 * <p>
 * | Name                                       	| Type     	| Comments                                                                                                                                                                                          	|
 * |--------------------------------------------	|----------	|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------	|
 * | STRING                                     	| Constant 	| A string literal enclosed in single or double quotes                                                                                                                                              	|
 * | decimal(precision: number, scale?: number) 	| Function 	| <code>precision</code> is the total number of digits. <code>scale</code> (optional) is the number of digits in fractional part with a default of 0. For example, use these type definitions: decimal(11,5), decimal(15) 	|
 * | char(length: number)                       	| Function 	| Fixed length character data, with a specified length between 1 and 255, such as char(10)                                                                                                          	|
 * | varchar(length: number)                    	| Function 	| Variable length character data, with a specified length between 1 and 65535, such as varchar(10)                                                                                                  	|
 * <p>
 * <h4>Miscellaneous</h4>
 * <p>
 * | Name    	| Type     	| Comments                      	|
 * |---------	|----------	|-------------------------------	|
 * | BOOLEAN 	| Constant 	| Values are <code>true</code> and <code>false</code> 	|
 * | BINARY  	| Constant 	| Value is in binary            	|
 * <p>
 * <h3>Complex</h3>
 * <p>
 * | Name                                	| Type     	| Comments                                                          	|
 * |-------------------------------------	|----------	|-------------------------------------------------------------------	|
 * | array(itemType: Type)               	| Function 	| An array of some other type                                       	|
 * | map(keyType: Type, valueType: Type) 	| Function 	| A map of some primitive key type to any value type                	|
 * | struct(collumns: Column[])          	| Function 	| Nested structure containing individually named and typed collumns 	|
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.glue;
