/**
 * <h1>AWS APIGatewayv2 Authorizers</h1>
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#introduction">Introduction</a></li>
 * <li><a href="#http-apis">HTTP APIs</a>
 * <p>
 * <ul>
 * <li><a href="#default-authorization">Default Authorization</a></li>
 * <li><a href="#route-authorization">Route Authorization</a></li>
 * <li><a href="#jwt-authorizers">JWT Authorizers</a>
 * <p>
 * <ul>
 * <li><a href="#user-pool-authorizer">User Pool Authorizer</a></li>
 * </ul></li>
 * <li><a href="#lambda-authorizers">Lambda Authorizers</a></li>
 * <li><a href="#iam-authorizers">IAM Authorizers</a></li>
 * </ul></li>
 * <li><a href="#websocket-apis">WebSocket APIs</a>
 * <p>
 * <ul>
 * <li><a href="#lambda-authorizer">Lambda Authorizer</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Introduction</h2>
 * <p>
 * API Gateway supports multiple mechanisms for controlling and managing access to your HTTP API. They are mainly
 * classified into Lambda Authorizers, JWT authorizers and standard AWS IAM roles and policies. More information is
 * available at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-access-control.html">Controlling and managing access to an HTTP
 * API</a>.
 * <p>
 * <h2>HTTP APIs</h2>
 * <p>
 * Access control for Http Apis is managed by restricting which routes can be invoked via.
 * <p>
 * Authorizers and scopes can either be applied to the api, or specifically for each route.
 * <p>
 * <h3>Default Authorization</h3>
 * <p>
 * When using default authorization, all routes of the api will inherit the configuration.
 * <p>
 * In the example below, all routes will require the <code>manage:books</code> scope present in order to invoke the integration.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpJwtAuthorizer;
 * 
 * 
 * String issuer = "https://test.us.auth0.com";
 * HttpJwtAuthorizer authorizer = HttpJwtAuthorizer.Builder.create("DefaultAuthorizer", issuer)
 *         .jwtAudience(List.of("3131231"))
 *         .build();
 * 
 * HttpApi api = HttpApi.Builder.create(this, "HttpApi")
 *         .defaultAuthorizer(authorizer)
 *         .defaultAuthorizationScopes(List.of("manage:books"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Route Authorization</h3>
 * <p>
 * Authorization can also configured for each Route. When a route authorization is configured, it takes precedence over default authorization.
 * <p>
 * The example below showcases default authorization, along with route authorization. It also shows how to remove authorization entirely for a route.
 * <p>
 * <ul>
 * <li><code>GET /books</code> and <code>GET /books/{id}</code> use the default authorizer settings on the api</li>
 * <li><code>POST /books</code> will require the [write:books] scope</li>
 * <li><code>POST /login</code> removes the default authorizer (unauthenticated route)</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpJwtAuthorizer;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.HttpUrlIntegration;
 * 
 * 
 * String issuer = "https://test.us.auth0.com";
 * HttpJwtAuthorizer authorizer = HttpJwtAuthorizer.Builder.create("DefaultAuthorizer", issuer)
 *         .jwtAudience(List.of("3131231"))
 *         .build();
 * 
 * HttpApi api = HttpApi.Builder.create(this, "HttpApi")
 *         .defaultAuthorizer(authorizer)
 *         .defaultAuthorizationScopes(List.of("read:books"))
 *         .build();
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books")
 *         .methods(List.of(HttpMethod.GET))
 *         .build());
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIdIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books/{id}")
 *         .methods(List.of(HttpMethod.GET))
 *         .build());
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books")
 *         .methods(List.of(HttpMethod.POST))
 *         .authorizationScopes(List.of("write:books"))
 *         .build());
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("LoginIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/login")
 *         .methods(List.of(HttpMethod.POST))
 *         .authorizer(new HttpNoneAuthorizer())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>JWT Authorizers</h3>
 * <p>
 * JWT authorizers allow the use of JSON Web Tokens (JWTs) as part of <a href="https://openid.net/specs/openid-connect-core-1_0.html">OpenID Connect</a> and <a href="https://oauth.net/2/">OAuth 2.0</a> frameworks to allow and restrict clients from accessing HTTP APIs.
 * <p>
 * When configured, API Gateway validates the JWT submitted by the client, and allows or denies access based on its content.
 * <p>
 * The location of the token is defined by the <code>identitySource</code> which defaults to the http <code>Authorization</code> header. However it also
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html#http-api-lambda-authorizer.identity-sources">supports a number of other options</a>.
 * It then decodes the JWT and validates the signature and claims, against the options defined in the authorizer and route (scopes).
 * For more information check the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-jwt-authorizer.html">JWT Authorizer documentation</a>.
 * <p>
 * Clients that fail authorization are presented with either 2 responses:
 * <p>
 * <ul>
 * <li><code>401 - Unauthorized</code> - When the JWT validation fails</li>
 * <li><code>403 - Forbidden</code> - When the JWT validation is successful but the required scopes are not met</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpJwtAuthorizer;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.HttpUrlIntegration;
 * 
 * 
 * String issuer = "https://test.us.auth0.com";
 * HttpJwtAuthorizer authorizer = HttpJwtAuthorizer.Builder.create("BooksAuthorizer", issuer)
 *         .jwtAudience(List.of("3131231"))
 *         .build();
 * 
 * HttpApi api = new HttpApi(this, "HttpApi");
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books")
 *         .authorizer(authorizer)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>User Pool Authorizer</h4>
 * <p>
 * User Pool Authorizer is a type of JWT Authorizer that uses a Cognito user pool and app client to control who can access your Api. After a successful authorization from the app client, the generated access token will be used as the JWT.
 * <p>
 * Clients accessing an API that uses a user pool authorizer must first sign in to a user pool and obtain an identity or access token.
 * They must then use this token in the specified <code>identitySource</code> for the API call. More information is available at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html">using Amazon Cognito user
 * pools as authorizer</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpUserPoolAuthorizer;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.HttpUrlIntegration;
 * 
 * 
 * UserPool userPool = new UserPool(this, "UserPool");
 * 
 * HttpUserPoolAuthorizer authorizer = new HttpUserPoolAuthorizer("BooksAuthorizer", userPool);
 * 
 * HttpApi api = new HttpApi(this, "HttpApi");
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books")
 *         .authorizer(authorizer)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Lambda Authorizers</h3>
 * <p>
 * Lambda authorizers use a Lambda function to control access to your HTTP API. When a client calls your API, API Gateway invokes your Lambda function and uses the response to determine whether the client can access your API.
 * <p>
 * Lambda authorizers depending on their response, fall into either two types - Simple or IAM. You can learn about differences <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html#http-api-lambda-authorizer.payload-format-response">here</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpLambdaAuthorizer;
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpLambdaResponseType;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.HttpUrlIntegration;
 * 
 * // This function handles your auth logic
 * Function authHandler;
 * 
 * 
 * HttpLambdaAuthorizer authorizer = HttpLambdaAuthorizer.Builder.create("BooksAuthorizer", authHandler)
 *         .responseTypes(List.of(HttpLambdaResponseType.SIMPLE))
 *         .build();
 * 
 * HttpApi api = new HttpApi(this, "HttpApi");
 * 
 * api.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books")
 *         .authorizer(authorizer)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>IAM Authorizers</h3>
 * <p>
 * API Gateway supports IAM via the included <code>HttpIamAuthorizer</code> and grant syntax:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.HttpIamAuthorizer;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.HttpUrlIntegration;
 * 
 * AnyPrincipal principal;
 * 
 * 
 * HttpIamAuthorizer authorizer = new HttpIamAuthorizer();
 * 
 * HttpApi httpApi = HttpApi.Builder.create(this, "HttpApi")
 *         .defaultAuthorizer(authorizer)
 *         .build();
 * 
 * HttpRoute[] routes = httpApi.addRoutes(AddRoutesOptions.builder()
 *         .integration(new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.myproxy.internal"))
 *         .path("/books/{book}")
 *         .build());
 * 
 * routes[0].grantInvoke(principal);
 * </pre></blockquote>
 * <p>
 * <h2>WebSocket APIs</h2>
 * <p>
 * You can set an authorizer to your WebSocket API's <code>$connect</code> route to control access to your API.
 * <p>
 * <h3>Lambda Authorizer</h3>
 * <p>
 * Lambda authorizers use a Lambda function to control access to your WebSocket API. When a client connects to your API, API Gateway invokes your Lambda function and uses the response to determine whether the client can access your API.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.authorizers.WebSocketLambdaAuthorizer;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.WebSocketLambdaIntegration;
 * 
 * // This function handles your auth logic
 * Function authHandler;
 * 
 * // This function handles your WebSocket requests
 * Function handler;
 * 
 * 
 * WebSocketLambdaAuthorizer authorizer = new WebSocketLambdaAuthorizer("Authorizer", authHandler);
 * 
 * WebSocketLambdaIntegration integration = new WebSocketLambdaIntegration("Integration", handler);
 * 
 * WebSocketApi.Builder.create(this, "WebSocketApi")
 *         .connectRouteOptions(WebSocketRouteOptions.builder()
 *                 .integration(integration)
 *                 .authorizer(authorizer)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.apigatewayv2.authorizers;
