/**
 * <h1>Amazon CloudWatch Construct Library</h1>
 * <p>
 * <h2>Metric objects</h2>
 * <p>
 * Metric objects represent a metric that is emitted by AWS services or your own
 * application, such as <code>CPUUsage</code>, <code>FailureCount</code> or <code>Bandwidth</code>.
 * <p>
 * Metric objects can be constructed directly or are exposed by resources as
 * attributes. Resources that expose metrics will have functions that look
 * like <code>metricXxx()</code> which will return a Metric object, initialized with defaults
 * that make sense.
 * <p>
 * For example, <code>lambda.Function</code> objects have the <code>fn.metricErrors()</code> method, which
 * represents the amount of errors reported by that Lambda function:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric errors = fn.metricErrors();
 * </pre></blockquote>
 * <p>
 * <code>Metric</code> objects can be account and region aware. You can specify <code>account</code> and <code>region</code> as properties of the metric, or use the <code>metric.attachTo(Construct)</code> method. <code>metric.attachTo()</code> will automatically copy the <code>region</code> and <code>account</code> fields of the <code>Construct</code>, which can come from anywhere in the Construct tree.
 * <p>
 * You can also instantiate <code>Metric</code> objects to reference any
 * <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html">published metric</a>
 * that's not exposed using a convenience method on the CDK construct.
 * For example:
 * <p>
 * <blockquote><pre>
 * HostedZone hostedZone = HostedZone.Builder.create(this, "MyHostedZone").zoneName("example.org").build();
 * Metric metric = Metric.Builder.create()
 *         .namespace("AWS/Route53")
 *         .metricName("DNSQueries")
 *         .dimensionsMap(Map.of(
 *                 "HostedZoneId", hostedZone.getHostedZoneId()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Instantiating a new Metric object</h3>
 * <p>
 * If you want to reference a metric that is not yet exposed by an existing construct,
 * you can instantiate a <code>Metric</code> object to represent it. For example:
 * <p>
 * <blockquote><pre>
 * Metric metric = Metric.Builder.create()
 *         .namespace("MyNamespace")
 *         .metricName("MyMetric")
 *         .dimensionsMap(Map.of(
 *                 "ProcessingStep", "Download"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Metric Math</h3>
 * <p>
 * Math expressions are supported by instantiating the <code>MathExpression</code> class.
 * For example, a math expression that sums two other metrics looks like this:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * MathExpression allProblems = MathExpression.Builder.create()
 *         .expression("errors + throttles")
 *         .usingMetrics(Map.of(
 *                 "errors", fn.metricErrors(),
 *                 "faults", fn.metricThrottles()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can use <code>MathExpression</code> objects like any other metric, including using
 * them in other math expressions:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * MathExpression allProblems;
 * 
 * 
 * MathExpression problemPercentage = MathExpression.Builder.create()
 *         .expression("(problems / invocations) * 100")
 *         .usingMetrics(Map.of(
 *                 "problems", allProblems,
 *                 "invocations", fn.metricInvocations()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Search Expressions</h3>
 * <p>
 * Math expressions also support search expressions. For example, the following
 * search expression returns all CPUUtilization metrics that it finds, with the
 * graph showing the Average statistic with an aggregation period of 5 minutes:
 * <p>
 * <blockquote><pre>
 * MathExpression cpuUtilization = MathExpression.Builder.create()
 *         .expression("SEARCH('{AWS/EC2,InstanceId} MetricName=\"CPUUtilization\"', 'Average', 300)")
 * 
 *         // Specifying '' as the label suppresses the default behavior
 *         // of using the expression as metric label. This is especially appropriate
 *         // when using expressions that return multiple time series (like SEARCH()
 *         // or METRICS()), to show the labels of the retrieved metrics only.
 *         .label("")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Cross-account and cross-region search expressions are also supported. Use
 * the <code>searchAccount</code> and <code>searchRegion</code> properties to specify the account
 * and/or region to evaluate the search expression against.
 * <p>
 * <h3>Aggregation</h3>
 * <p>
 * To graph or alarm on metrics you must aggregate them first, using a function
 * like <code>Average</code> or a percentile function like <code>P99</code>. By default, most Metric objects
 * returned by CDK libraries will be configured as <code>Average</code> over <code>300 seconds</code> (5 minutes).
 * The exception is if the metric represents a count of discrete events, such as
 * failures. In that case, the Metric object will be configured as <code>Sum</code> over <code>300 seconds</code>, i.e. it represents the number of times that event occurred over the
 * time period.
 * <p>
 * If you want to change the default aggregation of the Metric object (for example,
 * the function or the period), you can do so by passing additional parameters
 * to the metric function call:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric minuteErrorRate = fn.metricErrors(MetricOptions.builder()
 *         .statistic("avg")
 *         .period(Duration.minutes(1))
 *         .label("Lambda failure rate")
 *         .build());
 * </pre></blockquote>
 * <p>
 * This function also allows changing the metric label or color (which will be
 * useful when embedding them in graphs, see below).
 * <p>
 * <blockquote>
 * <p>
 * Rates versus Sums
 * <p>
 * The reason for using <code>Sum</code> to count discrete events is that <em>some</em> events are
 * emitted as either <code>0</code> or <code>1</code> (for example <code>Errors</code> for a Lambda) and some are
 * only emitted as <code>1</code> (for example <code>NumberOfMessagesPublished</code> for an SNS
 * topic).
 * <p>
 * In case <code>0</code>-metrics are emitted, it makes sense to take the <code>Average</code> of this
 * metric: the result will be the fraction of errors over all executions.
 * <p>
 * If <code>0</code>-metrics are not emitted, the <code>Average</code> will always be equal to <code>1</code>,
 * and not be very useful.
 * <p>
 * In order to simplify the mental model of <code>Metric</code> objects, we default to
 * aggregating using <code>Sum</code>, which will be the same for both metrics types. If you
 * happen to know the Metric you want to alarm on makes sense as a rate
 * (<code>Average</code>) you can always choose to change the statistic.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Labels</h3>
 * <p>
 * Metric labels are displayed in the legend of graphs that include the metrics.
 * <p>
 * You can use <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html">dynamic labels</a>
 * to show summary information about the displayed time series
 * in the legend. For example, if you use:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric minuteErrorRate = fn.metricErrors(MetricOptions.builder()
 *         .statistic("sum")
 *         .period(Duration.hours(1))
 * 
 *         // Show the maximum hourly error count in the legend
 *         .label("[max: ${MAX}] Lambda failure rate")
 *         .build());
 * </pre></blockquote>
 * <p>
 * As the metric label, the maximum value in the visible range will
 * be shown next to the time series name in the graph's legend.
 * <p>
 * If the metric is a math expression producing more than one time series, the
 * maximum will be individually calculated and shown for each time series produce
 * by the math expression.
 * <p>
 * <h2>Alarms</h2>
 * <p>
 * Alarms can be created on metrics in one of two ways. Either create an <code>Alarm</code>
 * object, passing the <code>Metric</code> object to set the alarm on:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Alarm.Builder.create(this, "Alarm")
 *         .metric(fn.metricErrors())
 *         .threshold(100)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can call <code>metric.createAlarm()</code>:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * fn.metricErrors().createAlarm(this, "Alarm", CreateAlarmOptions.builder()
 *         .threshold(100)
 *         .evaluationPeriods(2)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The most important properties to set while creating an Alarms are:
 * <p>
 * <ul>
 * <li><code>threshold</code>: the value to compare the metric against.</li>
 * <li><code>comparisonOperator</code>: the comparison operation to use, defaults to <code>metric &gt;= threshold</code>.</li>
 * <li><code>evaluationPeriods</code>: how many consecutive periods the metric has to be
 * breaching the the threshold for the alarm to trigger.</li>
 * </ul>
 * <p>
 * To create a cross-account alarm, make sure you have enabled <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Cross-Account-Cross-Region.html">cross-account functionality</a> in CloudWatch. Then, set the <code>account</code> property in the <code>Metric</code> object either manually or via the <code>metric.attachTo()</code> method.
 * <p>
 * <h3>Alarm Actions</h3>
 * <p>
 * To add actions to an alarm, use the integration classes from the
 * <code>&#64;aws-cdk/aws-cloudwatch-actions</code> package. For example, to post a message to
 * an SNS topic when an alarm breaches, do the following:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * Alarm alarm;
 * 
 * 
 * Topic topic = new Topic(this, "Topic");
 * alarm.addAlarmAction(new SnsAction(topic));
 * </pre></blockquote>
 * <p>
 * <h4>Notification formats</h4>
 * <p>
 * Alarms can be created in one of two "formats":
 * <p>
 * <ul>
 * <li>With "top-level parameters" (these are the classic style of CloudWatch Alarms).</li>
 * <li>With a list of metrics specifications (these are the modern style of CloudWatch Alarms).</li>
 * </ul>
 * <p>
 * For backwards compatibility, CDK will try to create classic, top-level CloudWatch alarms
 * as much as possible, unless you are using features that cannot be expressed in that format.
 * Features that require the new-style alarm format are:
 * <p>
 * <ul>
 * <li>Metric math</li>
 * <li>Cross-account metrics</li>
 * <li>Labels</li>
 * </ul>
 * <p>
 * The difference between these two does not impact the functionality of the alarm
 * in any way, <em>except</em> that the format of the notifications the Alarm generates is
 * different between them. This affects both the notifications sent out over SNS,
 * as well as the EventBridge events generated by this Alarm. If you are writing
 * code to consume these notifications, be sure to handle both formats.
 * <p>
 * <h3>Composite Alarms</h3>
 * <p>
 * <a href="https://aws.amazon.com/about-aws/whats-new/2020/03/amazon-cloudwatch-now-allows-you-to-combine-multiple-alarms/">Composite Alarms</a>
 * can be created from existing Alarm resources.
 * <p>
 * <blockquote><pre>
 * Alarm alarm1;
 * Alarm alarm2;
 * Alarm alarm3;
 * Alarm alarm4;
 * 
 * 
 * IAlarmRule alarmRule = AlarmRule.anyOf(AlarmRule.allOf(AlarmRule.anyOf(alarm1, AlarmRule.fromAlarm(alarm2, AlarmState.OK), alarm3), AlarmRule.not(AlarmRule.fromAlarm(alarm4, AlarmState.INSUFFICIENT_DATA))), AlarmRule.fromBoolean(false));
 * 
 * CompositeAlarm.Builder.create(this, "MyAwesomeCompositeAlarm")
 *         .alarmRule(alarmRule)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>A note on units</h3>
 * <p>
 * In CloudWatch, Metrics datums are emitted with units, such as <code>seconds</code> or
 * <code>bytes</code>. When <code>Metric</code> objects are given a <code>unit</code> attribute, it will be used to
 * <em>filter</em> the stream of metric datums for datums emitted using the same <code>unit</code>
 * attribute.
 * <p>
 * In particular, the <code>unit</code> field is <em>not</em> used to rescale datums or alarm threshold
 * values (for example, it cannot be used to specify an alarm threshold in
 * <em>Megabytes</em> if the metric stream is being emitted as <em>bytes</em>).
 * <p>
 * You almost certainly don't want to specify the <code>unit</code> property when creating
 * <code>Metric</code> objects (which will retrieve all datums regardless of their unit),
 * unless you have very specific requirements. Note that in any case, CloudWatch
 * only supports filtering by <code>unit</code> for Alarms, not in Dashboard graphs.
 * <p>
 * Please see the following GitHub issue for a discussion on real unit
 * calculations in CDK: https://github.com/aws/aws-cdk/issues/5595
 * <p>
 * <h2>Dashboards</h2>
 * <p>
 * Dashboards are set of Widgets stored server-side which can be accessed quickly
 * from the AWS console. Available widgets are graphs of a metric over time, the
 * current value of a metric, or a static piece of Markdown which explains what the
 * graphs mean.
 * <p>
 * The following widgets are available:
 * <p>
 * <ul>
 * <li><code>GraphWidget</code> -- shows any number of metrics on both the left and right
 * vertical axes.</li>
 * <li><code>AlarmWidget</code> -- shows the graph and alarm line for a single alarm.</li>
 * <li><code>SingleValueWidget</code> -- shows the current value of a set of metrics.</li>
 * <li><code>TextWidget</code> -- shows some static Markdown.</li>
 * <li><code>AlarmStatusWidget</code> -- shows the status of your alarms in a grid view.</li>
 * </ul>
 * <p>
 * <h3>Graph widget</h3>
 * <p>
 * A graph widget can display any number of metrics on either the <code>left</code> or
 * <code>right</code> vertical axis:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Metric executionCountMetric;
 * Metric errorCountMetric;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         .title("Executions vs error rate")
 * 
 *         .left(List.of(executionCountMetric))
 * 
 *         .right(List.of(errorCountMetric.with(MetricOptions.builder()
 *                 .statistic("average")
 *                 .label("Error rate")
 *                 .color(Color.GREEN)
 *                 .build())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Using the methods <code>addLeftMetric()</code> and <code>addRightMetric()</code> you can add metrics to a graph widget later on.
 * <p>
 * Graph widgets can also display annotations attached to the left or the right y-axis.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .leftAnnotations(List.of(HorizontalAnnotation.builder().value(1800).label(Duration.minutes(30).toHumanString()).color(Color.RED).build(), HorizontalAnnotation.builder().value(3600).label("1 hour").color("#2ca02c").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The graph legend can be adjusted from the default position at bottom of the widget.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .legendPosition(LegendPosition.RIGHT)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The graph can publish live data within the last minute that has not been fully aggregated.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .liveData(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The graph view can be changed from default 'timeSeries' to 'bar' or 'pie'.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .view(GraphWidgetView.BAR)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Alarm widget</h3>
 * <p>
 * An alarm widget shows the graph and the alarm line of a single alarm:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * 
 * 
 * dashboard.addWidgets(AlarmWidget.Builder.create()
 *         .title("Errors")
 *         .alarm(errorAlarm)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Single value widget</h3>
 * <p>
 * A single-value widget shows the latest value of a set of metrics (as opposed
 * to a graph of the value over time):
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Metric visitorCount;
 * Metric purchaseCount;
 * 
 * 
 * dashboard.addWidgets(SingleValueWidget.Builder.create()
 *         .metrics(List.of(visitorCount, purchaseCount))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Show as many digits as can fit, before rounding.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(SingleValueWidget.Builder.create()
 *         .metrics(List.of())
 * 
 *         .fullPrecision(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Text widget</h3>
 * <p>
 * A text widget shows an arbitrary piece of MarkDown. Use this to add explanations
 * to your dashboard:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TextWidget.Builder.create()
 *         .markdown("# Key Performance Indicators")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Alarm Status widget</h3>
 * <p>
 * An alarm status widget displays instantly the status of any type of alarms and gives the
 * ability to aggregate one or more alarms together in a small surface.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * 
 * 
 * dashboard.addWidgets(
 * AlarmStatusWidget.Builder.create()
 *         .alarms(List.of(errorAlarm))
 *         .build());
 * </pre></blockquote>
 * <p>
 * An alarm status widget only showing firing alarms, sorted by state and timestamp:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * 
 * 
 * dashboard.addWidgets(AlarmStatusWidget.Builder.create()
 *         .title("Errors")
 *         .alarms(List.of(errorAlarm))
 *         .sortBy(AlarmStatusWidgetSortBy.STATE_UPDATED_TIMESTAMP)
 *         .states(List.of(AlarmState.ALARM))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Query results widget</h3>
 * <p>
 * A <code>LogQueryWidget</code> shows the results of a query from Logs Insights:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(LogQueryWidget.Builder.create()
 *         .logGroupNames(List.of("my-log-group"))
 *         .view(LogQueryVisualizationType.TABLE)
 *         // The lines will be automatically combined using '\n|'.
 *         .queryLines(List.of("fields &#64;message", "filter &#64;message like /Error/"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Custom widget</h3>
 * <p>
 * A <code>CustomWidget</code> shows the result of an AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * // Import or create a lambda function
 * IFunction fn = Function.fromFunctionArn(dashboard, "Function", "arn:aws:lambda:us-east-1:123456789012:function:MyFn");
 * 
 * dashboard.addWidgets(CustomWidget.Builder.create()
 *         .functionArn(fn.getFunctionArn())
 *         .title("My lambda baked widget")
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can learn more about custom widgets in the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/add_custom_widget_dashboard.html">Amazon Cloudwatch User Guide</a>.
 * <p>
 * <h3>Dashboard Layout</h3>
 * <p>
 * The widgets on a dashboard are visually laid out in a grid that is 24 columns
 * wide. Normally you specify X and Y coordinates for the widgets on a Dashboard,
 * but because this is inconvenient to do manually, the library contains a simple
 * layout system to help you lay out your dashboards the way you want them to.
 * <p>
 * Widgets have a <code>width</code> and <code>height</code> property, and they will be automatically
 * laid out either horizontally or vertically stacked to fill out the available
 * space.
 * <p>
 * Widgets are added to a Dashboard by calling <code>add(widget1, widget2, ...)</code>.
 * Widgets given in the same call will be laid out horizontally. Widgets given
 * in different calls will be laid out vertically. To make more complex layouts,
 * you can use the following widgets to pack widgets together in different ways:
 * <p>
 * <ul>
 * <li><code>Column</code>: stack two or more widgets vertically.</li>
 * <li><code>Row</code>: lay out two or more widgets horizontally.</li>
 * <li><code>Spacer</code>: take up empty space</li>
 * </ul>
 */
package software.amazon.awscdk.services.cloudwatch;
