/**
 * <h1>Amazon EKS Construct Library</h1>
 * <p>
 * **This module is available for backwards compatibility purposes only (<a href="https://github.com/aws/aws-cdk/pull/5540">details</a>). It will
 * no longer be released with the CDK starting March 1st, 2020. See [issue
 * <p>
 * <h2>5544](https://github.com/aws/aws-cdk/issues/5544) for upgrade instructions.**</h2>
 * <p>
 * <hr>
 * <p>
 * This construct library allows you to define <a href="https://aws.amazon.com/eks/">Amazon Elastic Container Service
 * for Kubernetes (EKS)</a> clusters programmatically.
 * This library also supports programmatically defining Kubernetes resource
 * manifests within EKS clusters.
 * <p>
 * This example defines an Amazon EKS cluster with the following configuration:
 * <p>
 * <ul>
 * <li>2x <strong>m5.large</strong> instances (this instance type suits most common use-cases, and is good value for money)</li>
 * <li>Dedicated VPC with default configuration (see <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#vpc">ec2.Vpc</a>)</li>
 * <li>A Kubernetes pod with a container based on the <a href="https://github.com/paulbouwer/hello-kubernetes">paulbouwer/hello-kubernetes</a> image.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster = new Cluster(this, "hello-eks");
 * 
 * cluster.addResource("mypod", Map.of(
 *         "apiVersion", "v1",
 *         "kind", "Pod",
 *         "metadata", Map.of("name", "mypod"),
 *         "spec", Map.of(
 *                 "containers", List.of(Map.of(
 *                         "name", "hello",
 *                         "image", "paulbouwer/hello-kubernetes:1.5",
 *                         "ports", List.of(Map.of("containerPort", 8080)))))));
 * </pre></blockquote>
 * <p>
 * Here is a <a href="https://github.com/aws/aws-cdk/blob/master/packages/&#64;aws-cdk/aws-eks-legacy/test/integ.eks-kubectl.lit.ts">complete sample</a>.
 * <p>
 * <h3>Capacity</h3>
 * <p>
 * By default, <code>eks.Cluster</code> is created with x2 <code>m5.large</code> instances.
 * <p>
 * <blockquote><pre>
 * new Cluster(this, "cluster-two-m5-large");
 * </pre></blockquote>
 * <p>
 * The quantity and instance type for the default capacity can be specified through
 * the <code>defaultCapacity</code> and <code>defaultCapacityInstance</code> props:
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "cluster")
 *         .defaultCapacity(10)
 *         .defaultCapacityInstance(new InstanceType("m2.xlarge"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To disable the default capacity, simply set <code>defaultCapacity</code> to <code>0</code>:
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "cluster-with-no-capacity").defaultCapacity(0).build();
 * </pre></blockquote>
 * <p>
 * The <code>cluster.defaultCapacity</code> property will reference the <code>AutoScalingGroup</code>
 * resource for the default capacity. It will be <code>undefined</code> if <code>defaultCapacity</code>
 * is set to <code>0</code>:
 * <p>
 * <blockquote><pre>
 * Cluster cluster = new Cluster(this, "my-cluster");
 * cluster.defaultCapacity.scaleOnCpuUtilization("up", CpuUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(80)
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can add customized capacity through <code>cluster.addCapacity()</code> or
 * <code>cluster.addAutoScalingGroup()</code>:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addCapacity("frontend-nodes", CapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.medium"))
 *         .desiredCapacity(3)
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Spot Capacity</h3>
 * <p>
 * If <code>spotPrice</code> is specified, the capacity will be purchased from spot instances:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.addCapacity("spot", CapacityOptions.builder()
 *         .spotPrice("0.1094")
 *         .instanceType(new InstanceType("t3.large"))
 *         .maxCapacity(10)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Spot instance nodes will be labeled with <code>lifecycle=Ec2Spot</code> and tainted with <code>PreferNoSchedule</code>.
 * <p>
 * The <a href="https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler">Spot Termination Handler</a>
 * DaemonSet will be installed on these nodes. The termination handler leverages
 * <a href="https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/">EC2 Spot Instance Termination Notices</a>
 * to gracefully stop all pods running on spot nodes that are about to be
 * terminated.
 * <p>
 * <h3>Bootstrapping</h3>
 * <p>
 * When adding capacity, you can specify options for
 * <a href="https://github.com/awslabs/amazon-eks-ami/blob/master/files/bootstrap.sh">/etc/eks/boostrap.sh</a>
 * which is responsible for associating the node to the EKS cluster. For example,
 * you can use <code>kubeletExtraArgs</code> to add custom node labels or taints.
 * <p>
 * <blockquote><pre>
 * // up to ten spot instances
 * Cluster cluster;
 * 
 * cluster.addCapacity("spot", CapacityOptions.builder()
 *         .instanceType(new InstanceType("t3.large"))
 *         .desiredCapacity(2)
 *         .bootstrapOptions(BootstrapOptions.builder()
 *                 .kubeletExtraArgs("--node-labels foo=bar,goo=far")
 *                 .awsApiRetryAttempts(5)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * To disable bootstrapping altogether (i.e. to fully customize user-data), set <code>bootstrapEnabled</code> to <code>false</code> when you add
 * the capacity.
 * <p>
 * <h3>Masters Role</h3>
 * <p>
 * The Amazon EKS construct library allows you to specify an IAM role that will be
 * granted <code>system:masters</code> privileges on your cluster.
 * <p>
 * Without specifying a <code>mastersRole</code>, you will not be able to interact manually
 * with the cluster.
 * <p>
 * The following example defines an IAM role that can be assumed by all users
 * in the account and shows how to use the <code>mastersRole</code> property to map this
 * role to the Kubernetes <code>system:masters</code> group:
 * <p>
 * <blockquote><pre>
 * // first define the role
 * Role clusterAdmin = Role.Builder.create(this, "AdminRole")
 *         .assumedBy(new AccountRootPrincipal())
 *         .build();
 * 
 * // now define the cluster and map role to "masters" RBAC group
 * // now define the cluster and map role to "masters" RBAC group
 * Cluster.Builder.create(this, "Cluster")
 *         .mastersRole(clusterAdmin)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you <code>cdk deploy</code> this CDK app, you will notice that an output will be printed
 * with the <code>update-kubeconfig</code> command.
 * <p>
 * Something like this:
 * <p>
 * <blockquote><pre>
 * Outputs:
 * eks-integ-defaults.ClusterConfigCommand43AAE40F = aws eks update-kubeconfig --name cluster-ba7c166b-c4f3-421c-bf8a-6812e4036a33 --role-arn arn:aws:iam::112233445566:role/eks-integ-defaults-Role1ABCC5F0-1EFK2W5ZJD98Y
 * </pre></blockquote>
 * <p>
 * Copy &amp; paste the "<code>aws eks update-kubeconfig ...</code>" command to your shell in
 * order to connect to your EKS cluster with the "masters" role.
 * <p>
 * Now, given <a href="https://aws.amazon.com/cli/">AWS CLI</a> is configured to use AWS
 * credentials for a user that is trusted by the masters role, you should be able
 * to interact with your cluster through <code>kubectl</code> (the above example will trust
 * all users in the account).
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * $ aws eks update-kubeconfig --name cluster-ba7c166b-c4f3-421c-bf8a-6812e4036a33 --role-arn arn:aws:iam::112233445566:role/eks-integ-defaults-Role1ABCC5F0-1EFK2W5ZJD98Y
 * Added new context arn:aws:eks:eu-west-2:112233445566:cluster/cluster-ba7c166b-c4f3-421c-bf8a-6812e4036a33 to /Users/boom/.kube/config
 * 
 * $ kubectl get nodes # list all nodes
 * NAME                                         STATUS   ROLES    AGE   VERSION
 * ip-10-0-147-66.eu-west-2.compute.internal    Ready    &lt;none&gt;   21m   v1.13.7-eks-c57ff8
 * ip-10-0-169-151.eu-west-2.compute.internal   Ready    &lt;none&gt;   21m   v1.13.7-eks-c57ff8
 * 
 * $ kubectl get all -n kube-system
 * NAME                           READY   STATUS    RESTARTS   AGE
 * pod/aws-node-fpmwv             1/1     Running   0          21m
 * pod/aws-node-m9htf             1/1     Running   0          21m
 * pod/coredns-5cb4fb54c7-q222j   1/1     Running   0          23m
 * pod/coredns-5cb4fb54c7-v9nxx   1/1     Running   0          23m
 * pod/kube-proxy-d4jrh           1/1     Running   0          21m
 * pod/kube-proxy-q7hh7           1/1     Running   0          21m
 * 
 * NAME               TYPE        CLUSTER-IP    EXTERNAL-IP   PORT(S)         AGE
 * service/kube-dns   ClusterIP   172.20.0.10   &lt;none&gt;        53/UDP,53/TCP   23m
 * 
 * NAME                        DESIRED   CURRENT   READY   UP-TO-DATE   AVAILABLE   NODE SELECTOR   AGE
 * daemonset.apps/aws-node     2         2         2       2            2           &lt;none&gt;          23m
 * daemonset.apps/kube-proxy   2         2         2       2            2           &lt;none&gt;          23m
 * 
 * NAME                      READY   UP-TO-DATE   AVAILABLE   AGE
 * deployment.apps/coredns   2/2     2            2           23m
 * 
 * NAME                                 DESIRED   CURRENT   READY   AGE
 * replicaset.apps/coredns-5cb4fb54c7   2         2         2       23m
 * </pre></blockquote>
 * <p>
 * For your convenience, an AWS CloudFormation output will automatically be
 * included in your template and will be printed when running <code>cdk deploy</code>.
 * <p>
 * <strong>NOTE</strong>: if the cluster is configured with <code>kubectlEnabled: false</code>, it
 * will be created with the role/user that created the AWS CloudFormation
 * stack. See <a href="#kubectl-support">Kubectl Support</a> for details.
 * <p>
 * <h3>Kubernetes Resources</h3>
 * <p>
 * The <code>KubernetesResource</code> construct or <code>cluster.addResource</code> method can be used
 * to apply Kubernetes resource manifests to this cluster.
 * <p>
 * The following examples will deploy the <a href="https://github.com/paulbouwer/hello-kubernetes">paulbouwer/hello-kubernetes</a>
 * service on the cluster:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * Map&lt;String, String&gt; appLabel = Map.of("app", "hello-kubernetes");
 * 
 * Map&lt;String, Object&gt; deployment = Map.of(
 *         "apiVersion", "apps/v1",
 *         "kind", "Deployment",
 *         "metadata", Map.of("name", "hello-kubernetes"),
 *         "spec", Map.of(
 *                 "replicas", 3,
 *                 "selector", Map.of("matchLabels", appLabel),
 *                 "template", Map.of(
 *                         "metadata", Map.of("labels", appLabel),
 *                         "spec", Map.of(
 *                                 "containers", List.of(Map.of(
 *                                         "name", "hello-kubernetes",
 *                                         "image", "paulbouwer/hello-kubernetes:1.5",
 *                                         "ports", List.of(Map.of("containerPort", 8080))))))));
 * 
 * Map&lt;String, Object&gt; service = Map.of(
 *         "apiVersion", "v1",
 *         "kind", "Service",
 *         "metadata", Map.of("name", "hello-kubernetes"),
 *         "spec", Map.of(
 *                 "type", "LoadBalancer",
 *                 "ports", List.of(Map.of("port", 80, "targetPort", 8080)),
 *                 "selector", appLabel));
 * // option 1: use a construct
 * // option 1: use a construct
 * KubernetesResource.Builder.create(this, "hello-kub")
 *         .cluster(cluster)
 *         .manifest(List.of(deployment, service))
 *         .build();
 * 
 * // or, option2: use `addResource`
 * cluster.addResource("hello-kub", service, deployment);
 * </pre></blockquote>
 * <p>
 * Since Kubernetes resources are implemented as CloudFormation resources in the
 * CDK. This means that if the resource is deleted from your code (or the stack is
 * deleted), the next <code>cdk deploy</code> will issue a <code>kubectl delete</code> command and the
 * Kubernetes resources will be deleted.
 * <p>
 * <h3>AWS IAM Mapping</h3>
 * <p>
 * As described in the <a href="https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html">Amazon EKS User Guide</a>,
 * you can map AWS IAM users and roles to <a href="https://kubernetes.io/docs/reference/access-authn-authz/rbac">Kubernetes Role-based access control (RBAC)</a>.
 * <p>
 * The Amazon EKS construct manages the <strong>aws-auth ConfigMap</strong> Kubernetes resource
 * on your behalf and exposes an API through the <code>cluster.awsAuth</code> for mapping
 * users, roles and accounts.
 * <p>
 * Furthermore, when auto-scaling capacity is added to the cluster (through
 * <code>cluster.addCapacity</code> or <code>cluster.addAutoScalingGroup</code>), the IAM instance role
 * of the auto-scaling group will be automatically mapped to RBAC so nodes can
 * connect to the cluster. No manual mapping is required any longer.
 * <p>
 * <blockquote>
 * <p>
 * NOTE: <code>cluster.awsAuth</code> will throw an error if your cluster is created with <code>kubectlEnabled: false</code>.
 * <p>
 * </blockquote>
 * <p>
 * For example, let's say you want to grant an IAM user administrative privileges
 * on your cluster:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * User adminUser = new User(this, "Admin");
 * cluster.awsAuth.addUserMapping(adminUser, Mapping.builder().groups(List.of("system:masters")).build());
 * </pre></blockquote>
 * <p>
 * A convenience method for mapping a role to the <code>system:masters</code> group is also available:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * Role role;
 * 
 * cluster.awsAuth.addMastersRole(role);
 * </pre></blockquote>
 * <p>
 * <h3>Node ssh Access</h3>
 * <p>
 * If you want to be able to SSH into your worker nodes, you must already
 * have an SSH key in the region you're connecting to and pass it, and you must
 * be able to connect to the hosts (meaning they must have a public IP and you
 * should be allowed to connect to them on port 22):
 * <p>
 * <blockquote><pre>
 * AutoScalingGroup asg = cluster.addCapacity("Nodes", CapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.medium"))
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .keyName("my-key-name")
 *         .build());
 * 
 * // Replace with desired IP
 * asg.connections.allowFrom(Peer.ipv4("1.2.3.4/32"), Port.tcp(22));
 * </pre></blockquote>
 * <p>
 * If you want to SSH into nodes in a private subnet, you should set up a
 * bastion host in a public subnet. That setup is recommended, but is
 * unfortunately beyond the scope of this documentation.
 * <p>
 * <h3>kubectl Support</h3>
 * <p>
 * When you create an Amazon EKS cluster, the IAM entity user or role, such as a
 * <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers.html">federated user</a>
 * that creates the cluster, is automatically granted <code>system:masters</code> permissions
 * in the cluster's RBAC configuration.
 * <p>
 * In order to allow programmatically defining <strong>Kubernetes resources</strong> in your AWS
 * CDK app and provisioning them through AWS CloudFormation, we will need to assume
 * this "masters" role every time we want to issue <code>kubectl</code> operations against your
 * cluster.
 * <p>
 * At the moment, the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html">AWS::EKS::Cluster</a>
 * AWS CloudFormation resource does not support this behavior, so in order to
 * support "programmatic kubectl", such as applying manifests
 * and mapping IAM roles from within your CDK application, the Amazon EKS
 * construct library uses a custom resource for provisioning the cluster.
 * This custom resource is executed with an IAM role that we can then use
 * to issue <code>kubectl</code> commands.
 * <p>
 * The default behavior of this library is to use this custom resource in order
 * to retain programmatic control over the cluster. In other words: to allow
 * you to define Kubernetes resources in your CDK code instead of having to
 * manage your Kubernetes applications through a separate system.
 * <p>
 * One of the implications of this design is that, by default, the user who
 * provisioned the AWS CloudFormation stack (executed <code>cdk deploy</code>) will
 * not have administrative privileges on the EKS cluster.
 * <p>
 * <ol>
 * <li>Additional resources will be synthesized into your template (the AWS Lambda
 * function, the role and policy).</li>
 * <li>As described in <a href="#interacting-with-your-cluster">Interacting with Your Cluster</a>,
 * if you wish to be able to manually interact with your cluster, you will need
 * to map an IAM role or user to the <code>system:masters</code> group. This can be either
 * done by specifying a <code>mastersRole</code> when the cluster is defined, calling
 * <code>cluster.awsAuth.addMastersRole</code> or explicitly mapping an IAM role or IAM user to the
 * relevant Kubernetes RBAC groups using <code>cluster.addRoleMapping</code> and/or
 * <code>cluster.addUserMapping</code>.</li>
 * </ol>
 * <p>
 * If you wish to disable the programmatic kubectl behavior and use the standard
 * AWS::EKS::Cluster resource, you can specify <code>kubectlEnabled: false</code> when you define
 * the cluster:
 * <p>
 * <blockquote><pre>
 * Cluster.Builder.create(this, "cluster")
 *         .kubectlEnabled(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Take care</strong>: a change in this property will cause the cluster to be destroyed
 * and a new cluster to be created.
 * <p>
 * When kubectl is disabled, you should be aware of the following:
 * <p>
 * <ol>
 * <li>When you log-in to your cluster, you don't need to specify <code>--role-arn</code> as
 * long as you are using the same user that created the cluster.</li>
 * <li>As described in the Amazon EKS User Guide, you will need to manually
 * edit the <a href="https://docs.aws.amazon.com/eks/latest/userguide/add-user-role.html">aws-auth ConfigMap</a>
 * when you add capacity in order to map the IAM instance role to RBAC to allow nodes to join the cluster.</li>
 * <li>Any <code>eks.Cluster</code> APIs that depend on programmatic kubectl support will fail
 * with an error: <code>cluster.addResource</code>, <code>cluster.addChart</code>, <code>cluster.awsAuth</code>, <code>props.mastersRole</code>.</li>
 * </ol>
 * <p>
 * <h3>Helm Charts</h3>
 * <p>
 * The <code>HelmChart</code> construct or <code>cluster.addChart</code> method can be used
 * to add Kubernetes resources to this cluster using Helm.
 * <p>
 * The following example will install the <a href="https://kubernetes.github.io/ingress-nginx/">NGINX Ingress Controller</a>
 * to you cluster using Helm.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // option 1: use a construct
 * // option 1: use a construct
 * HelmChart.Builder.create(this, "NginxIngress")
 *         .cluster(cluster)
 *         .chart("nginx-ingress")
 *         .repository("https://helm.nginx.com/stable")
 *         .namespace("kube-system")
 *         .build();
 * 
 * // or, option2: use `addChart`
 * cluster.addChart("NginxIngress", HelmChartOptions.builder()
 *         .chart("nginx-ingress")
 *         .repository("https://helm.nginx.com/stable")
 *         .namespace("kube-system")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Helm charts will be installed and updated using <code>helm upgrade --install</code>.
 * This means that if the chart is added to CDK with the same release name, it will try to update
 * the chart in the cluster. The chart will exists as CloudFormation resource.
 * <p>
 * Helm charts are implemented as CloudFormation resources in CDK.
 * This means that if the chart is deleted from your code (or the stack is
 * deleted), the next <code>cdk deploy</code> will issue a <code>helm uninstall</code> command and the
 * Helm chart will be deleted.
 * <p>
 * When there is no <code>release</code> defined, the chart will be installed with a unique name allocated
 * based on the construct path.
 * <p>
 * <h3>Roadmap</h3>
 * <p>
 * <ul>
 * <li>[ ] AutoScaling (combine EC2 and Kubernetes scaling)</li>
 * </ul>
 */
package software.amazon.awscdk.services.eks.legacy;
