/**
 * <h1>Amazon Elasticsearch Service Construct Library</h1>
 * <p>
 * <blockquote>
 * <p>
 * Instead of this module, we recommend using the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-opensearchservice-readme.html">&#64;aws-cdk/aws-opensearchservice</a> module. See <a href="https://aws.amazon.com/opensearch-service/faqs/#Name_change">Amazon OpenSearch Service FAQs</a> for details. See <a href="#migrating-to-opensearch">Migrating to OpenSearch</a> for migration instructions.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Quick start</h2>
 * <p>
 * Create a development cluster by simply specifying the version:
 * <p>
 * <blockquote><pre>
 * Domain devDomain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To perform version upgrades without replacing the entire domain, specify the <code>enableVersionUpgrade</code> property.
 * <p>
 * <blockquote><pre>
 * Domain devDomain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_10)
 *         .enableVersionUpgrade(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Create a production grade cluster by also specifying things like capacity and az distribution
 * <p>
 * <blockquote><pre>
 * Domain prodDomain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .capacity(CapacityConfig.builder()
 *                 .masterNodes(5)
 *                 .dataNodes(20)
 *                 .build())
 *         .ebs(EbsOptions.builder()
 *                 .volumeSize(20)
 *                 .build())
 *         .zoneAwareness(ZoneAwarenessConfig.builder()
 *                 .availabilityZoneCount(3)
 *                 .build())
 *         .logging(LoggingOptions.builder()
 *                 .slowSearchLogEnabled(true)
 *                 .appLogEnabled(true)
 *                 .slowIndexLogEnabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * This creates an Elasticsearch cluster and automatically sets up log groups for
 * logging the domain logs and slow search logs.
 * <p>
 * <h2>A note about SLR</h2>
 * <p>
 * Some cluster configurations (e.g VPC access) require the existence of the <a href="https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/slr-es.html"><code>AWSServiceRoleForAmazonElasticsearchService</code></a> Service-Linked Role.
 * <p>
 * When performing such operations via the AWS Console, this SLR is created automatically when needed. However, this is not the behavior when using CloudFormation. If an SLR is needed, but doesn't exist, you will encounter a failure message simlar to:
 * <p>
 * <blockquote><pre>
 * Before you can proceed, you must enable a service-linked role to give Amazon ES...
 * </pre></blockquote>
 * <p>
 * To resolve this, you need to <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/using-service-linked-roles.html#create-service-linked-role">create</a> the SLR. We recommend using the AWS CLI:
 * <p>
 * <blockquote><pre>
 * aws iam create-service-linked-role --aws-service-name es.amazonaws.com
 * </pre></blockquote>
 * <p>
 * You can also create it using the CDK, <strong>but note that only the first application deploying this will succeed</strong>:
 * <p>
 * <blockquote><pre>
 * CfnServiceLinkedRole slr = CfnServiceLinkedRole.Builder.create(this, "ElasticSLR")
 *         .awsServiceName("es.amazonaws.com")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing domains</h2>
 * <p>
 * To import an existing domain into your CDK application, use the <code>Domain.fromDomainEndpoint</code> factory method.
 * This method accepts a domain endpoint of an already existing domain:
 * <p>
 * <blockquote><pre>
 * String domainEndpoint = "https://my-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com";
 * IDomain domain = Domain.fromDomainEndpoint(this, "ImportedDomain", domainEndpoint);
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * <h3>IAM</h3>
 * <p>
 * Helper methods also exist for managing access to the domain.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * Domain domain;
 * 
 * 
 * // Grant write access to the app-search index
 * domain.grantIndexWrite("app-search", fn);
 * 
 * // Grant read access to the 'app-search/_search' path
 * domain.grantPathRead("app-search/_search", fn);
 * </pre></blockquote>
 * <p>
 * <h2>Encryption</h2>
 * <p>
 * The domain can also be created with encryption enabled:
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_4)
 *         .ebs(EbsOptions.builder()
 *                 .volumeSize(100)
 *                 .volumeType(EbsDeviceVolumeType.GENERAL_PURPOSE_SSD)
 *                 .build())
 *         .nodeToNodeEncryption(true)
 *         .encryptionAtRest(EncryptionAtRestOptions.builder()
 *                 .enabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * This sets up the domain with node to node encryption and encryption at
 * rest. You can also choose to supply your own KMS key to use for encryption at
 * rest.
 * <p>
 * <h2>VPC Support</h2>
 * <p>
 * Elasticsearch domains can be placed inside a VPC, providing a secure communication between Amazon ES and other services within the VPC without the need for an internet gateway, NAT device, or VPN connection.
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html">VPC Support for Amazon Elasticsearch Service Domains</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * Vpc vpc = new Vpc(this, "Vpc");
 * DomainProps domainProps = DomainProps.builder()
 *         .version(ElasticsearchVersion.V7_1)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .vpc(vpc)
 *         // must be enabled since our VPC contains multiple private subnets.
 *         .zoneAwareness(ZoneAwarenessConfig.builder()
 *                 .enabled(true)
 *                 .build())
 *         .capacity(CapacityConfig.builder()
 *                 // must be an even number since the default az count is 2.
 *                 .dataNodes(2)
 *                 .build())
 *         .build();
 * new Domain(this, "Domain", domainProps);
 * </pre></blockquote>
 * <p>
 * In addition, you can use the <code>vpcSubnets</code> property to control which specific subnets will be used, and the <code>securityGroups</code> property to control
 * which security groups will be attached to the domain. By default, CDK will select all <em>private</em> subnets in the VPC, and create one dedicated security group.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * Helper methods exist to access common domain metrics for example:
 * <p>
 * <blockquote><pre>
 * Domain domain;
 * 
 * Metric freeStorageSpace = domain.metricFreeStorageSpace();
 * Metric masterSysMemoryUtilization = domain.metric("MasterSysMemoryUtilization");
 * </pre></blockquote>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Fine grained access control</h2>
 * <p>
 * The domain can also be created with a master user configured. The password can
 * be supplied or dynamically created if not supplied.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .enforceHttps(true)
 *         .nodeToNodeEncryption(true)
 *         .encryptionAtRest(EncryptionAtRestOptions.builder()
 *                 .enabled(true)
 *                 .build())
 *         .fineGrainedAccessControl(AdvancedSecurityOptions.builder()
 *                 .masterUserName("master-user")
 *                 .build())
 *         .build();
 * 
 * SecretValue masterUserPassword = domain.getMasterUserPassword();
 * </pre></blockquote>
 * <p>
 * <h2>Using unsigned basic auth</h2>
 * <p>
 * For convenience, the domain can be configured to allow unsigned HTTP requests
 * that use basic auth. Unless the domain is configured to be part of a VPC this
 * means anyone can access the domain using the configured master username and
 * password.
 * <p>
 * To enable unsigned basic auth access the domain is configured with an access
 * policy that allows anyonmous requests, HTTPS required, node to node encryption,
 * encryption at rest and fine grained access control.
 * <p>
 * If the above settings are not set they will be configured as part of enabling
 * unsigned basic auth. If they are set with conflicting values, an error will be
 * thrown.
 * <p>
 * If no master user is configured a default master user is created with the
 * username <code>admin</code>.
 * <p>
 * If no password is configured a default master user password is created and
 * stored in the AWS Secrets Manager as secret. The secret has the prefix
 * <code>&lt;domain id&gt;MasterUser</code>.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .useUnsignedBasicAuth(true)
 *         .build();
 * 
 * SecretValue masterUserPassword = domain.getMasterUserPassword();
 * </pre></blockquote>
 * <p>
 * <h2>Custom access policies</h2>
 * <p>
 * If the domain requires custom access control it can be configured either as a
 * constructor property, or later by means of a helper method.
 * <p>
 * For simple permissions the <code>accessPolicies</code> constructor may be sufficient:
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .accessPolicies(List.of(
 *             PolicyStatement.Builder.create()
 *                     .actions(List.of("es:*ESHttpPost", "es:ESHttpPut*"))
 *                     .effect(Effect.ALLOW)
 *                     .principals(List.of(new AccountPrincipal("123456789012")))
 *                     .resources(List.of("*"))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more complex use-cases, for example, to set the domain up to receive data from a
 * <a href="https://aws.amazon.com/premiumsupport/knowledge-center/kinesis-firehose-cross-account-streaming/">cross-account Kinesis Firehose</a> the <code>addAccessPolicies</code> helper method
 * allows for policies that include the explicit domain ARN.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .build();
 * 
 * domain.addAccessPolicies(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("es:ESHttpPost", "es:ESHttpPut"))
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(new AccountPrincipal("123456789012")))
 *         .resources(List.of(domain.getDomainArn(), String.format("%s/*", domain.getDomainArn())))
 *         .build(),
 * PolicyStatement.Builder.create()
 *         .actions(List.of("es:ESHttpGet"))
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(new AccountPrincipal("123456789012")))
 *         .resources(List.of(String.format("%s/_all/_settings", domain.getDomainArn()), String.format("%s/_cluster/stats", domain.getDomainArn()), String.format("%s/index-name*&#47;_mapping/type-name", domain.getDomainArn()), String.format("%s/roletest*&#47;_mapping/roletest", domain.getDomainArn()), String.format("%s/_nodes", domain.getDomainArn()), String.format("%s/_nodes/stats", domain.getDomainArn()), String.format("%s/_nodes/*&#47;stats", domain.getDomainArn()), String.format("%s/_stats", domain.getDomainArn()), String.format("%s/index-name*&#47;_stats", domain.getDomainArn()), String.format("%s/roletest*&#47;_stat", domain.getDomainArn())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Audit logs</h2>
 * <p>
 * Audit logs can be enabled for a domain, but only when fine grained access control is enabled.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_1)
 *         .enforceHttps(true)
 *         .nodeToNodeEncryption(true)
 *         .encryptionAtRest(EncryptionAtRestOptions.builder()
 *                 .enabled(true)
 *                 .build())
 *         .fineGrainedAccessControl(AdvancedSecurityOptions.builder()
 *                 .masterUserName("master-user")
 *                 .build())
 *         .logging(LoggingOptions.builder()
 *                 .auditLogEnabled(true)
 *                 .slowSearchLogEnabled(true)
 *                 .appLogEnabled(true)
 *                 .slowIndexLogEnabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>UltraWarm</h2>
 * <p>
 * UltraWarm nodes can be enabled to provide a cost-effective way to store large amounts of read-only data.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_10)
 *         .capacity(CapacityConfig.builder()
 *                 .masterNodes(2)
 *                 .warmNodes(2)
 *                 .warmInstanceType("ultrawarm1.medium.elasticsearch")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Custom endpoint</h2>
 * <p>
 * Custom endpoints can be configured to reach the ES domain under a custom domain name.
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_7)
 *         .customEndpoint(CustomEndpointOptions.builder()
 *                 .domainName("search.example.com")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * It is also possible to specify a custom certificate instead of the auto-generated one.
 * <p>
 * Additionally, an automatic CNAME-Record is created if a hosted zone is provided for the custom endpoint
 * <p>
 * <h2>Advanced options</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-advanced-options">Advanced options</a> can used to configure additional options.
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .version(ElasticsearchVersion.V7_7)
 *         .advancedOptions(Map.of(
 *                 "rest.action.multi.allow_explicit_index", "false",
 *                 "indices.fielddata.cache.size", "25",
 *                 "indices.query.bool.max_clause_count", "2048"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Migrating to OpenSearch</h2>
 * <p>
 * To migrate from this module (<code>&#64;aws-cdk/aws-elasticsearch</code>) to the new <code>&#64;aws-cdk/aws-opensearchservice</code> module, you must modify your CDK application to refer to the new module (including some associated changes) and then perform a CloudFormation resource deletion/import.
 * <p>
 * <h3>Necessary CDK Modifications</h3>
 * <p>
 * Make the following modifications to your CDK application to migrate to the <code>&#64;aws-cdk/aws-opensearchservice</code> module.
 * <p>
 * <ul>
 * <li>Rewrite module imports to use <code>'&#64;aws-cdk/aws-opensearchservice</code> to <code>'&#64;aws-cdk/aws-elasticsearch</code>.
 * For example:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.services.elasticsearch.Domain;
 * </pre></blockquote>
 * <p>
 * ...becomes...
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.services.opensearchservice.Domain;
 * </pre></blockquote></li>
 * <li>Replace instances of <code>es.ElasticsearchVersion</code> with <code>opensearch.EngineVersion</code>.
 * For example:
 * <p>
 * <blockquote><pre>
 * ElasticsearchVersion version = ElasticsearchVersion.V7_1;
 * </pre></blockquote>
 * <p>
 * ...becomes...
 * <p>
 * <blockquote><pre>
 * EngineVersion version = EngineVersion.ELASTICSEARCH_7_1;
 * </pre></blockquote></li>
 * <li>Replace the <code>cognitoKibanaAuth</code> property of <code>DomainProps</code> with <code>cognitoDashboardsAuth</code>.
 * For example:
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .cognitoKibanaAuth(CognitoOptions.builder()
 *                 .identityPoolId("test-identity-pool-id")
 *                 .userPoolId("test-user-pool-id")
 *                 .role(role)
 *                 .build())
 *         .version(elasticsearchVersion)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ...becomes...
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .cognitoDashboardsAuth(CognitoOptions.builder()
 *                 .identityPoolId("test-identity-pool-id")
 *                 .userPoolId("test-user-pool-id")
 *                 .role(role)
 *                 .build())
 *         .version(openSearchVersion)
 *         .build();
 * </pre></blockquote></li>
 * <li>Rewrite instance type suffixes from <code>.elasticsearch</code> to <code>.search</code>.
 * For example:
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .capacity(CapacityConfig.builder()
 *                 .masterNodeInstanceType("r5.large.elasticsearch")
 *                 .build())
 *         .version(elasticsearchVersion)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ...becomes...
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .capacity(CapacityConfig.builder()
 *                 .masterNodeInstanceType("r5.large.search")
 *                 .build())
 *         .version(openSearchVersion)
 *         .build();
 * </pre></blockquote></li>
 * <li>Any <code>CfnInclude</code>'d domains will need to be re-written in their original template in
 * order to be successfully included as a <code>opensearch.CfnDomain</code></li>
 * </ul>
 * <p>
 * <h3>CloudFormation Migration</h3>
 * <p>
 * Follow these steps to migrate your application without data loss:
 * <p>
 * <ul>
 * <li>Ensure that the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_core.RemovalPolicy.html">removal policy</a> on your domains are set to <code>RemovalPolicy.RETAIN</code>. This is the default for the domain construct, so nothing is required unless you have specifically set the removal policy to some other value.</li>
 * <li>Remove the domain resource from your CloudFormation stacks by manually modifying the synthesized templates used to create the CloudFormation stacks. This may also involve modifying or deleting dependent resources, such as the custom resources that CDK creates to manage the domain's access policy or any other resource you have connected to the domain. You will need to search for references to each domain's logical ID to determine which other resources refer to it and replace or delete those references. Do not remove resources that are dependencies of the domain or you will have to recreate or import them before importing the domain. After modification, deploy the stacks through the AWS Management Console or using the AWS CLI.</li>
 * <li>Migrate your CDK application to use the new <code>&#64;aws-cdk/aws-opensearchservice</code> module by applying the necessary modifications listed above. Synthesize your application and obtain the resulting stack templates.</li>
 * <li>Copy just the definition of the domain from the "migrated" templates to the corresponding "stripped" templates that you deployed above. <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-existing-stack.html">Import</a> the orphaned domains into your CloudFormation stacks using these templates.</li>
 * <li>Synthesize and deploy your CDK application to reconfigure/recreate the modified dependent resources. The CloudFormation stacks should now contain the same resources as existed prior to migration.</li>
 * <li>Proceed with development as normal!</li>
 * </ul>
 */
package software.amazon.awscdk.services.elasticsearch;
