/**
 * <h1>Actions for AWS IoT Rule</h1>
 * <p>
 * This library contains integration classes to send data to any number of
 * supported AWS Services. Instances of these classes should be passed to
 * <code>TopicRule</code> defined in <code>&#64;aws-cdk/aws-iot</code>.
 * <p>
 * Currently supported are:
 * <p>
 * <ul>
 * <li>Republish a message to another MQTT topic</li>
 * <li>Invoke a Lambda function</li>
 * <li>Put objects to a S3 bucket</li>
 * <li>Put logs to CloudWatch Logs</li>
 * <li>Capture CloudWatch metrics</li>
 * <li>Change state for a CloudWatch alarm</li>
 * <li>Put records to Kinesis Data stream</li>
 * <li>Put records to Kinesis Data Firehose stream</li>
 * <li>Send messages to SQS queues</li>
 * <li>Publish messages on SNS topics</li>
 * </ul>
 * <p>
 * <h2>Republish a message to another MQTT topic</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that republish a message to
 * another MQTT topic when it is triggered.
 * <p>
 * <blockquote><pre>
 * TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id, timestamp() as timestamp, temperature FROM 'device/+/data'"))
 *         .actions(List.of(
 *             IotRepublishMqttAction.Builder.create("${topic()}/republish")
 *                     .qualityOfService(MqttQualityOfService.AT_LEAST_ONCE)
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Invoke a Lambda function</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that invoke a Lambda function
 * when it is triggered.
 * <p>
 * <blockquote><pre>
 * Function func = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_14_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("\n    exports.handler = (event) =&gt; {\n      console.log(\"It is test for lambda action of AWS IoT Rule.\", event);\n    };"))
 *         .build();
 * 
 * TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id, timestamp() as timestamp, temperature FROM 'device/+/data'"))
 *         .actions(List.of(new LambdaFunctionAction(func)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put objects to a S3 bucket</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that put objects to a S3 bucket
 * when it is triggered.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id FROM 'device/+/data'"))
 *         .actions(List.of(new S3PutObjectAction(bucket)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The property <code>key</code> of <code>S3PutObjectAction</code> is given the value <code>${topic()}/${timestamp()}</code> by default. This <code>${topic()}</code>
 * and <code>${timestamp()}</code> is called Substitution templates. For more information see
 * <a href="https://docs.aws.amazon.com/iot/latest/developerguide/iot-substitution-templates.html">this documentation</a>.
 * In above sample, <code>${topic()}</code> is replaced by a given MQTT topic as <code>device/001/data</code>. And <code>${timestamp()}</code> is replaced
 * by the number of the current timestamp in milliseconds as <code>1636289461203</code>. So if the MQTT broker receives an MQTT topic
 * <code>device/001/data</code> on <code>2021-11-07T00:00:00.000Z</code>, the S3 bucket object will be put to <code>device/001/data/1636243200000</code>.
 * <p>
 * You can also set specific <code>key</code> as following:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id, year, month, day FROM 'device/+/data'"))
 *         .actions(List.of(
 *             S3PutObjectAction.Builder.create(bucket)
 *                     .key("${year}/${month}/${day}/${topic(2)}")
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you wanna set access control to the S3 bucket object, you can specify <code>accessControl</code> as following:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT * FROM 'device/+/data'"))
 *         .actions(List.of(
 *             S3PutObjectAction.Builder.create(bucket)
 *                     .accessControl(BucketAccessControl.PUBLIC_READ)
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put logs to CloudWatch Logs</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that put logs to CloudWatch Logs
 * when it is triggered.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup");
 * 
 * TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id FROM 'device/+/data'"))
 *         .actions(List.of(new CloudWatchLogsAction(logGroup)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Capture CloudWatch metrics</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that capture CloudWatch metrics
 * when it is triggered.
 * <p>
 * <blockquote><pre>
 * TopicRule topicRule = TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id, namespace, unit, value, timestamp FROM 'device/+/data'"))
 *         .actions(List.of(
 *             CloudWatchPutMetricAction.Builder.create()
 *                     .metricName("${topic(2)}")
 *                     .metricNamespace("${namespace}")
 *                     .metricUnit("${unit}")
 *                     .metricValue("${value}")
 *                     .metricTimestamp("${timestamp}")
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Change the state of an Amazon CloudWatch alarm</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that changes the state of an Amazon CloudWatch alarm when it is triggered:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Metric metric = Metric.Builder.create()
 *         .namespace("MyNamespace")
 *         .metricName("MyMetric")
 *         .dimensions(Map.of("MyDimension", "MyDimensionValue"))
 *         .build();
 * Alarm alarm = Alarm.Builder.create(this, "MyAlarm")
 *         .metric(metric)
 *         .threshold(100)
 *         .evaluationPeriods(3)
 *         .datapointsToAlarm(2)
 *         .build();
 * 
 * TopicRule topicRule = TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id FROM 'device/+/data'"))
 *         .actions(List.of(
 *             CloudWatchSetAlarmStateAction.Builder.create(alarm)
 *                     .reason("AWS Iot Rule action is triggered")
 *                     .alarmStateToSet(AlarmState.ALARM)
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put records to Kinesis Data stream</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that put records to Kinesis Data
 * stream when it is triggered.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Stream stream = new Stream(this, "MyStream");
 * 
 * TopicRule topicRule = TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT * FROM 'device/+/data'"))
 *         .actions(List.of(
 *             KinesisPutRecordAction.Builder.create(stream)
 *                     .partitionKey("${newuuid()}")
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put records to Kinesis Data Firehose stream</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that put records to Put records
 * to Kinesis Data Firehose stream when it is triggered.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Bucket bucket = new Bucket(this, "MyBucket");
 * DeliveryStream stream = DeliveryStream.Builder.create(this, "MyStream")
 *         .destinations(List.of(new S3Bucket(bucket)))
 *         .build();
 * 
 * TopicRule topicRule = TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT * FROM 'device/+/data'"))
 *         .actions(List.of(
 *             FirehosePutRecordAction.Builder.create(stream)
 *                     .batchMode(true)
 *                     .recordSeparator(FirehoseRecordSeparator.NEWLINE)
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Send messages to an SQS queue</h2>
 * <p>
 * The code snippet below creates an AWS IoT Rule that send messages
 * to an SQS queue when it is triggered:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Queue queue = new Queue(this, "MyQueue");
 * 
 * TopicRule topicRule = TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id, year, month, day FROM 'device/+/data'"))
 *         .actions(List.of(
 *             SqsQueueAction.Builder.create(queue)
 *                     .useBase64(true)
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Publish messages on an SNS topic</h2>
 * <p>
 * The code snippet below creates and AWS IoT Rule that publishes messages to an SNS topic when it is triggered:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * Topic topic = new Topic(this, "MyTopic");
 * 
 * TopicRule topicRule = TopicRule.Builder.create(this, "TopicRule")
 *         .sql(IotSql.fromStringAsVer20160323("SELECT topic(2) as device_id, year, month, day FROM 'device/+/data'"))
 *         .actions(List.of(
 *             SnsTopicAction.Builder.create(topic)
 *                     .messageFormat(SnsActionMessageFormat.JSON)
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.iot.actions;
