/**
 * <h1>Amazon Neptune Construct Library</h1>
 * <p>
 * Amazon Neptune is a fast, reliable, fully managed graph database service that makes it easy to build and run applications that work with highly connected datasets. The core of Neptune is a purpose-built, high-performance graph database engine. This engine is optimized for storing billions of relationships and querying the graph with milliseconds latency. Neptune supports the popular graph query languages Apache TinkerPop Gremlin and W3C’s SPARQL, enabling you to build queries that efficiently navigate highly connected datasets.
 * <p>
 * The <code>&#64;aws-cdk/aws-neptune</code> package contains primitives for setting up Neptune database clusters and instances.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * </pre></blockquote>
 * <p>
 * <h2>Starting a Neptune Database</h2>
 * <p>
 * To set up a Neptune database, define a <code>DatabaseCluster</code>. You must always launch a database in a VPC.
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.R5_LARGE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default only writer instance is provisioned with this construct.
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster, use the <code>.connections</code> attribute. Neptune databases have a default port, so
 * you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * cluster.connections.allowDefaultPortFromAnyIpv4("Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoints to access your database cluster will be available as the <code>.clusterEndpoint</code> and <code>.clusterReadEndpoint</code>
 * attributes:
 * <p>
 * <blockquote><pre>
 * String writeAddress = cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * <h2>IAM Authentication</h2>
 * <p>
 * You can also authenticate to a database cluster using AWS Identity and Access Management (IAM) database authentication;
 * See <a href="https://docs.aws.amazon.com/neptune/latest/userguide/iam-auth.html">https://docs.aws.amazon.com/neptune/latest/userguide/iam-auth.html</a> for more information and a list of supported
 * versions and limitations.
 * <p>
 * The following example shows enabling IAM authentication for a database cluster and granting connection access to an IAM role.
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.R5_LARGE)
 *         .iamAuthentication(true)
 *         .build();
 * Role role = Role.Builder.create(this, "DBRole").assumedBy(new AccountPrincipal(this.account)).build();
 * cluster.grantConnect(role);
 * </pre></blockquote>
 * <p>
 * <h2>Customizing parameters</h2>
 * <p>
 * Neptune allows configuring database behavior by supplying custom parameter groups.  For more details, refer to the
 * following link: <a href="https://docs.aws.amazon.com/neptune/latest/userguide/parameters.html">https://docs.aws.amazon.com/neptune/latest/userguide/parameters.html</a>
 * <p>
 * <blockquote><pre>
 * ClusterParameterGroup clusterParams = ClusterParameterGroup.Builder.create(this, "ClusterParams")
 *         .description("Cluster parameter group")
 *         .parameters(Map.of(
 *                 "neptune_enable_audit_log", "1"))
 *         .build();
 * 
 * ParameterGroup dbParams = ParameterGroup.Builder.create(this, "DbParams")
 *         .description("Db parameter group")
 *         .parameters(Map.of(
 *                 "neptune_query_timeout", "120000"))
 *         .build();
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.R5_LARGE)
 *         .clusterParameterGroup(clusterParams)
 *         .parameterGroup(dbParams)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding replicas</h2>
 * <p>
 * <code>DatabaseCluster</code> allows launching replicas along with the writer instance. This can be specified using the <code>instanceCount</code>
 * attribute.
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.R5_LARGE)
 *         .instances(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Additionally it is also possible to add replicas using <code>DatabaseInstance</code> for an existing cluster.
 * <p>
 * <blockquote><pre>
 * DatabaseInstance replica1 = DatabaseInstance.Builder.create(this, "Instance")
 *         .cluster(cluster)
 *         .instanceType(InstanceType.R5_LARGE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Automatic minor version upgrades</h2>
 * <p>
 * By setting <code>autoMinorVersionUpgrade</code> to true, Neptune will automatically update
 * the engine of the entire cluster to the latest minor version after a stabilization
 * window of 2 to 3 weeks.
 * <p>
 * <blockquote><pre>
 * DatabaseCluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.R5_LARGE)
 *         .autoMinorVersionUpgrade(true)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.neptune;
