/**
 * <h1>Amazon CloudWatch Synthetics Construct Library</h1>
 * <p>
 * Amazon CloudWatch Synthetics allow you to monitor your application by generating <strong>synthetic</strong> traffic. The traffic is produced by a <strong>canary</strong>: a configurable script that runs on a schedule. You configure the canary script to follow the same routes and perform the same actions as a user, which allows you to continually verify your user experience even when you don't have any traffic on your applications.
 * <p>
 * <h2>Canary</h2>
 * <p>
 * To illustrate how to use a canary, assume your application defines the following endpoint:
 * <p>
 * <blockquote><pre>
 * % curl "https://api.example.com/user/books/topbook/"
 * The Hitchhikers Guide to the Galaxy
 * 
 * </pre></blockquote>
 * <p>
 * The below code defines a canary that will hit the <code>books/topbook</code> endpoint every 5 minutes:
 * <p>
 * <blockquote><pre>
 * Canary canary = Canary.Builder.create(this, "MyCanary")
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_1)
 *         .environmentVariables(Map.of(
 *                 "stage", "prod"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following is an example of an <code>index.js</code> file which exports the <code>handler</code> function:
 * <p>
 * <blockquote><pre>
 * const synthetics = require('Synthetics');
 * const log = require('SyntheticsLogger');
 * 
 * const pageLoadBlueprint = async function () {
 *   // Configure the stage of the API using environment variables
 *   const url = `https://api.example.com/${process.env.stage}/user/books/topbook/`;
 * 
 *   const page = await synthetics.getPage();
 *   const response = await page.goto(url, { waitUntil: 'domcontentloaded', timeout: 30000 });
 *   // Wait for page to render. Increase or decrease wait time based on endpoint being monitored.
 *   await page.waitFor(15000);
 *   // This will take a screenshot that will be included in test output artifacts.
 *   await synthetics.takeScreenshot('loaded', 'loaded');
 *   const pageTitle = await page.title();
 *   log.info('Page title: ' + pageTitle);
 *   if (response.status() !== 200) {
 *     throw 'Failed to load page!';
 *   }
 * };
 * 
 * exports.handler = async () =&gt; {
 *   return await pageLoadBlueprint();
 * };
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> The function <strong>must</strong> be called <code>handler</code>.
 * <p>
 * </blockquote>
 * <p>
 * The canary will automatically produce a CloudWatch Dashboard:
 * <p>
 * <img alt="UI Screenshot" src="images/ui-screenshot.png">
 * <p>
 * The Canary code will be executed in a lambda function created by Synthetics on your behalf. The Lambda function includes a custom <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html">runtime</a> provided by Synthetics. The provided runtime includes a variety of handy tools such as <a href="https://www.npmjs.com/package/puppeteer-core">Puppeteer</a> (for nodejs based one) and Chromium.
 * <p>
 * To learn more about Synthetics capabilities, check out the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries.html">docs</a>.
 * <p>
 * <h3>Canary Schedule</h3>
 * <p>
 * You can specify the schedule on which a canary runs by providing a
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-synthetics.Schedule.html"><code>Schedule</code></a>
 * object to the <code>schedule</code> property.
 * <p>
 * Configure a run rate of up to 60 minutes with <code>Schedule.rate</code>:
 * <p>
 * <blockquote><pre>
 * Schedule schedule = Schedule.rate(Duration.minutes(5));
 * </pre></blockquote>
 * <p>
 * You can also specify a <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_cron.html">cron expression</a> with <code>Schedule.cron</code>:
 * <p>
 * <blockquote><pre>
 * Schedule schedule = Schedule.cron(CronOptions.builder()
 *         .hour("0,8,16")
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you want the canary to run just once upon deployment, you can use <code>Schedule.once()</code>.
 * <p>
 * <h3>Configuring the Canary Script</h3>
 * <p>
 * To configure the script the canary executes, use the <code>test</code> property. The <code>test</code> property accepts a <code>Test</code> instance that can be initialized by the <code>Test</code> class static methods. Currently, the only implemented method is <code>Test.custom()</code>, which allows you to bring your own code. In the future, other methods will be added. <code>Test.custom()</code> accepts <code>code</code> and <code>handler</code> properties -- both are required by Synthetics to create a lambda function on your behalf.
 * <p>
 * The <code>synthetics.Code</code> class exposes static methods to bundle your code artifacts:
 * <p>
 * <ul>
 * <li><code>code.fromInline(code)</code> - specify an inline script.</li>
 * <li><code>code.fromAsset(path)</code> - specify a .zip file or a directory in the local filesystem which will be zipped and uploaded to S3 on deployment. See the above Note for directory structure.</li>
 * <li><code>code.fromBucket(bucket, key[, objectVersion])</code> - specify an S3 object that contains the .zip file of your runtime code. See the above Note for directory structure.</li>
 * </ul>
 * <p>
 * Using the <code>Code</code> class static initializers:
 * <p>
 * <blockquote><pre>
 * // To supply the code from a S3 bucket:
 * import software.amazon.awscdk.core.*;
 * // To supply the code inline:
 * // To supply the code inline:
 * Canary.Builder.create(this, "Inline Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromInline("/* Synthetics handler code *&#47;"))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_4)
 *         .build();
 * 
 * // To supply the code from your local filesystem:
 * // To supply the code from your local filesystem:
 * Canary.Builder.create(this, "Asset Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_4)
 *         .build();
 * Bucket bucket = new Bucket(this, "Code Bucket");
 * Canary.Builder.create(this, "Bucket Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromBucket(bucket, "canary.zip"))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_4)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> Synthetics have a specified folder structure for canaries. For Node scripts supplied via <code>code.fromAsset()</code> or <code>code.fromBucket()</code>, the canary resource requires the following folder structure:
 * <p>
 * <blockquote><pre>
 * canary/
 * ├── nodejs/
 *    ├── node_modules/
 *         ├── &lt;filename&gt;.js
 * </pre></blockquote>
 * <p>
 * For Python scripts supplied via <code>code.fromAsset()</code> or <code>code.fromBucket()</code>, the canary resource requires the following folder structure:
 * <p>
 * <blockquote><pre>
 * canary/
 * ├── python/
 *     ├── &lt;filename&gt;.py
 * </pre></blockquote>
 * <p>
 * See Synthetics <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html">docs</a>.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Running a canary on a VPC</h3>
 * <p>
 * You can specify what <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_VPC.html">VPC a canary executes in</a>.
 * This can allow for monitoring services that may be internal to a specific VPC. To place a canary within a VPC, you can specify the <code>vpc</code> property with the desired <code>VPC</code> to place then canary in.
 * This will automatically attach the appropriate IAM permissions to attach to the VPC. This will also create a Security Group and attach to the default subnets for the VPC unless specified via <code>vpcSubnets</code> and <code>securityGroups</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * IVpc vpc;
 * 
 * Canary.Builder.create(this, "Vpc Canary")
 *         .test(Test.custom(CustomTestOptions.builder()
 *                 .code(Code.fromAsset(join(__dirname, "canary")))
 *                 .handler("index.handler")
 *                 .build()))
 *         .runtime(Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_4)
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> By default, the Synthetics runtime needs access to the S3 and CloudWatch APIs, which will fail in a private subnet without internet access enabled (e.g. an isolated subnnet).
 * <p>
 * Ensure that the Canary is placed in a VPC either with internet connectivity or with VPC Endpoints for S3 and CloudWatch enabled and configured.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_VPC.html">Synthetics VPC docs</a>.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Alarms</h3>
 * <p>
 * You can configure a CloudWatch Alarm on a canary metric. Metrics are emitted by CloudWatch automatically and can be accessed by the following APIs:
 * <p>
 * <ul>
 * <li><code>canary.metricSuccessPercent()</code> - percentage of successful canary runs over a given time</li>
 * <li><code>canary.metricDuration()</code> - how much time each canary run takes, in seconds.</li>
 * <li><code>canary.metricFailed()</code> - number of failed canary runs over a given time</li>
 * </ul>
 * <p>
 * Create an alarm that tracks the canary metric:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * Canary canary;
 * 
 * Alarm.Builder.create(this, "CanaryAlarm")
 *         .metric(canary.metricSuccessPercent())
 *         .evaluationPeriods(2)
 *         .threshold(90)
 *         .comparisonOperator(ComparisonOperator.LESS_THAN_THRESHOLD)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.synthetics;
