/**
 * <h1>Amazon Redshift Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Experimental" src="https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * The APIs of higher level constructs in this module are experimental and under active development.
 * They are subject to non-backward compatible changes or removal in any future version. These are
 * not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be
 * announced in the release notes. This means that while you may use them, you may need to update
 * your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <h2>Starting a Redshift Cluster Database</h2>
 * <p>
 * To set up a Redshift cluster, define a <code>Cluster</code>. It will be launched in a VPC.
 * You can specify a VPC, otherwise one will be created. The nodes are always launched in private subnets and are encrypted by default.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * 
 * Vpc vpc = new Vpc(this, "Vpc");
 * Cluster cluster = Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager.
 * <p>
 * A default database named <code>default_db</code> will be created in the cluster. To change the name of this database set the <code>defaultDatabaseName</code> attribute in the constructor properties.
 * <p>
 * By default, the cluster will not be publicly accessible.
 * Depending on your use case, you can make the cluster publicly accessible with the <code>publiclyAccessible</code> property.
 * <p>
 * <h2>Adding a logging bucket for database audit logging to S3</h2>
 * <p>
 * Amazon Redshift logs information about connections and user activities in your database. These logs help you to monitor the database for security and troubleshooting purposes, a process called database auditing. To send these logs to an S3 bucket, specify the <code>loggingProperties</code> when creating a new cluster.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.s3.*;
 * 
 * 
 * Vpc vpc = new Vpc(this, "Vpc");
 * IBucket bucket = Bucket.fromBucketName(this, "bucket", "logging-bucket");
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .loggingProperties(LoggingProperties.builder()
 *                 .loggingBucket(bucket)
 *                 .loggingKeyPrefix("prefix")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster, use the <code>.connections</code> attribute. Redshift Clusters have
 * a default port, so you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * cluster.connections.allowDefaultPortFromAnyIpv4("Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoint to access your database cluster will be available as the <code>.clusterEndpoint</code> attribute:
 * <p>
 * <blockquote><pre>
 * cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * <h2>Database Resources</h2>
 * <p>
 * This module allows for the creation of non-CloudFormation database resources such as users
 * and tables. This allows you to manage identities, permissions, and stateful resources
 * within your Redshift cluster from your CDK application.
 * <p>
 * Because these resources are not available in CloudFormation, this library leverages
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html">custom
 * resources</a>
 * to manage them. In addition to the IAM permissions required to make Redshift service
 * calls, the execution role for the custom resource handler requires database credentials to
 * create resources within the cluster.
 * <p>
 * These database credentials can be supplied explicitly through the <code>adminUser</code> properties
 * of the various database resource constructs. Alternatively, the credentials can be
 * automatically pulled from the Redshift cluster's default administrator
 * credentials. However, this option is only available if the password for the credentials
 * was generated by the CDK application (ie., no value vas provided for <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-redshift.Login.html#masterpasswordspan-classapi-icon-api-icon-experimental-titlethis-api-element-is-experimental-it-may-change-without-noticespan">the <code>masterPassword</code>
 * property</a>
 * of
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-redshift.Cluster.html#masteruserspan-classapi-icon-api-icon-experimental-titlethis-api-element-is-experimental-it-may-change-without-noticespan"><code>Cluster.masterUser</code></a>).
 * <p>
 * <h3>Creating Users</h3>
 * <p>
 * Create a user within a Redshift cluster database by instantiating a <code>User</code> construct. This
 * will generate a username and password, store the credentials in a <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-secretsmanager.Secret.html">AWS Secrets Manager
 * <code>Secret</code></a>,
 * and make a query to the Redshift cluster to create a new database user with the
 * credentials.
 * <p>
 * <blockquote><pre>
 * User.Builder.create(this, "User")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the user credentials are encrypted with your AWS account's default Secrets
 * Manager encryption key. You can specify the encryption key used for this purpose by
 * supplying a key in the <code>encryptionKey</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Key encryptionKey = new Key(this, "Key");
 * User.Builder.create(this, "User")
 *         .encryptionKey(encryptionKey)
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, a username is automatically generated from the user construct ID and its path
 * in the construct tree. You can specify a particular username by providing a value for the
 * <code>username</code> property. Usernames must be valid identifiers; see: <a href="https://docs.aws.amazon.com/redshift/latest/dg/r_names.html">Names and
 * identifiers</a> in the <em>Amazon
 * Redshift Database Developer Guide</em>.
 * <p>
 * <blockquote><pre>
 * User.Builder.create(this, "User")
 *         .username("myuser")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The user password is generated by AWS Secrets Manager using the default configuration
 * found in
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-secretsmanager.SecretStringGenerator.html"><code>secretsmanager.SecretStringGenerator</code></a>,
 * except with password length <code>30</code> and some SQL-incompliant characters excluded. The
 * plaintext for the password will never be present in the CDK application; instead, a
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">CloudFormation Dynamic
 * Reference</a>
 * will be used wherever the password value is required.
 * <p>
 * <h3>Creating Tables</h3>
 * <p>
 * Create a table within a Redshift cluster database by instantiating a <code>Table</code>
 * construct. This will make a query to the Redshift cluster to create a new database table
 * with the supplied schema.
 * <p>
 * <blockquote><pre>
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Tables greater than v2.114.1 can have their table name changed, for versions &lt;= v2.114.1, this would not be possible.
 * Therefore, changing of table names for &lt;= v2.114.1 have been disabled.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Table.Builder.create(this, "Table")
 *         .tableName("oldTableName") // This value can be change for versions greater than v2.114.1
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The table can be configured to have distStyle attribute and a distKey column:
 * <p>
 * <blockquote><pre>
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").distKey(true).build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .distStyle(TableDistStyle.KEY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The table can also be configured to have sortStyle attribute and sortKey columns:
 * <p>
 * <blockquote><pre>
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").sortKey(true).build(), Column.builder().name("col2").dataType("float").sortKey(true).build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .sortStyle(TableSortStyle.COMPOUND)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Tables and their respective columns can be configured to contain comments:
 * <p>
 * <blockquote><pre>
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").comment("This is a column comment").build(), Column.builder().name("col2").dataType("float").comment("This is a another column comment").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .tableComment("This is a table comment")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Table columns can be configured to use a specific compression encoding:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.redshift.alpha.ColumnEncoding;
 * 
 * 
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").encoding(ColumnEncoding.TEXT32K).build(), Column.builder().name("col2").dataType("float").encoding(ColumnEncoding.DELTA32K).build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Table columns can also contain an <code>id</code> attribute, which can allow table columns to be renamed.
 * <p>
 * <strong>NOTE</strong> To use the <code>id</code> attribute, you must also enable the <code>&#64;aws-cdk/aws-redshift:columnId</code> feature flag.
 * <p>
 * <blockquote><pre>
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().id("col1").name("col1").dataType("varchar(4)").build(), Column.builder().id("col2").name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Granting Privileges</h3>
 * <p>
 * You can give a user privileges to perform certain actions on a table by using the
 * <code>Table.grant()</code> method.
 * <p>
 * <blockquote><pre>
 * User user = User.Builder.create(this, "User")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * Table table = Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * 
 * table.grant(user, TableAction.DROP, TableAction.SELECT);
 * </pre></blockquote>
 * <p>
 * Take care when managing privileges via the CDK, as attempting to manage a user's
 * privileges on the same table in multiple CDK applications could lead to accidentally
 * overriding these permissions. Consider the following two CDK applications which both refer
 * to the same user and table. In application 1, the resources are created and the user is
 * given <code>INSERT</code> permissions on the table:
 * <p>
 * <blockquote><pre>
 * String databaseName = "databaseName";
 * String username = "myuser";
 * String tableName = "mytable";
 * 
 * User user = User.Builder.create(this, "User")
 *         .username(username)
 *         .cluster(cluster)
 *         .databaseName(databaseName)
 *         .build();
 * Table table = Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName(databaseName)
 *         .build();
 * table.grant(user, TableAction.INSERT);
 * </pre></blockquote>
 * <p>
 * In application 2, the resources are imported and the user is given <code>INSERT</code> permissions on
 * the table:
 * <p>
 * <blockquote><pre>
 * String databaseName = "databaseName";
 * String username = "myuser";
 * String tableName = "mytable";
 * 
 * IUser user = User.fromUserAttributes(this, "User", UserAttributes.builder()
 *         .username(username)
 *         .password(SecretValue.unsafePlainText("NOT_FOR_PRODUCTION"))
 *         .cluster(cluster)
 *         .databaseName(databaseName)
 *         .build());
 * ITable table = Table.fromTableAttributes(this, "Table", TableAttributes.builder()
 *         .tableName(tableName)
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build());
 * table.grant(user, TableAction.INSERT);
 * </pre></blockquote>
 * <p>
 * Both applications attempt to grant the user the appropriate privilege on the table by
 * submitting a <code>GRANT USER</code> SQL query to the Redshift cluster. Note that the latter of these
 * two calls will have no effect since the user has already been granted the privilege.
 * <p>
 * Now, if application 1 were to remove the call to <code>grant</code>, a <code>REVOKE USER</code> SQL query is
 * submitted to the Redshift cluster. In general, application 1 does not know that
 * application 2 has also granted this permission and thus cannot decide not to issue the
 * revocation. This leads to the undesirable state where application 2 still contains the
 * call to <code>grant</code> but the user does not have the specified permission.
 * <p>
 * Note that this does not occur when duplicate privileges are granted within the same
 * application, as such privileges are de-duplicated before any SQL query is submitted.
 * <p>
 * <h2>Rotating credentials</h2>
 * <p>
 * When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:
 * <p>
 * <blockquote><pre>
 * cluster.addRotationSingleUser();
 * </pre></blockquote>
 * <p>
 * The multi user rotation scheme is also available:
 * <p>
 * <blockquote><pre>
 * User user = User.Builder.create(this, "User")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * cluster.addRotationMultiUser("MultiUserRotation", RotationMultiUserOptions.builder()
 *         .secret(user.getSecret())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Adding Parameters</h2>
 * <p>
 * You can add a parameter to a parameter group with<code>ClusterParameterGroup.addParameter()</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.redshift.alpha.ClusterParameterGroup;
 * 
 * 
 * ClusterParameterGroup params = ClusterParameterGroup.Builder.create(this, "Params")
 *         .description("desc")
 *         .parameters(Map.of(
 *                 "require_ssl", "true"))
 *         .build();
 * 
 * params.addParameter("enable_user_activity_logging", "true");
 * </pre></blockquote>
 * <p>
 * Additionally, you can add a parameter to the cluster's associated parameter group with <code>Cluster.addToParameterGroup()</code>. If the cluster does not have an associated parameter group, a new parameter group is created.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.*;
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .masterPassword(SecretValue.unsafePlainText("tooshort"))
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * 
 * cluster.addToParameterGroup("enable_user_activity_logging", "true");
 * </pre></blockquote>
 * <p>
 * <h2>Rebooting for Parameter Updates</h2>
 * <p>
 * In most cases, existing clusters <a href="https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html">must be manually rebooted</a> to apply parameter changes. You can automate parameter related reboots by setting the cluster's <code>rebootForParameterChanges</code> property to <code>true</code> , or by using <code>Cluster.enableRebootForParameterChanges()</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.*;
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .masterPassword(SecretValue.unsafePlainText("tooshort"))
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * 
 * cluster.addToParameterGroup("enable_user_activity_logging", "true");
 * cluster.enableRebootForParameterChanges();
 * </pre></blockquote>
 * <p>
 * <h2>Elastic IP</h2>
 * <p>
 * If you configure your cluster to be publicly accessible, you can optionally select an <em>elastic IP address</em> to use for the external IP address. An elastic IP address is a static IP address that is associated with your AWS account. You can use an elastic IP address to connect to your cluster from outside the VPC. An elastic IP address gives you the ability to change your underlying configuration without affecting the IP address that clients use to connect to your cluster. This approach can be helpful for situations such as recovery after a failure.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.*;
 * Vpc vpc;
 * 
 * 
 * Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .masterPassword(SecretValue.unsafePlainText("tooshort"))
 *                 .build())
 *         .vpc(vpc)
 *         .publiclyAccessible(true)
 *         .elasticIp("10.123.123.255")
 *         .build();
 * </pre></blockquote>
 * <p>
 * If the Cluster is in a VPC and you want to connect to it using the private IP address from within the cluster, it is important to enable <em>DNS resolution</em> and <em>DNS hostnames</em> in the VPC config. If these parameters would not be set, connections from within the VPC would connect to the elastic IP address and not the private IP address.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * Vpc vpc = Vpc.Builder.create(this, "VPC")
 *         .enableDnsSupport(true)
 *         .enableDnsHostnames(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that if there is already an existing, public accessible Cluster, which VPC configuration is changed to use <em>DNS hostnames</em> and <em>DNS resolution</em>, connections still use the elastic IP address until the cluster is resized.
 * <p>
 * <h3>Elastic IP vs. Cluster node public IP</h3>
 * <p>
 * The elastic IP address is an external IP address for accessing the cluster outside of a VPC. It's not related to the cluster node public IP addresses and private IP addresses that are accessible via the <code>clusterEndpoint</code> property. The public and private cluster node IP addresses appear regardless of whether the cluster is publicly accessible or not. They are used only in certain circumstances to configure ingress rules on the remote host. These circumstances occur when you load data from an Amazon EC2 instance or other remote host using a Secure Shell (SSH) connection.
 * <p>
 * <h3>Attach Elastic IP after Cluster creation</h3>
 * <p>
 * In some cases, you might want to associate the cluster with an elastic IP address or change an elastic IP address that is associated with the cluster. To attach an elastic IP address after the cluster is created, first update the cluster so that it is not publicly accessible, then make it both publicly accessible and add an Elastic IP address in the same operation.
 * <p>
 * <h2>Enhanced VPC Routing</h2>
 * <p>
 * When you use Amazon Redshift enhanced VPC routing, Amazon Redshift forces all COPY and UNLOAD traffic between your cluster and your data repositories through your virtual private cloud (VPC) based on the Amazon VPC service. By using enhanced VPC routing, you can use standard VPC features, such as VPC security groups, network access control lists (ACLs), VPC endpoints, VPC endpoint policies, internet gateways, and Domain Name System (DNS) servers, as described in the Amazon VPC User Guide. You use these features to tightly manage the flow of data between your Amazon Redshift cluster and other resources. When you use enhanced VPC routing to route traffic through your VPC, you can also use VPC flow logs to monitor COPY and UNLOAD traffic.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.*;
 * Vpc vpc;
 * 
 * 
 * Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .masterPassword(SecretValue.unsafePlainText("tooshort"))
 *                 .build())
 *         .vpc(vpc)
 *         .enhancedVpcRouting(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If enhanced VPC routing is not enabled, Amazon Redshift routes traffic through the internet, including traffic to other services within the AWS network.
 * <p>
 * <h2>Default IAM role</h2>
 * <p>
 * Some Amazon Redshift features require Amazon Redshift to access other AWS services on your behalf. For your Amazon Redshift clusters to act on your behalf, you supply security credentials to your clusters. The preferred method to supply security credentials is to specify an AWS Identity and Access Management (IAM) role.
 * <p>
 * When you create an IAM role and set it as the default for the cluster using console, you don't have to provide the IAM role's Amazon Resource Name (ARN) to perform authentication and authorization.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.iam.*;
 * Vpc vpc;
 * 
 * 
 * Role defaultRole = Role.Builder.create(this, "DefaultRole")
 *         .assumedBy(new ServicePrincipal("redshift.amazonaws.com"))
 *         .build();
 * 
 * Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .roles(List.of(defaultRole))
 *         .defaultRole(defaultRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * A default role can also be added to a cluster using the <code>addDefaultIamRole</code> method.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.iam.*;
 * Vpc vpc;
 * 
 * 
 * Role defaultRole = Role.Builder.create(this, "DefaultRole")
 *         .assumedBy(new ServicePrincipal("redshift.amazonaws.com"))
 *         .build();
 * 
 * Cluster redshiftCluster = Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .roles(List.of(defaultRole))
 *         .build();
 * 
 * redshiftCluster.addDefaultIamRole(defaultRole);
 * </pre></blockquote>
 * <p>
 * <h2>IAM roles</h2>
 * <p>
 * Attaching IAM roles to a Redshift Cluster grants permissions to the Redshift service to perform actions on your behalf.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.iam.*;
 * Vpc vpc;
 * 
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("redshift.amazonaws.com"))
 *         .build();
 * Cluster cluster = Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .roles(List.of(role))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Additional IAM roles can be attached to a cluster using the <code>addIamRole</code> method.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.iam.*;
 * Vpc vpc;
 * 
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("redshift.amazonaws.com"))
 *         .build();
 * Cluster cluster = Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * cluster.addIamRole(role);
 * </pre></blockquote>
 * <p>
 * <h2>Resizing</h2>
 * <p>
 * As your data warehousing needs change, it's possible to resize your Redshift cluster. If the cluster was deployed via CDK,
 * it's important to resize it via CDK so the change is registered in the AWS CloudFormation template.
 * There are two types of resize operations:
 * <p>
 * <ul>
 * <li>Elastic resize - Number of nodes and node type can be changed, but not at the same time. Elastic resize is the default behavior,
 * as it's a fast operation and typically completes in minutes. Elastic resize is only supported on clusters of the following types:
 * <p>
 * <ul>
 * <li>dc1.large (if your cluster is in a VPC)</li>
 * <li>dc1.8xlarge (if your cluster is in a VPC)</li>
 * <li>dc2.large</li>
 * <li>dc2.8xlarge</li>
 * <li>ds2.xlarge</li>
 * <li>ds2.8xlarge</li>
 * <li>ra3.xlplus</li>
 * <li>ra3.4xlarge</li>
 * <li>ra3.16xlarge</li>
 * </ul></li>
 * <li>Classic resize - Number of nodes, node type, or both, can be changed. This operation takes longer to complete,
 * but is useful when the resize operation doesn't meet the criteria of an elastic resize. If you prefer classic resizing,
 * you can set the <code>classicResizing</code> flag when creating the cluster.</li>
 * </ul>
 * <p>
 * There are other constraints to be aware of, for example, elastic resizing does not support single-node clusters and there are
 * limits on the number of nodes you can add to a cluster. See the <a href="https://docs.aws.amazon.com/redshift/latest/mgmt/managing-cluster-operations.html#rs-resize-tutorial">AWS Redshift Documentation</a> and <a href="https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResizeCluster.html">AWS API Documentation</a> for more details.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.redshift.alpha;
