/**
 * <h2>Amazon S3 Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge" alt="Stability: Experimental" /></p>
 * <blockquote>
 * <p>This API is still under active development and subject to non-backward
 * compatible changes or removal in any future version. Use of the API is not recommended in production
 * environments. Experimental APIs are not subject to the Semantic Versioning model.</p>
 * </blockquote>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Define an unencrypted S3 bucket.</p>
 * <pre><code class="language-ts">new Bucket(this, 'MyFirstBucket');
 * </code></pre>
 * <p><code>Bucket</code> constructs expose the following deploy-time attributes:</p>
 * <ul>
 * <li><code>bucketArn</code> - the ARN of the bucket (i.e. <code>arn:aws:s3:::bucket_name</code>)</li>
 * <li><code>bucketName</code> - the name of the bucket (i.e. <code>bucket_name</code>)</li>
 * <li><code>bucketUrl</code> - the URL of the bucket (i.e.
 * <code>https://s3.us-west-1.amazonaws.com/onlybucket</code>)</li>
 * <li><code>arnForObjects(...pattern)</code> - the ARN of an object or objects within the
 * bucket (i.e.
 * <code>arn:aws:s3:::my_corporate_bucket/exampleobject.png</code> or
 * <code>arn:aws:s3:::my_corporate_bucket/Development/*</code>)</li>
 * <li><code>urlForObject(key)</code> - the URL of an object within the bucket (i.e.
 * <code>https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey</code>)</li>
 * </ul>
 * <h3>Encryption</h3>
 * <p>Define a KMS-encrypted bucket:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyUnencryptedBucket', {
 *     encryption: BucketEncryption.Kms
 * });
 * 
 * // you can access the encryption key:
 * assert(bucket.encryptionKey instanceof kms.Key);
 * </code></pre>
 * <p>You can also supply your own key:</p>
 * <pre><code class="language-ts">const myKmsKey = new kms.EncryptionKey(this, 'MyKey');
 * 
 * const bucket = new Bucket(this, 'MyEncryptedBucket', {
 *     encryption: BucketEncryption.Kms,
 *     encryptionKey: myKmsKey
 * });
 * 
 * assert(bucket.encryptionKey === myKmsKey);
 * </code></pre>
 * <p>Use <code>BucketEncryption.ManagedKms</code> to use the S3 master KMS key:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'Buck', {
 *     encryption: BucketEncryption.ManagedKms
 * });
 * 
 * assert(bucket.encryptionKey == null);
 * </code></pre>
 * <h3>Permissions</h3>
 * <p>A bucket policy will be automatically created for the bucket upon the first call to
 * <code>addToResourcePolicy(statement)</code>:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBucket');
 * bucket.addToResourcePolicy(new iam.PolicyStatement()
 *   .addActions('s3:GetObject')
 *   .addResources(bucket.arnForObjects('file.txt'))
 *   .addAccountRootPrincipal());
 * </code></pre>
 * <p>Most of the time, you won't have to manipulate the bucket policy directly.
 * Instead, buckets have &quot;grant&quot; methods called to give prepackaged sets of permissions
 * to other resources. For example:</p>
 * <pre><code class="language-ts">const lambda = new lambda.Function(this, 'Lambda', { /* ... *{@literal /} });
 * 
 * const bucket = new Bucket(this, 'MyBucket');
 * bucket.grantReadWrite(lambda.role);
 * </code></pre>
 * <p>Will give the Lambda's execution role permissions to read and write
 * from the bucket.</p>
 * <h3>Sharing buckets between stacks</h3>
 * <p>To use a bucket in a different stack in the same CDK application, pass the object to the other stack:</p>
 * <pre><code class="language-ts">
 * /**
 *  * Stack that defines the bucket
 *  *{@literal /}
 * class Producer extends cdk.Stack {
 *     public readonly myBucket: s3.Bucket;
 * 
 *     constructor(scope: cdk.App, id: string, props?: cdk.StackProps) {
 *         super(scope, id, props);
 * 
 *         const bucket = new s3.Bucket(this, 'MyBucket', {
 *           removalPolicy: cdk.RemovalPolicy.Destroy
 *         });
 *         this.myBucket = bucket;
 *     }
 * }
 * 
 * interface ConsumerProps extends cdk.StackProps {
 *     userBucket: s3.IBucket;
 * }
 * 
 * /**
 *  * Stack that consumes the bucket
 *  *{@literal /}
 * class Consumer extends cdk.Stack {
 *     constructor(scope: cdk.App, id: string, props: ConsumerProps) {
 *         super(scope, id, props);
 * 
 *         const user = new iam.User(this, 'MyUser');
 *         props.userBucket.grantReadWrite(user);
 *     }
 * }
 * 
 * const producer = new Producer(app, 'ProducerStack');
 * new Consumer(app, 'ConsumerStack', { userBucket: producer.myBucket });
 * </code></pre>
 * <h3>Importing existing buckets</h3>
 * <p>To import an existing bucket into your CDK application, use the <code>Bucket.import</code> factory method.  This method accepts a
 * <code>BucketImportProps</code> which describes the properties of the already existing bucket:</p>
 * <pre><code class="language-ts">const bucket = Bucket.import(this, {
 *     bucketArn: 'arn:aws:s3:::my-bucket'
 * });
 * 
 * // now you can just call methods on the bucket
 * bucket.grantReadWrite(user);
 * </code></pre>
 * <h3>Bucket Notifications</h3>
 * <p>The Amazon S3 notification feature enables you to receive notifications when
 * certain events happen in your bucket as described under <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html">S3 Bucket
 * Notifications</a> of the S3 Developer Guide.</p>
 * <p>To subscribe for bucket notifications, use the <code>bucket.onEvent</code> method. The
 * <code>bucket.onObjectCreated</code> and <code>bucket.onObjectRemoved</code> can also be used for these
 * common use cases.</p>
 * <p>The following example will subscribe an SNS topic to be notified of all
 * ``s3:ObjectCreated:*` events:</p>
 * <pre><code class="language-ts">const myTopic = new sns.Topic(this, 'MyTopic');
 * bucket.onEvent(s3.EventType.ObjectCreated, myTopic);
 * </code></pre>
 * <p>This call will also ensure that the topic policy can accept notifications for
 * this specific bucket.</p>
 * <p>The following destinations are currently supported:</p>
 * <ul>
 * <li><code>sns.Topic</code></li>
 * <li><code>sqs.Queue</code></li>
 * <li><code>lambda.Function</code></li>
 * </ul>
 * <p>It is also possible to specify S3 object key filters when subscribing. The
 * following example will notify <code>myQueue</code> when objects prefixed with <code>foo/</code> and
 * have the <code>.jpg</code> suffix are removed from the bucket.</p>
 * <pre><code class="language-ts">bucket.onEvent(s3.EventType.ObjectRemoved, myQueue, { prefix: 'foo/', suffix: '.jpg' });
 * </code></pre>
 * <h3>Block Public Access</h3>
 * <p>Use <code>blockPublicAccess</code> to specify <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html">block public access settings</a> on the bucket.</p>
 * <p>Enable all block public access settings:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBlockedBucket', {
 *     blockPublicAccess: BlockPublicAccess.BlockAll
 * });
 * </code></pre>
 * <p>Block and ignore public ACLs:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBlockedBucket', {
 *     blockPublicAccess: BlockPublicAccess.BlockAcls
 * });
 * </code></pre>
 * <p>Alternatively, specify the settings manually:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBlockedBucket', {
 *     blockPublicAccess: new BlockPublicAccess({ blockPublicPolicy: true })
 * });
 * </code></pre>
 * <p>When <code>blockPublicPolicy</code> is set to <code>true</code>, <code>grantPublicRead()</code> throws an error.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.s3;
