/**
 * <h2>Amazon S3 Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Define an unencrypted S3 bucket.</p>
 * <pre><code class="language-ts">new Bucket(this, 'MyFirstBucket');
 * </code></pre>
 * <p><code>Bucket</code> constructs expose the following deploy-time attributes:</p>
 * <ul>
 * <li><code>bucketArn</code> - the ARN of the bucket (i.e. <code>arn:aws:s3:::bucket_name</code>)</li>
 * <li><code>bucketName</code> - the name of the bucket (i.e. <code>bucket_name</code>)</li>
 * <li><code>bucketWebsiteUrl</code> - the Website URL of the bucket (i.e.
 * <code>http://bucket_name.s3-website-us-west-1.amazonaws.com</code>)</li>
 * <li><code>bucketDomainName</code> - the URL of the bucket (i.e. <code>bucket_name.s3.amazonaws.com</code>)</li>
 * <li><code>bucketDualStackDomainName</code> - the dual-stack URL of the bucket (i.e.
 * <code>bucket_name.s3.dualstack.eu-west-1.amazonaws.com</code>)</li>
 * <li><code>bucketRegionalDomainName</code> - the regional URL of the bucket (i.e.
 * <code>bucket_name.s3.eu-west-1.amazonaws.com</code>)</li>
 * <li><code>arnForObjects(pattern)</code> - the ARN of an object or objects within the bucket (i.e.
 * <code>arn:aws:s3:::bucket_name/exampleobject.png</code> or
 * <code>arn:aws:s3:::bucket_name/Development/*</code>)</li>
 * <li><code>urlForObject(key)</code> - the URL of an object within the bucket (i.e.
 * <code>https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey</code>)</li>
 * </ul>
 * <h3>Encryption</h3>
 * <p>Define a KMS-encrypted bucket:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyUnencryptedBucket', {
 *     encryption: BucketEncryption.KMS
 * });
 * 
 * // you can access the encryption key:
 * assert(bucket.encryptionKey instanceof kms.Key);
 * </code></pre>
 * <p>You can also supply your own key:</p>
 * <pre><code class="language-ts">const myKmsKey = new kms.Key(this, 'MyKey');
 * 
 * const bucket = new Bucket(this, 'MyEncryptedBucket', {
 *     encryption: BucketEncryption.KMS,
 *     encryptionKey: myKmsKey
 * });
 * 
 * assert(bucket.encryptionKey === myKmsKey);
 * </code></pre>
 * <p>Use <code>BucketEncryption.ManagedKms</code> to use the S3 master KMS key:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'Buck', {
 *     encryption: BucketEncryption.KMS_MANAGED
 * });
 * 
 * assert(bucket.encryptionKey == null);
 * </code></pre>
 * <h3>Permissions</h3>
 * <p>A bucket policy will be automatically created for the bucket upon the first call to
 * <code>addToResourcePolicy(statement)</code>:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBucket');
 * bucket.addToResourcePolicy(new iam.PolicyStatement({
 *   actions: ['s3:GetObject'],
 *   resources: [bucket.arnForObjects('file.txt')],
 *   principals: [new iam.AccountRootPrincipal()],
 * }));
 * </code></pre>
 * <p>Most of the time, you won't have to manipulate the bucket policy directly.
 * Instead, buckets have &quot;grant&quot; methods called to give prepackaged sets of permissions
 * to other resources. For example:</p>
 * <pre><code class="language-ts">const lambda = new lambda.Function(this, 'Lambda', { /* ... *{@literal /} });
 * 
 * const bucket = new Bucket(this, 'MyBucket');
 * bucket.grantReadWrite(lambda);
 * </code></pre>
 * <p>Will give the Lambda's execution role permissions to read and write
 * from the bucket.</p>
 * <h3>Sharing buckets between stacks</h3>
 * <p>To use a bucket in a different stack in the same CDK application, pass the object to the other stack:</p>
 * <pre><code class="language-ts">
 * /**
 *  * Stack that defines the bucket
 *  *{@literal /}
 * class Producer extends cdk.Stack {
 *     public readonly myBucket: s3.Bucket;
 * 
 *     constructor(scope: cdk.App, id: string, props?: cdk.StackProps) {
 *         super(scope, id, props);
 * 
 *         const bucket = new s3.Bucket(this, 'MyBucket', {
 *           removalPolicy: cdk.RemovalPolicy.DESTROY
 *         });
 *         this.myBucket = bucket;
 *     }
 * }
 * 
 * interface ConsumerProps extends cdk.StackProps {
 *     userBucket: s3.IBucket;
 * }
 * 
 * /**
 *  * Stack that consumes the bucket
 *  *{@literal /}
 * class Consumer extends cdk.Stack {
 *     constructor(scope: cdk.App, id: string, props: ConsumerProps) {
 *         super(scope, id, props);
 * 
 *         const user = new iam.User(this, 'MyUser');
 *         props.userBucket.grantReadWrite(user);
 *     }
 * }
 * 
 * const producer = new Producer(app, 'ProducerStack');
 * new Consumer(app, 'ConsumerStack', { userBucket: producer.myBucket });
 * </code></pre>
 * <h3>Importing existing buckets</h3>
 * <p>To import an existing bucket into your CDK application, use the <code>Bucket.fromBucketAttributes</code>
 * factory method. This method accepts <code>BucketAttributes</code> which describes the properties of an already
 * existing bucket:</p>
 * <pre><code class="language-ts">const bucket = Bucket.fromBucketAttributes(this, 'ImportedBucket', {
 *     bucketArn: 'arn:aws:s3:::my-bucket'
 * });
 * 
 * // now you can just call methods on the bucket
 * bucket.grantReadWrite(user);
 * </code></pre>
 * <p>Alternatively, short-hand factories are available as <code>Bucket.fromBucketName</code> and
 * <code>Bucket.fromBucketArn</code>, which will derive all bucket attributes from the bucket
 * name or ARN respectively:</p>
 * <pre><code class="language-ts">const byName = Bucket.fromBucketName(this, 'BucketByName', 'my-bucket');
 * const byArn  = Bucket.fromBucketArn(this, 'BucketByArn', 'arn:aws:s3:::my-bucket');
 * </code></pre>
 * <h3>Bucket Notifications</h3>
 * <p>The Amazon S3 notification feature enables you to receive notifications when
 * certain events happen in your bucket as described under <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html">S3 Bucket
 * Notifications</a> of the S3 Developer Guide.</p>
 * <p>To subscribe for bucket notifications, use the <code>bucket.addEventNotification</code> method. The
 * <code>bucket.addObjectCreatedNotification</code> and <code>bucket.addObjectRemovedNotification</code> can also be used for
 * these common use cases.</p>
 * <p>The following example will subscribe an SNS topic to be notified of all <code>s3:ObjectCreated:*</code> events:</p>
 * <pre><code class="language-ts">import s3n = require('@aws-cdk/aws-s3-notifications');
 * 
 * const myTopic = new sns.Topic(this, 'MyTopic');
 * bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.SnsDestination(topic));
 * </code></pre>
 * <p>This call will also ensure that the topic policy can accept notifications for
 * this specific bucket.</p>
 * <p>Supported S3 notification targets are exposed by the <code>@aws-cdk/aws-s3-notifications</code> package.</p>
 * <p>It is also possible to specify S3 object key filters when subscribing. The
 * following example will notify <code>myQueue</code> when objects prefixed with <code>foo/</code> and
 * have the <code>.jpg</code> suffix are removed from the bucket.</p>
 * <pre><code class="language-ts">bucket.addEventNotification(s3.EventType.OBJECT_REMOVED,
 *   new s3n.SqsDestination(myQueue),
 *   { prefix: 'foo/', suffix: '.jpg' });
 * </code></pre>
 * <h3>Block Public Access</h3>
 * <p>Use <code>blockPublicAccess</code> to specify <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html">block public access settings</a> on the bucket.</p>
 * <p>Enable all block public access settings:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBlockedBucket', {
 *     blockPublicAccess: BlockPublicAccess.BLOCK_ALL
 * });
 * </code></pre>
 * <p>Block and ignore public ACLs:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBlockedBucket', {
 *     blockPublicAccess: BlockPublicAccess.BLOCK_ACLS
 * });
 * </code></pre>
 * <p>Alternatively, specify the settings manually:</p>
 * <pre><code class="language-ts">const bucket = new Bucket(this, 'MyBlockedBucket', {
 *     blockPublicAccess: new BlockPublicAccess({ blockPublicPolicy: true })
 * });
 * </code></pre>
 * <p>When <code>blockPublicPolicy</code> is set to <code>true</code>, <code>grantPublicRead()</code> throws an error.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.s3;
