package software.amazon.awscdk.services.sagemaker;

/**
 * A CloudFormation `AWS::SageMaker::Domain`.
 * <p>
 * Creates a <code>Domain</code> used by Amazon SageMaker Studio. A domain consists of an associated Amazon Elastic File System (EFS) volume, a list of authorized users, and a variety of security, application, policy, and Amazon Virtual Private Cloud (VPC) configurations. Users within a domain can share notebook files and other artifacts with each other.
 * <p>
 * <em>EFS storage</em>
 * <p>
 * When a domain is created, an EFS volume is created for use by all of the users within the domain. Each user receives a private home directory within the EFS volume for notebooks, Git repositories, and data files.
 * <p>
 * SageMaker uses the AWS Key Management Service ( AWS KMS) to encrypt the EFS volume attached to the domain with an AWS managed key by default. For more control, you can specify a customer managed key. For more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/encryption-at-rest.html">Protect Data at Rest Using Encryption</a> .
 * <p>
 * <em>VPC configuration</em>
 * <p>
 * All SageMaker Studio traffic between the domain and the EFS volume is through the specified VPC and subnets. For other Studio traffic, you can specify the <code>AppNetworkAccessType</code> parameter. <code>AppNetworkAccessType</code> corresponds to the network access type that you choose when you onboard to Studio. The following options are available:
 * <p>
 * <ul>
 * <li><code>PublicInternetOnly</code> - Non-EFS traffic goes through a VPC managed by Amazon SageMaker, which allows internet access. This is the default value.</li>
 * <li><code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets. Internet access is disabled by default. To allow internet access, you must specify a NAT gateway.</li>
 * </ul>
 * <p>
 * When internet access is disabled, you won't be able to run a Studio notebook or to train or host models unless your VPC has an interface endpoint to the SageMaker API and runtime or a NAT gateway and your security groups allow outbound connections.
 * <p>
 * <blockquote>
 * <p>
 * NFS traffic over TCP on port 2049 needs to be allowed in both inbound and outbound rules in order to launch a SageMaker Studio app successfully.
 * <p>
 * </blockquote>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/studio-notebooks-and-internet-access.html">Connect SageMaker Studio Notebooks to Resources in a VPC</a> .
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // The code below shows an example of how to instantiate this type.
 * // The values are placeholders you should change.
 * import software.amazon.awscdk.services.sagemaker.*;
 * CfnDomain cfnDomain = CfnDomain.Builder.create(this, "MyCfnDomain")
 *         .authMode("authMode")
 *         .defaultUserSettings(UserSettingsProperty.builder()
 *                 .executionRole("executionRole")
 *                 .jupyterServerAppSettings(JupyterServerAppSettingsProperty.builder()
 *                         .defaultResourceSpec(ResourceSpecProperty.builder()
 *                                 .instanceType("instanceType")
 *                                 .lifecycleConfigArn("lifecycleConfigArn")
 *                                 .sageMakerImageArn("sageMakerImageArn")
 *                                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
 *                                 .build())
 *                         .build())
 *                 .kernelGatewayAppSettings(KernelGatewayAppSettingsProperty.builder()
 *                         .customImages(List.of(CustomImageProperty.builder()
 *                                 .appImageConfigName("appImageConfigName")
 *                                 .imageName("imageName")
 *                                 // the properties below are optional
 *                                 .imageVersionNumber(123)
 *                                 .build()))
 *                         .defaultResourceSpec(ResourceSpecProperty.builder()
 *                                 .instanceType("instanceType")
 *                                 .lifecycleConfigArn("lifecycleConfigArn")
 *                                 .sageMakerImageArn("sageMakerImageArn")
 *                                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
 *                                 .build())
 *                         .build())
 *                 .rSessionAppSettings(RSessionAppSettingsProperty.builder()
 *                         .customImages(List.of(CustomImageProperty.builder()
 *                                 .appImageConfigName("appImageConfigName")
 *                                 .imageName("imageName")
 *                                 // the properties below are optional
 *                                 .imageVersionNumber(123)
 *                                 .build()))
 *                         .defaultResourceSpec(ResourceSpecProperty.builder()
 *                                 .instanceType("instanceType")
 *                                 .lifecycleConfigArn("lifecycleConfigArn")
 *                                 .sageMakerImageArn("sageMakerImageArn")
 *                                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
 *                                 .build())
 *                         .build())
 *                 .rStudioServerProAppSettings(RStudioServerProAppSettingsProperty.builder()
 *                         .accessStatus("accessStatus")
 *                         .userGroup("userGroup")
 *                         .build())
 *                 .securityGroups(List.of("securityGroups"))
 *                 .sharingSettings(SharingSettingsProperty.builder()
 *                         .notebookOutputOption("notebookOutputOption")
 *                         .s3KmsKeyId("s3KmsKeyId")
 *                         .s3OutputPath("s3OutputPath")
 *                         .build())
 *                 .build())
 *         .domainName("domainName")
 *         .subnetIds(List.of("subnetIds"))
 *         .vpcId("vpcId")
 *         // the properties below are optional
 *         .appNetworkAccessType("appNetworkAccessType")
 *         .appSecurityGroupManagement("appSecurityGroupManagement")
 *         .defaultSpaceSettings(DefaultSpaceSettingsProperty.builder()
 *                 .executionRole("executionRole")
 *                 .jupyterServerAppSettings(JupyterServerAppSettingsProperty.builder()
 *                         .defaultResourceSpec(ResourceSpecProperty.builder()
 *                                 .instanceType("instanceType")
 *                                 .lifecycleConfigArn("lifecycleConfigArn")
 *                                 .sageMakerImageArn("sageMakerImageArn")
 *                                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
 *                                 .build())
 *                         .build())
 *                 .kernelGatewayAppSettings(KernelGatewayAppSettingsProperty.builder()
 *                         .customImages(List.of(CustomImageProperty.builder()
 *                                 .appImageConfigName("appImageConfigName")
 *                                 .imageName("imageName")
 *                                 // the properties below are optional
 *                                 .imageVersionNumber(123)
 *                                 .build()))
 *                         .defaultResourceSpec(ResourceSpecProperty.builder()
 *                                 .instanceType("instanceType")
 *                                 .lifecycleConfigArn("lifecycleConfigArn")
 *                                 .sageMakerImageArn("sageMakerImageArn")
 *                                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
 *                                 .build())
 *                         .build())
 *                 .securityGroups(List.of("securityGroups"))
 *                 .build())
 *         .domainSettings(DomainSettingsProperty.builder()
 *                 .rStudioServerProDomainSettings(RStudioServerProDomainSettingsProperty.builder()
 *                         .domainExecutionRoleArn("domainExecutionRoleArn")
 *                         // the properties below are optional
 *                         .defaultResourceSpec(ResourceSpecProperty.builder()
 *                                 .instanceType("instanceType")
 *                                 .lifecycleConfigArn("lifecycleConfigArn")
 *                                 .sageMakerImageArn("sageMakerImageArn")
 *                                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
 *                                 .build())
 *                         .rStudioConnectUrl("rStudioConnectUrl")
 *                         .rStudioPackageManagerUrl("rStudioPackageManagerUrl")
 *                         .build())
 *                 .securityGroupIds(List.of("securityGroupIds"))
 *                 .build())
 *         .kmsKeyId("kmsKeyId")
 *         .tags(List.of(CfnTag.builder()
 *                 .key("key")
 *                 .value("value")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 */
@javax.annotation.Generated(value = "jsii-pacmak/1.74.0 (build 6d08790)", date = "2023-02-21T19:24:02.084Z")
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
@software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain")
public class CfnDomain extends software.amazon.awscdk.core.CfnResource implements software.amazon.awscdk.core.IInspectable {

    protected CfnDomain(final software.amazon.jsii.JsiiObjectRef objRef) {
        super(objRef);
    }

    protected CfnDomain(final software.amazon.jsii.JsiiObject.InitializationMode initializationMode) {
        super(initializationMode);
    }

    static {
        CFN_RESOURCE_TYPE_NAME = software.amazon.jsii.JsiiObject.jsiiStaticGet(software.amazon.awscdk.services.sagemaker.CfnDomain.class, "CFN_RESOURCE_TYPE_NAME", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * Create a new `AWS::SageMaker::Domain`.
     * <p>
     * @param scope - scope in which this resource is defined. This parameter is required.
     * @param id - scoped id of the resource. This parameter is required.
     * @param props - resource properties. This parameter is required.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public CfnDomain(final @org.jetbrains.annotations.NotNull software.amazon.awscdk.core.Construct scope, final @org.jetbrains.annotations.NotNull java.lang.String id, final @org.jetbrains.annotations.NotNull software.amazon.awscdk.services.sagemaker.CfnDomainProps props) {
        super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
        software.amazon.jsii.JsiiEngine.getInstance().createNewObject(this, new Object[] { java.util.Objects.requireNonNull(scope, "scope is required"), java.util.Objects.requireNonNull(id, "id is required"), java.util.Objects.requireNonNull(props, "props is required") });
    }

    /**
     * Examines the CloudFormation resource and discloses attributes.
     * <p>
     * @param inspector - tree inspector to collect and process attributes. This parameter is required.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    @Override
    public void inspect(final @org.jetbrains.annotations.NotNull software.amazon.awscdk.core.TreeInspector inspector) {
        software.amazon.jsii.Kernel.call(this, "inspect", software.amazon.jsii.NativeType.VOID, new Object[] { java.util.Objects.requireNonNull(inspector, "inspector is required") });
    }

    /**
     * @param props This parameter is required.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    @Override
    protected @org.jetbrains.annotations.NotNull java.util.Map<java.lang.String, java.lang.Object> renderProperties(final @org.jetbrains.annotations.NotNull java.util.Map<java.lang.String, java.lang.Object> props) {
        return java.util.Collections.unmodifiableMap(software.amazon.jsii.Kernel.call(this, "renderProperties", software.amazon.jsii.NativeType.mapOf(software.amazon.jsii.NativeType.forClass(java.lang.Object.class)), new Object[] { java.util.Objects.requireNonNull(props, "props is required") }));
    }

    /**
     * The CloudFormation resource type name for this resource class.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public final static java.lang.String CFN_RESOURCE_TYPE_NAME;

    /**
     * The Amazon Resource Name (ARN) of the Domain, such as `arn:aws:sagemaker:us-west-2:account-id:domain/my-domain-name` .
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAttrDomainArn() {
        return software.amazon.jsii.Kernel.get(this, "attrDomainArn", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The Domain ID.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAttrDomainId() {
        return software.amazon.jsii.Kernel.get(this, "attrDomainId", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The ID of the Amazon Elastic File System (EFS) managed by this Domain.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAttrHomeEfsFileSystemId() {
        return software.amazon.jsii.Kernel.get(this, "attrHomeEfsFileSystemId", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The ID of the security group that authorizes traffic between the `RSessionGateway` apps and the `RStudioServerPro` app.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAttrSecurityGroupIdForDomainBoundary() {
        return software.amazon.jsii.Kernel.get(this, "attrSecurityGroupIdForDomainBoundary", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The IAM Identity Center managed application instance ID.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAttrSingleSignOnManagedApplicationInstanceId() {
        return software.amazon.jsii.Kernel.get(this, "attrSingleSignOnManagedApplicationInstanceId", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The URL for the Domain.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAttrUrl() {
        return software.amazon.jsii.Kernel.get(this, "attrUrl", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     */
    @Override
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    protected @org.jetbrains.annotations.NotNull java.util.Map<java.lang.String, java.lang.Object> getCfnProperties() {
        return java.util.Collections.unmodifiableMap(software.amazon.jsii.Kernel.get(this, "cfnProperties", software.amazon.jsii.NativeType.mapOf(software.amazon.jsii.NativeType.forClass(java.lang.Object.class))));
    }

    /**
     * Tags to associated with the Domain.
     * <p>
     * Each tag consists of a key and an optional value. Tag keys must be unique per resource. Tags are searchable using the Search API.
     * <p>
     * Tags that you specify for the Domain are also added to all apps that are launched in the Domain.
     * <p>
     * <em>Array members</em> : Minimum number of 0 items. Maximum number of 50 items.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull software.amazon.awscdk.core.TagManager getTags() {
        return software.amazon.jsii.Kernel.get(this, "tags", software.amazon.jsii.NativeType.forClass(software.amazon.awscdk.core.TagManager.class));
    }

    /**
     * The mode of authentication that members use to access the Domain.
     * <p>
     * <em>Valid Values</em> : <code>SSO | IAM</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getAuthMode() {
        return software.amazon.jsii.Kernel.get(this, "authMode", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The mode of authentication that members use to access the Domain.
     * <p>
     * <em>Valid Values</em> : <code>SSO | IAM</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setAuthMode(final @org.jetbrains.annotations.NotNull java.lang.String value) {
        software.amazon.jsii.Kernel.set(this, "authMode", java.util.Objects.requireNonNull(value, "authMode is required"));
    }

    /**
     * The default user settings.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.Object getDefaultUserSettings() {
        return software.amazon.jsii.Kernel.get(this, "defaultUserSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
    }

    /**
     * The default user settings.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDefaultUserSettings(final @org.jetbrains.annotations.NotNull software.amazon.awscdk.core.IResolvable value) {
        software.amazon.jsii.Kernel.set(this, "defaultUserSettings", java.util.Objects.requireNonNull(value, "defaultUserSettings is required"));
    }

    /**
     * The default user settings.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDefaultUserSettings(final @org.jetbrains.annotations.NotNull software.amazon.awscdk.services.sagemaker.CfnDomain.UserSettingsProperty value) {
        software.amazon.jsii.Kernel.set(this, "defaultUserSettings", java.util.Objects.requireNonNull(value, "defaultUserSettings is required"));
    }

    /**
     * The domain name.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getDomainName() {
        return software.amazon.jsii.Kernel.get(this, "domainName", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The domain name.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDomainName(final @org.jetbrains.annotations.NotNull java.lang.String value) {
        software.amazon.jsii.Kernel.set(this, "domainName", java.util.Objects.requireNonNull(value, "domainName is required"));
    }

    /**
     * The VPC subnets that Studio uses for communication.
     * <p>
     * <em>Length Constraints</em> : Maximum length of 32.
     * <p>
     * <em>Array members</em> : Minimum number of 1 item. Maximum number of 16 items.
     * <p>
     * <em>Pattern</em> : <code>[-0-9a-zA-Z]+</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.util.List<java.lang.String> getSubnetIds() {
        return java.util.Collections.unmodifiableList(software.amazon.jsii.Kernel.get(this, "subnetIds", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(java.lang.String.class))));
    }

    /**
     * The VPC subnets that Studio uses for communication.
     * <p>
     * <em>Length Constraints</em> : Maximum length of 32.
     * <p>
     * <em>Array members</em> : Minimum number of 1 item. Maximum number of 16 items.
     * <p>
     * <em>Pattern</em> : <code>[-0-9a-zA-Z]+</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setSubnetIds(final @org.jetbrains.annotations.NotNull java.util.List<java.lang.String> value) {
        software.amazon.jsii.Kernel.set(this, "subnetIds", java.util.Objects.requireNonNull(value, "subnetIds is required"));
    }

    /**
     * The ID of the Amazon Virtual Private Cloud (Amazon VPC) that Studio uses for communication.
     * <p>
     * <em>Length Constraints</em> : Maximum length of 32.
     * <p>
     * <em>Pattern</em> : <code>[-0-9a-zA-Z]+</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.NotNull java.lang.String getVpcId() {
        return software.amazon.jsii.Kernel.get(this, "vpcId", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The ID of the Amazon Virtual Private Cloud (Amazon VPC) that Studio uses for communication.
     * <p>
     * <em>Length Constraints</em> : Maximum length of 32.
     * <p>
     * <em>Pattern</em> : <code>[-0-9a-zA-Z]+</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setVpcId(final @org.jetbrains.annotations.NotNull java.lang.String value) {
        software.amazon.jsii.Kernel.set(this, "vpcId", java.util.Objects.requireNonNull(value, "vpcId is required"));
    }

    /**
     * Specifies the VPC used for non-EFS traffic. The default value is `PublicInternetOnly` .
     * <p>
     * <ul>
     * <li><code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker , which allows direct internet access</li>
     * <li><code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets</li>
     * </ul>
     * <p>
     * <em>Valid Values</em> : <code>PublicInternetOnly | VpcOnly</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.Nullable java.lang.String getAppNetworkAccessType() {
        return software.amazon.jsii.Kernel.get(this, "appNetworkAccessType", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * Specifies the VPC used for non-EFS traffic. The default value is `PublicInternetOnly` .
     * <p>
     * <ul>
     * <li><code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker , which allows direct internet access</li>
     * <li><code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets</li>
     * </ul>
     * <p>
     * <em>Valid Values</em> : <code>PublicInternetOnly | VpcOnly</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setAppNetworkAccessType(final @org.jetbrains.annotations.Nullable java.lang.String value) {
        software.amazon.jsii.Kernel.set(this, "appNetworkAccessType", value);
    }

    /**
     * The entity that creates and manages the required security groups for inter-app communication in `VpcOnly` mode.
     * <p>
     * Required when <code>CreateDomain.AppNetworkAccessType</code> is <code>VpcOnly</code> and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.Nullable java.lang.String getAppSecurityGroupManagement() {
        return software.amazon.jsii.Kernel.get(this, "appSecurityGroupManagement", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * The entity that creates and manages the required security groups for inter-app communication in `VpcOnly` mode.
     * <p>
     * Required when <code>CreateDomain.AppNetworkAccessType</code> is <code>VpcOnly</code> and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setAppSecurityGroupManagement(final @org.jetbrains.annotations.Nullable java.lang.String value) {
        software.amazon.jsii.Kernel.set(this, "appSecurityGroupManagement", value);
    }

    /**
     * `AWS::SageMaker::Domain.DefaultSpaceSettings`.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.Nullable java.lang.Object getDefaultSpaceSettings() {
        return software.amazon.jsii.Kernel.get(this, "defaultSpaceSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
    }

    /**
     * `AWS::SageMaker::Domain.DefaultSpaceSettings`.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDefaultSpaceSettings(final @org.jetbrains.annotations.Nullable software.amazon.awscdk.core.IResolvable value) {
        software.amazon.jsii.Kernel.set(this, "defaultSpaceSettings", value);
    }

    /**
     * `AWS::SageMaker::Domain.DefaultSpaceSettings`.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDefaultSpaceSettings(final @org.jetbrains.annotations.Nullable software.amazon.awscdk.services.sagemaker.CfnDomain.DefaultSpaceSettingsProperty value) {
        software.amazon.jsii.Kernel.set(this, "defaultSpaceSettings", value);
    }

    /**
     * A collection of settings that apply to the `SageMaker Domain` .
     * <p>
     * These settings are specified through the <code>CreateDomain</code> API call.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.Nullable java.lang.Object getDomainSettings() {
        return software.amazon.jsii.Kernel.get(this, "domainSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
    }

    /**
     * A collection of settings that apply to the `SageMaker Domain` .
     * <p>
     * These settings are specified through the <code>CreateDomain</code> API call.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDomainSettings(final @org.jetbrains.annotations.Nullable software.amazon.awscdk.core.IResolvable value) {
        software.amazon.jsii.Kernel.set(this, "domainSettings", value);
    }

    /**
     * A collection of settings that apply to the `SageMaker Domain` .
     * <p>
     * These settings are specified through the <code>CreateDomain</code> API call.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setDomainSettings(final @org.jetbrains.annotations.Nullable software.amazon.awscdk.services.sagemaker.CfnDomain.DomainSettingsProperty value) {
        software.amazon.jsii.Kernel.set(this, "domainSettings", value);
    }

    /**
     * SageMaker uses AWS KMS to encrypt the EFS volume attached to the Domain with an AWS managed customer master key (CMK) by default.
     * <p>
     * For more control, specify a customer managed CMK.
     * <p>
     * <em>Length Constraints</em> : Maximum length of 2048.
     * <p>
     * <em>Pattern</em> : <code>.*</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public @org.jetbrains.annotations.Nullable java.lang.String getKmsKeyId() {
        return software.amazon.jsii.Kernel.get(this, "kmsKeyId", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
    }

    /**
     * SageMaker uses AWS KMS to encrypt the EFS volume attached to the Domain with an AWS managed customer master key (CMK) by default.
     * <p>
     * For more control, specify a customer managed CMK.
     * <p>
     * <em>Length Constraints</em> : Maximum length of 2048.
     * <p>
     * <em>Pattern</em> : <code>.*</code>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public void setKmsKeyId(final @org.jetbrains.annotations.Nullable java.lang.String value) {
        software.amazon.jsii.Kernel.set(this, "kmsKeyId", value);
    }
    /**
     * A custom SageMaker image.
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/studio-byoi.html">Bring your own SageMaker image</a> .
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * CustomImageProperty customImageProperty = CustomImageProperty.builder()
     *         .appImageConfigName("appImageConfigName")
     *         .imageName("imageName")
     *         // the properties below are optional
     *         .imageVersionNumber(123)
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.CustomImageProperty")
    @software.amazon.jsii.Jsii.Proxy(CustomImageProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface CustomImageProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * The name of the AppImageConfig.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @org.jetbrains.annotations.NotNull java.lang.String getAppImageConfigName();

        /**
         * The name of the CustomImage.
         * <p>
         * Must be unique to your account.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @org.jetbrains.annotations.NotNull java.lang.String getImageName();

        /**
         * The version number of the CustomImage.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Number getImageVersionNumber() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link CustomImageProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link CustomImageProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<CustomImageProperty> {
            java.lang.String appImageConfigName;
            java.lang.String imageName;
            java.lang.Number imageVersionNumber;

            /**
             * Sets the value of {@link CustomImageProperty#getAppImageConfigName}
             * @param appImageConfigName The name of the AppImageConfig. This parameter is required.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder appImageConfigName(java.lang.String appImageConfigName) {
                this.appImageConfigName = appImageConfigName;
                return this;
            }

            /**
             * Sets the value of {@link CustomImageProperty#getImageName}
             * @param imageName The name of the CustomImage. This parameter is required.
             *                  Must be unique to your account.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder imageName(java.lang.String imageName) {
                this.imageName = imageName;
                return this;
            }

            /**
             * Sets the value of {@link CustomImageProperty#getImageVersionNumber}
             * @param imageVersionNumber The version number of the CustomImage.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder imageVersionNumber(java.lang.Number imageVersionNumber) {
                this.imageVersionNumber = imageVersionNumber;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link CustomImageProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public CustomImageProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link CustomImageProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements CustomImageProperty {
            private final java.lang.String appImageConfigName;
            private final java.lang.String imageName;
            private final java.lang.Number imageVersionNumber;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.appImageConfigName = software.amazon.jsii.Kernel.get(this, "appImageConfigName", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.imageName = software.amazon.jsii.Kernel.get(this, "imageName", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.imageVersionNumber = software.amazon.jsii.Kernel.get(this, "imageVersionNumber", software.amazon.jsii.NativeType.forClass(java.lang.Number.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.appImageConfigName = java.util.Objects.requireNonNull(builder.appImageConfigName, "appImageConfigName is required");
                this.imageName = java.util.Objects.requireNonNull(builder.imageName, "imageName is required");
                this.imageVersionNumber = builder.imageVersionNumber;
            }

            @Override
            public final java.lang.String getAppImageConfigName() {
                return this.appImageConfigName;
            }

            @Override
            public final java.lang.String getImageName() {
                return this.imageName;
            }

            @Override
            public final java.lang.Number getImageVersionNumber() {
                return this.imageVersionNumber;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                data.set("appImageConfigName", om.valueToTree(this.getAppImageConfigName()));
                data.set("imageName", om.valueToTree(this.getImageName()));
                if (this.getImageVersionNumber() != null) {
                    data.set("imageVersionNumber", om.valueToTree(this.getImageVersionNumber()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.CustomImageProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                CustomImageProperty.Jsii$Proxy that = (CustomImageProperty.Jsii$Proxy) o;

                if (!appImageConfigName.equals(that.appImageConfigName)) return false;
                if (!imageName.equals(that.imageName)) return false;
                return this.imageVersionNumber != null ? this.imageVersionNumber.equals(that.imageVersionNumber) : that.imageVersionNumber == null;
            }

            @Override
            public final int hashCode() {
                int result = this.appImageConfigName.hashCode();
                result = 31 * result + (this.imageName.hashCode());
                result = 31 * result + (this.imageVersionNumber != null ? this.imageVersionNumber.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * A collection of settings that apply to spaces created in the Domain.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * DefaultSpaceSettingsProperty defaultSpaceSettingsProperty = DefaultSpaceSettingsProperty.builder()
     *         .executionRole("executionRole")
     *         .jupyterServerAppSettings(JupyterServerAppSettingsProperty.builder()
     *                 .defaultResourceSpec(ResourceSpecProperty.builder()
     *                         .instanceType("instanceType")
     *                         .lifecycleConfigArn("lifecycleConfigArn")
     *                         .sageMakerImageArn("sageMakerImageArn")
     *                         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                         .build())
     *                 .build())
     *         .kernelGatewayAppSettings(KernelGatewayAppSettingsProperty.builder()
     *                 .customImages(List.of(CustomImageProperty.builder()
     *                         .appImageConfigName("appImageConfigName")
     *                         .imageName("imageName")
     *                         // the properties below are optional
     *                         .imageVersionNumber(123)
     *                         .build()))
     *                 .defaultResourceSpec(ResourceSpecProperty.builder()
     *                         .instanceType("instanceType")
     *                         .lifecycleConfigArn("lifecycleConfigArn")
     *                         .sageMakerImageArn("sageMakerImageArn")
     *                         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                         .build())
     *                 .build())
     *         .securityGroups(List.of("securityGroups"))
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.DefaultSpaceSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(DefaultSpaceSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface DefaultSpaceSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * The execution role for the space.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getExecutionRole() {
            return null;
        }

        /**
         * The JupyterServer app settings.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getJupyterServerAppSettings() {
            return null;
        }

        /**
         * The KernelGateway app settings.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getKernelGatewayAppSettings() {
            return null;
        }

        /**
         * The security groups for the Amazon Virtual Private Cloud that the space uses for communication.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.util.List<java.lang.String> getSecurityGroups() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link DefaultSpaceSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link DefaultSpaceSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<DefaultSpaceSettingsProperty> {
            java.lang.String executionRole;
            java.lang.Object jupyterServerAppSettings;
            java.lang.Object kernelGatewayAppSettings;
            java.util.List<java.lang.String> securityGroups;

            /**
             * Sets the value of {@link DefaultSpaceSettingsProperty#getExecutionRole}
             * @param executionRole The execution role for the space.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder executionRole(java.lang.String executionRole) {
                this.executionRole = executionRole;
                return this;
            }

            /**
             * Sets the value of {@link DefaultSpaceSettingsProperty#getJupyterServerAppSettings}
             * @param jupyterServerAppSettings The JupyterServer app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder jupyterServerAppSettings(software.amazon.awscdk.core.IResolvable jupyterServerAppSettings) {
                this.jupyterServerAppSettings = jupyterServerAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link DefaultSpaceSettingsProperty#getJupyterServerAppSettings}
             * @param jupyterServerAppSettings The JupyterServer app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder jupyterServerAppSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.JupyterServerAppSettingsProperty jupyterServerAppSettings) {
                this.jupyterServerAppSettings = jupyterServerAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link DefaultSpaceSettingsProperty#getKernelGatewayAppSettings}
             * @param kernelGatewayAppSettings The KernelGateway app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder kernelGatewayAppSettings(software.amazon.awscdk.core.IResolvable kernelGatewayAppSettings) {
                this.kernelGatewayAppSettings = kernelGatewayAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link DefaultSpaceSettingsProperty#getKernelGatewayAppSettings}
             * @param kernelGatewayAppSettings The KernelGateway app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder kernelGatewayAppSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.KernelGatewayAppSettingsProperty kernelGatewayAppSettings) {
                this.kernelGatewayAppSettings = kernelGatewayAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link DefaultSpaceSettingsProperty#getSecurityGroups}
             * @param securityGroups The security groups for the Amazon Virtual Private Cloud that the space uses for communication.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder securityGroups(java.util.List<java.lang.String> securityGroups) {
                this.securityGroups = securityGroups;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link DefaultSpaceSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public DefaultSpaceSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link DefaultSpaceSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements DefaultSpaceSettingsProperty {
            private final java.lang.String executionRole;
            private final java.lang.Object jupyterServerAppSettings;
            private final java.lang.Object kernelGatewayAppSettings;
            private final java.util.List<java.lang.String> securityGroups;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.executionRole = software.amazon.jsii.Kernel.get(this, "executionRole", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.jupyterServerAppSettings = software.amazon.jsii.Kernel.get(this, "jupyterServerAppSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.kernelGatewayAppSettings = software.amazon.jsii.Kernel.get(this, "kernelGatewayAppSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.securityGroups = software.amazon.jsii.Kernel.get(this, "securityGroups", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(java.lang.String.class)));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.executionRole = builder.executionRole;
                this.jupyterServerAppSettings = builder.jupyterServerAppSettings;
                this.kernelGatewayAppSettings = builder.kernelGatewayAppSettings;
                this.securityGroups = builder.securityGroups;
            }

            @Override
            public final java.lang.String getExecutionRole() {
                return this.executionRole;
            }

            @Override
            public final java.lang.Object getJupyterServerAppSettings() {
                return this.jupyterServerAppSettings;
            }

            @Override
            public final java.lang.Object getKernelGatewayAppSettings() {
                return this.kernelGatewayAppSettings;
            }

            @Override
            public final java.util.List<java.lang.String> getSecurityGroups() {
                return this.securityGroups;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getExecutionRole() != null) {
                    data.set("executionRole", om.valueToTree(this.getExecutionRole()));
                }
                if (this.getJupyterServerAppSettings() != null) {
                    data.set("jupyterServerAppSettings", om.valueToTree(this.getJupyterServerAppSettings()));
                }
                if (this.getKernelGatewayAppSettings() != null) {
                    data.set("kernelGatewayAppSettings", om.valueToTree(this.getKernelGatewayAppSettings()));
                }
                if (this.getSecurityGroups() != null) {
                    data.set("securityGroups", om.valueToTree(this.getSecurityGroups()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.DefaultSpaceSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                DefaultSpaceSettingsProperty.Jsii$Proxy that = (DefaultSpaceSettingsProperty.Jsii$Proxy) o;

                if (this.executionRole != null ? !this.executionRole.equals(that.executionRole) : that.executionRole != null) return false;
                if (this.jupyterServerAppSettings != null ? !this.jupyterServerAppSettings.equals(that.jupyterServerAppSettings) : that.jupyterServerAppSettings != null) return false;
                if (this.kernelGatewayAppSettings != null ? !this.kernelGatewayAppSettings.equals(that.kernelGatewayAppSettings) : that.kernelGatewayAppSettings != null) return false;
                return this.securityGroups != null ? this.securityGroups.equals(that.securityGroups) : that.securityGroups == null;
            }

            @Override
            public final int hashCode() {
                int result = this.executionRole != null ? this.executionRole.hashCode() : 0;
                result = 31 * result + (this.jupyterServerAppSettings != null ? this.jupyterServerAppSettings.hashCode() : 0);
                result = 31 * result + (this.kernelGatewayAppSettings != null ? this.kernelGatewayAppSettings.hashCode() : 0);
                result = 31 * result + (this.securityGroups != null ? this.securityGroups.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * A collection of settings that apply to the `SageMaker Domain` .
     * <p>
     * These settings are specified through the <code>CreateDomain</code> API call.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * DomainSettingsProperty domainSettingsProperty = DomainSettingsProperty.builder()
     *         .rStudioServerProDomainSettings(RStudioServerProDomainSettingsProperty.builder()
     *                 .domainExecutionRoleArn("domainExecutionRoleArn")
     *                 // the properties below are optional
     *                 .defaultResourceSpec(ResourceSpecProperty.builder()
     *                         .instanceType("instanceType")
     *                         .lifecycleConfigArn("lifecycleConfigArn")
     *                         .sageMakerImageArn("sageMakerImageArn")
     *                         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                         .build())
     *                 .rStudioConnectUrl("rStudioConnectUrl")
     *                 .rStudioPackageManagerUrl("rStudioPackageManagerUrl")
     *                 .build())
     *         .securityGroupIds(List.of("securityGroupIds"))
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.DomainSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(DomainSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface DomainSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * A collection of settings that configure the `RStudioServerPro` Domain-level app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getRStudioServerProDomainSettings() {
            return null;
        }

        /**
         * The security groups for the Amazon Virtual Private Cloud that the `Domain` uses for communication between Domain-level apps and user apps.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.util.List<java.lang.String> getSecurityGroupIds() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link DomainSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link DomainSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<DomainSettingsProperty> {
            java.lang.Object rStudioServerProDomainSettings;
            java.util.List<java.lang.String> securityGroupIds;

            /**
             * Sets the value of {@link DomainSettingsProperty#getRStudioServerProDomainSettings}
             * @param rStudioServerProDomainSettings A collection of settings that configure the `RStudioServerPro` Domain-level app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rStudioServerProDomainSettings(software.amazon.awscdk.core.IResolvable rStudioServerProDomainSettings) {
                this.rStudioServerProDomainSettings = rStudioServerProDomainSettings;
                return this;
            }

            /**
             * Sets the value of {@link DomainSettingsProperty#getRStudioServerProDomainSettings}
             * @param rStudioServerProDomainSettings A collection of settings that configure the `RStudioServerPro` Domain-level app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rStudioServerProDomainSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.RStudioServerProDomainSettingsProperty rStudioServerProDomainSettings) {
                this.rStudioServerProDomainSettings = rStudioServerProDomainSettings;
                return this;
            }

            /**
             * Sets the value of {@link DomainSettingsProperty#getSecurityGroupIds}
             * @param securityGroupIds The security groups for the Amazon Virtual Private Cloud that the `Domain` uses for communication between Domain-level apps and user apps.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder securityGroupIds(java.util.List<java.lang.String> securityGroupIds) {
                this.securityGroupIds = securityGroupIds;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link DomainSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public DomainSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link DomainSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements DomainSettingsProperty {
            private final java.lang.Object rStudioServerProDomainSettings;
            private final java.util.List<java.lang.String> securityGroupIds;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.rStudioServerProDomainSettings = software.amazon.jsii.Kernel.get(this, "rStudioServerProDomainSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.securityGroupIds = software.amazon.jsii.Kernel.get(this, "securityGroupIds", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(java.lang.String.class)));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.rStudioServerProDomainSettings = builder.rStudioServerProDomainSettings;
                this.securityGroupIds = builder.securityGroupIds;
            }

            @Override
            public final java.lang.Object getRStudioServerProDomainSettings() {
                return this.rStudioServerProDomainSettings;
            }

            @Override
            public final java.util.List<java.lang.String> getSecurityGroupIds() {
                return this.securityGroupIds;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getRStudioServerProDomainSettings() != null) {
                    data.set("rStudioServerProDomainSettings", om.valueToTree(this.getRStudioServerProDomainSettings()));
                }
                if (this.getSecurityGroupIds() != null) {
                    data.set("securityGroupIds", om.valueToTree(this.getSecurityGroupIds()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.DomainSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                DomainSettingsProperty.Jsii$Proxy that = (DomainSettingsProperty.Jsii$Proxy) o;

                if (this.rStudioServerProDomainSettings != null ? !this.rStudioServerProDomainSettings.equals(that.rStudioServerProDomainSettings) : that.rStudioServerProDomainSettings != null) return false;
                return this.securityGroupIds != null ? this.securityGroupIds.equals(that.securityGroupIds) : that.securityGroupIds == null;
            }

            @Override
            public final int hashCode() {
                int result = this.rStudioServerProDomainSettings != null ? this.rStudioServerProDomainSettings.hashCode() : 0;
                result = 31 * result + (this.securityGroupIds != null ? this.securityGroupIds.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * The JupyterServer app settings.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * JupyterServerAppSettingsProperty jupyterServerAppSettingsProperty = JupyterServerAppSettingsProperty.builder()
     *         .defaultResourceSpec(ResourceSpecProperty.builder()
     *                 .instanceType("instanceType")
     *                 .lifecycleConfigArn("lifecycleConfigArn")
     *                 .sageMakerImageArn("sageMakerImageArn")
     *                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                 .build())
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.JupyterServerAppSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(JupyterServerAppSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface JupyterServerAppSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * The default instance type and the Amazon Resource Name (ARN) of the default SageMaker image used by the JupyterServer app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getDefaultResourceSpec() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link JupyterServerAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link JupyterServerAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<JupyterServerAppSettingsProperty> {
            java.lang.Object defaultResourceSpec;

            /**
             * Sets the value of {@link JupyterServerAppSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec The default instance type and the Amazon Resource Name (ARN) of the default SageMaker image used by the JupyterServer app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.core.IResolvable defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Sets the value of {@link JupyterServerAppSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec The default instance type and the Amazon Resource Name (ARN) of the default SageMaker image used by the JupyterServer app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.services.sagemaker.CfnDomain.ResourceSpecProperty defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link JupyterServerAppSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public JupyterServerAppSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link JupyterServerAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements JupyterServerAppSettingsProperty {
            private final java.lang.Object defaultResourceSpec;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.defaultResourceSpec = software.amazon.jsii.Kernel.get(this, "defaultResourceSpec", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.defaultResourceSpec = builder.defaultResourceSpec;
            }

            @Override
            public final java.lang.Object getDefaultResourceSpec() {
                return this.defaultResourceSpec;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getDefaultResourceSpec() != null) {
                    data.set("defaultResourceSpec", om.valueToTree(this.getDefaultResourceSpec()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.JupyterServerAppSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                JupyterServerAppSettingsProperty.Jsii$Proxy that = (JupyterServerAppSettingsProperty.Jsii$Proxy) o;

                return this.defaultResourceSpec != null ? this.defaultResourceSpec.equals(that.defaultResourceSpec) : that.defaultResourceSpec == null;
            }

            @Override
            public final int hashCode() {
                int result = this.defaultResourceSpec != null ? this.defaultResourceSpec.hashCode() : 0;
                return result;
            }
        }
    }
    /**
     * The KernelGateway app settings.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * KernelGatewayAppSettingsProperty kernelGatewayAppSettingsProperty = KernelGatewayAppSettingsProperty.builder()
     *         .customImages(List.of(CustomImageProperty.builder()
     *                 .appImageConfigName("appImageConfigName")
     *                 .imageName("imageName")
     *                 // the properties below are optional
     *                 .imageVersionNumber(123)
     *                 .build()))
     *         .defaultResourceSpec(ResourceSpecProperty.builder()
     *                 .instanceType("instanceType")
     *                 .lifecycleConfigArn("lifecycleConfigArn")
     *                 .sageMakerImageArn("sageMakerImageArn")
     *                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                 .build())
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.KernelGatewayAppSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(KernelGatewayAppSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface KernelGatewayAppSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * A list of custom SageMaker images that are configured to run as a KernelGateway app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getCustomImages() {
            return null;
        }

        /**
         * The default instance type and the Amazon Resource Name (ARN) of the default SageMaker image used by the KernelGateway app.
         * <p>
         * <blockquote>
         * <p>
         * The Amazon SageMaker Studio UI does not use the default instance type value set here. The default instance type set here is used when Apps are created using the AWS Command Line Interface or AWS CloudFormation and the instance type parameter value is not passed.
         * <p>
         * </blockquote>
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getDefaultResourceSpec() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link KernelGatewayAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link KernelGatewayAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<KernelGatewayAppSettingsProperty> {
            java.lang.Object customImages;
            java.lang.Object defaultResourceSpec;

            /**
             * Sets the value of {@link KernelGatewayAppSettingsProperty#getCustomImages}
             * @param customImages A list of custom SageMaker images that are configured to run as a KernelGateway app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder customImages(software.amazon.awscdk.core.IResolvable customImages) {
                this.customImages = customImages;
                return this;
            }

            /**
             * Sets the value of {@link KernelGatewayAppSettingsProperty#getCustomImages}
             * @param customImages A list of custom SageMaker images that are configured to run as a KernelGateway app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder customImages(java.util.List<? extends java.lang.Object> customImages) {
                this.customImages = customImages;
                return this;
            }

            /**
             * Sets the value of {@link KernelGatewayAppSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec The default instance type and the Amazon Resource Name (ARN) of the default SageMaker image used by the KernelGateway app.
             *                            <blockquote>
             *                            <p>
             *                            The Amazon SageMaker Studio UI does not use the default instance type value set here. The default instance type set here is used when Apps are created using the AWS Command Line Interface or AWS CloudFormation and the instance type parameter value is not passed.
             *                            <p>
             *                            </blockquote>
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.core.IResolvable defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Sets the value of {@link KernelGatewayAppSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec The default instance type and the Amazon Resource Name (ARN) of the default SageMaker image used by the KernelGateway app.
             *                            <blockquote>
             *                            <p>
             *                            The Amazon SageMaker Studio UI does not use the default instance type value set here. The default instance type set here is used when Apps are created using the AWS Command Line Interface or AWS CloudFormation and the instance type parameter value is not passed.
             *                            <p>
             *                            </blockquote>
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.services.sagemaker.CfnDomain.ResourceSpecProperty defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link KernelGatewayAppSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public KernelGatewayAppSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link KernelGatewayAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements KernelGatewayAppSettingsProperty {
            private final java.lang.Object customImages;
            private final java.lang.Object defaultResourceSpec;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.customImages = software.amazon.jsii.Kernel.get(this, "customImages", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.defaultResourceSpec = software.amazon.jsii.Kernel.get(this, "defaultResourceSpec", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.customImages = builder.customImages;
                this.defaultResourceSpec = builder.defaultResourceSpec;
            }

            @Override
            public final java.lang.Object getCustomImages() {
                return this.customImages;
            }

            @Override
            public final java.lang.Object getDefaultResourceSpec() {
                return this.defaultResourceSpec;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getCustomImages() != null) {
                    data.set("customImages", om.valueToTree(this.getCustomImages()));
                }
                if (this.getDefaultResourceSpec() != null) {
                    data.set("defaultResourceSpec", om.valueToTree(this.getDefaultResourceSpec()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.KernelGatewayAppSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                KernelGatewayAppSettingsProperty.Jsii$Proxy that = (KernelGatewayAppSettingsProperty.Jsii$Proxy) o;

                if (this.customImages != null ? !this.customImages.equals(that.customImages) : that.customImages != null) return false;
                return this.defaultResourceSpec != null ? this.defaultResourceSpec.equals(that.defaultResourceSpec) : that.defaultResourceSpec == null;
            }

            @Override
            public final int hashCode() {
                int result = this.customImages != null ? this.customImages.hashCode() : 0;
                result = 31 * result + (this.defaultResourceSpec != null ? this.defaultResourceSpec.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * A collection of settings that apply to an `RSessionGateway` app.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * RSessionAppSettingsProperty rSessionAppSettingsProperty = RSessionAppSettingsProperty.builder()
     *         .customImages(List.of(CustomImageProperty.builder()
     *                 .appImageConfigName("appImageConfigName")
     *                 .imageName("imageName")
     *                 // the properties below are optional
     *                 .imageVersionNumber(123)
     *                 .build()))
     *         .defaultResourceSpec(ResourceSpecProperty.builder()
     *                 .instanceType("instanceType")
     *                 .lifecycleConfigArn("lifecycleConfigArn")
     *                 .sageMakerImageArn("sageMakerImageArn")
     *                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                 .build())
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.RSessionAppSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(RSessionAppSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface RSessionAppSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * A list of custom SageMaker images that are configured to run as a RSession app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getCustomImages() {
            return null;
        }

        /**
         * Specifies the ARNs of a SageMaker image and SageMaker image version, and the instance type that the version runs on.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getDefaultResourceSpec() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link RSessionAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link RSessionAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<RSessionAppSettingsProperty> {
            java.lang.Object customImages;
            java.lang.Object defaultResourceSpec;

            /**
             * Sets the value of {@link RSessionAppSettingsProperty#getCustomImages}
             * @param customImages A list of custom SageMaker images that are configured to run as a RSession app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder customImages(software.amazon.awscdk.core.IResolvable customImages) {
                this.customImages = customImages;
                return this;
            }

            /**
             * Sets the value of {@link RSessionAppSettingsProperty#getCustomImages}
             * @param customImages A list of custom SageMaker images that are configured to run as a RSession app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder customImages(java.util.List<? extends java.lang.Object> customImages) {
                this.customImages = customImages;
                return this;
            }

            /**
             * Sets the value of {@link RSessionAppSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec Specifies the ARNs of a SageMaker image and SageMaker image version, and the instance type that the version runs on.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.core.IResolvable defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Sets the value of {@link RSessionAppSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec Specifies the ARNs of a SageMaker image and SageMaker image version, and the instance type that the version runs on.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.services.sagemaker.CfnDomain.ResourceSpecProperty defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link RSessionAppSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public RSessionAppSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link RSessionAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements RSessionAppSettingsProperty {
            private final java.lang.Object customImages;
            private final java.lang.Object defaultResourceSpec;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.customImages = software.amazon.jsii.Kernel.get(this, "customImages", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.defaultResourceSpec = software.amazon.jsii.Kernel.get(this, "defaultResourceSpec", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.customImages = builder.customImages;
                this.defaultResourceSpec = builder.defaultResourceSpec;
            }

            @Override
            public final java.lang.Object getCustomImages() {
                return this.customImages;
            }

            @Override
            public final java.lang.Object getDefaultResourceSpec() {
                return this.defaultResourceSpec;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getCustomImages() != null) {
                    data.set("customImages", om.valueToTree(this.getCustomImages()));
                }
                if (this.getDefaultResourceSpec() != null) {
                    data.set("defaultResourceSpec", om.valueToTree(this.getDefaultResourceSpec()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.RSessionAppSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                RSessionAppSettingsProperty.Jsii$Proxy that = (RSessionAppSettingsProperty.Jsii$Proxy) o;

                if (this.customImages != null ? !this.customImages.equals(that.customImages) : that.customImages != null) return false;
                return this.defaultResourceSpec != null ? this.defaultResourceSpec.equals(that.defaultResourceSpec) : that.defaultResourceSpec == null;
            }

            @Override
            public final int hashCode() {
                int result = this.customImages != null ? this.customImages.hashCode() : 0;
                result = 31 * result + (this.defaultResourceSpec != null ? this.defaultResourceSpec.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * A collection of settings that configure user interaction with the `RStudioServerPro` app.
     * <p>
     * <code>RStudioServerProAppSettings</code> cannot be updated. The <code>RStudioServerPro</code> app must be deleted and a new one created to make any changes.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * RStudioServerProAppSettingsProperty rStudioServerProAppSettingsProperty = RStudioServerProAppSettingsProperty.builder()
     *         .accessStatus("accessStatus")
     *         .userGroup("userGroup")
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.RStudioServerProAppSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(RStudioServerProAppSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface RStudioServerProAppSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * Indicates whether the current user has access to the `RStudioServerPro` app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getAccessStatus() {
            return null;
        }

        /**
         * The level of permissions that the user has within the `RStudioServerPro` app.
         * <p>
         * This value defaults to <code>User</code>. The <code>Admin</code> value allows the user access to the RStudio Administrative Dashboard.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getUserGroup() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link RStudioServerProAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link RStudioServerProAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<RStudioServerProAppSettingsProperty> {
            java.lang.String accessStatus;
            java.lang.String userGroup;

            /**
             * Sets the value of {@link RStudioServerProAppSettingsProperty#getAccessStatus}
             * @param accessStatus Indicates whether the current user has access to the `RStudioServerPro` app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder accessStatus(java.lang.String accessStatus) {
                this.accessStatus = accessStatus;
                return this;
            }

            /**
             * Sets the value of {@link RStudioServerProAppSettingsProperty#getUserGroup}
             * @param userGroup The level of permissions that the user has within the `RStudioServerPro` app.
             *                  This value defaults to <code>User</code>. The <code>Admin</code> value allows the user access to the RStudio Administrative Dashboard.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder userGroup(java.lang.String userGroup) {
                this.userGroup = userGroup;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link RStudioServerProAppSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public RStudioServerProAppSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link RStudioServerProAppSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements RStudioServerProAppSettingsProperty {
            private final java.lang.String accessStatus;
            private final java.lang.String userGroup;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.accessStatus = software.amazon.jsii.Kernel.get(this, "accessStatus", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.userGroup = software.amazon.jsii.Kernel.get(this, "userGroup", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.accessStatus = builder.accessStatus;
                this.userGroup = builder.userGroup;
            }

            @Override
            public final java.lang.String getAccessStatus() {
                return this.accessStatus;
            }

            @Override
            public final java.lang.String getUserGroup() {
                return this.userGroup;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getAccessStatus() != null) {
                    data.set("accessStatus", om.valueToTree(this.getAccessStatus()));
                }
                if (this.getUserGroup() != null) {
                    data.set("userGroup", om.valueToTree(this.getUserGroup()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.RStudioServerProAppSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                RStudioServerProAppSettingsProperty.Jsii$Proxy that = (RStudioServerProAppSettingsProperty.Jsii$Proxy) o;

                if (this.accessStatus != null ? !this.accessStatus.equals(that.accessStatus) : that.accessStatus != null) return false;
                return this.userGroup != null ? this.userGroup.equals(that.userGroup) : that.userGroup == null;
            }

            @Override
            public final int hashCode() {
                int result = this.accessStatus != null ? this.accessStatus.hashCode() : 0;
                result = 31 * result + (this.userGroup != null ? this.userGroup.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * A collection of settings that configure the `RStudioServerPro` Domain-level app.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * RStudioServerProDomainSettingsProperty rStudioServerProDomainSettingsProperty = RStudioServerProDomainSettingsProperty.builder()
     *         .domainExecutionRoleArn("domainExecutionRoleArn")
     *         // the properties below are optional
     *         .defaultResourceSpec(ResourceSpecProperty.builder()
     *                 .instanceType("instanceType")
     *                 .lifecycleConfigArn("lifecycleConfigArn")
     *                 .sageMakerImageArn("sageMakerImageArn")
     *                 .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                 .build())
     *         .rStudioConnectUrl("rStudioConnectUrl")
     *         .rStudioPackageManagerUrl("rStudioPackageManagerUrl")
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.RStudioServerProDomainSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(RStudioServerProDomainSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface RStudioServerProDomainSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * The ARN of the execution role for the `RStudioServerPro` Domain-level app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @org.jetbrains.annotations.NotNull java.lang.String getDomainExecutionRoleArn();

        /**
         * A collection that defines the default `InstanceType` , `SageMakerImageArn` , and `SageMakerImageVersionArn` for the Domain.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getDefaultResourceSpec() {
            return null;
        }

        /**
         * A URL pointing to an RStudio Connect server.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getRStudioConnectUrl() {
            return null;
        }

        /**
         * A URL pointing to an RStudio Package Manager server.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getRStudioPackageManagerUrl() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link RStudioServerProDomainSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link RStudioServerProDomainSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<RStudioServerProDomainSettingsProperty> {
            java.lang.String domainExecutionRoleArn;
            java.lang.Object defaultResourceSpec;
            java.lang.String rStudioConnectUrl;
            java.lang.String rStudioPackageManagerUrl;

            /**
             * Sets the value of {@link RStudioServerProDomainSettingsProperty#getDomainExecutionRoleArn}
             * @param domainExecutionRoleArn The ARN of the execution role for the `RStudioServerPro` Domain-level app. This parameter is required.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder domainExecutionRoleArn(java.lang.String domainExecutionRoleArn) {
                this.domainExecutionRoleArn = domainExecutionRoleArn;
                return this;
            }

            /**
             * Sets the value of {@link RStudioServerProDomainSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec A collection that defines the default `InstanceType` , `SageMakerImageArn` , and `SageMakerImageVersionArn` for the Domain.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.core.IResolvable defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Sets the value of {@link RStudioServerProDomainSettingsProperty#getDefaultResourceSpec}
             * @param defaultResourceSpec A collection that defines the default `InstanceType` , `SageMakerImageArn` , and `SageMakerImageVersionArn` for the Domain.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder defaultResourceSpec(software.amazon.awscdk.services.sagemaker.CfnDomain.ResourceSpecProperty defaultResourceSpec) {
                this.defaultResourceSpec = defaultResourceSpec;
                return this;
            }

            /**
             * Sets the value of {@link RStudioServerProDomainSettingsProperty#getRStudioConnectUrl}
             * @param rStudioConnectUrl A URL pointing to an RStudio Connect server.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rStudioConnectUrl(java.lang.String rStudioConnectUrl) {
                this.rStudioConnectUrl = rStudioConnectUrl;
                return this;
            }

            /**
             * Sets the value of {@link RStudioServerProDomainSettingsProperty#getRStudioPackageManagerUrl}
             * @param rStudioPackageManagerUrl A URL pointing to an RStudio Package Manager server.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rStudioPackageManagerUrl(java.lang.String rStudioPackageManagerUrl) {
                this.rStudioPackageManagerUrl = rStudioPackageManagerUrl;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link RStudioServerProDomainSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public RStudioServerProDomainSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link RStudioServerProDomainSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements RStudioServerProDomainSettingsProperty {
            private final java.lang.String domainExecutionRoleArn;
            private final java.lang.Object defaultResourceSpec;
            private final java.lang.String rStudioConnectUrl;
            private final java.lang.String rStudioPackageManagerUrl;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.domainExecutionRoleArn = software.amazon.jsii.Kernel.get(this, "domainExecutionRoleArn", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.defaultResourceSpec = software.amazon.jsii.Kernel.get(this, "defaultResourceSpec", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.rStudioConnectUrl = software.amazon.jsii.Kernel.get(this, "rStudioConnectUrl", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.rStudioPackageManagerUrl = software.amazon.jsii.Kernel.get(this, "rStudioPackageManagerUrl", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.domainExecutionRoleArn = java.util.Objects.requireNonNull(builder.domainExecutionRoleArn, "domainExecutionRoleArn is required");
                this.defaultResourceSpec = builder.defaultResourceSpec;
                this.rStudioConnectUrl = builder.rStudioConnectUrl;
                this.rStudioPackageManagerUrl = builder.rStudioPackageManagerUrl;
            }

            @Override
            public final java.lang.String getDomainExecutionRoleArn() {
                return this.domainExecutionRoleArn;
            }

            @Override
            public final java.lang.Object getDefaultResourceSpec() {
                return this.defaultResourceSpec;
            }

            @Override
            public final java.lang.String getRStudioConnectUrl() {
                return this.rStudioConnectUrl;
            }

            @Override
            public final java.lang.String getRStudioPackageManagerUrl() {
                return this.rStudioPackageManagerUrl;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                data.set("domainExecutionRoleArn", om.valueToTree(this.getDomainExecutionRoleArn()));
                if (this.getDefaultResourceSpec() != null) {
                    data.set("defaultResourceSpec", om.valueToTree(this.getDefaultResourceSpec()));
                }
                if (this.getRStudioConnectUrl() != null) {
                    data.set("rStudioConnectUrl", om.valueToTree(this.getRStudioConnectUrl()));
                }
                if (this.getRStudioPackageManagerUrl() != null) {
                    data.set("rStudioPackageManagerUrl", om.valueToTree(this.getRStudioPackageManagerUrl()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.RStudioServerProDomainSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                RStudioServerProDomainSettingsProperty.Jsii$Proxy that = (RStudioServerProDomainSettingsProperty.Jsii$Proxy) o;

                if (!domainExecutionRoleArn.equals(that.domainExecutionRoleArn)) return false;
                if (this.defaultResourceSpec != null ? !this.defaultResourceSpec.equals(that.defaultResourceSpec) : that.defaultResourceSpec != null) return false;
                if (this.rStudioConnectUrl != null ? !this.rStudioConnectUrl.equals(that.rStudioConnectUrl) : that.rStudioConnectUrl != null) return false;
                return this.rStudioPackageManagerUrl != null ? this.rStudioPackageManagerUrl.equals(that.rStudioPackageManagerUrl) : that.rStudioPackageManagerUrl == null;
            }

            @Override
            public final int hashCode() {
                int result = this.domainExecutionRoleArn.hashCode();
                result = 31 * result + (this.defaultResourceSpec != null ? this.defaultResourceSpec.hashCode() : 0);
                result = 31 * result + (this.rStudioConnectUrl != null ? this.rStudioConnectUrl.hashCode() : 0);
                result = 31 * result + (this.rStudioPackageManagerUrl != null ? this.rStudioPackageManagerUrl.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * Specifies the ARN's of a SageMaker image and SageMaker image version, and the instance type that the version runs on.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * ResourceSpecProperty resourceSpecProperty = ResourceSpecProperty.builder()
     *         .instanceType("instanceType")
     *         .lifecycleConfigArn("lifecycleConfigArn")
     *         .sageMakerImageArn("sageMakerImageArn")
     *         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.ResourceSpecProperty")
    @software.amazon.jsii.Jsii.Proxy(ResourceSpecProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface ResourceSpecProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * The instance type that the image version runs on.
         * <p>
         * <blockquote>
         * <p>
         * <em>JupyterServer apps</em> only support the <code>system</code> value.
         * <p>
         * For <em>KernelGateway apps</em> , the <code>system</code> value is translated to <code>ml.t3.medium</code> . KernelGateway apps also support all other values for available instance types.
         * <p>
         * </blockquote>
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getInstanceType() {
            return null;
        }

        /**
         * The Amazon Resource Name (ARN) of the Lifecycle Configuration attached to the Resource.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getLifecycleConfigArn() {
            return null;
        }

        /**
         * The ARN of the SageMaker image that the image version belongs to.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getSageMakerImageArn() {
            return null;
        }

        /**
         * The ARN of the image version created on the instance.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getSageMakerImageVersionArn() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link ResourceSpecProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link ResourceSpecProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<ResourceSpecProperty> {
            java.lang.String instanceType;
            java.lang.String lifecycleConfigArn;
            java.lang.String sageMakerImageArn;
            java.lang.String sageMakerImageVersionArn;

            /**
             * Sets the value of {@link ResourceSpecProperty#getInstanceType}
             * @param instanceType The instance type that the image version runs on.
             *                     <blockquote>
             *                     <p>
             *                     <em>JupyterServer apps</em> only support the <code>system</code> value.
             *                     <p>
             *                     For <em>KernelGateway apps</em> , the <code>system</code> value is translated to <code>ml.t3.medium</code> . KernelGateway apps also support all other values for available instance types.
             *                     <p>
             *                     </blockquote>
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder instanceType(java.lang.String instanceType) {
                this.instanceType = instanceType;
                return this;
            }

            /**
             * Sets the value of {@link ResourceSpecProperty#getLifecycleConfigArn}
             * @param lifecycleConfigArn The Amazon Resource Name (ARN) of the Lifecycle Configuration attached to the Resource.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder lifecycleConfigArn(java.lang.String lifecycleConfigArn) {
                this.lifecycleConfigArn = lifecycleConfigArn;
                return this;
            }

            /**
             * Sets the value of {@link ResourceSpecProperty#getSageMakerImageArn}
             * @param sageMakerImageArn The ARN of the SageMaker image that the image version belongs to.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder sageMakerImageArn(java.lang.String sageMakerImageArn) {
                this.sageMakerImageArn = sageMakerImageArn;
                return this;
            }

            /**
             * Sets the value of {@link ResourceSpecProperty#getSageMakerImageVersionArn}
             * @param sageMakerImageVersionArn The ARN of the image version created on the instance.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder sageMakerImageVersionArn(java.lang.String sageMakerImageVersionArn) {
                this.sageMakerImageVersionArn = sageMakerImageVersionArn;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link ResourceSpecProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public ResourceSpecProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link ResourceSpecProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements ResourceSpecProperty {
            private final java.lang.String instanceType;
            private final java.lang.String lifecycleConfigArn;
            private final java.lang.String sageMakerImageArn;
            private final java.lang.String sageMakerImageVersionArn;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.instanceType = software.amazon.jsii.Kernel.get(this, "instanceType", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.lifecycleConfigArn = software.amazon.jsii.Kernel.get(this, "lifecycleConfigArn", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.sageMakerImageArn = software.amazon.jsii.Kernel.get(this, "sageMakerImageArn", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.sageMakerImageVersionArn = software.amazon.jsii.Kernel.get(this, "sageMakerImageVersionArn", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.instanceType = builder.instanceType;
                this.lifecycleConfigArn = builder.lifecycleConfigArn;
                this.sageMakerImageArn = builder.sageMakerImageArn;
                this.sageMakerImageVersionArn = builder.sageMakerImageVersionArn;
            }

            @Override
            public final java.lang.String getInstanceType() {
                return this.instanceType;
            }

            @Override
            public final java.lang.String getLifecycleConfigArn() {
                return this.lifecycleConfigArn;
            }

            @Override
            public final java.lang.String getSageMakerImageArn() {
                return this.sageMakerImageArn;
            }

            @Override
            public final java.lang.String getSageMakerImageVersionArn() {
                return this.sageMakerImageVersionArn;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getInstanceType() != null) {
                    data.set("instanceType", om.valueToTree(this.getInstanceType()));
                }
                if (this.getLifecycleConfigArn() != null) {
                    data.set("lifecycleConfigArn", om.valueToTree(this.getLifecycleConfigArn()));
                }
                if (this.getSageMakerImageArn() != null) {
                    data.set("sageMakerImageArn", om.valueToTree(this.getSageMakerImageArn()));
                }
                if (this.getSageMakerImageVersionArn() != null) {
                    data.set("sageMakerImageVersionArn", om.valueToTree(this.getSageMakerImageVersionArn()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.ResourceSpecProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                ResourceSpecProperty.Jsii$Proxy that = (ResourceSpecProperty.Jsii$Proxy) o;

                if (this.instanceType != null ? !this.instanceType.equals(that.instanceType) : that.instanceType != null) return false;
                if (this.lifecycleConfigArn != null ? !this.lifecycleConfigArn.equals(that.lifecycleConfigArn) : that.lifecycleConfigArn != null) return false;
                if (this.sageMakerImageArn != null ? !this.sageMakerImageArn.equals(that.sageMakerImageArn) : that.sageMakerImageArn != null) return false;
                return this.sageMakerImageVersionArn != null ? this.sageMakerImageVersionArn.equals(that.sageMakerImageVersionArn) : that.sageMakerImageVersionArn == null;
            }

            @Override
            public final int hashCode() {
                int result = this.instanceType != null ? this.instanceType.hashCode() : 0;
                result = 31 * result + (this.lifecycleConfigArn != null ? this.lifecycleConfigArn.hashCode() : 0);
                result = 31 * result + (this.sageMakerImageArn != null ? this.sageMakerImageArn.hashCode() : 0);
                result = 31 * result + (this.sageMakerImageVersionArn != null ? this.sageMakerImageVersionArn.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * Specifies options when sharing an Amazon SageMaker Studio notebook.
     * <p>
     * These settings are specified as part of <code>DefaultUserSettings</code> when the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateDomain.html">CreateDomain</a> API is called, and as part of <code>UserSettings</code> when the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateUserProfile.html">CreateUserProfile</a> API is called.
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * SharingSettingsProperty sharingSettingsProperty = SharingSettingsProperty.builder()
     *         .notebookOutputOption("notebookOutputOption")
     *         .s3KmsKeyId("s3KmsKeyId")
     *         .s3OutputPath("s3OutputPath")
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.SharingSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(SharingSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface SharingSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * Whether to include the notebook cell output when sharing the notebook.
         * <p>
         * The default is <code>Disabled</code> .
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getNotebookOutputOption() {
            return null;
        }

        /**
         * When `NotebookOutputOption` is `Allowed` , the AWS Key Management Service (KMS) encryption key ID used to encrypt the notebook cell output in the Amazon S3 bucket.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getS3KmsKeyId() {
            return null;
        }

        /**
         * When `NotebookOutputOption` is `Allowed` , the Amazon S3 bucket used to store the shared notebook snapshots.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getS3OutputPath() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link SharingSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link SharingSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<SharingSettingsProperty> {
            java.lang.String notebookOutputOption;
            java.lang.String s3KmsKeyId;
            java.lang.String s3OutputPath;

            /**
             * Sets the value of {@link SharingSettingsProperty#getNotebookOutputOption}
             * @param notebookOutputOption Whether to include the notebook cell output when sharing the notebook.
             *                             The default is <code>Disabled</code> .
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder notebookOutputOption(java.lang.String notebookOutputOption) {
                this.notebookOutputOption = notebookOutputOption;
                return this;
            }

            /**
             * Sets the value of {@link SharingSettingsProperty#getS3KmsKeyId}
             * @param s3KmsKeyId When `NotebookOutputOption` is `Allowed` , the AWS Key Management Service (KMS) encryption key ID used to encrypt the notebook cell output in the Amazon S3 bucket.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder s3KmsKeyId(java.lang.String s3KmsKeyId) {
                this.s3KmsKeyId = s3KmsKeyId;
                return this;
            }

            /**
             * Sets the value of {@link SharingSettingsProperty#getS3OutputPath}
             * @param s3OutputPath When `NotebookOutputOption` is `Allowed` , the Amazon S3 bucket used to store the shared notebook snapshots.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder s3OutputPath(java.lang.String s3OutputPath) {
                this.s3OutputPath = s3OutputPath;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link SharingSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public SharingSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link SharingSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements SharingSettingsProperty {
            private final java.lang.String notebookOutputOption;
            private final java.lang.String s3KmsKeyId;
            private final java.lang.String s3OutputPath;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.notebookOutputOption = software.amazon.jsii.Kernel.get(this, "notebookOutputOption", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.s3KmsKeyId = software.amazon.jsii.Kernel.get(this, "s3KmsKeyId", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.s3OutputPath = software.amazon.jsii.Kernel.get(this, "s3OutputPath", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.notebookOutputOption = builder.notebookOutputOption;
                this.s3KmsKeyId = builder.s3KmsKeyId;
                this.s3OutputPath = builder.s3OutputPath;
            }

            @Override
            public final java.lang.String getNotebookOutputOption() {
                return this.notebookOutputOption;
            }

            @Override
            public final java.lang.String getS3KmsKeyId() {
                return this.s3KmsKeyId;
            }

            @Override
            public final java.lang.String getS3OutputPath() {
                return this.s3OutputPath;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getNotebookOutputOption() != null) {
                    data.set("notebookOutputOption", om.valueToTree(this.getNotebookOutputOption()));
                }
                if (this.getS3KmsKeyId() != null) {
                    data.set("s3KmsKeyId", om.valueToTree(this.getS3KmsKeyId()));
                }
                if (this.getS3OutputPath() != null) {
                    data.set("s3OutputPath", om.valueToTree(this.getS3OutputPath()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.SharingSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                SharingSettingsProperty.Jsii$Proxy that = (SharingSettingsProperty.Jsii$Proxy) o;

                if (this.notebookOutputOption != null ? !this.notebookOutputOption.equals(that.notebookOutputOption) : that.notebookOutputOption != null) return false;
                if (this.s3KmsKeyId != null ? !this.s3KmsKeyId.equals(that.s3KmsKeyId) : that.s3KmsKeyId != null) return false;
                return this.s3OutputPath != null ? this.s3OutputPath.equals(that.s3OutputPath) : that.s3OutputPath == null;
            }

            @Override
            public final int hashCode() {
                int result = this.notebookOutputOption != null ? this.notebookOutputOption.hashCode() : 0;
                result = 31 * result + (this.s3KmsKeyId != null ? this.s3KmsKeyId.hashCode() : 0);
                result = 31 * result + (this.s3OutputPath != null ? this.s3OutputPath.hashCode() : 0);
                return result;
            }
        }
    }
    /**
     * A collection of settings that apply to users of Amazon SageMaker Studio.
     * <p>
     * These settings are specified when the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateUserProfile.html">CreateUserProfile</a> API is called, and as <code>DefaultUserSettings</code> when the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateDomain.html">CreateDomain</a> API is called.
     * <p>
     * <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in <code>UserSettings</code> , the values specified in <code>CreateUserProfile</code> take precedence over those specified in <code>CreateDomain</code> .
     * <p>
     * Example:
     * <p>
     * <blockquote><pre>
     * // The code below shows an example of how to instantiate this type.
     * // The values are placeholders you should change.
     * import software.amazon.awscdk.services.sagemaker.*;
     * UserSettingsProperty userSettingsProperty = UserSettingsProperty.builder()
     *         .executionRole("executionRole")
     *         .jupyterServerAppSettings(JupyterServerAppSettingsProperty.builder()
     *                 .defaultResourceSpec(ResourceSpecProperty.builder()
     *                         .instanceType("instanceType")
     *                         .lifecycleConfigArn("lifecycleConfigArn")
     *                         .sageMakerImageArn("sageMakerImageArn")
     *                         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                         .build())
     *                 .build())
     *         .kernelGatewayAppSettings(KernelGatewayAppSettingsProperty.builder()
     *                 .customImages(List.of(CustomImageProperty.builder()
     *                         .appImageConfigName("appImageConfigName")
     *                         .imageName("imageName")
     *                         // the properties below are optional
     *                         .imageVersionNumber(123)
     *                         .build()))
     *                 .defaultResourceSpec(ResourceSpecProperty.builder()
     *                         .instanceType("instanceType")
     *                         .lifecycleConfigArn("lifecycleConfigArn")
     *                         .sageMakerImageArn("sageMakerImageArn")
     *                         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                         .build())
     *                 .build())
     *         .rSessionAppSettings(RSessionAppSettingsProperty.builder()
     *                 .customImages(List.of(CustomImageProperty.builder()
     *                         .appImageConfigName("appImageConfigName")
     *                         .imageName("imageName")
     *                         // the properties below are optional
     *                         .imageVersionNumber(123)
     *                         .build()))
     *                 .defaultResourceSpec(ResourceSpecProperty.builder()
     *                         .instanceType("instanceType")
     *                         .lifecycleConfigArn("lifecycleConfigArn")
     *                         .sageMakerImageArn("sageMakerImageArn")
     *                         .sageMakerImageVersionArn("sageMakerImageVersionArn")
     *                         .build())
     *                 .build())
     *         .rStudioServerProAppSettings(RStudioServerProAppSettingsProperty.builder()
     *                 .accessStatus("accessStatus")
     *                 .userGroup("userGroup")
     *                 .build())
     *         .securityGroups(List.of("securityGroups"))
     *         .sharingSettings(SharingSettingsProperty.builder()
     *                 .notebookOutputOption("notebookOutputOption")
     *                 .s3KmsKeyId("s3KmsKeyId")
     *                 .s3OutputPath("s3OutputPath")
     *                 .build())
     *         .build();
     * </pre></blockquote>
     */
    @software.amazon.jsii.Jsii(module = software.amazon.awscdk.services.sagemaker.$Module.class, fqn = "@aws-cdk/aws-sagemaker.CfnDomain.UserSettingsProperty")
    @software.amazon.jsii.Jsii.Proxy(UserSettingsProperty.Jsii$Proxy.class)
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static interface UserSettingsProperty extends software.amazon.jsii.JsiiSerializable {

        /**
         * The execution role for the user.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.String getExecutionRole() {
            return null;
        }

        /**
         * The Jupyter server's app settings.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getJupyterServerAppSettings() {
            return null;
        }

        /**
         * The kernel gateway app settings.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getKernelGatewayAppSettings() {
            return null;
        }

        /**
         * A collection of settings that configure the `RSessionGateway` app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getRSessionAppSettings() {
            return null;
        }

        /**
         * A collection of settings that configure user interaction with the `RStudioServerPro` app.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getRStudioServerProAppSettings() {
            return null;
        }

        /**
         * The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
         * <p>
         * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>PublicInternetOnly</code> .
         * <p>
         * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code> .
         * <p>
         * Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the number of security groups that you can specify is one less than the maximum number shown.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.util.List<java.lang.String> getSecurityGroups() {
            return null;
        }

        /**
         * Specifies options for sharing SageMaker Studio notebooks.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        default @org.jetbrains.annotations.Nullable java.lang.Object getSharingSettings() {
            return null;
        }

        /**
         * @return a {@link Builder} of {@link UserSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        static Builder builder() {
            return new Builder();
        }
        /**
         * A builder for {@link UserSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static final class Builder implements software.amazon.jsii.Builder<UserSettingsProperty> {
            java.lang.String executionRole;
            java.lang.Object jupyterServerAppSettings;
            java.lang.Object kernelGatewayAppSettings;
            java.lang.Object rSessionAppSettings;
            java.lang.Object rStudioServerProAppSettings;
            java.util.List<java.lang.String> securityGroups;
            java.lang.Object sharingSettings;

            /**
             * Sets the value of {@link UserSettingsProperty#getExecutionRole}
             * @param executionRole The execution role for the user.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder executionRole(java.lang.String executionRole) {
                this.executionRole = executionRole;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getJupyterServerAppSettings}
             * @param jupyterServerAppSettings The Jupyter server's app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder jupyterServerAppSettings(software.amazon.awscdk.core.IResolvable jupyterServerAppSettings) {
                this.jupyterServerAppSettings = jupyterServerAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getJupyterServerAppSettings}
             * @param jupyterServerAppSettings The Jupyter server's app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder jupyterServerAppSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.JupyterServerAppSettingsProperty jupyterServerAppSettings) {
                this.jupyterServerAppSettings = jupyterServerAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getKernelGatewayAppSettings}
             * @param kernelGatewayAppSettings The kernel gateway app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder kernelGatewayAppSettings(software.amazon.awscdk.core.IResolvable kernelGatewayAppSettings) {
                this.kernelGatewayAppSettings = kernelGatewayAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getKernelGatewayAppSettings}
             * @param kernelGatewayAppSettings The kernel gateway app settings.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder kernelGatewayAppSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.KernelGatewayAppSettingsProperty kernelGatewayAppSettings) {
                this.kernelGatewayAppSettings = kernelGatewayAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getRSessionAppSettings}
             * @param rSessionAppSettings A collection of settings that configure the `RSessionGateway` app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rSessionAppSettings(software.amazon.awscdk.core.IResolvable rSessionAppSettings) {
                this.rSessionAppSettings = rSessionAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getRSessionAppSettings}
             * @param rSessionAppSettings A collection of settings that configure the `RSessionGateway` app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rSessionAppSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.RSessionAppSettingsProperty rSessionAppSettings) {
                this.rSessionAppSettings = rSessionAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getRStudioServerProAppSettings}
             * @param rStudioServerProAppSettings A collection of settings that configure user interaction with the `RStudioServerPro` app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rStudioServerProAppSettings(software.amazon.awscdk.core.IResolvable rStudioServerProAppSettings) {
                this.rStudioServerProAppSettings = rStudioServerProAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getRStudioServerProAppSettings}
             * @param rStudioServerProAppSettings A collection of settings that configure user interaction with the `RStudioServerPro` app.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder rStudioServerProAppSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.RStudioServerProAppSettingsProperty rStudioServerProAppSettings) {
                this.rStudioServerProAppSettings = rStudioServerProAppSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getSecurityGroups}
             * @param securityGroups The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
             *                       Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>PublicInternetOnly</code> .
             *                       <p>
             *                       Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code> .
             *                       <p>
             *                       Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the number of security groups that you can specify is one less than the maximum number shown.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder securityGroups(java.util.List<java.lang.String> securityGroups) {
                this.securityGroups = securityGroups;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getSharingSettings}
             * @param sharingSettings Specifies options for sharing SageMaker Studio notebooks.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder sharingSettings(software.amazon.awscdk.core.IResolvable sharingSettings) {
                this.sharingSettings = sharingSettings;
                return this;
            }

            /**
             * Sets the value of {@link UserSettingsProperty#getSharingSettings}
             * @param sharingSettings Specifies options for sharing SageMaker Studio notebooks.
             * @return {@code this}
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            public Builder sharingSettings(software.amazon.awscdk.services.sagemaker.CfnDomain.SharingSettingsProperty sharingSettings) {
                this.sharingSettings = sharingSettings;
                return this;
            }

            /**
             * Builds the configured instance.
             * @return a new instance of {@link UserSettingsProperty}
             * @throws NullPointerException if any required attribute was not provided
             */
            @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
            @Override
            public UserSettingsProperty build() {
                return new Jsii$Proxy(this);
            }
        }

        /**
         * An implementation for {@link UserSettingsProperty}
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @software.amazon.jsii.Internal
        final class Jsii$Proxy extends software.amazon.jsii.JsiiObject implements UserSettingsProperty {
            private final java.lang.String executionRole;
            private final java.lang.Object jupyterServerAppSettings;
            private final java.lang.Object kernelGatewayAppSettings;
            private final java.lang.Object rSessionAppSettings;
            private final java.lang.Object rStudioServerProAppSettings;
            private final java.util.List<java.lang.String> securityGroups;
            private final java.lang.Object sharingSettings;

            /**
             * Constructor that initializes the object based on values retrieved from the JsiiObject.
             * @param objRef Reference to the JSII managed object.
             */
            protected Jsii$Proxy(final software.amazon.jsii.JsiiObjectRef objRef) {
                super(objRef);
                this.executionRole = software.amazon.jsii.Kernel.get(this, "executionRole", software.amazon.jsii.NativeType.forClass(java.lang.String.class));
                this.jupyterServerAppSettings = software.amazon.jsii.Kernel.get(this, "jupyterServerAppSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.kernelGatewayAppSettings = software.amazon.jsii.Kernel.get(this, "kernelGatewayAppSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.rSessionAppSettings = software.amazon.jsii.Kernel.get(this, "rSessionAppSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.rStudioServerProAppSettings = software.amazon.jsii.Kernel.get(this, "rStudioServerProAppSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
                this.securityGroups = software.amazon.jsii.Kernel.get(this, "securityGroups", software.amazon.jsii.NativeType.listOf(software.amazon.jsii.NativeType.forClass(java.lang.String.class)));
                this.sharingSettings = software.amazon.jsii.Kernel.get(this, "sharingSettings", software.amazon.jsii.NativeType.forClass(java.lang.Object.class));
            }

            /**
             * Constructor that initializes the object based on literal property values passed by the {@link Builder}.
             */
            protected Jsii$Proxy(final Builder builder) {
                super(software.amazon.jsii.JsiiObject.InitializationMode.JSII);
                this.executionRole = builder.executionRole;
                this.jupyterServerAppSettings = builder.jupyterServerAppSettings;
                this.kernelGatewayAppSettings = builder.kernelGatewayAppSettings;
                this.rSessionAppSettings = builder.rSessionAppSettings;
                this.rStudioServerProAppSettings = builder.rStudioServerProAppSettings;
                this.securityGroups = builder.securityGroups;
                this.sharingSettings = builder.sharingSettings;
            }

            @Override
            public final java.lang.String getExecutionRole() {
                return this.executionRole;
            }

            @Override
            public final java.lang.Object getJupyterServerAppSettings() {
                return this.jupyterServerAppSettings;
            }

            @Override
            public final java.lang.Object getKernelGatewayAppSettings() {
                return this.kernelGatewayAppSettings;
            }

            @Override
            public final java.lang.Object getRSessionAppSettings() {
                return this.rSessionAppSettings;
            }

            @Override
            public final java.lang.Object getRStudioServerProAppSettings() {
                return this.rStudioServerProAppSettings;
            }

            @Override
            public final java.util.List<java.lang.String> getSecurityGroups() {
                return this.securityGroups;
            }

            @Override
            public final java.lang.Object getSharingSettings() {
                return this.sharingSettings;
            }

            @Override
            @software.amazon.jsii.Internal
            public com.fasterxml.jackson.databind.JsonNode $jsii$toJson() {
                final com.fasterxml.jackson.databind.ObjectMapper om = software.amazon.jsii.JsiiObjectMapper.INSTANCE;
                final com.fasterxml.jackson.databind.node.ObjectNode data = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();

                if (this.getExecutionRole() != null) {
                    data.set("executionRole", om.valueToTree(this.getExecutionRole()));
                }
                if (this.getJupyterServerAppSettings() != null) {
                    data.set("jupyterServerAppSettings", om.valueToTree(this.getJupyterServerAppSettings()));
                }
                if (this.getKernelGatewayAppSettings() != null) {
                    data.set("kernelGatewayAppSettings", om.valueToTree(this.getKernelGatewayAppSettings()));
                }
                if (this.getRSessionAppSettings() != null) {
                    data.set("rSessionAppSettings", om.valueToTree(this.getRSessionAppSettings()));
                }
                if (this.getRStudioServerProAppSettings() != null) {
                    data.set("rStudioServerProAppSettings", om.valueToTree(this.getRStudioServerProAppSettings()));
                }
                if (this.getSecurityGroups() != null) {
                    data.set("securityGroups", om.valueToTree(this.getSecurityGroups()));
                }
                if (this.getSharingSettings() != null) {
                    data.set("sharingSettings", om.valueToTree(this.getSharingSettings()));
                }

                final com.fasterxml.jackson.databind.node.ObjectNode struct = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                struct.set("fqn", om.valueToTree("@aws-cdk/aws-sagemaker.CfnDomain.UserSettingsProperty"));
                struct.set("data", data);

                final com.fasterxml.jackson.databind.node.ObjectNode obj = com.fasterxml.jackson.databind.node.JsonNodeFactory.instance.objectNode();
                obj.set("$jsii.struct", struct);

                return obj;
            }

            @Override
            public final boolean equals(final Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;

                UserSettingsProperty.Jsii$Proxy that = (UserSettingsProperty.Jsii$Proxy) o;

                if (this.executionRole != null ? !this.executionRole.equals(that.executionRole) : that.executionRole != null) return false;
                if (this.jupyterServerAppSettings != null ? !this.jupyterServerAppSettings.equals(that.jupyterServerAppSettings) : that.jupyterServerAppSettings != null) return false;
                if (this.kernelGatewayAppSettings != null ? !this.kernelGatewayAppSettings.equals(that.kernelGatewayAppSettings) : that.kernelGatewayAppSettings != null) return false;
                if (this.rSessionAppSettings != null ? !this.rSessionAppSettings.equals(that.rSessionAppSettings) : that.rSessionAppSettings != null) return false;
                if (this.rStudioServerProAppSettings != null ? !this.rStudioServerProAppSettings.equals(that.rStudioServerProAppSettings) : that.rStudioServerProAppSettings != null) return false;
                if (this.securityGroups != null ? !this.securityGroups.equals(that.securityGroups) : that.securityGroups != null) return false;
                return this.sharingSettings != null ? this.sharingSettings.equals(that.sharingSettings) : that.sharingSettings == null;
            }

            @Override
            public final int hashCode() {
                int result = this.executionRole != null ? this.executionRole.hashCode() : 0;
                result = 31 * result + (this.jupyterServerAppSettings != null ? this.jupyterServerAppSettings.hashCode() : 0);
                result = 31 * result + (this.kernelGatewayAppSettings != null ? this.kernelGatewayAppSettings.hashCode() : 0);
                result = 31 * result + (this.rSessionAppSettings != null ? this.rSessionAppSettings.hashCode() : 0);
                result = 31 * result + (this.rStudioServerProAppSettings != null ? this.rStudioServerProAppSettings.hashCode() : 0);
                result = 31 * result + (this.securityGroups != null ? this.securityGroups.hashCode() : 0);
                result = 31 * result + (this.sharingSettings != null ? this.sharingSettings.hashCode() : 0);
                return result;
            }
        }
    }

    /**
     * A fluent builder for {@link software.amazon.awscdk.services.sagemaker.CfnDomain}.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    public static final class Builder implements software.amazon.jsii.Builder<software.amazon.awscdk.services.sagemaker.CfnDomain> {
        /**
         * @return a new instance of {@link Builder}.
         * @param scope - scope in which this resource is defined. This parameter is required.
         * @param id - scoped id of the resource. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public static Builder create(final software.amazon.awscdk.core.Construct scope, final java.lang.String id) {
            return new Builder(scope, id);
        }

        private final software.amazon.awscdk.core.Construct scope;
        private final java.lang.String id;
        private final software.amazon.awscdk.services.sagemaker.CfnDomainProps.Builder props;

        private Builder(final software.amazon.awscdk.core.Construct scope, final java.lang.String id) {
            this.scope = scope;
            this.id = id;
            this.props = new software.amazon.awscdk.services.sagemaker.CfnDomainProps.Builder();
        }

        /**
         * The mode of authentication that members use to access the Domain.
         * <p>
         * <em>Valid Values</em> : <code>SSO | IAM</code>
         * <p>
         * @return {@code this}
         * @param authMode The mode of authentication that members use to access the Domain. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder authMode(final java.lang.String authMode) {
            this.props.authMode(authMode);
            return this;
        }

        /**
         * The default user settings.
         * <p>
         * @return {@code this}
         * @param defaultUserSettings The default user settings. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder defaultUserSettings(final software.amazon.awscdk.core.IResolvable defaultUserSettings) {
            this.props.defaultUserSettings(defaultUserSettings);
            return this;
        }
        /**
         * The default user settings.
         * <p>
         * @return {@code this}
         * @param defaultUserSettings The default user settings. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder defaultUserSettings(final software.amazon.awscdk.services.sagemaker.CfnDomain.UserSettingsProperty defaultUserSettings) {
            this.props.defaultUserSettings(defaultUserSettings);
            return this;
        }

        /**
         * The domain name.
         * <p>
         * @return {@code this}
         * @param domainName The domain name. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder domainName(final java.lang.String domainName) {
            this.props.domainName(domainName);
            return this;
        }

        /**
         * The VPC subnets that Studio uses for communication.
         * <p>
         * <em>Length Constraints</em> : Maximum length of 32.
         * <p>
         * <em>Array members</em> : Minimum number of 1 item. Maximum number of 16 items.
         * <p>
         * <em>Pattern</em> : <code>[-0-9a-zA-Z]+</code>
         * <p>
         * @return {@code this}
         * @param subnetIds The VPC subnets that Studio uses for communication. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder subnetIds(final java.util.List<java.lang.String> subnetIds) {
            this.props.subnetIds(subnetIds);
            return this;
        }

        /**
         * The ID of the Amazon Virtual Private Cloud (Amazon VPC) that Studio uses for communication.
         * <p>
         * <em>Length Constraints</em> : Maximum length of 32.
         * <p>
         * <em>Pattern</em> : <code>[-0-9a-zA-Z]+</code>
         * <p>
         * @return {@code this}
         * @param vpcId The ID of the Amazon Virtual Private Cloud (Amazon VPC) that Studio uses for communication. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder vpcId(final java.lang.String vpcId) {
            this.props.vpcId(vpcId);
            return this;
        }

        /**
         * Specifies the VPC used for non-EFS traffic. The default value is `PublicInternetOnly` .
         * <p>
         * <ul>
         * <li><code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker , which allows direct internet access</li>
         * <li><code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets</li>
         * </ul>
         * <p>
         * <em>Valid Values</em> : <code>PublicInternetOnly | VpcOnly</code>
         * <p>
         * @return {@code this}
         * @param appNetworkAccessType Specifies the VPC used for non-EFS traffic. The default value is `PublicInternetOnly` . This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder appNetworkAccessType(final java.lang.String appNetworkAccessType) {
            this.props.appNetworkAccessType(appNetworkAccessType);
            return this;
        }

        /**
         * The entity that creates and manages the required security groups for inter-app communication in `VpcOnly` mode.
         * <p>
         * Required when <code>CreateDomain.AppNetworkAccessType</code> is <code>VpcOnly</code> and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided.
         * <p>
         * @return {@code this}
         * @param appSecurityGroupManagement The entity that creates and manages the required security groups for inter-app communication in `VpcOnly` mode. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder appSecurityGroupManagement(final java.lang.String appSecurityGroupManagement) {
            this.props.appSecurityGroupManagement(appSecurityGroupManagement);
            return this;
        }

        /**
         * `AWS::SageMaker::Domain.DefaultSpaceSettings`.
         * <p>
         * @return {@code this}
         * @param defaultSpaceSettings `AWS::SageMaker::Domain.DefaultSpaceSettings`. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder defaultSpaceSettings(final software.amazon.awscdk.core.IResolvable defaultSpaceSettings) {
            this.props.defaultSpaceSettings(defaultSpaceSettings);
            return this;
        }
        /**
         * `AWS::SageMaker::Domain.DefaultSpaceSettings`.
         * <p>
         * @return {@code this}
         * @param defaultSpaceSettings `AWS::SageMaker::Domain.DefaultSpaceSettings`. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder defaultSpaceSettings(final software.amazon.awscdk.services.sagemaker.CfnDomain.DefaultSpaceSettingsProperty defaultSpaceSettings) {
            this.props.defaultSpaceSettings(defaultSpaceSettings);
            return this;
        }

        /**
         * A collection of settings that apply to the `SageMaker Domain` .
         * <p>
         * These settings are specified through the <code>CreateDomain</code> API call.
         * <p>
         * @return {@code this}
         * @param domainSettings A collection of settings that apply to the `SageMaker Domain` . This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder domainSettings(final software.amazon.awscdk.core.IResolvable domainSettings) {
            this.props.domainSettings(domainSettings);
            return this;
        }
        /**
         * A collection of settings that apply to the `SageMaker Domain` .
         * <p>
         * These settings are specified through the <code>CreateDomain</code> API call.
         * <p>
         * @return {@code this}
         * @param domainSettings A collection of settings that apply to the `SageMaker Domain` . This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder domainSettings(final software.amazon.awscdk.services.sagemaker.CfnDomain.DomainSettingsProperty domainSettings) {
            this.props.domainSettings(domainSettings);
            return this;
        }

        /**
         * SageMaker uses AWS KMS to encrypt the EFS volume attached to the Domain with an AWS managed customer master key (CMK) by default.
         * <p>
         * For more control, specify a customer managed CMK.
         * <p>
         * <em>Length Constraints</em> : Maximum length of 2048.
         * <p>
         * <em>Pattern</em> : <code>.*</code>
         * <p>
         * @return {@code this}
         * @param kmsKeyId SageMaker uses AWS KMS to encrypt the EFS volume attached to the Domain with an AWS managed customer master key (CMK) by default. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder kmsKeyId(final java.lang.String kmsKeyId) {
            this.props.kmsKeyId(kmsKeyId);
            return this;
        }

        /**
         * Tags to associated with the Domain.
         * <p>
         * Each tag consists of a key and an optional value. Tag keys must be unique per resource. Tags are searchable using the Search API.
         * <p>
         * Tags that you specify for the Domain are also added to all apps that are launched in the Domain.
         * <p>
         * <em>Array members</em> : Minimum number of 0 items. Maximum number of 50 items.
         * <p>
         * @return {@code this}
         * @param tags Tags to associated with the Domain. This parameter is required.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        public Builder tags(final java.util.List<? extends software.amazon.awscdk.core.CfnTag> tags) {
            this.props.tags(tags);
            return this;
        }

        /**
         * @returns a newly built instance of {@link software.amazon.awscdk.services.sagemaker.CfnDomain}.
         */
        @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
        @Override
        public software.amazon.awscdk.services.sagemaker.CfnDomain build() {
            return new software.amazon.awscdk.services.sagemaker.CfnDomain(
                this.scope,
                this.id,
                this.props.build()
            );
        }
    }
}
