/**
 * <h1>AWS Service Catalog Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <a href="https://docs.aws.amazon.com/servicecatalog/latest/dg/what-is-service-catalog.html">AWS Service Catalog</a>
 * enables organizations to create and manage catalogs of products for their end users that are approved for use on AWS.
 * <p>
 * <h2>Table Of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#portfolio">Portfolio</a>
 * <p>
 * <ul>
 * <li><a href="#granting-access-to-a-portfolio">Granting access to a portfolio</a></li>
 * <li><a href="#sharing-a-portfolio-with-another-aws-account">Sharing a portfolio with another AWS account</a></li>
 * </ul></li>
 * <li><a href="#product">Product</a>
 * <p>
 * <ul>
 * <li><a href="#creating-a-product-from-local-asset">Creating a product from a local asset</a></li>
 * <li><a href="#creating-a-product-from-a-stack">Creating a product from a stack</a></li>
 * <li><a href="#creating-a-product-from-a-stack-with-a-history-of-all-previous-versions">Creating a Product from a stack with a history of previous versions</a></li>
 * <li><a href="#adding-a-product-to-a-portfolio">Adding a product to a portfolio</a></li>
 * </ul></li>
 * <li><a href="#tag-options">TagOptions</a></li>
 * <li><a href="#constraints">Constraints</a>
 * <p>
 * <ul>
 * <li><a href="#tag-update-constraint">Tag update constraint</a></li>
 * <li><a href="#notify-on-stack-events">Notify on stack events</a></li>
 * <li><a href="#cloudformation-template-parameters-constraint">CloudFormation template parameters constraint</a></li>
 * <li><a href="#set-launch-role">Set launch role</a></li>
 * <li><a href="#deploy-with-stacksets">Deploy with StackSets</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * The <code>&#64;aws-cdk/aws-servicecatalog</code> package contains resources that enable users to automate governance and management of their AWS resources at scale.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.servicecatalog.*;
 * </pre></blockquote>
 * <p>
 * <h2>Portfolio</h2>
 * <p>
 * AWS Service Catalog portfolios allow administrators to organize, manage, and distribute cloud resources for their end users.
 * Using the CDK, a new portfolio can be created with the <code>Portfolio</code> construct:
 * <p>
 * <blockquote><pre>
 * Portfolio.Builder.create(this, "Portfolio")
 *         .displayName("MyPortfolio")
 *         .providerName("MyTeam")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify optional metadata properties such as <code>description</code> and <code>messageLanguage</code>
 * to help better catalog and manage your portfolios.
 * <p>
 * <blockquote><pre>
 * Portfolio.Builder.create(this, "Portfolio")
 *         .displayName("MyFirstPortfolio")
 *         .providerName("SCAdmin")
 *         .description("Portfolio for a project")
 *         .messageLanguage(MessageLanguage.EN)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Read more at <a href="https://docs.aws.amazon.com/servicecatalog/latest/adminguide/catalogs_portfolios.html">Creating and Managing Portfolios</a>.
 * <p>
 * To reference an existing portfolio into your CDK application, use the <code>Portfolio.fromPortfolioArn()</code> factory method:
 * <p>
 * <blockquote><pre>
 * IPortfolio portfolio = Portfolio.fromPortfolioArn(this, "ReferencedPortfolio", "arn:aws:catalog:region:account-id:portfolio/port-abcdefghi");
 * </pre></blockquote>
 * <p>
 * <h3>Granting access to a portfolio</h3>
 * <p>
 * You can grant access to and manage the <code>IAM</code> users, groups, or roles that have access to the products within a portfolio.
 * Entities with granted access will be able to utilize the portfolios resources and products via the console or AWS CLI.
 * Once resources are deployed end users will be able to access them via the console or service catalog CLI.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Portfolio portfolio;
 * 
 * 
 * User user = new User(this, "User");
 * portfolio.giveAccessToUser(user);
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new AccountRootPrincipal())
 *         .build();
 * portfolio.giveAccessToRole(role);
 * 
 * Group group = new Group(this, "Group");
 * portfolio.giveAccessToGroup(group);
 * </pre></blockquote>
 * <p>
 * <h3>Sharing a portfolio with another AWS account</h3>
 * <p>
 * You can use account-to-account sharing to distribute a reference to your portfolio to other AWS accounts by passing the recipient account number.
 * After the share is initiated, the recipient account can accept the share via CLI or console by importing the portfolio ID.
 * Changes made to the shared portfolio will automatically propagate to recipients.
 * <p>
 * <blockquote><pre>
 * Portfolio portfolio;
 * 
 * portfolio.shareWithAccount("012345678901");
 * </pre></blockquote>
 * <p>
 * <h2>Product</h2>
 * <p>
 * Products are version friendly infrastructure-as-code templates that admins create and add to portfolios for end users to provision and create AWS resources.
 * Service Catalog supports products from AWS Marketplace or ones defined by a CloudFormation template.
 * The CDK currently only supports adding products of type CloudFormation.
 * Using the CDK, a new Product can be created with the <code>CloudFormationProduct</code> construct.
 * You can use <code>CloudFormationTemplate.fromUrl</code> to create a Product from a CloudFormation template directly from a URL that points to the template in S3, GitHub, or CodeCommit:
 * <p>
 * <blockquote><pre>
 * CloudFormationProduct product = CloudFormationProduct.Builder.create(this, "MyFirstProduct")
 *         .productName("My Product")
 *         .owner("Product Owner")
 *         .productVersions(List.of(CloudFormationProductVersion.builder()
 *                 .productVersionName("v1")
 *                 .cloudFormationTemplate(CloudFormationTemplate.fromUrl("https://raw.githubusercontent.com/awslabs/aws-cloudformation-templates/master/aws/services/ServiceCatalog/Product.yaml"))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Creating a product from a local asset</h3>
 * <p>
 * A <code>CloudFormationProduct</code> can also be created by using a CloudFormation template held locally on disk using Assets.
 * Assets are files that are uploaded to an S3 Bucket before deployment.
 * <code>CloudFormationTemplate.fromAsset</code> can be utilized to create a Product by passing the path to a local template file on your disk:
 * <p>
 * <blockquote><pre>
 * import path.*;
 * 
 * 
 * CloudFormationProduct product = CloudFormationProduct.Builder.create(this, "Product")
 *         .productName("My Product")
 *         .owner("Product Owner")
 *         .productVersions(List.of(CloudFormationProductVersion.builder()
 *                 .productVersionName("v1")
 *                 .cloudFormationTemplate(CloudFormationTemplate.fromUrl("https://raw.githubusercontent.com/awslabs/aws-cloudformation-templates/master/aws/services/ServiceCatalog/Product.yaml"))
 *                 .build(), CloudFormationProductVersion.builder()
 *                 .productVersionName("v2")
 *                 .cloudFormationTemplate(CloudFormationTemplate.fromAsset(join(__dirname, "development-environment.template.json")))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Creating a product from a stack</h3>
 * <p>
 * You can create a Service Catalog <code>CloudFormationProduct</code> entirely defined with CDK code using a service catalog <code>ProductStack</code>.
 * A separate child stack for your product is created and you can add resources like you would for any other CDK stack,
 * such as an S3 Bucket, IAM roles, and EC2 instances. This stack is passed in as a product version to your
 * product.  This will not create a separate CloudFormation stack during deployment.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * public class S3BucketProduct extends ProductStack {
 *     public S3BucketProduct(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         new Bucket(this, "BucketProduct");
 *     }
 * }
 * 
 * CloudFormationProduct product = CloudFormationProduct.Builder.create(this, "Product")
 *         .productName("My Product")
 *         .owner("Product Owner")
 *         .productVersions(List.of(CloudFormationProductVersion.builder()
 *                 .productVersionName("v1")
 *                 .cloudFormationTemplate(CloudFormationTemplate.fromProductStack(new S3BucketProduct(this, "S3BucketProduct")))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Creating a Product from a stack with a history of previous versions</h3>
 * <p>
 * The default behavior of Service Catalog is to overwrite each product version upon deployment.
 * This applies to Product Stacks as well, where only the latest changes to your Product Stack will
 * be deployed.
 * To keep a history of the revisions of a ProductStack available in Service Catalog,
 * you would need to define a ProductStack for each historical copy.
 * <p>
 * You can instead create a <code>ProductStackHistory</code> to maintain snapshots of all previous versions.
 * The <code>ProductStackHistory</code> can be created by passing the base <code>productStack</code>,
 * a <code>currentVersionName</code> for your current version and a <code>locked</code> boolean.
 * The <code>locked</code> boolean which when set to true will prevent your <code>currentVersionName</code>
 * from being overwritten when there is an existing snapshot for that version.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * public class S3BucketProduct extends ProductStack {
 *     public S3BucketProduct(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         new Bucket(this, "BucketProduct");
 *     }
 * }
 * 
 * ProductStackHistory productStackHistory = ProductStackHistory.Builder.create(this, "ProductStackHistory")
 *         .productStack(new S3BucketProduct(this, "S3BucketProduct"))
 *         .currentVersionName("v1")
 *         .currentVersionLocked(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * We can deploy the current version <code>v1</code> by using <code>productStackHistory.currentVersion()</code>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * public class S3BucketProduct extends ProductStack {
 *     public S3BucketProduct(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         new Bucket(this, "BucketProductV2");
 *     }
 * }
 * 
 * ProductStackHistory productStackHistory = ProductStackHistory.Builder.create(this, "ProductStackHistory")
 *         .productStack(new S3BucketProduct(this, "S3BucketProduct"))
 *         .currentVersionName("v2")
 *         .currentVersionLocked(true)
 *         .build();
 * 
 * CloudFormationProduct product = CloudFormationProduct.Builder.create(this, "MyFirstProduct")
 *         .productName("My Product")
 *         .owner("Product Owner")
 *         .productVersions(List.of(productStackHistory.currentVersion()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Using <code>ProductStackHistory</code> all deployed templates for the ProductStack will be written to disk,
 * so that they will still be available in the future as the definition of the <code>ProductStack</code> subclass changes over time.
 * <strong>It is very important</strong> that you commit these old versions to source control as these versions
 * determine whether a version has already been deployed and can also be deployed themselves.
 * <p>
 * After using <code>ProductStackHistory</code> to deploy version <code>v1</code> of your <code>ProductStack</code>, we
 * make changes to the <code>ProductStack</code> and update the <code>currentVersionName</code> to <code>v2</code>.
 * We still want our <code>v1</code> version to still be deployed, so we reference it by calling <code>productStackHistory.versionFromSnapshot('v1')</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.core.*;
 * 
 * 
 * public class S3BucketProduct extends ProductStack {
 *     public S3BucketProduct(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         new Bucket(this, "BucketProductV2");
 *     }
 * }
 * 
 * ProductStackHistory productStackHistory = ProductStackHistory.Builder.create(this, "ProductStackHistory")
 *         .productStack(new S3BucketProduct(this, "S3BucketProduct"))
 *         .currentVersionName("v2")
 *         .currentVersionLocked(true)
 *         .build();
 * 
 * CloudFormationProduct product = CloudFormationProduct.Builder.create(this, "MyFirstProduct")
 *         .productName("My Product")
 *         .owner("Product Owner")
 *         .productVersions(List.of(productStackHistory.currentVersion(), productStackHistory.versionFromSnapshot("v1")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Adding a product to a portfolio</h3>
 * <p>
 * You add products to a portfolio to organize and distribute your catalog at scale.  Adding a product to a portfolio creates an association,
 * and the product will become visible within the portfolio side in both the Service Catalog console and AWS CLI.
 * You can add a product to multiple portfolios depending on your organizational structure and how you would like to group access to products.
 * <p>
 * <blockquote><pre>
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * portfolio.addProduct(product);
 * </pre></blockquote>
 * <p>
 * <h2>Tag Options</h2>
 * <p>
 * TagOptions allow administrators to easily manage tags on provisioned products by providing a template for a selection of tags that end users choose from.
 * TagOptions are created by specifying a tag key with a set of allowed values and can be associated with both portfolios and products.
 * When launching a product, both the TagOptions associated with the product and the containing portfolio are made available.
 * <p>
 * At the moment, TagOptions can only be deactivated in the console.
 * <p>
 * <blockquote><pre>
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * TagOptions tagOptionsForPortfolio = TagOptions.Builder.create(this, "OrgTagOptions")
 *         .allowedValuesForTags(Map.of(
 *                 "Group", List.of("finance", "engineering", "marketing", "research"),
 *                 "CostCenter", List.of("01", "02", "03")))
 *         .build();
 * portfolio.associateTagOptions(tagOptionsForPortfolio);
 * 
 * TagOptions tagOptionsForProduct = TagOptions.Builder.create(this, "ProductTagOptions")
 *         .allowedValuesForTags(Map.of(
 *                 "Environment", List.of("dev", "alpha", "prod")))
 *         .build();
 * product.associateTagOptions(tagOptionsForProduct);
 * </pre></blockquote>
 * <p>
 * <h2>Constraints</h2>
 * <p>
 * Constraints are governance gestures that you place on product-portfolio associations that allow you to manage minimal launch permissions, notifications, and other optional actions that end users can perform on products.
 * Using the CDK, if you do not explicitly associate a product to a portfolio and add a constraint, it will automatically add an association for you.
 * <p>
 * There are rules around how constraints are applied to portfolio-product associations.
 * For example, you can only have a single "launch role" constraint applied to a portfolio-product association.
 * If a misconfigured constraint is added, <code>synth</code> will fail with an error message.
 * <p>
 * Read more at <a href="https://docs.aws.amazon.com/servicecatalog/latest/adminguide/constraints.html">Service Catalog Constraints</a>.
 * <p>
 * <h3>Tag update constraint</h3>
 * <p>
 * Tag update constraints allow or disallow end users to update tags on resources associated with an AWS Service Catalog product upon provisioning.
 * By default, if a Tag Update constraint is not configured, tag updating is not permitted.
 * If tag updating is allowed, then new tags associated with the product or portfolio will be applied to provisioned resources during a provisioned product update.
 * <p>
 * <blockquote><pre>
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * portfolio.addProduct(product);
 * portfolio.constrainTagUpdates(product);
 * </pre></blockquote>
 * <p>
 * If you want to disable this feature later on, you can update it by setting the "allow" parameter to <code>false</code>:
 * <p>
 * <blockquote><pre>
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * // to disable tag updates:
 * portfolio.constrainTagUpdates(product, TagUpdateConstraintOptions.builder()
 *         .allow(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Notify on stack events</h3>
 * <p>
 * Allows users to subscribe an AWS <code>SNS</code> topic to a provisioned product's CloudFormation stack events.
 * When an end user provisions a product it creates a CloudFormation stack that notifies the subscribed topic on creation, edit, and delete events.
 * An individual <code>SNS</code> topic may only have a single subscription to any given portfolio-product association.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * 
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * Topic topic1 = new Topic(this, "Topic1");
 * portfolio.notifyOnStackEvents(product, topic1);
 * 
 * Topic topic2 = new Topic(this, "Topic2");
 * portfolio.notifyOnStackEvents(product, topic2, CommonConstraintOptions.builder()
 *         .description("description for topic2")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>CloudFormation template parameters constraint</h3>
 * <p>
 * CloudFormation template parameter constraints allow you to configure the provisioning parameters that are available to end users when they launch a product.
 * Template constraint rules consist of one or more assertions that define the default and/or allowable values for a product’s provisioning parameters.
 * You can configure multiple parameter constraints to govern the different provisioning parameters within your products.
 * For example, a rule might define the <code>EC2</code> instance types that users can choose from when launching a product that includes one or more <code>EC2</code> instances.
 * Parameter rules have an optional <code>condition</code> field that allow for rule application to consider conditional evaluations.
 * If a <code>condition</code> is specified, all  assertions will be applied if the condition evaluates to true.
 * For information on rule-specific intrinsic functions to define rule conditions and assertions,
 * see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-rules.html">AWS Rule Functions</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.core.*;
 * 
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * portfolio.constrainCloudFormationParameters(product, CloudFormationRuleConstraintOptions.builder()
 *         .rule(TemplateRule.builder()
 *                 .ruleName("testInstanceType")
 *                 .condition(Fn.conditionEquals(Fn.ref("Environment"), "test"))
 *                 .assertions(List.of(TemplateRuleAssertion.builder()
 *                         .assert(Fn.conditionContains(List.of("t2.micro", "t2.small"), Fn.ref("InstanceType")))
 *                         .description("For test environment, the instance type should be small")
 *                         .build()))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Set launch role</h3>
 * <p>
 * Allows you to configure a specific <code>IAM</code> role that Service Catalog assumes on behalf of the end user when launching a product.
 * By setting a launch role constraint, you can maintain least permissions for an end user when launching a product.
 * For example, a launch role can grant permissions for specific resource creation like an <code>S3</code> bucket that the user.
 * The launch role must be assumed by the Service Catalog principal.
 * You can only have one launch role set for a portfolio-product association,
 * and you cannot set a launch role on a product that already has a StackSets deployment configured.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * Role launchRole = Role.Builder.create(this, "LaunchRole")
 *         .assumedBy(new ServicePrincipal("servicecatalog.amazonaws.com"))
 *         .build();
 * 
 * portfolio.setLaunchRole(product, launchRole);
 * </pre></blockquote>
 * <p>
 * You can also set the launch role using just the name of a role which is locally deployed in end user accounts.
 * This is useful for when roles and users are separately managed outside of the CDK.
 * The given role must exist in both the account that creates the launch role constraint,
 * as well as in any end user accounts that wish to provision a product with the launch role.
 * <p>
 * You can do this by passing in the role with an explicitly set name:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * Role launchRole = Role.Builder.create(this, "LaunchRole")
 *         .roleName("MyRole")
 *         .assumedBy(new ServicePrincipal("servicecatalog.amazonaws.com"))
 *         .build();
 * 
 * portfolio.setLocalLaunchRole(product, launchRole);
 * </pre></blockquote>
 * <p>
 * Or you can simply pass in a role name and CDK will create a role with that name that trusts service catalog in the account:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * String roleName = "MyRole";
 * IRole launchRole = portfolio.setLocalLaunchRoleName(product, roleName);
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/servicecatalog/latest/adminguide/constraints-launch.html">Launch Constraint</a> documentation
 * to understand the permissions that launch roles need.
 * <p>
 * <h3>Deploy with StackSets</h3>
 * <p>
 * A StackSets deployment constraint allows you to configure product deployment options using
 * <a href="https://docs.aws.amazon.com/servicecatalog/latest/adminguide/using-stacksets.html">AWS CloudFormation StackSets</a>.
 * You can specify one or more accounts and regions into which stack instances will launch when the product is provisioned.
 * There is an additional field <code>allowStackSetInstanceOperations</code> that sets ability for end users to create, edit, or delete the stacks created by the StackSet.
 * By default, this field is set to <code>false</code>.
 * When launching a StackSets product, end users can select from the list of accounts and regions configured in the constraint to determine where the Stack Instances will deploy and the order of deployment.
 * You can only define one StackSets deployment configuration per portfolio-product association,
 * and you cannot both set a launch role and StackSets deployment configuration for an assocation.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Portfolio portfolio;
 * CloudFormationProduct product;
 * 
 * 
 * Role adminRole = Role.Builder.create(this, "AdminRole")
 *         .assumedBy(new AccountRootPrincipal())
 *         .build();
 * 
 * portfolio.deployWithStackSets(product, StackSetsConstraintOptions.builder()
 *         .accounts(List.of("012345678901", "012345678902", "012345678903"))
 *         .regions(List.of("us-west-1", "us-east-1", "us-west-2", "us-east-1"))
 *         .adminRole(adminRole)
 *         .executionRoleName("SCStackSetExecutionRole") // Name of role deployed in end users accounts.
 *         .allowStackSetInstanceOperations(true)
 *         .build());
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.servicecatalog;
