/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.accessanalyzer.model.AccessAnalyzerException;
import software.amazon.awssdk.services.accessanalyzer.model.AccessAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.AccessDeniedException;
import software.amazon.awssdk.services.accessanalyzer.model.ApplyArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ApplyArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CancelPolicyGenerationRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CancelPolicyGenerationResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ConflictException;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAccessPreviewRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAccessPreviewResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CreateArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAccessPreviewRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAccessPreviewResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzedResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzedResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetFindingRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetFindingResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetGeneratedPolicyRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetGeneratedPolicyResponse;
import software.amazon.awssdk.services.accessanalyzer.model.InternalServerException;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.accessanalyzer.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.accessanalyzer.model.StartPolicyGenerationRequest;
import software.amazon.awssdk.services.accessanalyzer.model.StartPolicyGenerationResponse;
import software.amazon.awssdk.services.accessanalyzer.model.StartResourceScanRequest;
import software.amazon.awssdk.services.accessanalyzer.model.StartResourceScanResponse;
import software.amazon.awssdk.services.accessanalyzer.model.TagResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.TagResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ThrottlingException;
import software.amazon.awssdk.services.accessanalyzer.model.UntagResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UntagResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ValidationException;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyPublisher;
import software.amazon.awssdk.services.accessanalyzer.transform.ApplyArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CancelPolicyGenerationRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateAccessPreviewRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.DeleteAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.DeleteArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAccessPreviewRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAnalyzedResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetFindingRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetGeneratedPolicyRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAccessPreviewFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAccessPreviewsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAnalyzedResourcesRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAnalyzersRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListArchiveRulesRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListPolicyGenerationsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.StartPolicyGenerationRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.StartResourceScanRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UpdateArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UpdateFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ValidatePolicyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AccessAnalyzerAsyncClient}.
 *
 * @see AccessAnalyzerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAccessAnalyzerAsyncClient implements AccessAnalyzerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAccessAnalyzerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAccessAnalyzerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retroactively applies the archive rule to existing findings that meet the archive rule criteria.
     * </p>
     *
     * @param applyArchiveRuleRequest
     *        Retroactively applies an archive rule.
     * @return A Java Future containing the result of the ApplyArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ApplyArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ApplyArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ApplyArchiveRuleResponse> applyArchiveRule(ApplyArchiveRuleRequest applyArchiveRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, applyArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ApplyArchiveRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ApplyArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ApplyArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ApplyArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ApplyArchiveRuleRequest, ApplyArchiveRuleResponse>()
                            .withOperationName("ApplyArchiveRule")
                            .withMarshaller(new ApplyArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(applyArchiveRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = applyArchiveRuleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ApplyArchiveRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the requested policy generation.
     * </p>
     *
     * @param cancelPolicyGenerationRequest
     * @return A Java Future containing the result of the CancelPolicyGeneration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.CancelPolicyGeneration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CancelPolicyGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelPolicyGenerationResponse> cancelPolicyGeneration(
            CancelPolicyGenerationRequest cancelPolicyGenerationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelPolicyGenerationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelPolicyGeneration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelPolicyGenerationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelPolicyGenerationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelPolicyGenerationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelPolicyGenerationRequest, CancelPolicyGenerationResponse>()
                            .withOperationName("CancelPolicyGeneration")
                            .withMarshaller(new CancelPolicyGenerationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelPolicyGenerationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = cancelPolicyGenerationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CancelPolicyGenerationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an access preview that allows you to preview Access Analyzer findings for your resource before deploying
     * resource permissions.
     * </p>
     *
     * @param createAccessPreviewRequest
     * @return A Java Future containing the result of the CreateAccessPreview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ServiceQuotaExceededException Service quote met error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.CreateAccessPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAccessPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessPreviewResponse> createAccessPreview(
            CreateAccessPreviewRequest createAccessPreviewRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessPreview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAccessPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAccessPreviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAccessPreviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessPreviewRequest, CreateAccessPreviewResponse>()
                            .withOperationName("CreateAccessPreview")
                            .withMarshaller(new CreateAccessPreviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAccessPreviewRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAccessPreviewRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateAccessPreviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an analyzer for your account.
     * </p>
     *
     * @param createAnalyzerRequest
     *        Creates an analyzer.
     * @return A Java Future containing the result of the CreateAnalyzer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ServiceQuotaExceededException Service quote met error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.CreateAnalyzer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAnalyzerResponse> createAnalyzer(CreateAnalyzerRequest createAnalyzerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAnalyzerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAnalyzer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAnalyzerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAnalyzerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAnalyzerRequest, CreateAnalyzerResponse>()
                            .withOperationName("CreateAnalyzer")
                            .withMarshaller(new CreateAnalyzerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAnalyzerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAnalyzerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateAnalyzerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an archive rule for the specified analyzer. Archive rules automatically archive new findings that meet
     * the criteria you define when you create the rule.
     * </p>
     * <p>
     * To learn about filter keys that you can use to create an archive rule, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     *
     * @param createArchiveRuleRequest
     *        Creates an archive rule.
     * @return A Java Future containing the result of the CreateArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ServiceQuotaExceededException Service quote met error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.CreateArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateArchiveRuleResponse> createArchiveRule(CreateArchiveRuleRequest createArchiveRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateArchiveRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateArchiveRuleRequest, CreateArchiveRuleResponse>()
                            .withOperationName("CreateArchiveRule")
                            .withMarshaller(new CreateArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createArchiveRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createArchiveRuleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateArchiveRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified analyzer. When you delete an analyzer, Access Analyzer is disabled for the account or
     * organization in the current or specific Region. All findings that were generated by the analyzer are deleted. You
     * cannot undo this action.
     * </p>
     *
     * @param deleteAnalyzerRequest
     *        Deletes an analyzer.
     * @return A Java Future containing the result of the DeleteAnalyzer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.DeleteAnalyzer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAnalyzerResponse> deleteAnalyzer(DeleteAnalyzerRequest deleteAnalyzerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAnalyzerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAnalyzer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAnalyzerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAnalyzerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAnalyzerRequest, DeleteAnalyzerResponse>()
                            .withOperationName("DeleteAnalyzer")
                            .withMarshaller(new DeleteAnalyzerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAnalyzerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAnalyzerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAnalyzerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified archive rule.
     * </p>
     *
     * @param deleteArchiveRuleRequest
     *        Deletes an archive rule.
     * @return A Java Future containing the result of the DeleteArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.DeleteArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteArchiveRuleResponse> deleteArchiveRule(DeleteArchiveRuleRequest deleteArchiveRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteArchiveRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteArchiveRuleRequest, DeleteArchiveRuleResponse>()
                            .withOperationName("DeleteArchiveRule")
                            .withMarshaller(new DeleteArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteArchiveRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteArchiveRuleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteArchiveRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an access preview for the specified analyzer.
     * </p>
     *
     * @param getAccessPreviewRequest
     * @return A Java Future containing the result of the GetAccessPreview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetAccessPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAccessPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPreviewResponse> getAccessPreview(GetAccessPreviewRequest getAccessPreviewRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPreview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccessPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccessPreviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccessPreviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPreviewRequest, GetAccessPreviewResponse>()
                            .withOperationName("GetAccessPreview")
                            .withMarshaller(new GetAccessPreviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessPreviewRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPreviewRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAccessPreviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a resource that was analyzed.
     * </p>
     *
     * @param getAnalyzedResourceRequest
     *        Retrieves an analyzed resource.
     * @return A Java Future containing the result of the GetAnalyzedResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetAnalyzedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAnalyzedResourceResponse> getAnalyzedResource(
            GetAnalyzedResourceRequest getAnalyzedResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnalyzedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnalyzedResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAnalyzedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAnalyzedResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAnalyzedResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAnalyzedResourceRequest, GetAnalyzedResourceResponse>()
                            .withOperationName("GetAnalyzedResource")
                            .withMarshaller(new GetAnalyzedResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAnalyzedResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAnalyzedResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetAnalyzedResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified analyzer.
     * </p>
     *
     * @param getAnalyzerRequest
     *        Retrieves an analyzer.
     * @return A Java Future containing the result of the GetAnalyzer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetAnalyzer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAnalyzerResponse> getAnalyzer(GetAnalyzerRequest getAnalyzerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnalyzerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnalyzer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAnalyzerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAnalyzerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAnalyzerRequest, GetAnalyzerResponse>()
                            .withOperationName("GetAnalyzer").withMarshaller(new GetAnalyzerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAnalyzerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAnalyzerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAnalyzerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an archive rule.
     * </p>
     * <p>
     * To learn about filter keys that you can use to create an archive rule, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     *
     * @param getArchiveRuleRequest
     *        Retrieves an archive rule.
     * @return A Java Future containing the result of the GetArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetArchiveRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetArchiveRuleResponse> getArchiveRule(GetArchiveRuleRequest getArchiveRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArchiveRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetArchiveRuleRequest, GetArchiveRuleResponse>()
                            .withOperationName("GetArchiveRule")
                            .withMarshaller(new GetArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getArchiveRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getArchiveRuleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetArchiveRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified finding.
     * </p>
     *
     * @param getFindingRequest
     *        Retrieves a finding.
     * @return A Java Future containing the result of the GetFinding operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetFinding
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetFinding" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFindingResponse> getFinding(GetFindingRequest getFindingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFindingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFinding");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFindingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFindingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFindingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFindingRequest, GetFindingResponse>().withOperationName("GetFinding")
                            .withMarshaller(new GetFindingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFindingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getFindingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetFindingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the policy that was generated using <code>StartPolicyGeneration</code>.
     * </p>
     *
     * @param getGeneratedPolicyRequest
     * @return A Java Future containing the result of the GetGeneratedPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetGeneratedPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetGeneratedPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGeneratedPolicyResponse> getGeneratedPolicy(GetGeneratedPolicyRequest getGeneratedPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeneratedPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeneratedPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGeneratedPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGeneratedPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGeneratedPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGeneratedPolicyRequest, GetGeneratedPolicyResponse>()
                            .withOperationName("GetGeneratedPolicy")
                            .withMarshaller(new GetGeneratedPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGeneratedPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getGeneratedPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetGeneratedPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of access preview findings generated by the specified access preview.
     * </p>
     *
     * @param listAccessPreviewFindingsRequest
     * @return A Java Future containing the result of the ListAccessPreviewFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAccessPreviewFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviewFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPreviewFindingsResponse> listAccessPreviewFindings(
            ListAccessPreviewFindingsRequest listAccessPreviewFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPreviewFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPreviewFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessPreviewFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccessPreviewFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessPreviewFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPreviewFindingsRequest, ListAccessPreviewFindingsResponse>()
                            .withOperationName("ListAccessPreviewFindings")
                            .withMarshaller(new ListAccessPreviewFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccessPreviewFindingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAccessPreviewFindingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListAccessPreviewFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of access preview findings generated by the specified access preview.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPreviewFindings(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsPublisher publisher = client.listAccessPreviewFindingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsPublisher publisher = client.listAccessPreviewFindingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPreviewFindings(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccessPreviewFindingsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAccessPreviewFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviewFindings"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAccessPreviewFindingsPublisher listAccessPreviewFindingsPaginator(
            ListAccessPreviewFindingsRequest listAccessPreviewFindingsRequest) {
        return new ListAccessPreviewFindingsPublisher(this, applyPaginatorUserAgent(listAccessPreviewFindingsRequest));
    }

    /**
     * <p>
     * Retrieves a list of access previews for the specified analyzer.
     * </p>
     *
     * @param listAccessPreviewsRequest
     * @return A Java Future containing the result of the ListAccessPreviews operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAccessPreviews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPreviewsResponse> listAccessPreviews(ListAccessPreviewsRequest listAccessPreviewsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPreviewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPreviews");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessPreviewsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccessPreviewsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessPreviewsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPreviewsRequest, ListAccessPreviewsResponse>()
                            .withOperationName("ListAccessPreviews")
                            .withMarshaller(new ListAccessPreviewsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccessPreviewsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAccessPreviewsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListAccessPreviewsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of access previews for the specified analyzer.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPreviews(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsPublisher publisher = client.listAccessPreviewsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsPublisher publisher = client.listAccessPreviewsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPreviews(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccessPreviewsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAccessPreviews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviews"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAccessPreviewsPublisher listAccessPreviewsPaginator(ListAccessPreviewsRequest listAccessPreviewsRequest) {
        return new ListAccessPreviewsPublisher(this, applyPaginatorUserAgent(listAccessPreviewsRequest));
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     *
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return A Java Future containing the result of the ListAnalyzedResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnalyzedResourcesResponse> listAnalyzedResources(
            ListAnalyzedResourcesRequest listAnalyzedResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnalyzedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnalyzedResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnalyzedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnalyzedResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnalyzedResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnalyzedResourcesRequest, ListAnalyzedResourcesResponse>()
                            .withOperationName("ListAnalyzedResources")
                            .withMarshaller(new ListAnalyzedResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnalyzedResourcesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAnalyzedResourcesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListAnalyzedResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnalyzedResources(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesPublisher publisher = client.listAnalyzedResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesPublisher publisher = client.listAnalyzedResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnalyzedResources(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnalyzedResourcesPublisher listAnalyzedResourcesPaginator(ListAnalyzedResourcesRequest listAnalyzedResourcesRequest) {
        return new ListAnalyzedResourcesPublisher(this, applyPaginatorUserAgent(listAnalyzedResourcesRequest));
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     *
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return A Java Future containing the result of the ListAnalyzers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnalyzersResponse> listAnalyzers(ListAnalyzersRequest listAnalyzersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnalyzersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnalyzers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnalyzersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAnalyzersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnalyzersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnalyzersRequest, ListAnalyzersResponse>()
                            .withOperationName("ListAnalyzers")
                            .withMarshaller(new ListAnalyzersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnalyzersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAnalyzersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListAnalyzersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnalyzers(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersPublisher publisher = client.listAnalyzersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersPublisher publisher = client.listAnalyzersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnalyzers(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest)} operation.</b>
     * </p>
     *
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListAnalyzersPublisher listAnalyzersPaginator(ListAnalyzersRequest listAnalyzersRequest) {
        return new ListAnalyzersPublisher(this, applyPaginatorUserAgent(listAnalyzersRequest));
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     *
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return A Java Future containing the result of the ListArchiveRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListArchiveRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListArchiveRulesResponse> listArchiveRules(ListArchiveRulesRequest listArchiveRulesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listArchiveRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListArchiveRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListArchiveRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListArchiveRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListArchiveRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListArchiveRulesRequest, ListArchiveRulesResponse>()
                            .withOperationName("ListArchiveRules")
                            .withMarshaller(new ListArchiveRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listArchiveRulesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listArchiveRulesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListArchiveRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listArchiveRules(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesPublisher publisher = client.listArchiveRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesPublisher publisher = client.listArchiveRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listArchiveRules(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListArchiveRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    public ListArchiveRulesPublisher listArchiveRulesPaginator(ListArchiveRulesRequest listArchiveRulesRequest) {
        return new ListArchiveRulesPublisher(this, applyPaginatorUserAgent(listArchiveRulesRequest));
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <p>
     * To learn about filter keys that you can use to retrieve a list of findings, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     *
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return A Java Future containing the result of the ListFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFindingsResponse> listFindings(ListFindingsRequest listFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFindingsRequest, ListFindingsResponse>()
                            .withOperationName("ListFindings").withMarshaller(new ListFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFindingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFindingsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <p>
     * To learn about filter keys that you can use to retrieve a list of findings, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFindings(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsPublisher publisher = client.listFindingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsPublisher publisher = client.listFindingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFindings(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest)} operation.</b>
     * </p>
     *
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    public ListFindingsPublisher listFindingsPaginator(ListFindingsRequest listFindingsRequest) {
        return new ListFindingsPublisher(this, applyPaginatorUserAgent(listFindingsRequest));
    }

    /**
     * <p>
     * Lists all of the policy generations requested in the last seven days.
     * </p>
     *
     * @param listPolicyGenerationsRequest
     * @return A Java Future containing the result of the ListPolicyGenerations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListPolicyGenerations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListPolicyGenerations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPolicyGenerationsResponse> listPolicyGenerations(
            ListPolicyGenerationsRequest listPolicyGenerationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPolicyGenerationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicyGenerations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPolicyGenerationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPolicyGenerationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPolicyGenerationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPolicyGenerationsRequest, ListPolicyGenerationsResponse>()
                            .withOperationName("ListPolicyGenerations")
                            .withMarshaller(new ListPolicyGenerationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPolicyGenerationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPolicyGenerationsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListPolicyGenerationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the policy generations requested in the last seven days.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPolicyGenerations(software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsPublisher publisher = client.listPolicyGenerationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsPublisher publisher = client.listPolicyGenerationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPolicyGenerations(software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPolicyGenerationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListPolicyGenerations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListPolicyGenerations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPolicyGenerationsPublisher listPolicyGenerationsPaginator(ListPolicyGenerationsRequest listPolicyGenerationsRequest) {
        return new ListPolicyGenerationsPublisher(this, applyPaginatorUserAgent(listPolicyGenerationsRequest));
    }

    /**
     * <p>
     * Retrieves a list of tags applied to the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Retrieves a list of tags applied to the specified resource.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the policy generation request.
     * </p>
     *
     * @param startPolicyGenerationRequest
     * @return A Java Future containing the result of the StartPolicyGeneration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ServiceQuotaExceededException Service quote met error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.StartPolicyGeneration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartPolicyGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPolicyGenerationResponse> startPolicyGeneration(
            StartPolicyGenerationRequest startPolicyGenerationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPolicyGenerationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPolicyGeneration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPolicyGenerationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPolicyGenerationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPolicyGenerationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPolicyGenerationRequest, StartPolicyGenerationResponse>()
                            .withOperationName("StartPolicyGeneration")
                            .withMarshaller(new StartPolicyGenerationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startPolicyGenerationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startPolicyGenerationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<StartPolicyGenerationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Immediately starts a scan of the policies applied to the specified resource.
     * </p>
     *
     * @param startResourceScanRequest
     *        Starts a scan of the policies applied to the specified resource.
     * @return A Java Future containing the result of the StartResourceScan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.StartResourceScan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartResourceScan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartResourceScanResponse> startResourceScan(StartResourceScanRequest startResourceScanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startResourceScanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartResourceScan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartResourceScanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartResourceScanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartResourceScanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartResourceScanRequest, StartResourceScanResponse>()
                            .withOperationName("StartResourceScan")
                            .withMarshaller(new StartResourceScanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startResourceScanRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startResourceScanRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartResourceScanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a tag to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     *        Adds a tag to the specified resource.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     *        Removes a tag from the specified resource.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the criteria and values for the specified archive rule.
     * </p>
     *
     * @param updateArchiveRuleRequest
     *        Updates the specified archive rule.
     * @return A Java Future containing the result of the UpdateArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.UpdateArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateArchiveRuleResponse> updateArchiveRule(UpdateArchiveRuleRequest updateArchiveRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateArchiveRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateArchiveRuleRequest, UpdateArchiveRuleResponse>()
                            .withOperationName("UpdateArchiveRule")
                            .withMarshaller(new UpdateArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateArchiveRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateArchiveRuleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateArchiveRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status for the specified findings.
     * </p>
     *
     * @param updateFindingsRequest
     *        Updates findings with the new values provided in the request.
     * @return A Java Future containing the result of the UpdateFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.UpdateFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFindingsResponse> updateFindings(UpdateFindingsRequest updateFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFindingsRequest, UpdateFindingsResponse>()
                            .withOperationName("UpdateFindings")
                            .withMarshaller(new UpdateFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFindingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFindingsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests the validation of a policy and returns a list of findings. The findings help you identify issues and
     * provide actionable recommendations to resolve the issue and enable you to author functional policies that meet
     * security best practices.
     * </p>
     *
     * @param validatePolicyRequest
     * @return A Java Future containing the result of the ValidatePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ValidatePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ValidatePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidatePolicyResponse> validatePolicy(ValidatePolicyRequest validatePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidatePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidatePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidatePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidatePolicyRequest, ValidatePolicyResponse>()
                            .withOperationName("ValidatePolicy")
                            .withMarshaller(new ValidatePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(validatePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = validatePolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ValidatePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests the validation of a policy and returns a list of findings. The findings help you identify issues and
     * provide actionable recommendations to resolve the issue and enable you to author functional policies that meet
     * security best practices.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #validatePolicy(software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyPublisher publisher = client.validatePolicyPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyPublisher publisher = client.validatePolicyPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #validatePolicy(software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyRequest)}
     * operation.</b>
     * </p>
     *
     * @param validatePolicyRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ValidatePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ValidatePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    public ValidatePolicyPublisher validatePolicyPaginator(ValidatePolicyRequest validatePolicyRequest) {
        return new ValidatePolicyPublisher(this, applyPaginatorUserAgent(validatePolicyRequest));
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AccessAnalyzerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends AccessAnalyzerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
