/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for an Amazon S3 access point for the bucket. You can propose up to 10 access points per bucket. If
 * the proposed Amazon S3 access point configuration is for an existing bucket, the access preview uses the proposed
 * access point configuration in place of the existing access points. To propose an access point without a policy, you
 * can provide an empty string as the access point policy. For more information, see <a
 * href="https://docs.aws.amazon.com/https:/docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html"
 * >Creating access points</a>. For more information about access point policy limits, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points-restrictions-limitations.html">Access points
 * restrictions and limitations</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3AccessPointConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3AccessPointConfiguration.Builder, S3AccessPointConfiguration> {
    private static final SdkField<String> ACCESS_POINT_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accessPointPolicy").getter(getter(S3AccessPointConfiguration::accessPointPolicy))
            .setter(setter(Builder::accessPointPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessPointPolicy").build()).build();

    private static final SdkField<NetworkOriginConfiguration> NETWORK_ORIGIN_FIELD = SdkField
            .<NetworkOriginConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkOrigin")
            .getter(getter(S3AccessPointConfiguration::networkOrigin)).setter(setter(Builder::networkOrigin))
            .constructor(NetworkOriginConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkOrigin").build()).build();

    private static final SdkField<S3PublicAccessBlockConfiguration> PUBLIC_ACCESS_BLOCK_FIELD = SdkField
            .<S3PublicAccessBlockConfiguration> builder(MarshallingType.SDK_POJO).memberName("publicAccessBlock")
            .getter(getter(S3AccessPointConfiguration::publicAccessBlock)).setter(setter(Builder::publicAccessBlock))
            .constructor(S3PublicAccessBlockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicAccessBlock").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCESS_POINT_POLICY_FIELD,
            NETWORK_ORIGIN_FIELD, PUBLIC_ACCESS_BLOCK_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accessPointPolicy;

    private final NetworkOriginConfiguration networkOrigin;

    private final S3PublicAccessBlockConfiguration publicAccessBlock;

    private S3AccessPointConfiguration(BuilderImpl builder) {
        this.accessPointPolicy = builder.accessPointPolicy;
        this.networkOrigin = builder.networkOrigin;
        this.publicAccessBlock = builder.publicAccessBlock;
    }

    /**
     * <p>
     * The access point policy.
     * </p>
     * 
     * @return The access point policy.
     */
    public final String accessPointPolicy() {
        return accessPointPolicy;
    }

    /**
     * <p>
     * The proposed <code>Internet</code> and <code>VpcConfiguration</code> to apply to this Amazon S3 access point. If
     * the access preview is for a new resource and neither is specified, the access preview uses <code>Internet</code>
     * for the network origin. If the access preview is for an existing resource and neither is specified, the access
     * preview uses the exiting network origin.
     * </p>
     * 
     * @return The proposed <code>Internet</code> and <code>VpcConfiguration</code> to apply to this Amazon S3 access
     *         point. If the access preview is for a new resource and neither is specified, the access preview uses
     *         <code>Internet</code> for the network origin. If the access preview is for an existing resource and
     *         neither is specified, the access preview uses the exiting network origin.
     */
    public final NetworkOriginConfiguration networkOrigin() {
        return networkOrigin;
    }

    /**
     * <p>
     * The proposed <code>S3PublicAccessBlock</code> configuration to apply to this Amazon S3 Access Point.
     * </p>
     * 
     * @return The proposed <code>S3PublicAccessBlock</code> configuration to apply to this Amazon S3 Access Point.
     */
    public final S3PublicAccessBlockConfiguration publicAccessBlock() {
        return publicAccessBlock;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accessPointPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(networkOrigin());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccessBlock());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3AccessPointConfiguration)) {
            return false;
        }
        S3AccessPointConfiguration other = (S3AccessPointConfiguration) obj;
        return Objects.equals(accessPointPolicy(), other.accessPointPolicy())
                && Objects.equals(networkOrigin(), other.networkOrigin())
                && Objects.equals(publicAccessBlock(), other.publicAccessBlock());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3AccessPointConfiguration").add("AccessPointPolicy", accessPointPolicy())
                .add("NetworkOrigin", networkOrigin()).add("PublicAccessBlock", publicAccessBlock()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accessPointPolicy":
            return Optional.ofNullable(clazz.cast(accessPointPolicy()));
        case "networkOrigin":
            return Optional.ofNullable(clazz.cast(networkOrigin()));
        case "publicAccessBlock":
            return Optional.ofNullable(clazz.cast(publicAccessBlock()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3AccessPointConfiguration, T> g) {
        return obj -> g.apply((S3AccessPointConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3AccessPointConfiguration> {
        /**
         * <p>
         * The access point policy.
         * </p>
         * 
         * @param accessPointPolicy
         *        The access point policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessPointPolicy(String accessPointPolicy);

        /**
         * <p>
         * The proposed <code>Internet</code> and <code>VpcConfiguration</code> to apply to this Amazon S3 access point.
         * If the access preview is for a new resource and neither is specified, the access preview uses
         * <code>Internet</code> for the network origin. If the access preview is for an existing resource and neither
         * is specified, the access preview uses the exiting network origin.
         * </p>
         * 
         * @param networkOrigin
         *        The proposed <code>Internet</code> and <code>VpcConfiguration</code> to apply to this Amazon S3 access
         *        point. If the access preview is for a new resource and neither is specified, the access preview uses
         *        <code>Internet</code> for the network origin. If the access preview is for an existing resource and
         *        neither is specified, the access preview uses the exiting network origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkOrigin(NetworkOriginConfiguration networkOrigin);

        /**
         * <p>
         * The proposed <code>Internet</code> and <code>VpcConfiguration</code> to apply to this Amazon S3 access point.
         * If the access preview is for a new resource and neither is specified, the access preview uses
         * <code>Internet</code> for the network origin. If the access preview is for an existing resource and neither
         * is specified, the access preview uses the exiting network origin.
         * </p>
         * This is a convenience that creates an instance of the {@link NetworkOriginConfiguration.Builder} avoiding the
         * need to create one manually via {@link NetworkOriginConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkOriginConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #networkOrigin(NetworkOriginConfiguration)}.
         * 
         * @param networkOrigin
         *        a consumer that will call methods on {@link NetworkOriginConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkOrigin(NetworkOriginConfiguration)
         */
        default Builder networkOrigin(Consumer<NetworkOriginConfiguration.Builder> networkOrigin) {
            return networkOrigin(NetworkOriginConfiguration.builder().applyMutation(networkOrigin).build());
        }

        /**
         * <p>
         * The proposed <code>S3PublicAccessBlock</code> configuration to apply to this Amazon S3 Access Point.
         * </p>
         * 
         * @param publicAccessBlock
         *        The proposed <code>S3PublicAccessBlock</code> configuration to apply to this Amazon S3 Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccessBlock(S3PublicAccessBlockConfiguration publicAccessBlock);

        /**
         * <p>
         * The proposed <code>S3PublicAccessBlock</code> configuration to apply to this Amazon S3 Access Point.
         * </p>
         * This is a convenience that creates an instance of the {@link S3PublicAccessBlockConfiguration.Builder}
         * avoiding the need to create one manually via {@link S3PublicAccessBlockConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3PublicAccessBlockConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #publicAccessBlock(S3PublicAccessBlockConfiguration)}.
         * 
         * @param publicAccessBlock
         *        a consumer that will call methods on {@link S3PublicAccessBlockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccessBlock(S3PublicAccessBlockConfiguration)
         */
        default Builder publicAccessBlock(Consumer<S3PublicAccessBlockConfiguration.Builder> publicAccessBlock) {
            return publicAccessBlock(S3PublicAccessBlockConfiguration.builder().applyMutation(publicAccessBlock).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String accessPointPolicy;

        private NetworkOriginConfiguration networkOrigin;

        private S3PublicAccessBlockConfiguration publicAccessBlock;

        private BuilderImpl() {
        }

        private BuilderImpl(S3AccessPointConfiguration model) {
            accessPointPolicy(model.accessPointPolicy);
            networkOrigin(model.networkOrigin);
            publicAccessBlock(model.publicAccessBlock);
        }

        public final String getAccessPointPolicy() {
            return accessPointPolicy;
        }

        @Override
        public final Builder accessPointPolicy(String accessPointPolicy) {
            this.accessPointPolicy = accessPointPolicy;
            return this;
        }

        public final void setAccessPointPolicy(String accessPointPolicy) {
            this.accessPointPolicy = accessPointPolicy;
        }

        public final NetworkOriginConfiguration.Builder getNetworkOrigin() {
            return networkOrigin != null ? networkOrigin.toBuilder() : null;
        }

        @Override
        public final Builder networkOrigin(NetworkOriginConfiguration networkOrigin) {
            this.networkOrigin = networkOrigin;
            return this;
        }

        public final void setNetworkOrigin(NetworkOriginConfiguration.BuilderImpl networkOrigin) {
            this.networkOrigin = networkOrigin != null ? networkOrigin.build() : null;
        }

        public final S3PublicAccessBlockConfiguration.Builder getPublicAccessBlock() {
            return publicAccessBlock != null ? publicAccessBlock.toBuilder() : null;
        }

        @Override
        public final Builder publicAccessBlock(S3PublicAccessBlockConfiguration publicAccessBlock) {
            this.publicAccessBlock = publicAccessBlock;
            return this;
        }

        public final void setPublicAccessBlock(S3PublicAccessBlockConfiguration.BuilderImpl publicAccessBlock) {
            this.publicAccessBlock = publicAccessBlock != null ? publicAccessBlock.build() : null;
        }

        @Override
        public S3AccessPointConfiguration build() {
            return new S3AccessPointConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
