/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A proposed grant configuration for a KMS key. For more information, see <a
 * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html">CreateGrant</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KmsGrantConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<KmsGrantConfiguration.Builder, KmsGrantConfiguration> {
    private static final SdkField<KmsGrantConstraints> CONSTRAINTS_FIELD = SdkField
            .<KmsGrantConstraints> builder(MarshallingType.SDK_POJO).memberName("constraints")
            .getter(getter(KmsGrantConfiguration::constraints)).setter(setter(Builder::constraints))
            .constructor(KmsGrantConstraints::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("constraints").build()).build();

    private static final SdkField<String> GRANTEE_PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("granteePrincipal").getter(getter(KmsGrantConfiguration::granteePrincipal))
            .setter(setter(Builder::granteePrincipal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("granteePrincipal").build()).build();

    private static final SdkField<String> ISSUING_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("issuingAccount").getter(getter(KmsGrantConfiguration::issuingAccount))
            .setter(setter(Builder::issuingAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuingAccount").build()).build();

    private static final SdkField<List<String>> OPERATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("operations")
            .getter(getter(KmsGrantConfiguration::operationsAsStrings))
            .setter(setter(Builder::operationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("operations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RETIRING_PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("retiringPrincipal").getter(getter(KmsGrantConfiguration::retiringPrincipal))
            .setter(setter(Builder::retiringPrincipal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retiringPrincipal").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONSTRAINTS_FIELD,
            GRANTEE_PRINCIPAL_FIELD, ISSUING_ACCOUNT_FIELD, OPERATIONS_FIELD, RETIRING_PRINCIPAL_FIELD));

    private static final long serialVersionUID = 1L;

    private final KmsGrantConstraints constraints;

    private final String granteePrincipal;

    private final String issuingAccount;

    private final List<String> operations;

    private final String retiringPrincipal;

    private KmsGrantConfiguration(BuilderImpl builder) {
        this.constraints = builder.constraints;
        this.granteePrincipal = builder.granteePrincipal;
        this.issuingAccount = builder.issuingAccount;
        this.operations = builder.operations;
        this.retiringPrincipal = builder.retiringPrincipal;
    }

    /**
     * <p>
     * Use this structure to propose allowing <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">cryptographic
     * operations</a> in the grant only when the operation request includes the specified <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">encryption
     * context</a>.
     * </p>
     * 
     * @return Use this structure to propose allowing <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
     *         >cryptographic operations</a> in the grant only when the operation request includes the specified <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">encryption
     *         context</a>.
     */
    public final KmsGrantConstraints constraints() {
        return constraints;
    }

    /**
     * <p>
     * The principal that is given permission to perform the operations that the grant permits.
     * </p>
     * 
     * @return The principal that is given permission to perform the operations that the grant permits.
     */
    public final String granteePrincipal() {
        return granteePrincipal;
    }

    /**
     * <p>
     * The Amazon Web Services account under which the grant was issued. The account is used to propose KMS grants
     * issued by accounts other than the owner of the key.
     * </p>
     * 
     * @return The Amazon Web Services account under which the grant was issued. The account is used to propose KMS
     *         grants issued by accounts other than the owner of the key.
     */
    public final String issuingAccount() {
        return issuingAccount;
    }

    /**
     * <p>
     * A list of operations that the grant permits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOperations} method.
     * </p>
     * 
     * @return A list of operations that the grant permits.
     */
    public final List<KmsGrantOperation> operations() {
        return KmsGrantOperationsListCopier.copyStringToEnum(operations);
    }

    /**
     * For responses, this returns true if the service returned a value for the Operations property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOperations() {
        return operations != null && !(operations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of operations that the grant permits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOperations} method.
     * </p>
     * 
     * @return A list of operations that the grant permits.
     */
    public final List<String> operationsAsStrings() {
        return operations;
    }

    /**
     * <p>
     * The principal that is given permission to retire the grant by using <a
     * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html">RetireGrant</a> operation.
     * </p>
     * 
     * @return The principal that is given permission to retire the grant by using <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html">RetireGrant</a>
     *         operation.
     */
    public final String retiringPrincipal() {
        return retiringPrincipal;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(constraints());
        hashCode = 31 * hashCode + Objects.hashCode(granteePrincipal());
        hashCode = 31 * hashCode + Objects.hashCode(issuingAccount());
        hashCode = 31 * hashCode + Objects.hashCode(hasOperations() ? operationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(retiringPrincipal());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KmsGrantConfiguration)) {
            return false;
        }
        KmsGrantConfiguration other = (KmsGrantConfiguration) obj;
        return Objects.equals(constraints(), other.constraints()) && Objects.equals(granteePrincipal(), other.granteePrincipal())
                && Objects.equals(issuingAccount(), other.issuingAccount()) && hasOperations() == other.hasOperations()
                && Objects.equals(operationsAsStrings(), other.operationsAsStrings())
                && Objects.equals(retiringPrincipal(), other.retiringPrincipal());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KmsGrantConfiguration").add("Constraints", constraints())
                .add("GranteePrincipal", granteePrincipal()).add("IssuingAccount", issuingAccount())
                .add("Operations", hasOperations() ? operationsAsStrings() : null).add("RetiringPrincipal", retiringPrincipal())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "constraints":
            return Optional.ofNullable(clazz.cast(constraints()));
        case "granteePrincipal":
            return Optional.ofNullable(clazz.cast(granteePrincipal()));
        case "issuingAccount":
            return Optional.ofNullable(clazz.cast(issuingAccount()));
        case "operations":
            return Optional.ofNullable(clazz.cast(operationsAsStrings()));
        case "retiringPrincipal":
            return Optional.ofNullable(clazz.cast(retiringPrincipal()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KmsGrantConfiguration, T> g) {
        return obj -> g.apply((KmsGrantConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KmsGrantConfiguration> {
        /**
         * <p>
         * Use this structure to propose allowing <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         * >cryptographic operations</a> in the grant only when the operation request includes the specified <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">encryption
         * context</a>.
         * </p>
         * 
         * @param constraints
         *        Use this structure to propose allowing <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         *        >cryptographic operations</a> in the grant only when the operation request includes the specified <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">encryption
         *        context</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constraints(KmsGrantConstraints constraints);

        /**
         * <p>
         * Use this structure to propose allowing <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         * >cryptographic operations</a> in the grant only when the operation request includes the specified <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">encryption
         * context</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link KmsGrantConstraints.Builder} avoiding the
         * need to create one manually via {@link KmsGrantConstraints#builder()}.
         *
         * When the {@link Consumer} completes, {@link KmsGrantConstraints.Builder#build()} is called immediately and
         * its result is passed to {@link #constraints(KmsGrantConstraints)}.
         * 
         * @param constraints
         *        a consumer that will call methods on {@link KmsGrantConstraints.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #constraints(KmsGrantConstraints)
         */
        default Builder constraints(Consumer<KmsGrantConstraints.Builder> constraints) {
            return constraints(KmsGrantConstraints.builder().applyMutation(constraints).build());
        }

        /**
         * <p>
         * The principal that is given permission to perform the operations that the grant permits.
         * </p>
         * 
         * @param granteePrincipal
         *        The principal that is given permission to perform the operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder granteePrincipal(String granteePrincipal);

        /**
         * <p>
         * The Amazon Web Services account under which the grant was issued. The account is used to propose KMS grants
         * issued by accounts other than the owner of the key.
         * </p>
         * 
         * @param issuingAccount
         *        The Amazon Web Services account under which the grant was issued. The account is used to propose KMS
         *        grants issued by accounts other than the owner of the key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuingAccount(String issuingAccount);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationsWithStrings(Collection<String> operations);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationsWithStrings(String... operations);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operations(Collection<KmsGrantOperation> operations);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operations(KmsGrantOperation... operations);

        /**
         * <p>
         * The principal that is given permission to retire the grant by using <a
         * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html">RetireGrant</a> operation.
         * </p>
         * 
         * @param retiringPrincipal
         *        The principal that is given permission to retire the grant by using <a
         *        href="https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html">RetireGrant</a>
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retiringPrincipal(String retiringPrincipal);
    }

    static final class BuilderImpl implements Builder {
        private KmsGrantConstraints constraints;

        private String granteePrincipal;

        private String issuingAccount;

        private List<String> operations = DefaultSdkAutoConstructList.getInstance();

        private String retiringPrincipal;

        private BuilderImpl() {
        }

        private BuilderImpl(KmsGrantConfiguration model) {
            constraints(model.constraints);
            granteePrincipal(model.granteePrincipal);
            issuingAccount(model.issuingAccount);
            operationsWithStrings(model.operations);
            retiringPrincipal(model.retiringPrincipal);
        }

        public final KmsGrantConstraints.Builder getConstraints() {
            return constraints != null ? constraints.toBuilder() : null;
        }

        public final void setConstraints(KmsGrantConstraints.BuilderImpl constraints) {
            this.constraints = constraints != null ? constraints.build() : null;
        }

        @Override
        public final Builder constraints(KmsGrantConstraints constraints) {
            this.constraints = constraints;
            return this;
        }

        public final String getGranteePrincipal() {
            return granteePrincipal;
        }

        public final void setGranteePrincipal(String granteePrincipal) {
            this.granteePrincipal = granteePrincipal;
        }

        @Override
        public final Builder granteePrincipal(String granteePrincipal) {
            this.granteePrincipal = granteePrincipal;
            return this;
        }

        public final String getIssuingAccount() {
            return issuingAccount;
        }

        public final void setIssuingAccount(String issuingAccount) {
            this.issuingAccount = issuingAccount;
        }

        @Override
        public final Builder issuingAccount(String issuingAccount) {
            this.issuingAccount = issuingAccount;
            return this;
        }

        public final Collection<String> getOperations() {
            if (operations instanceof SdkAutoConstructList) {
                return null;
            }
            return operations;
        }

        public final void setOperations(Collection<String> operations) {
            this.operations = KmsGrantOperationsListCopier.copy(operations);
        }

        @Override
        public final Builder operationsWithStrings(Collection<String> operations) {
            this.operations = KmsGrantOperationsListCopier.copy(operations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operationsWithStrings(String... operations) {
            operationsWithStrings(Arrays.asList(operations));
            return this;
        }

        @Override
        public final Builder operations(Collection<KmsGrantOperation> operations) {
            this.operations = KmsGrantOperationsListCopier.copyEnumToString(operations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operations(KmsGrantOperation... operations) {
            operations(Arrays.asList(operations));
            return this;
        }

        public final String getRetiringPrincipal() {
            return retiringPrincipal;
        }

        public final void setRetiringPrincipal(String retiringPrincipal) {
            this.retiringPrincipal = retiringPrincipal;
        }

        @Override
        public final Builder retiringPrincipal(String retiringPrincipal) {
            this.retiringPrincipal = retiringPrincipal;
            return this;
        }

        @Override
        public KmsGrantConfiguration build() {
            return new KmsGrantConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
