/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The proposed access control configuration for an Amazon EBS volume snapshot. You can propose a configuration for a
 * new Amazon EBS volume snapshot or an Amazon EBS volume snapshot that you own by specifying the user IDs, groups, and
 * optional KMS encryption key. For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html"
 * >ModifySnapshotAttribute</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EbsSnapshotConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EbsSnapshotConfiguration.Builder, EbsSnapshotConfiguration> {
    private static final SdkField<List<String>> USER_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("userIds")
            .getter(getter(EbsSnapshotConfiguration::userIds))
            .setter(setter(Builder::userIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("groups")
            .getter(getter(EbsSnapshotConfiguration::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyId").getter(getter(EbsSnapshotConfiguration::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_IDS_FIELD, GROUPS_FIELD,
            KMS_KEY_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> userIds;

    private final List<String> groups;

    private final String kmsKeyId;

    private EbsSnapshotConfiguration(BuilderImpl builder) {
        this.userIds = builder.userIds;
        this.groups = builder.groups;
        this.kmsKeyId = builder.kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasUserIds() {
        return userIds != null && !(userIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the Amazon Web Services accounts that have access to the Amazon EBS volume snapshot.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
     * <code>userIds</code>, then the access preview uses the existing shared <code>userIds</code> for the snapshot.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the access preview is for a new resource and you do not specify the <code>userIds</code>, then the access
     * preview considers the snapshot without any <code>userIds</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To propose deletion of existing shared <code>accountIds</code>, you can specify an empty list for
     * <code>userIds</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserIds} method.
     * </p>
     * 
     * @return The IDs of the Amazon Web Services accounts that have access to the Amazon EBS volume snapshot.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
     *         <code>userIds</code>, then the access preview uses the existing shared <code>userIds</code> for the
     *         snapshot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the access preview is for a new resource and you do not specify the <code>userIds</code>, then the
     *         access preview considers the snapshot without any <code>userIds</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To propose deletion of existing shared <code>accountIds</code>, you can specify an empty list for
     *         <code>userIds</code>.
     *         </p>
     *         </li>
     */
    public final List<String> userIds() {
        return userIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Groups property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The groups that have access to the Amazon EBS volume snapshot. If the value <code>all</code> is specified, then
     * the Amazon EBS volume snapshot is public.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the <code>groups</code>
     * , then the access preview uses the existing shared <code>groups</code> for the snapshot.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the access preview is for a new resource and you do not specify the <code>groups</code>, then the access
     * preview considers the snapshot without any <code>groups</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To propose deletion of existing shared <code>groups</code>, you can specify an empty list for <code>groups</code>
     * .
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGroups} method.
     * </p>
     * 
     * @return The groups that have access to the Amazon EBS volume snapshot. If the value <code>all</code> is
     *         specified, then the Amazon EBS volume snapshot is public.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
     *         <code>groups</code>, then the access preview uses the existing shared <code>groups</code> for the
     *         snapshot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the access preview is for a new resource and you do not specify the <code>groups</code>, then the
     *         access preview considers the snapshot without any <code>groups</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To propose deletion of existing shared <code>groups</code>, you can specify an empty list for
     *         <code>groups</code>.
     *         </p>
     *         </li>
     */
    public final List<String> groups() {
        return groups;
    }

    /**
     * <p>
     * The KMS key identifier for an encrypted Amazon EBS volume snapshot. The KMS key identifier is the key ARN, key
     * ID, alias ARN, or alias name for the KMS key.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
     * <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
     * <code>kmsKeyId</code> of the snapshot.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the access preview is for a new resource and you do not specify the <code>kmsKeyId</code>, the access preview
     * considers the snapshot as unencrypted.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The KMS key identifier for an encrypted Amazon EBS volume snapshot. The KMS key identifier is the key
     *         ARN, key ID, alias ARN, or alias name for the KMS key.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
     *         <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
     *         <code>kmsKeyId</code> of the snapshot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the access preview is for a new resource and you do not specify the <code>kmsKeyId</code>, the access
     *         preview considers the snapshot as unencrypted.
     *         </p>
     *         </li>
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasUserIds() ? userIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasGroups() ? groups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EbsSnapshotConfiguration)) {
            return false;
        }
        EbsSnapshotConfiguration other = (EbsSnapshotConfiguration) obj;
        return hasUserIds() == other.hasUserIds() && Objects.equals(userIds(), other.userIds())
                && hasGroups() == other.hasGroups() && Objects.equals(groups(), other.groups())
                && Objects.equals(kmsKeyId(), other.kmsKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EbsSnapshotConfiguration").add("UserIds", hasUserIds() ? userIds() : null)
                .add("Groups", hasGroups() ? groups() : null).add("KmsKeyId", kmsKeyId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "userIds":
            return Optional.ofNullable(clazz.cast(userIds()));
        case "groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "kmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EbsSnapshotConfiguration, T> g) {
        return obj -> g.apply((EbsSnapshotConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EbsSnapshotConfiguration> {
        /**
         * <p>
         * The IDs of the Amazon Web Services accounts that have access to the Amazon EBS volume snapshot.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         * <code>userIds</code>, then the access preview uses the existing shared <code>userIds</code> for the snapshot.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the access preview is for a new resource and you do not specify the <code>userIds</code>, then the access
         * preview considers the snapshot without any <code>userIds</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To propose deletion of existing shared <code>accountIds</code>, you can specify an empty list for
         * <code>userIds</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param userIds
         *        The IDs of the Amazon Web Services accounts that have access to the Amazon EBS volume snapshot.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         *        <code>userIds</code>, then the access preview uses the existing shared <code>userIds</code> for the
         *        snapshot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the access preview is for a new resource and you do not specify the <code>userIds</code>, then the
         *        access preview considers the snapshot without any <code>userIds</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To propose deletion of existing shared <code>accountIds</code>, you can specify an empty list for
         *        <code>userIds</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userIds(Collection<String> userIds);

        /**
         * <p>
         * The IDs of the Amazon Web Services accounts that have access to the Amazon EBS volume snapshot.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         * <code>userIds</code>, then the access preview uses the existing shared <code>userIds</code> for the snapshot.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the access preview is for a new resource and you do not specify the <code>userIds</code>, then the access
         * preview considers the snapshot without any <code>userIds</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To propose deletion of existing shared <code>accountIds</code>, you can specify an empty list for
         * <code>userIds</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param userIds
         *        The IDs of the Amazon Web Services accounts that have access to the Amazon EBS volume snapshot.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         *        <code>userIds</code>, then the access preview uses the existing shared <code>userIds</code> for the
         *        snapshot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the access preview is for a new resource and you do not specify the <code>userIds</code>, then the
         *        access preview considers the snapshot without any <code>userIds</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To propose deletion of existing shared <code>accountIds</code>, you can specify an empty list for
         *        <code>userIds</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userIds(String... userIds);

        /**
         * <p>
         * The groups that have access to the Amazon EBS volume snapshot. If the value <code>all</code> is specified,
         * then the Amazon EBS volume snapshot is public.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         * <code>groups</code>, then the access preview uses the existing shared <code>groups</code> for the snapshot.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the access preview is for a new resource and you do not specify the <code>groups</code>, then the access
         * preview considers the snapshot without any <code>groups</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To propose deletion of existing shared <code>groups</code>, you can specify an empty list for
         * <code>groups</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param groups
         *        The groups that have access to the Amazon EBS volume snapshot. If the value <code>all</code> is
         *        specified, then the Amazon EBS volume snapshot is public.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         *        <code>groups</code>, then the access preview uses the existing shared <code>groups</code> for the
         *        snapshot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the access preview is for a new resource and you do not specify the <code>groups</code>, then the
         *        access preview considers the snapshot without any <code>groups</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To propose deletion of existing shared <code>groups</code>, you can specify an empty list for
         *        <code>groups</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * The groups that have access to the Amazon EBS volume snapshot. If the value <code>all</code> is specified,
         * then the Amazon EBS volume snapshot is public.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         * <code>groups</code>, then the access preview uses the existing shared <code>groups</code> for the snapshot.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the access preview is for a new resource and you do not specify the <code>groups</code>, then the access
         * preview considers the snapshot without any <code>groups</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To propose deletion of existing shared <code>groups</code>, you can specify an empty list for
         * <code>groups</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param groups
         *        The groups that have access to the Amazon EBS volume snapshot. If the value <code>all</code> is
         *        specified, then the Amazon EBS volume snapshot is public.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         *        <code>groups</code>, then the access preview uses the existing shared <code>groups</code> for the
         *        snapshot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the access preview is for a new resource and you do not specify the <code>groups</code>, then the
         *        access preview considers the snapshot without any <code>groups</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To propose deletion of existing shared <code>groups</code>, you can specify an empty list for
         *        <code>groups</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * The KMS key identifier for an encrypted Amazon EBS volume snapshot. The KMS key identifier is the key ARN,
         * key ID, alias ARN, or alias name for the KMS key.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         * <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
         * <code>kmsKeyId</code> of the snapshot.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the access preview is for a new resource and you do not specify the <code>kmsKeyId</code>, the access
         * preview considers the snapshot as unencrypted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        The KMS key identifier for an encrypted Amazon EBS volume snapshot. The KMS key identifier is the key
         *        ARN, key ID, alias ARN, or alias name for the KMS key.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the configuration is for an existing Amazon EBS volume snapshot and you do not specify the
         *        <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
         *        <code>kmsKeyId</code> of the snapshot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the access preview is for a new resource and you do not specify the <code>kmsKeyId</code>, the
         *        access preview considers the snapshot as unencrypted.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private List<String> userIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> groups = DefaultSdkAutoConstructList.getInstance();

        private String kmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(EbsSnapshotConfiguration model) {
            userIds(model.userIds);
            groups(model.groups);
            kmsKeyId(model.kmsKeyId);
        }

        public final Collection<String> getUserIds() {
            if (userIds instanceof SdkAutoConstructList) {
                return null;
            }
            return userIds;
        }

        public final void setUserIds(Collection<String> userIds) {
            this.userIds = EbsUserIdListCopier.copy(userIds);
        }

        @Override
        public final Builder userIds(Collection<String> userIds) {
            this.userIds = EbsUserIdListCopier.copy(userIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userIds(String... userIds) {
            userIds(Arrays.asList(userIds));
            return this;
        }

        public final Collection<String> getGroups() {
            if (groups instanceof SdkAutoConstructList) {
                return null;
            }
            return groups;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = EbsGroupListCopier.copy(groups);
        }

        @Override
        public final Builder groups(Collection<String> groups) {
            this.groups = EbsGroupListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        @Override
        public EbsSnapshotConfiguration build() {
            return new EbsSnapshotConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
