/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.accessanalyzer.model.AccessAnalyzerException;
import software.amazon.awssdk.services.accessanalyzer.model.AccessAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.AccessDeniedException;
import software.amazon.awssdk.services.accessanalyzer.model.ApplyArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ApplyArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CancelPolicyGenerationRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CancelPolicyGenerationResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ConflictException;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAccessPreviewRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAccessPreviewResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CreateArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAccessPreviewRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAccessPreviewResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzedResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzedResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetFindingRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetFindingResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetGeneratedPolicyRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetGeneratedPolicyResponse;
import software.amazon.awssdk.services.accessanalyzer.model.InternalServerException;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.accessanalyzer.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.accessanalyzer.model.StartPolicyGenerationRequest;
import software.amazon.awssdk.services.accessanalyzer.model.StartPolicyGenerationResponse;
import software.amazon.awssdk.services.accessanalyzer.model.StartResourceScanRequest;
import software.amazon.awssdk.services.accessanalyzer.model.StartResourceScanResponse;
import software.amazon.awssdk.services.accessanalyzer.model.TagResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.TagResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ThrottlingException;
import software.amazon.awssdk.services.accessanalyzer.model.UntagResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UntagResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ValidationException;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsIterable;
import software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyIterable;
import software.amazon.awssdk.services.accessanalyzer.transform.ApplyArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CancelPolicyGenerationRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateAccessPreviewRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.DeleteAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.DeleteArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAccessPreviewRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAnalyzedResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetFindingRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetGeneratedPolicyRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAccessPreviewFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAccessPreviewsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAnalyzedResourcesRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAnalyzersRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListArchiveRulesRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListPolicyGenerationsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.StartPolicyGenerationRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.StartResourceScanRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UpdateArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UpdateFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ValidatePolicyRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AccessAnalyzerClient}.
 *
 * @see AccessAnalyzerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAccessAnalyzerClient implements AccessAnalyzerClient {
    private static final Logger log = Logger.loggerFor(DefaultAccessAnalyzerClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AccessAnalyzerServiceClientConfiguration serviceClientConfiguration;

    protected DefaultAccessAnalyzerClient(AccessAnalyzerServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retroactively applies the archive rule to existing findings that meet the archive rule criteria.
     * </p>
     *
     * @param applyArchiveRuleRequest
     *        Retroactively applies an archive rule.
     * @return Result of the ApplyArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ApplyArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ApplyArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ApplyArchiveRuleResponse applyArchiveRule(ApplyArchiveRuleRequest applyArchiveRuleRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ApplyArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ApplyArchiveRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, applyArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ApplyArchiveRule");

            return clientHandler.execute(new ClientExecutionParams<ApplyArchiveRuleRequest, ApplyArchiveRuleResponse>()
                    .withOperationName("ApplyArchiveRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(applyArchiveRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ApplyArchiveRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels the requested policy generation.
     * </p>
     *
     * @param cancelPolicyGenerationRequest
     * @return Result of the CancelPolicyGeneration operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.CancelPolicyGeneration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CancelPolicyGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelPolicyGenerationResponse cancelPolicyGeneration(CancelPolicyGenerationRequest cancelPolicyGenerationRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelPolicyGenerationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelPolicyGenerationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelPolicyGenerationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelPolicyGeneration");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelPolicyGenerationRequest, CancelPolicyGenerationResponse>()
                            .withOperationName("CancelPolicyGeneration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(cancelPolicyGenerationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelPolicyGenerationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an access preview that allows you to preview IAM Access Analyzer findings for your resource before
     * deploying resource permissions.
     * </p>
     *
     * @param createAccessPreviewRequest
     * @return Result of the CreateAccessPreview operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.CreateAccessPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAccessPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAccessPreviewResponse createAccessPreview(CreateAccessPreviewRequest createAccessPreviewRequest)
            throws ResourceNotFoundException, ConflictException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAccessPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAccessPreviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessPreview");

            return clientHandler.execute(new ClientExecutionParams<CreateAccessPreviewRequest, CreateAccessPreviewResponse>()
                    .withOperationName("CreateAccessPreview").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAccessPreviewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAccessPreviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an analyzer for your account.
     * </p>
     *
     * @param createAnalyzerRequest
     *        Creates an analyzer.
     * @return Result of the CreateAnalyzer operation returned by the service.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.CreateAnalyzer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAnalyzerResponse createAnalyzer(CreateAnalyzerRequest createAnalyzerRequest) throws ConflictException,
            ValidationException, InternalServerException, ServiceQuotaExceededException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAnalyzerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAnalyzerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAnalyzer");

            return clientHandler.execute(new ClientExecutionParams<CreateAnalyzerRequest, CreateAnalyzerResponse>()
                    .withOperationName("CreateAnalyzer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAnalyzerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAnalyzerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an archive rule for the specified analyzer. Archive rules automatically archive new findings that meet
     * the criteria you define when you create the rule.
     * </p>
     * <p>
     * To learn about filter keys that you can use to create an archive rule, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     *
     * @param createArchiveRuleRequest
     *        Creates an archive rule.
     * @return Result of the CreateArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.CreateArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateArchiveRuleResponse createArchiveRule(CreateArchiveRuleRequest createArchiveRuleRequest)
            throws ResourceNotFoundException, ConflictException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateArchiveRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateArchiveRule");

            return clientHandler.execute(new ClientExecutionParams<CreateArchiveRuleRequest, CreateArchiveRuleResponse>()
                    .withOperationName("CreateArchiveRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createArchiveRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateArchiveRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified analyzer. When you delete an analyzer, IAM Access Analyzer is disabled for the account or
     * organization in the current or specific Region. All findings that were generated by the analyzer are deleted. You
     * cannot undo this action.
     * </p>
     *
     * @param deleteAnalyzerRequest
     *        Deletes an analyzer.
     * @return Result of the DeleteAnalyzer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.DeleteAnalyzer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAnalyzerResponse deleteAnalyzer(DeleteAnalyzerRequest deleteAnalyzerRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAnalyzerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAnalyzerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAnalyzer");

            return clientHandler.execute(new ClientExecutionParams<DeleteAnalyzerRequest, DeleteAnalyzerResponse>()
                    .withOperationName("DeleteAnalyzer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAnalyzerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAnalyzerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified archive rule.
     * </p>
     *
     * @param deleteArchiveRuleRequest
     *        Deletes an archive rule.
     * @return Result of the DeleteArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.DeleteArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteArchiveRuleResponse deleteArchiveRule(DeleteArchiveRuleRequest deleteArchiveRuleRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteArchiveRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteArchiveRule");

            return clientHandler.execute(new ClientExecutionParams<DeleteArchiveRuleRequest, DeleteArchiveRuleResponse>()
                    .withOperationName("DeleteArchiveRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteArchiveRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteArchiveRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an access preview for the specified analyzer.
     * </p>
     *
     * @param getAccessPreviewRequest
     * @return Result of the GetAccessPreview operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.GetAccessPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAccessPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAccessPreviewResponse getAccessPreview(GetAccessPreviewRequest getAccessPreviewRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccessPreviewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAccessPreviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPreview");

            return clientHandler.execute(new ClientExecutionParams<GetAccessPreviewRequest, GetAccessPreviewResponse>()
                    .withOperationName("GetAccessPreview").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAccessPreviewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccessPreviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a resource that was analyzed.
     * </p>
     *
     * @param getAnalyzedResourceRequest
     *        Retrieves an analyzed resource.
     * @return Result of the GetAnalyzedResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.GetAnalyzedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAnalyzedResourceResponse getAnalyzedResource(GetAnalyzedResourceRequest getAnalyzedResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAnalyzedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAnalyzedResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnalyzedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnalyzedResource");

            return clientHandler.execute(new ClientExecutionParams<GetAnalyzedResourceRequest, GetAnalyzedResourceResponse>()
                    .withOperationName("GetAnalyzedResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAnalyzedResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAnalyzedResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified analyzer.
     * </p>
     *
     * @param getAnalyzerRequest
     *        Retrieves an analyzer.
     * @return Result of the GetAnalyzer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.GetAnalyzer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAnalyzerResponse getAnalyzer(GetAnalyzerRequest getAnalyzerRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAnalyzerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnalyzerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnalyzer");

            return clientHandler.execute(new ClientExecutionParams<GetAnalyzerRequest, GetAnalyzerResponse>()
                    .withOperationName("GetAnalyzer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAnalyzerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAnalyzerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an archive rule.
     * </p>
     * <p>
     * To learn about filter keys that you can use to create an archive rule, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     *
     * @param getArchiveRuleRequest
     *        Retrieves an archive rule.
     * @return Result of the GetArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.GetArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetArchiveRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetArchiveRuleResponse getArchiveRule(GetArchiveRuleRequest getArchiveRuleRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetArchiveRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArchiveRule");

            return clientHandler.execute(new ClientExecutionParams<GetArchiveRuleRequest, GetArchiveRuleResponse>()
                    .withOperationName("GetArchiveRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getArchiveRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetArchiveRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified finding.
     * </p>
     *
     * @param getFindingRequest
     *        Retrieves a finding.
     * @return Result of the GetFinding operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.GetFinding
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetFinding" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFindingResponse getFinding(GetFindingRequest getFindingRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFindingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFindingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFindingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFinding");

            return clientHandler
                    .execute(new ClientExecutionParams<GetFindingRequest, GetFindingResponse>().withOperationName("GetFinding")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFindingRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetFindingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the policy that was generated using <code>StartPolicyGeneration</code>.
     * </p>
     *
     * @param getGeneratedPolicyRequest
     * @return Result of the GetGeneratedPolicy operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.GetGeneratedPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetGeneratedPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGeneratedPolicyResponse getGeneratedPolicy(GetGeneratedPolicyRequest getGeneratedPolicyRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGeneratedPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetGeneratedPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeneratedPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeneratedPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetGeneratedPolicyRequest, GetGeneratedPolicyResponse>()
                    .withOperationName("GetGeneratedPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGeneratedPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGeneratedPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of access preview findings generated by the specified access preview.
     * </p>
     *
     * @param listAccessPreviewFindingsRequest
     * @return Result of the ListAccessPreviewFindings operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAccessPreviewFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviewFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccessPreviewFindingsResponse listAccessPreviewFindings(
            ListAccessPreviewFindingsRequest listAccessPreviewFindingsRequest) throws ResourceNotFoundException,
            ConflictException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccessPreviewFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAccessPreviewFindingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPreviewFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPreviewFindings");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAccessPreviewFindingsRequest, ListAccessPreviewFindingsResponse>()
                            .withOperationName("ListAccessPreviewFindings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAccessPreviewFindingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAccessPreviewFindingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of access preview findings generated by the specified access preview.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPreviewFindings(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsIterable responses = client.listAccessPreviewFindingsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsIterable responses = client
     *             .listAccessPreviewFindingsPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewFindingsIterable responses = client.listAccessPreviewFindingsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPreviewFindings(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewFindingsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccessPreviewFindingsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAccessPreviewFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviewFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccessPreviewFindingsIterable listAccessPreviewFindingsPaginator(
            ListAccessPreviewFindingsRequest listAccessPreviewFindingsRequest) throws ResourceNotFoundException,
            ConflictException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, AccessAnalyzerException {
        return new ListAccessPreviewFindingsIterable(this, applyPaginatorUserAgent(listAccessPreviewFindingsRequest));
    }

    /**
     * <p>
     * Retrieves a list of access previews for the specified analyzer.
     * </p>
     *
     * @param listAccessPreviewsRequest
     * @return Result of the ListAccessPreviews operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAccessPreviews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccessPreviewsResponse listAccessPreviews(ListAccessPreviewsRequest listAccessPreviewsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccessPreviewsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAccessPreviewsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPreviewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPreviews");

            return clientHandler.execute(new ClientExecutionParams<ListAccessPreviewsRequest, ListAccessPreviewsResponse>()
                    .withOperationName("ListAccessPreviews").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAccessPreviewsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAccessPreviewsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of access previews for the specified analyzer.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPreviews(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsIterable responses = client.listAccessPreviewsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsIterable responses = client
     *             .listAccessPreviewsPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAccessPreviewsIterable responses = client.listAccessPreviewsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPreviews(software.amazon.awssdk.services.accessanalyzer.model.ListAccessPreviewsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccessPreviewsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAccessPreviews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccessPreviewsIterable listAccessPreviewsPaginator(ListAccessPreviewsRequest listAccessPreviewsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        return new ListAccessPreviewsIterable(this, applyPaginatorUserAgent(listAccessPreviewsRequest));
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     *
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return Result of the ListAnalyzedResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAnalyzedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnalyzedResourcesResponse listAnalyzedResources(ListAnalyzedResourcesRequest listAnalyzedResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAnalyzedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAnalyzedResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnalyzedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnalyzedResources");

            return clientHandler.execute(new ClientExecutionParams<ListAnalyzedResourcesRequest, ListAnalyzedResourcesResponse>()
                    .withOperationName("ListAnalyzedResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAnalyzedResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAnalyzedResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnalyzedResources(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesIterable responses = client.listAnalyzedResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesIterable responses = client
     *             .listAnalyzedResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesIterable responses = client.listAnalyzedResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnalyzedResources(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAnalyzedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnalyzedResourcesIterable listAnalyzedResourcesPaginator(ListAnalyzedResourcesRequest listAnalyzedResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        return new ListAnalyzedResourcesIterable(this, applyPaginatorUserAgent(listAnalyzedResourcesRequest));
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     *
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return Result of the ListAnalyzers operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAnalyzers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAnalyzersResponse listAnalyzers(ListAnalyzersRequest listAnalyzersRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAnalyzersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAnalyzersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnalyzersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnalyzers");

            return clientHandler.execute(new ClientExecutionParams<ListAnalyzersRequest, ListAnalyzersResponse>()
                    .withOperationName("ListAnalyzers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAnalyzersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAnalyzersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnalyzers(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersIterable responses = client.listAnalyzersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersIterable responses = client
     *             .listAnalyzersPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersIterable responses = client.listAnalyzersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnalyzers(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest)} operation.</b>
     * </p>
     *
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListAnalyzers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAnalyzersIterable listAnalyzersPaginator(ListAnalyzersRequest listAnalyzersRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            AccessAnalyzerException {
        return new ListAnalyzersIterable(this, applyPaginatorUserAgent(listAnalyzersRequest));
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     *
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return Result of the ListArchiveRules operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListArchiveRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListArchiveRulesResponse listArchiveRules(ListArchiveRulesRequest listArchiveRulesRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListArchiveRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListArchiveRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listArchiveRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListArchiveRules");

            return clientHandler.execute(new ClientExecutionParams<ListArchiveRulesRequest, ListArchiveRulesResponse>()
                    .withOperationName("ListArchiveRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listArchiveRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListArchiveRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listArchiveRules(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesIterable responses = client.listArchiveRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesIterable responses = client
     *             .listArchiveRulesPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesIterable responses = client.listArchiveRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listArchiveRules(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListArchiveRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListArchiveRulesIterable listArchiveRulesPaginator(ListArchiveRulesRequest listArchiveRulesRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        return new ListArchiveRulesIterable(this, applyPaginatorUserAgent(listArchiveRulesRequest));
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <p>
     * To learn about filter keys that you can use to retrieve a list of findings, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     *
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return Result of the ListFindings operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFindingsResponse listFindings(ListFindingsRequest listFindingsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFindingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFindings");

            return clientHandler.execute(new ClientExecutionParams<ListFindingsRequest, ListFindingsResponse>()
                    .withOperationName("ListFindings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFindingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFindingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <p>
     * To learn about filter keys that you can use to retrieve a list of findings, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFindings(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsIterable responses = client.listFindingsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsIterable responses = client
     *             .listFindingsPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsIterable responses = client.listFindingsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFindings(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest)} operation.</b>
     * </p>
     *
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFindingsIterable listFindingsPaginator(ListFindingsRequest listFindingsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        return new ListFindingsIterable(this, applyPaginatorUserAgent(listFindingsRequest));
    }

    /**
     * <p>
     * Lists all of the policy generations requested in the last seven days.
     * </p>
     *
     * @param listPolicyGenerationsRequest
     * @return Result of the ListPolicyGenerations operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListPolicyGenerations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListPolicyGenerations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPolicyGenerationsResponse listPolicyGenerations(ListPolicyGenerationsRequest listPolicyGenerationsRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPolicyGenerationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPolicyGenerationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPolicyGenerationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicyGenerations");

            return clientHandler.execute(new ClientExecutionParams<ListPolicyGenerationsRequest, ListPolicyGenerationsResponse>()
                    .withOperationName("ListPolicyGenerations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPolicyGenerationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPolicyGenerationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the policy generations requested in the last seven days.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPolicyGenerations(software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsIterable responses = client.listPolicyGenerationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsIterable responses = client
     *             .listPolicyGenerationsPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListPolicyGenerationsIterable responses = client.listPolicyGenerationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPolicyGenerations(software.amazon.awssdk.services.accessanalyzer.model.ListPolicyGenerationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPolicyGenerationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListPolicyGenerations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListPolicyGenerations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPolicyGenerationsIterable listPolicyGenerationsPaginator(ListPolicyGenerationsRequest listPolicyGenerationsRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        return new ListPolicyGenerationsIterable(this, applyPaginatorUserAgent(listPolicyGenerationsRequest));
    }

    /**
     * <p>
     * Retrieves a list of tags applied to the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Retrieves a list of tags applied to the specified resource.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the policy generation request.
     * </p>
     *
     * @param startPolicyGenerationRequest
     * @return Result of the StartPolicyGeneration operation returned by the service.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.StartPolicyGeneration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartPolicyGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartPolicyGenerationResponse startPolicyGeneration(StartPolicyGenerationRequest startPolicyGenerationRequest)
            throws ConflictException, ValidationException, InternalServerException, ServiceQuotaExceededException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartPolicyGenerationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartPolicyGenerationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPolicyGenerationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPolicyGeneration");

            return clientHandler.execute(new ClientExecutionParams<StartPolicyGenerationRequest, StartPolicyGenerationResponse>()
                    .withOperationName("StartPolicyGeneration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startPolicyGenerationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartPolicyGenerationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Immediately starts a scan of the policies applied to the specified resource.
     * </p>
     *
     * @param startResourceScanRequest
     *        Starts a scan of the policies applied to the specified resource.
     * @return Result of the StartResourceScan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.StartResourceScan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartResourceScan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartResourceScanResponse startResourceScan(StartResourceScanRequest startResourceScanRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartResourceScanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartResourceScanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startResourceScanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartResourceScan");

            return clientHandler.execute(new ClientExecutionParams<StartResourceScanRequest, StartResourceScanResponse>()
                    .withOperationName("StartResourceScan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startResourceScanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartResourceScanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a tag to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     *        Adds a tag to the specified resource.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     *        Removes a tag from the specified resource.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the criteria and values for the specified archive rule.
     * </p>
     *
     * @param updateArchiveRuleRequest
     *        Updates the specified archive rule.
     * @return Result of the UpdateArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.UpdateArchiveRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateArchiveRuleResponse updateArchiveRule(UpdateArchiveRuleRequest updateArchiveRuleRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateArchiveRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateArchiveRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateArchiveRule");

            return clientHandler.execute(new ClientExecutionParams<UpdateArchiveRuleRequest, UpdateArchiveRuleResponse>()
                    .withOperationName("UpdateArchiveRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateArchiveRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateArchiveRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the status for the specified findings.
     * </p>
     *
     * @param updateFindingsRequest
     *        Updates findings with the new values provided in the request.
     * @return Result of the UpdateFindings operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.UpdateFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateFindingsResponse updateFindings(UpdateFindingsRequest updateFindingsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFindingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFindings");

            return clientHandler.execute(new ClientExecutionParams<UpdateFindingsRequest, UpdateFindingsResponse>()
                    .withOperationName("UpdateFindings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFindingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFindingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Requests the validation of a policy and returns a list of findings. The findings help you identify issues and
     * provide actionable recommendations to resolve the issue and enable you to author functional policies that meet
     * security best practices.
     * </p>
     *
     * @param validatePolicyRequest
     * @return Result of the ValidatePolicy operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ValidatePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ValidatePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ValidatePolicyResponse validatePolicy(ValidatePolicyRequest validatePolicyRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            AccessAnalyzerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidatePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ValidatePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AccessAnalyzer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePolicy");

            return clientHandler.execute(new ClientExecutionParams<ValidatePolicyRequest, ValidatePolicyResponse>()
                    .withOperationName("ValidatePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(validatePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ValidatePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Requests the validation of a policy and returns a list of findings. The findings help you identify issues and
     * provide actionable recommendations to resolve the issue and enable you to author functional policies that meet
     * security best practices.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #validatePolicy(software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyIterable responses = client.validatePolicyPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyIterable responses = client
     *             .validatePolicyPaginator(request);
     *     for (software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ValidatePolicyIterable responses = client.validatePolicyPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #validatePolicy(software.amazon.awssdk.services.accessanalyzer.model.ValidatePolicyRequest)}
     * operation.</b>
     * </p>
     *
     * @param validatePolicyRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccessAnalyzerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccessAnalyzerClient.ValidatePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ValidatePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ValidatePolicyIterable validatePolicyPaginator(ValidatePolicyRequest validatePolicyRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, AccessAnalyzerException {
        return new ValidatePolicyIterable(this, applyPaginatorUserAgent(validatePolicyRequest));
    }

    private <T extends AccessAnalyzerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AccessAnalyzerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final AccessAnalyzerServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
