/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The proposed access control configuration for an Amazon RDS DB snapshot. You can propose a configuration for a new
 * Amazon RDS DB snapshot or an Amazon RDS DB snapshot that you own by specifying the
 * <code>RdsDbSnapshotAttributeValue</code> and optional KMS encryption key. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshotAttribute.html"
 * >ModifyDBSnapshotAttribute</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RdsDbSnapshotConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<RdsDbSnapshotConfiguration.Builder, RdsDbSnapshotConfiguration> {
    private static final SdkField<Map<String, RdsDbSnapshotAttributeValue>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, RdsDbSnapshotAttributeValue>> builder(MarshallingType.MAP)
            .memberName("attributes")
            .getter(getter(RdsDbSnapshotConfiguration::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<RdsDbSnapshotAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(RdsDbSnapshotAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyId").getter(getter(RdsDbSnapshotConfiguration::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTRIBUTES_FIELD,
            KMS_KEY_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Map<String, RdsDbSnapshotAttributeValue> attributes;

    private final String kmsKeyId;

    private RdsDbSnapshotConfiguration(BuilderImpl builder) {
        this.attributes = builder.attributes;
        this.kmsKeyId = builder.kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The names and values of manual DB snapshot attributes. Manual DB snapshot attributes are used to authorize other
     * Amazon Web Services accounts to restore a manual DB snapshot. The only valid value for <code>attributeName</code>
     * for the attribute map is restore.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return The names and values of manual DB snapshot attributes. Manual DB snapshot attributes are used to
     *         authorize other Amazon Web Services accounts to restore a manual DB snapshot. The only valid value for
     *         <code>attributeName</code> for the attribute map is restore.
     */
    public final Map<String, RdsDbSnapshotAttributeValue> attributes() {
        return attributes;
    }

    /**
     * <p>
     * The KMS key identifier for an encrypted Amazon RDS DB snapshot. The KMS key identifier is the key ARN, key ID,
     * alias ARN, or alias name for the KMS key.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the configuration is for an existing Amazon RDS DB snapshot and you do not specify the <code>kmsKeyId</code>,
     * or you specify an empty string, then the access preview uses the existing <code>kmsKeyId</code> of the snapshot.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the access preview is for a new resource and you do not specify the specify the <code>kmsKeyId</code>, then
     * the access preview considers the snapshot as unencrypted.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The KMS key identifier for an encrypted Amazon RDS DB snapshot. The KMS key identifier is the key ARN,
     *         key ID, alias ARN, or alias name for the KMS key.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the configuration is for an existing Amazon RDS DB snapshot and you do not specify the
     *         <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
     *         <code>kmsKeyId</code> of the snapshot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the access preview is for a new resource and you do not specify the specify the <code>kmsKeyId</code>,
     *         then the access preview considers the snapshot as unencrypted.
     *         </p>
     *         </li>
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RdsDbSnapshotConfiguration)) {
            return false;
        }
        RdsDbSnapshotConfiguration other = (RdsDbSnapshotConfiguration) obj;
        return hasAttributes() == other.hasAttributes() && Objects.equals(attributes(), other.attributes())
                && Objects.equals(kmsKeyId(), other.kmsKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RdsDbSnapshotConfiguration").add("Attributes", hasAttributes() ? attributes() : null)
                .add("KmsKeyId", kmsKeyId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "kmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("attributes", ATTRIBUTES_FIELD);
        map.put("kmsKeyId", KMS_KEY_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RdsDbSnapshotConfiguration, T> g) {
        return obj -> g.apply((RdsDbSnapshotConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RdsDbSnapshotConfiguration> {
        /**
         * <p>
         * The names and values of manual DB snapshot attributes. Manual DB snapshot attributes are used to authorize
         * other Amazon Web Services accounts to restore a manual DB snapshot. The only valid value for
         * <code>attributeName</code> for the attribute map is restore.
         * </p>
         * 
         * @param attributes
         *        The names and values of manual DB snapshot attributes. Manual DB snapshot attributes are used to
         *        authorize other Amazon Web Services accounts to restore a manual DB snapshot. The only valid value for
         *        <code>attributeName</code> for the attribute map is restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, RdsDbSnapshotAttributeValue> attributes);

        /**
         * <p>
         * The KMS key identifier for an encrypted Amazon RDS DB snapshot. The KMS key identifier is the key ARN, key
         * ID, alias ARN, or alias name for the KMS key.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the configuration is for an existing Amazon RDS DB snapshot and you do not specify the
         * <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
         * <code>kmsKeyId</code> of the snapshot.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the access preview is for a new resource and you do not specify the specify the <code>kmsKeyId</code>,
         * then the access preview considers the snapshot as unencrypted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        The KMS key identifier for an encrypted Amazon RDS DB snapshot. The KMS key identifier is the key ARN,
         *        key ID, alias ARN, or alias name for the KMS key.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the configuration is for an existing Amazon RDS DB snapshot and you do not specify the
         *        <code>kmsKeyId</code>, or you specify an empty string, then the access preview uses the existing
         *        <code>kmsKeyId</code> of the snapshot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the access preview is for a new resource and you do not specify the specify the
         *        <code>kmsKeyId</code>, then the access preview considers the snapshot as unencrypted.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, RdsDbSnapshotAttributeValue> attributes = DefaultSdkAutoConstructMap.getInstance();

        private String kmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(RdsDbSnapshotConfiguration model) {
            attributes(model.attributes);
            kmsKeyId(model.kmsKeyId);
        }

        public final Map<String, RdsDbSnapshotAttributeValue.Builder> getAttributes() {
            Map<String, RdsDbSnapshotAttributeValue.Builder> result = RdsDbSnapshotAttributesMapCopier
                    .copyToBuilder(this.attributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setAttributes(Map<String, RdsDbSnapshotAttributeValue.BuilderImpl> attributes) {
            this.attributes = RdsDbSnapshotAttributesMapCopier.copyFromBuilder(attributes);
        }

        @Override
        public final Builder attributes(Map<String, RdsDbSnapshotAttributeValue> attributes) {
            this.attributes = RdsDbSnapshotAttributesMapCopier.copy(attributes);
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        @Override
        public RdsDbSnapshotConfiguration build() {
            return new RdsDbSnapshotConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
