/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the analyzer.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnalyzerSummary implements SdkPojo, Serializable, ToCopyableBuilder<AnalyzerSummary.Builder, AnalyzerSummary> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(AnalyzerSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(AnalyzerSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(AnalyzerSummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(AnalyzerSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> LAST_RESOURCE_ANALYZED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lastResourceAnalyzed").getter(getter(AnalyzerSummary::lastResourceAnalyzed))
            .setter(setter(Builder::lastResourceAnalyzed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastResourceAnalyzed").build())
            .build();

    private static final SdkField<Instant> LAST_RESOURCE_ANALYZED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastResourceAnalyzedAt")
            .getter(getter(AnalyzerSummary::lastResourceAnalyzedAt))
            .setter(setter(Builder::lastResourceAnalyzedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastResourceAnalyzedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(AnalyzerSummary::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AnalyzerSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<StatusReason> STATUS_REASON_FIELD = SdkField.<StatusReason> builder(MarshallingType.SDK_POJO)
            .memberName("statusReason").getter(getter(AnalyzerSummary::statusReason)).setter(setter(Builder::statusReason))
            .constructor(StatusReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<AnalyzerConfiguration> CONFIGURATION_FIELD = SdkField
            .<AnalyzerConfiguration> builder(MarshallingType.SDK_POJO).memberName("configuration")
            .getter(getter(AnalyzerSummary::configuration)).setter(setter(Builder::configuration))
            .constructor(AnalyzerConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            TYPE_FIELD, CREATED_AT_FIELD, LAST_RESOURCE_ANALYZED_FIELD, LAST_RESOURCE_ANALYZED_AT_FIELD, TAGS_FIELD,
            STATUS_FIELD, STATUS_REASON_FIELD, CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String type;

    private final Instant createdAt;

    private final String lastResourceAnalyzed;

    private final Instant lastResourceAnalyzedAt;

    private final Map<String, String> tags;

    private final String status;

    private final StatusReason statusReason;

    private final AnalyzerConfiguration configuration;

    private AnalyzerSummary(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.type = builder.type;
        this.createdAt = builder.createdAt;
        this.lastResourceAnalyzed = builder.lastResourceAnalyzed;
        this.lastResourceAnalyzedAt = builder.lastResourceAnalyzedAt;
        this.tags = builder.tags;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.configuration = builder.configuration;
    }

    /**
     * <p>
     * The ARN of the analyzer.
     * </p>
     * 
     * @return The ARN of the analyzer.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the analyzer.
     * </p>
     * 
     * @return The name of the analyzer.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link Type#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
     * @see Type
     */
    public final Type type() {
        return Type.fromValue(type);
    }

    /**
     * <p>
     * The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link Type#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
     * @see Type
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * A timestamp for the time at which the analyzer was created.
     * </p>
     * 
     * @return A timestamp for the time at which the analyzer was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The resource that was most recently analyzed by the analyzer.
     * </p>
     * 
     * @return The resource that was most recently analyzed by the analyzer.
     */
    public final String lastResourceAnalyzed() {
        return lastResourceAnalyzed;
    }

    /**
     * <p>
     * The time at which the most recently analyzed resource was analyzed.
     * </p>
     * 
     * @return The time at which the most recently analyzed resource was analyzed.
     */
    public final Instant lastResourceAnalyzedAt() {
        return lastResourceAnalyzedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags added to the analyzer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags added to the analyzer.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources and
     * generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing trusted access
     * for Identity and Access Management Access Analyzer from Organizations, causes the analyzer to stop generating new
     * findings. The status is <code>Creating</code> when the analyzer creation is in progress and <code>Failed</code>
     * when the analyzer creation has failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AnalyzerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources and
     *         generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing
     *         trusted access for Identity and Access Management Access Analyzer from Organizations, causes the analyzer
     *         to stop generating new findings. The status is <code>Creating</code> when the analyzer creation is in
     *         progress and <code>Failed</code> when the analyzer creation has failed.
     * @see AnalyzerStatus
     */
    public final AnalyzerStatus status() {
        return AnalyzerStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources and
     * generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing trusted access
     * for Identity and Access Management Access Analyzer from Organizations, causes the analyzer to stop generating new
     * findings. The status is <code>Creating</code> when the analyzer creation is in progress and <code>Failed</code>
     * when the analyzer creation has failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AnalyzerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources and
     *         generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing
     *         trusted access for Identity and Access Management Access Analyzer from Organizations, causes the analyzer
     *         to stop generating new findings. The status is <code>Creating</code> when the analyzer creation is in
     *         progress and <code>Failed</code> when the analyzer creation has failed.
     * @see AnalyzerStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The <code>statusReason</code> provides more details about the current status of the analyzer. For example, if the
     * creation for the analyzer fails, a <code>Failed</code> status is returned. For an analyzer with organization as
     * the type, this failure can be due to an issue with creating the service-linked roles required in the member
     * accounts of the Amazon Web Services organization.
     * </p>
     * 
     * @return The <code>statusReason</code> provides more details about the current status of the analyzer. For
     *         example, if the creation for the analyzer fails, a <code>Failed</code> status is returned. For an
     *         analyzer with organization as the type, this failure can be due to an issue with creating the
     *         service-linked roles required in the member accounts of the Amazon Web Services organization.
     */
    public final StatusReason statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * Specifies if the analyzer is an external access, unused access, or internal access analyzer.
     * </p>
     * 
     * @return Specifies if the analyzer is an external access, unused access, or internal access analyzer.
     */
    public final AnalyzerConfiguration configuration() {
        return configuration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastResourceAnalyzed());
        hashCode = 31 * hashCode + Objects.hashCode(lastResourceAnalyzedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnalyzerSummary)) {
            return false;
        }
        AnalyzerSummary other = (AnalyzerSummary) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastResourceAnalyzed(), other.lastResourceAnalyzed())
                && Objects.equals(lastResourceAnalyzedAt(), other.lastResourceAnalyzedAt()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && Objects.equals(configuration(), other.configuration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AnalyzerSummary").add("Arn", arn()).add("Name", name()).add("Type", typeAsString())
                .add("CreatedAt", createdAt()).add("LastResourceAnalyzed", lastResourceAnalyzed())
                .add("LastResourceAnalyzedAt", lastResourceAnalyzedAt()).add("Tags", hasTags() ? tags() : null)
                .add("Status", statusAsString()).add("StatusReason", statusReason()).add("Configuration", configuration())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "lastResourceAnalyzed":
            return Optional.ofNullable(clazz.cast(lastResourceAnalyzed()));
        case "lastResourceAnalyzedAt":
            return Optional.ofNullable(clazz.cast(lastResourceAnalyzedAt()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("name", NAME_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("lastResourceAnalyzed", LAST_RESOURCE_ANALYZED_FIELD);
        map.put("lastResourceAnalyzedAt", LAST_RESOURCE_ANALYZED_AT_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("configuration", CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AnalyzerSummary, T> g) {
        return obj -> g.apply((AnalyzerSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnalyzerSummary> {
        /**
         * <p>
         * The ARN of the analyzer.
         * </p>
         * 
         * @param arn
         *        The ARN of the analyzer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the analyzer.
         * </p>
         * 
         * @param name
         *        The name of the analyzer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
         * </p>
         * 
         * @param type
         *        The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
         * @see Type
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Type
         */
        Builder type(String type);

        /**
         * <p>
         * The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
         * </p>
         * 
         * @param type
         *        The type of analyzer, which corresponds to the zone of trust chosen for the analyzer.
         * @see Type
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Type
         */
        Builder type(Type type);

        /**
         * <p>
         * A timestamp for the time at which the analyzer was created.
         * </p>
         * 
         * @param createdAt
         *        A timestamp for the time at which the analyzer was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The resource that was most recently analyzed by the analyzer.
         * </p>
         * 
         * @param lastResourceAnalyzed
         *        The resource that was most recently analyzed by the analyzer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastResourceAnalyzed(String lastResourceAnalyzed);

        /**
         * <p>
         * The time at which the most recently analyzed resource was analyzed.
         * </p>
         * 
         * @param lastResourceAnalyzedAt
         *        The time at which the most recently analyzed resource was analyzed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastResourceAnalyzedAt(Instant lastResourceAnalyzedAt);

        /**
         * <p>
         * The tags added to the analyzer.
         * </p>
         * 
         * @param tags
         *        The tags added to the analyzer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources and
         * generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing trusted
         * access for Identity and Access Management Access Analyzer from Organizations, causes the analyzer to stop
         * generating new findings. The status is <code>Creating</code> when the analyzer creation is in progress and
         * <code>Failed</code> when the analyzer creation has failed.
         * </p>
         * 
         * @param status
         *        The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources
         *        and generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing
         *        trusted access for Identity and Access Management Access Analyzer from Organizations, causes the
         *        analyzer to stop generating new findings. The status is <code>Creating</code> when the analyzer
         *        creation is in progress and <code>Failed</code> when the analyzer creation has failed.
         * @see AnalyzerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyzerStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources and
         * generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing trusted
         * access for Identity and Access Management Access Analyzer from Organizations, causes the analyzer to stop
         * generating new findings. The status is <code>Creating</code> when the analyzer creation is in progress and
         * <code>Failed</code> when the analyzer creation has failed.
         * </p>
         * 
         * @param status
         *        The status of the analyzer. An <code>Active</code> analyzer successfully monitors supported resources
         *        and generates new findings. The analyzer is <code>Disabled</code> when a user action, such as removing
         *        trusted access for Identity and Access Management Access Analyzer from Organizations, causes the
         *        analyzer to stop generating new findings. The status is <code>Creating</code> when the analyzer
         *        creation is in progress and <code>Failed</code> when the analyzer creation has failed.
         * @see AnalyzerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyzerStatus
         */
        Builder status(AnalyzerStatus status);

        /**
         * <p>
         * The <code>statusReason</code> provides more details about the current status of the analyzer. For example, if
         * the creation for the analyzer fails, a <code>Failed</code> status is returned. For an analyzer with
         * organization as the type, this failure can be due to an issue with creating the service-linked roles required
         * in the member accounts of the Amazon Web Services organization.
         * </p>
         * 
         * @param statusReason
         *        The <code>statusReason</code> provides more details about the current status of the analyzer. For
         *        example, if the creation for the analyzer fails, a <code>Failed</code> status is returned. For an
         *        analyzer with organization as the type, this failure can be due to an issue with creating the
         *        service-linked roles required in the member accounts of the Amazon Web Services organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(StatusReason statusReason);

        /**
         * <p>
         * The <code>statusReason</code> provides more details about the current status of the analyzer. For example, if
         * the creation for the analyzer fails, a <code>Failed</code> status is returned. For an analyzer with
         * organization as the type, this failure can be due to an issue with creating the service-linked roles required
         * in the member accounts of the Amazon Web Services organization.
         * </p>
         * This is a convenience method that creates an instance of the {@link StatusReason.Builder} avoiding the need
         * to create one manually via {@link StatusReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StatusReason.Builder#build()} is called immediately and its
         * result is passed to {@link #statusReason(StatusReason)}.
         * 
         * @param statusReason
         *        a consumer that will call methods on {@link StatusReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusReason(StatusReason)
         */
        default Builder statusReason(Consumer<StatusReason.Builder> statusReason) {
            return statusReason(StatusReason.builder().applyMutation(statusReason).build());
        }

        /**
         * <p>
         * Specifies if the analyzer is an external access, unused access, or internal access analyzer.
         * </p>
         * 
         * @param configuration
         *        Specifies if the analyzer is an external access, unused access, or internal access analyzer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(AnalyzerConfiguration configuration);

        /**
         * <p>
         * Specifies if the analyzer is an external access, unused access, or internal access analyzer.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalyzerConfiguration.Builder} avoiding
         * the need to create one manually via {@link AnalyzerConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalyzerConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #configuration(AnalyzerConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link AnalyzerConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(AnalyzerConfiguration)
         */
        default Builder configuration(Consumer<AnalyzerConfiguration.Builder> configuration) {
            return configuration(AnalyzerConfiguration.builder().applyMutation(configuration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String type;

        private Instant createdAt;

        private String lastResourceAnalyzed;

        private Instant lastResourceAnalyzedAt;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private StatusReason statusReason;

        private AnalyzerConfiguration configuration;

        private BuilderImpl() {
        }

        private BuilderImpl(AnalyzerSummary model) {
            arn(model.arn);
            name(model.name);
            type(model.type);
            createdAt(model.createdAt);
            lastResourceAnalyzed(model.lastResourceAnalyzed);
            lastResourceAnalyzedAt(model.lastResourceAnalyzedAt);
            tags(model.tags);
            status(model.status);
            statusReason(model.statusReason);
            configuration(model.configuration);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(Type type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getLastResourceAnalyzed() {
            return lastResourceAnalyzed;
        }

        public final void setLastResourceAnalyzed(String lastResourceAnalyzed) {
            this.lastResourceAnalyzed = lastResourceAnalyzed;
        }

        @Override
        public final Builder lastResourceAnalyzed(String lastResourceAnalyzed) {
            this.lastResourceAnalyzed = lastResourceAnalyzed;
            return this;
        }

        public final Instant getLastResourceAnalyzedAt() {
            return lastResourceAnalyzedAt;
        }

        public final void setLastResourceAnalyzedAt(Instant lastResourceAnalyzedAt) {
            this.lastResourceAnalyzedAt = lastResourceAnalyzedAt;
        }

        @Override
        public final Builder lastResourceAnalyzedAt(Instant lastResourceAnalyzedAt) {
            this.lastResourceAnalyzedAt = lastResourceAnalyzedAt;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AnalyzerStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final StatusReason.Builder getStatusReason() {
            return statusReason != null ? statusReason.toBuilder() : null;
        }

        public final void setStatusReason(StatusReason.BuilderImpl statusReason) {
            this.statusReason = statusReason != null ? statusReason.build() : null;
        }

        @Override
        public final Builder statusReason(StatusReason statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final AnalyzerConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(AnalyzerConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(AnalyzerConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        @Override
        public AnalyzerSummary build() {
            return new AnalyzerSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
