/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.account;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.account.model.AccessDeniedException;
import software.amazon.awssdk.services.account.model.AccountException;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactRequest;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactResponse;
import software.amazon.awssdk.services.account.model.GetAlternateContactRequest;
import software.amazon.awssdk.services.account.model.GetAlternateContactResponse;
import software.amazon.awssdk.services.account.model.InternalServerException;
import software.amazon.awssdk.services.account.model.PutAlternateContactRequest;
import software.amazon.awssdk.services.account.model.PutAlternateContactResponse;
import software.amazon.awssdk.services.account.model.ResourceNotFoundException;
import software.amazon.awssdk.services.account.model.TooManyRequestsException;
import software.amazon.awssdk.services.account.model.ValidationException;
import software.amazon.awssdk.services.account.transform.DeleteAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.PutAlternateContactRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AccountClient}.
 *
 * @see AccountClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAccountClient implements AccountClient {
    private static final Logger log = Logger.loggerFor(DefaultAccountClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAccountClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified alternate contact from an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     *
     * @param deleteAlternateContactRequest
     * @return Result of the DeleteAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DeleteAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DeleteAlternateContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAlternateContactResponse deleteAlternateContact(DeleteAlternateContactRequest deleteAlternateContactRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAlternateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlternateContact");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAlternateContactRequest, DeleteAlternateContactResponse>()
                            .withOperationName("DeleteAlternateContact").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteAlternateContactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAlternateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     *
     * @param getAlternateContactRequest
     * @return Result of the GetAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAlternateContactResponse getAlternateContact(GetAlternateContactRequest getAlternateContactRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAlternateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAlternateContact");

            return clientHandler.execute(new ClientExecutionParams<GetAlternateContactRequest, GetAlternateContactResponse>()
                    .withOperationName("GetAlternateContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAlternateContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAlternateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     *
     * @param putAlternateContactRequest
     * @return Result of the PutAlternateContact operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutAlternateContactResponse putAlternateContact(PutAlternateContactRequest putAlternateContactRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAlternateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAlternateContact");

            return clientHandler.execute(new ClientExecutionParams<PutAlternateContactRequest, PutAlternateContactResponse>()
                    .withOperationName("PutAlternateContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putAlternateContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAlternateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AccountException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
