/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.account;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.account.internal.AccountServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateRequest;
import software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateResponse;
import software.amazon.awssdk.services.account.model.AccessDeniedException;
import software.amazon.awssdk.services.account.model.AccountException;
import software.amazon.awssdk.services.account.model.ConflictException;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactRequest;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactResponse;
import software.amazon.awssdk.services.account.model.DisableRegionRequest;
import software.amazon.awssdk.services.account.model.DisableRegionResponse;
import software.amazon.awssdk.services.account.model.EnableRegionRequest;
import software.amazon.awssdk.services.account.model.EnableRegionResponse;
import software.amazon.awssdk.services.account.model.GetAlternateContactRequest;
import software.amazon.awssdk.services.account.model.GetAlternateContactResponse;
import software.amazon.awssdk.services.account.model.GetContactInformationRequest;
import software.amazon.awssdk.services.account.model.GetContactInformationResponse;
import software.amazon.awssdk.services.account.model.GetPrimaryEmailRequest;
import software.amazon.awssdk.services.account.model.GetPrimaryEmailResponse;
import software.amazon.awssdk.services.account.model.GetRegionOptStatusRequest;
import software.amazon.awssdk.services.account.model.GetRegionOptStatusResponse;
import software.amazon.awssdk.services.account.model.InternalServerException;
import software.amazon.awssdk.services.account.model.ListRegionsRequest;
import software.amazon.awssdk.services.account.model.ListRegionsResponse;
import software.amazon.awssdk.services.account.model.PutAlternateContactRequest;
import software.amazon.awssdk.services.account.model.PutAlternateContactResponse;
import software.amazon.awssdk.services.account.model.PutContactInformationRequest;
import software.amazon.awssdk.services.account.model.PutContactInformationResponse;
import software.amazon.awssdk.services.account.model.ResourceNotFoundException;
import software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateRequest;
import software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateResponse;
import software.amazon.awssdk.services.account.model.TooManyRequestsException;
import software.amazon.awssdk.services.account.model.ValidationException;
import software.amazon.awssdk.services.account.transform.AcceptPrimaryEmailUpdateRequestMarshaller;
import software.amazon.awssdk.services.account.transform.DeleteAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.DisableRegionRequestMarshaller;
import software.amazon.awssdk.services.account.transform.EnableRegionRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetContactInformationRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetPrimaryEmailRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetRegionOptStatusRequestMarshaller;
import software.amazon.awssdk.services.account.transform.ListRegionsRequestMarshaller;
import software.amazon.awssdk.services.account.transform.PutAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.PutContactInformationRequestMarshaller;
import software.amazon.awssdk.services.account.transform.StartPrimaryEmailUpdateRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AccountAsyncClient}.
 *
 * @see AccountAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAccountAsyncClient implements AccountAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAccountAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAccountAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts the request that originated from <a>StartPrimaryEmailUpdate</a> to update the primary email address (also
     * known as the root user email address) for the specified account.
     * </p>
     *
     * @param acceptPrimaryEmailUpdateRequest
     * @return A Java Future containing the result of the AcceptPrimaryEmailUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation failed because it specified a resource that can't be found.</li>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>ConflictException The request could not be processed because of a conflict in the current status of
     *         the resource. For example, this happens if you try to enable a Region that is currently being disabled
     *         (in a status of DISABLING).</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.AcceptPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/AcceptPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptPrimaryEmailUpdateResponse> acceptPrimaryEmailUpdate(
            AcceptPrimaryEmailUpdateRequest acceptPrimaryEmailUpdateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptPrimaryEmailUpdateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptPrimaryEmailUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptPrimaryEmailUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptPrimaryEmailUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptPrimaryEmailUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptPrimaryEmailUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptPrimaryEmailUpdateRequest, AcceptPrimaryEmailUpdateResponse>()
                            .withOperationName("AcceptPrimaryEmailUpdate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcceptPrimaryEmailUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptPrimaryEmailUpdateRequest));
            CompletableFuture<AcceptPrimaryEmailUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified alternate contact from an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param deleteAlternateContactRequest
     * @return A Java Future containing the result of the DeleteAlternateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation failed because it specified a resource that can't be found.</li>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.DeleteAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DeleteAlternateContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAlternateContactResponse> deleteAlternateContact(
            DeleteAlternateContactRequest deleteAlternateContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAlternateContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlternateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAlternateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAlternateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAlternateContactRequest, DeleteAlternateContactResponse>()
                            .withOperationName("DeleteAlternateContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAlternateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAlternateContactRequest));
            CompletableFuture<DeleteAlternateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables (opts-out) a particular Region for an account.
     * </p>
     * <note>
     * <p>
     * The act of disabling a Region will remove all IAM access to any resources that reside in that Region.
     * </p>
     * </note>
     *
     * @param disableRegionRequest
     * @return A Java Future containing the result of the DisableRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>ConflictException The request could not be processed because of a conflict in the current status of
     *         the resource. For example, this happens if you try to enable a Region that is currently being disabled
     *         (in a status of DISABLING).</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.DisableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DisableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableRegionResponse> disableRegion(DisableRegionRequest disableRegionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableRegionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableRegionRequest, DisableRegionResponse>()
                            .withOperationName("DisableRegion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableRegionRequest));
            CompletableFuture<DisableRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables (opts-in) a particular Region for an account.
     * </p>
     *
     * @param enableRegionRequest
     * @return A Java Future containing the result of the EnableRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>ConflictException The request could not be processed because of a conflict in the current status of
     *         the resource. For example, this happens if you try to enable a Region that is currently being disabled
     *         (in a status of DISABLING).</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.EnableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/EnableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableRegionResponse> enableRegion(EnableRegionRequest enableRegionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableRegionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableRegionRequest, EnableRegionResponse>()
                            .withOperationName("EnableRegion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableRegionRequest));
            CompletableFuture<EnableRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param getAlternateContactRequest
     * @return A Java Future containing the result of the GetAlternateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation failed because it specified a resource that can't be found.</li>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.GetAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAlternateContactResponse> getAlternateContact(
            GetAlternateContactRequest getAlternateContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAlternateContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAlternateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAlternateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAlternateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAlternateContactRequest, GetAlternateContactResponse>()
                            .withOperationName("GetAlternateContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAlternateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAlternateContactRequest));
            CompletableFuture<GetAlternateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     *
     * @param getContactInformationRequest
     * @return A Java Future containing the result of the GetContactInformation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation failed because it specified a resource that can't be found.</li>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.GetContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactInformationResponse> getContactInformation(
            GetContactInformationRequest getContactInformationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactInformationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactInformation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContactInformationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactInformationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactInformationRequest, GetContactInformationResponse>()
                            .withOperationName("GetContactInformation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetContactInformationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getContactInformationRequest));
            CompletableFuture<GetContactInformationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the primary email address for the specified account.
     * </p>
     *
     * @param getPrimaryEmailRequest
     * @return A Java Future containing the result of the GetPrimaryEmail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation failed because it specified a resource that can't be found.</li>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.GetPrimaryEmail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetPrimaryEmail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPrimaryEmailResponse> getPrimaryEmail(GetPrimaryEmailRequest getPrimaryEmailRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPrimaryEmailRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPrimaryEmailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPrimaryEmail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPrimaryEmailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPrimaryEmailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPrimaryEmailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPrimaryEmailRequest, GetPrimaryEmailResponse>()
                            .withOperationName("GetPrimaryEmail").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPrimaryEmailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPrimaryEmailRequest));
            CompletableFuture<GetPrimaryEmailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the opt-in status of a particular Region.
     * </p>
     *
     * @param getRegionOptStatusRequest
     * @return A Java Future containing the result of the GetRegionOptStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.GetRegionOptStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetRegionOptStatus" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRegionOptStatusResponse> getRegionOptStatus(GetRegionOptStatusRequest getRegionOptStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegionOptStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegionOptStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegionOptStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRegionOptStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRegionOptStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRegionOptStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRegionOptStatusRequest, GetRegionOptStatusResponse>()
                            .withOperationName("GetRegionOptStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRegionOptStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRegionOptStatusRequest));
            CompletableFuture<GetRegionOptStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the Regions for a given account and their respective opt-in statuses. Optionally, this list can be
     * filtered by the <code>region-opt-status-contains</code> parameter.
     * </p>
     *
     * @param listRegionsRequest
     * @return A Java Future containing the result of the ListRegions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/ListRegions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRegionsResponse> listRegions(ListRegionsRequest listRegionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRegionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRegionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRegionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRegionsRequest, ListRegionsResponse>()
                            .withOperationName("ListRegions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRegionsRequest));
            CompletableFuture<ListRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param putAlternateContactRequest
     * @return A Java Future containing the result of the PutAlternateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.PutAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAlternateContactResponse> putAlternateContact(
            PutAlternateContactRequest putAlternateContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAlternateContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAlternateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAlternateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAlternateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAlternateContactRequest, PutAlternateContactResponse>()
                            .withOperationName("PutAlternateContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAlternateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAlternateContactRequest));
            CompletableFuture<PutAlternateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     *
     * @param putContactInformationRequest
     * @return A Java Future containing the result of the PutContactInformation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.PutContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutContactInformationResponse> putContactInformation(
            PutContactInformationRequest putContactInformationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putContactInformationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContactInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContactInformation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutContactInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutContactInformationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutContactInformationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutContactInformationRequest, PutContactInformationResponse>()
                            .withOperationName("PutContactInformation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutContactInformationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putContactInformationRequest));
            CompletableFuture<PutContactInformationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the process to update the primary email address for the specified account.
     * </p>
     *
     * @param startPrimaryEmailUpdateRequest
     * @return A Java Future containing the result of the StartPrimaryEmailUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation failed because it specified a resource that can't be found.</li>
     *         <li>ValidationException The operation failed because one of the input parameters was invalid.</li>
     *         <li>ConflictException The request could not be processed because of a conflict in the current status of
     *         the resource. For example, this happens if you try to enable a Region that is currently being disabled
     *         (in a status of DISABLING).</li>
     *         <li>AccessDeniedException The operation failed because the calling identity doesn't have the minimum
     *         required permissions.</li>
     *         <li>TooManyRequestsException The operation failed because it was called too frequently and exceeded a
     *         throttle limit.</li>
     *         <li>InternalServerException The operation failed because of an error internal to Amazon Web Services. Try
     *         your operation again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccountException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AccountAsyncClient.StartPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/StartPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPrimaryEmailUpdateResponse> startPrimaryEmailUpdate(
            StartPrimaryEmailUpdateRequest startPrimaryEmailUpdateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPrimaryEmailUpdateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPrimaryEmailUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPrimaryEmailUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPrimaryEmailUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPrimaryEmailUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPrimaryEmailUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPrimaryEmailUpdateRequest, StartPrimaryEmailUpdateResponse>()
                            .withOperationName("StartPrimaryEmailUpdate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartPrimaryEmailUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startPrimaryEmailUpdateRequest));
            CompletableFuture<StartPrimaryEmailUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AccountServiceClientConfiguration serviceClientConfiguration() {
        return new AccountServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AccountException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AccountServiceClientConfigurationBuilder serviceConfigBuilder = new AccountServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
