/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.account;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.account.internal.AccountServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateRequest;
import software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateResponse;
import software.amazon.awssdk.services.account.model.AccessDeniedException;
import software.amazon.awssdk.services.account.model.AccountException;
import software.amazon.awssdk.services.account.model.ConflictException;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactRequest;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactResponse;
import software.amazon.awssdk.services.account.model.DisableRegionRequest;
import software.amazon.awssdk.services.account.model.DisableRegionResponse;
import software.amazon.awssdk.services.account.model.EnableRegionRequest;
import software.amazon.awssdk.services.account.model.EnableRegionResponse;
import software.amazon.awssdk.services.account.model.GetAlternateContactRequest;
import software.amazon.awssdk.services.account.model.GetAlternateContactResponse;
import software.amazon.awssdk.services.account.model.GetContactInformationRequest;
import software.amazon.awssdk.services.account.model.GetContactInformationResponse;
import software.amazon.awssdk.services.account.model.GetPrimaryEmailRequest;
import software.amazon.awssdk.services.account.model.GetPrimaryEmailResponse;
import software.amazon.awssdk.services.account.model.GetRegionOptStatusRequest;
import software.amazon.awssdk.services.account.model.GetRegionOptStatusResponse;
import software.amazon.awssdk.services.account.model.InternalServerException;
import software.amazon.awssdk.services.account.model.ListRegionsRequest;
import software.amazon.awssdk.services.account.model.ListRegionsResponse;
import software.amazon.awssdk.services.account.model.PutAlternateContactRequest;
import software.amazon.awssdk.services.account.model.PutAlternateContactResponse;
import software.amazon.awssdk.services.account.model.PutContactInformationRequest;
import software.amazon.awssdk.services.account.model.PutContactInformationResponse;
import software.amazon.awssdk.services.account.model.ResourceNotFoundException;
import software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateRequest;
import software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateResponse;
import software.amazon.awssdk.services.account.model.TooManyRequestsException;
import software.amazon.awssdk.services.account.model.ValidationException;
import software.amazon.awssdk.services.account.transform.AcceptPrimaryEmailUpdateRequestMarshaller;
import software.amazon.awssdk.services.account.transform.DeleteAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.DisableRegionRequestMarshaller;
import software.amazon.awssdk.services.account.transform.EnableRegionRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetContactInformationRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetPrimaryEmailRequestMarshaller;
import software.amazon.awssdk.services.account.transform.GetRegionOptStatusRequestMarshaller;
import software.amazon.awssdk.services.account.transform.ListRegionsRequestMarshaller;
import software.amazon.awssdk.services.account.transform.PutAlternateContactRequestMarshaller;
import software.amazon.awssdk.services.account.transform.PutContactInformationRequestMarshaller;
import software.amazon.awssdk.services.account.transform.StartPrimaryEmailUpdateRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AccountClient}.
 *
 * @see AccountClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAccountClient implements AccountClient {
    private static final Logger log = Logger.loggerFor(DefaultAccountClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAccountClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts the request that originated from <a>StartPrimaryEmailUpdate</a> to update the primary email address (also
     * known as the root user email address) for the specified account.
     * </p>
     *
     * @param acceptPrimaryEmailUpdateRequest
     * @return Result of the AcceptPrimaryEmailUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING).
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.AcceptPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/AcceptPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcceptPrimaryEmailUpdateResponse acceptPrimaryEmailUpdate(
            AcceptPrimaryEmailUpdateRequest acceptPrimaryEmailUpdateRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptPrimaryEmailUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AcceptPrimaryEmailUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptPrimaryEmailUpdateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptPrimaryEmailUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptPrimaryEmailUpdate");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptPrimaryEmailUpdateRequest, AcceptPrimaryEmailUpdateResponse>()
                            .withOperationName("AcceptPrimaryEmailUpdate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(acceptPrimaryEmailUpdateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptPrimaryEmailUpdateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified alternate contact from an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param deleteAlternateContactRequest
     * @return Result of the DeleteAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DeleteAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DeleteAlternateContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAlternateContactResponse deleteAlternateContact(DeleteAlternateContactRequest deleteAlternateContactRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAlternateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAlternateContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlternateContact");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAlternateContactRequest, DeleteAlternateContactResponse>()
                            .withOperationName("DeleteAlternateContact").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteAlternateContactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAlternateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables (opts-out) a particular Region for an account.
     * </p>
     * <note>
     * <p>
     * The act of disabling a Region will remove all IAM access to any resources that reside in that Region.
     * </p>
     * </note>
     *
     * @param disableRegionRequest
     * @return Result of the DisableRegion operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING).
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DisableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DisableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DisableRegionResponse disableRegion(DisableRegionRequest disableRegionRequest) throws ValidationException,
            ConflictException, AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisableRegionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableRegionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableRegion");

            return clientHandler.execute(new ClientExecutionParams<DisableRegionRequest, DisableRegionResponse>()
                    .withOperationName("DisableRegion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableRegionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableRegionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables (opts-in) a particular Region for an account.
     * </p>
     *
     * @param enableRegionRequest
     * @return Result of the EnableRegion operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING).
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.EnableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/EnableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EnableRegionResponse enableRegion(EnableRegionRequest enableRegionRequest) throws ValidationException,
            ConflictException, AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableRegionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableRegionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableRegion");

            return clientHandler.execute(new ClientExecutionParams<EnableRegionRequest, EnableRegionResponse>()
                    .withOperationName("EnableRegion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableRegionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableRegionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param getAlternateContactRequest
     * @return Result of the GetAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAlternateContactResponse getAlternateContact(GetAlternateContactRequest getAlternateContactRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAlternateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAlternateContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAlternateContact");

            return clientHandler.execute(new ClientExecutionParams<GetAlternateContactRequest, GetAlternateContactResponse>()
                    .withOperationName("GetAlternateContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAlternateContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAlternateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     *
     * @param getContactInformationRequest
     * @return Result of the GetContactInformation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetContactInformationResponse getContactInformation(GetContactInformationRequest getContactInformationRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetContactInformationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactInformationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactInformation");

            return clientHandler.execute(new ClientExecutionParams<GetContactInformationRequest, GetContactInformationResponse>()
                    .withOperationName("GetContactInformation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getContactInformationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactInformationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the primary email address for the specified account.
     * </p>
     *
     * @param getPrimaryEmailRequest
     * @return Result of the GetPrimaryEmail operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetPrimaryEmail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetPrimaryEmail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPrimaryEmailResponse getPrimaryEmail(GetPrimaryEmailRequest getPrimaryEmailRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPrimaryEmailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPrimaryEmailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPrimaryEmailRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPrimaryEmailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPrimaryEmail");

            return clientHandler.execute(new ClientExecutionParams<GetPrimaryEmailRequest, GetPrimaryEmailResponse>()
                    .withOperationName("GetPrimaryEmail").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPrimaryEmailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPrimaryEmailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the opt-in status of a particular Region.
     * </p>
     *
     * @param getRegionOptStatusRequest
     * @return Result of the GetRegionOptStatus operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetRegionOptStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetRegionOptStatus" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRegionOptStatusResponse getRegionOptStatus(GetRegionOptStatusRequest getRegionOptStatusRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRegionOptStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRegionOptStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegionOptStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegionOptStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegionOptStatus");

            return clientHandler.execute(new ClientExecutionParams<GetRegionOptStatusRequest, GetRegionOptStatusResponse>()
                    .withOperationName("GetRegionOptStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRegionOptStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRegionOptStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the Regions for a given account and their respective opt-in statuses. Optionally, this list can be
     * filtered by the <code>region-opt-status-contains</code> parameter.
     * </p>
     *
     * @param listRegionsRequest
     * @return Result of the ListRegions operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/ListRegions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRegionsResponse listRegions(ListRegionsRequest listRegionsRequest) throws ValidationException,
            AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRegionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRegionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRegionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegions");

            return clientHandler.execute(new ClientExecutionParams<ListRegionsRequest, ListRegionsResponse>()
                    .withOperationName("ListRegions").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listRegionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRegionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param putAlternateContactRequest
     * @return Result of the PutAlternateContact operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutAlternateContactResponse putAlternateContact(PutAlternateContactRequest putAlternateContactRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAlternateContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAlternateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAlternateContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAlternateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAlternateContact");

            return clientHandler.execute(new ClientExecutionParams<PutAlternateContactRequest, PutAlternateContactResponse>()
                    .withOperationName("PutAlternateContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAlternateContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAlternateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     *
     * @param putContactInformationRequest
     * @return Result of the PutContactInformation operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutContactInformationResponse putContactInformation(PutContactInformationRequest putContactInformationRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutContactInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutContactInformationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putContactInformationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContactInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContactInformation");

            return clientHandler.execute(new ClientExecutionParams<PutContactInformationRequest, PutContactInformationResponse>()
                    .withOperationName("PutContactInformation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putContactInformationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutContactInformationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the process to update the primary email address for the specified account.
     * </p>
     *
     * @param startPrimaryEmailUpdateRequest
     * @return Result of the StartPrimaryEmailUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING).
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.StartPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/StartPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartPrimaryEmailUpdateResponse startPrimaryEmailUpdate(StartPrimaryEmailUpdateRequest startPrimaryEmailUpdateRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, AccessDeniedException,
            TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException, AccountException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartPrimaryEmailUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartPrimaryEmailUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPrimaryEmailUpdateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPrimaryEmailUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Account");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPrimaryEmailUpdate");

            return clientHandler
                    .execute(new ClientExecutionParams<StartPrimaryEmailUpdateRequest, StartPrimaryEmailUpdateResponse>()
                            .withOperationName("StartPrimaryEmailUpdate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startPrimaryEmailUpdateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartPrimaryEmailUpdateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AccountServiceClientConfigurationBuilder serviceConfigBuilder = new AccountServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AccountException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build());
    }

    @Override
    public final AccountServiceClientConfiguration serviceClientConfiguration() {
        return new AccountServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
