/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Permissions designate which private CA actions can be performed by an Amazon Web Services service or entity. In order
 * for ACM to automatically renew private certificates, you must give the ACM service principal all available
 * permissions (<code>IssueCertificate</code>, <code>GetCertificate</code>, and <code>ListPermissions</code>).
 * Permissions can be assigned with the <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html">CreatePermission</a> action,
 * removed with the <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html">DeletePermission</a> action,
 * and listed with the <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html">ListPermissions</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Permission implements SdkPojo, Serializable, ToCopyableBuilder<Permission.Builder, Permission> {
    private static final SdkField<String> CERTIFICATE_AUTHORITY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CertificateAuthorityArn").getter(getter(Permission::certificateAuthorityArn))
            .setter(setter(Builder::certificateAuthorityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateAuthorityArn").build())
            .build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(Permission::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<String> PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Principal").getter(getter(Permission::principal)).setter(setter(Builder::principal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Principal").build()).build();

    private static final SdkField<String> SOURCE_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceAccount").getter(getter(Permission::sourceAccount)).setter(setter(Builder::sourceAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceAccount").build()).build();

    private static final SdkField<List<String>> ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Actions")
            .getter(getter(Permission::actionsAsStrings))
            .setter(setter(Builder::actionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Policy")
            .getter(getter(Permission::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(CERTIFICATE_AUTHORITY_ARN_FIELD, CREATED_AT_FIELD, PRINCIPAL_FIELD, SOURCE_ACCOUNT_FIELD, ACTIONS_FIELD,
                    POLICY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certificateAuthorityArn;

    private final Instant createdAt;

    private final String principal;

    private final String sourceAccount;

    private final List<String> actions;

    private final String policy;

    private Permission(BuilderImpl builder) {
        this.certificateAuthorityArn = builder.certificateAuthorityArn;
        this.createdAt = builder.createdAt;
        this.principal = builder.principal;
        this.sourceAccount = builder.sourceAccount;
        this.actions = builder.actions;
        this.policy = builder.policy;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
     */
    public final String certificateAuthorityArn() {
        return certificateAuthorityArn;
    }

    /**
     * <p>
     * The time at which the permission was created.
     * </p>
     * 
     * @return The time at which the permission was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Amazon Web Services service or entity that holds the permission. At this time, the only valid principal is
     * <code>acm.amazonaws.com</code>.
     * </p>
     * 
     * @return The Amazon Web Services service or entity that holds the permission. At this time, the only valid
     *         principal is <code>acm.amazonaws.com</code>.
     */
    public final String principal() {
        return principal;
    }

    /**
     * <p>
     * The ID of the account that assigned the permission.
     * </p>
     * 
     * @return The ID of the account that assigned the permission.
     */
    public final String sourceAccount() {
        return sourceAccount;
    }

    /**
     * <p>
     * The private CA actions that can be performed by the designated Amazon Web Services service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return The private CA actions that can be performed by the designated Amazon Web Services service.
     */
    public final List<ActionType> actions() {
        return ActionListCopier.copyStringToEnum(actions);
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The private CA actions that can be performed by the designated Amazon Web Services service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return The private CA actions that can be performed by the designated Amazon Web Services service.
     */
    public final List<String> actionsAsStrings() {
        return actions;
    }

    /**
     * <p>
     * The name of the policy that is associated with the permission.
     * </p>
     * 
     * @return The name of the policy that is associated with the permission.
     */
    public final String policy() {
        return policy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(principal());
        hashCode = 31 * hashCode + Objects.hashCode(sourceAccount());
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Permission)) {
            return false;
        }
        Permission other = (Permission) obj;
        return Objects.equals(certificateAuthorityArn(), other.certificateAuthorityArn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(principal(), other.principal())
                && Objects.equals(sourceAccount(), other.sourceAccount()) && hasActions() == other.hasActions()
                && Objects.equals(actionsAsStrings(), other.actionsAsStrings()) && Objects.equals(policy(), other.policy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Permission").add("CertificateAuthorityArn", certificateAuthorityArn())
                .add("CreatedAt", createdAt()).add("Principal", principal()).add("SourceAccount", sourceAccount())
                .add("Actions", hasActions() ? actionsAsStrings() : null).add("Policy", policy()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateAuthorityArn":
            return Optional.ofNullable(clazz.cast(certificateAuthorityArn()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "Principal":
            return Optional.ofNullable(clazz.cast(principal()));
        case "SourceAccount":
            return Optional.ofNullable(clazz.cast(sourceAccount()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actionsAsStrings()));
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Permission, T> g) {
        return obj -> g.apply((Permission) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Permission> {
        /**
         * <p>
         * The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
         * </p>
         * 
         * @param certificateAuthorityArn
         *        The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityArn(String certificateAuthorityArn);

        /**
         * <p>
         * The time at which the permission was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the permission was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Amazon Web Services service or entity that holds the permission. At this time, the only valid principal
         * is <code>acm.amazonaws.com</code>.
         * </p>
         * 
         * @param principal
         *        The Amazon Web Services service or entity that holds the permission. At this time, the only valid
         *        principal is <code>acm.amazonaws.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principal(String principal);

        /**
         * <p>
         * The ID of the account that assigned the permission.
         * </p>
         * 
         * @param sourceAccount
         *        The ID of the account that assigned the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceAccount(String sourceAccount);

        /**
         * <p>
         * The private CA actions that can be performed by the designated Amazon Web Services service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated Amazon Web Services service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsWithStrings(Collection<String> actions);

        /**
         * <p>
         * The private CA actions that can be performed by the designated Amazon Web Services service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated Amazon Web Services service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsWithStrings(String... actions);

        /**
         * <p>
         * The private CA actions that can be performed by the designated Amazon Web Services service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated Amazon Web Services service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<ActionType> actions);

        /**
         * <p>
         * The private CA actions that can be performed by the designated Amazon Web Services service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated Amazon Web Services service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ActionType... actions);

        /**
         * <p>
         * The name of the policy that is associated with the permission.
         * </p>
         * 
         * @param policy
         *        The name of the policy that is associated with the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);
    }

    static final class BuilderImpl implements Builder {
        private String certificateAuthorityArn;

        private Instant createdAt;

        private String principal;

        private String sourceAccount;

        private List<String> actions = DefaultSdkAutoConstructList.getInstance();

        private String policy;

        private BuilderImpl() {
        }

        private BuilderImpl(Permission model) {
            certificateAuthorityArn(model.certificateAuthorityArn);
            createdAt(model.createdAt);
            principal(model.principal);
            sourceAccount(model.sourceAccount);
            actionsWithStrings(model.actions);
            policy(model.policy);
        }

        public final String getCertificateAuthorityArn() {
            return certificateAuthorityArn;
        }

        public final void setCertificateAuthorityArn(String certificateAuthorityArn) {
            this.certificateAuthorityArn = certificateAuthorityArn;
        }

        @Override
        public final Builder certificateAuthorityArn(String certificateAuthorityArn) {
            this.certificateAuthorityArn = certificateAuthorityArn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getPrincipal() {
            return principal;
        }

        public final void setPrincipal(String principal) {
            this.principal = principal;
        }

        @Override
        public final Builder principal(String principal) {
            this.principal = principal;
            return this;
        }

        public final String getSourceAccount() {
            return sourceAccount;
        }

        public final void setSourceAccount(String sourceAccount) {
            this.sourceAccount = sourceAccount;
        }

        @Override
        public final Builder sourceAccount(String sourceAccount) {
            this.sourceAccount = sourceAccount;
            return this;
        }

        public final Collection<String> getActions() {
            if (actions instanceof SdkAutoConstructList) {
                return null;
            }
            return actions;
        }

        public final void setActions(Collection<String> actions) {
            this.actions = ActionListCopier.copy(actions);
        }

        @Override
        public final Builder actionsWithStrings(Collection<String> actions) {
            this.actions = ActionListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actionsWithStrings(String... actions) {
            actionsWithStrings(Arrays.asList(actions));
            return this;
        }

        @Override
        public final Builder actions(Collection<ActionType> actions) {
            this.actions = ActionListCopier.copyEnumToString(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(ActionType... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        public final String getPolicy() {
            return policy;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        @Override
        public Permission build() {
            return new Permission(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
