/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration information for a certificate revocation list (CRL). Your private certificate authority (CA)
 * creates base CRLs. Delta CRLs are not supported. You can enable CRLs for your new or an existing private CA by
 * setting the <b>Enabled</b> parameter to <code>true</code>. Your private CA writes CRLs to an S3 bucket that you
 * specify in the <b>S3BucketName</b> parameter. You can hide the name of your bucket by specifying a value for the
 * <b>CustomCname</b> parameter. Your private CA copies the CNAME or the S3 bucket name to the <b>CRL Distribution
 * Points</b> extension of each certificate it issues. Your S3 bucket policy must give write permission to ACM Private
 * CA.
 * </p>
 * <p>
 * ACM Private CA assets that are stored in Amazon S3 can be protected with encryption. For more information, see <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#crl-encryption">Encrypting Your CRLs</a>.
 * </p>
 * <p>
 * Your private CA uses the value in the <b>ExpirationInDays</b> parameter to calculate the <b>nextUpdate</b> field in
 * the CRL. The CRL is refreshed prior to a certificate's expiration date or when a certificate is revoked. When a
 * certificate is revoked, it appears in the CRL until the certificate expires, and then in one additional CRL after
 * expiration, and it always appears in the audit report.
 * </p>
 * <p>
 * A CRL is typically updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL update
 * fails, ACM Private CA makes further attempts every 15 minutes.
 * </p>
 * <p>
 * CRLs contain the following fields:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Version</b>: The current version number defined in RFC 5280 is V2. The integer value is 0x1.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Signature Algorithm</b>: The name of the algorithm used to sign the CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Issuer</b>: The X.500 distinguished name of your private CA that issued the CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Last Update</b>: The issue date and time of this CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Next Update</b>: The day and time by which the next CRL will be issued.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Revoked Certificates</b>: List of revoked certificates. Each list item contains the following information.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Serial Number</b>: The serial number, in hexadecimal format, of the revoked certificate.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Revocation Date</b>: Date and time the certificate was revoked.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>CRL Entry Extensions</b>: Optional extensions for the CRL entry.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>X509v3 CRL Reason Code</b>: Reason the certificate was revoked.
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>CRL Extensions</b>: Optional extensions for the CRL.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>X509v3 Authority Key Identifier</b>: Identifies the public key associated with the private key used to sign the
 * certificate.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>X509v3 CRL Number:</b>: Decimal sequence number for the CRL.
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>Signature Algorithm</b>: Algorithm used by your private CA to sign the CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Signature Value</b>: Signature computed over the CRL.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Certificate revocation lists created by ACM Private CA are DER-encoded. You can use the following OpenSSL command to
 * list a CRL.
 * </p>
 * <p>
 * <code>openssl crl -inform DER -text -in <i>crl_path</i> -noout</code>
 * </p>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/crl-planning.html">Planning a
 * certificate revocation list (CRL)</a> in the <i>Certificate Manager Private Certificate Authority (PCA) User
 * Guide</i>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CrlConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CrlConfiguration.Builder, CrlConfiguration> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(CrlConfiguration::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<Integer> EXPIRATION_IN_DAYS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExpirationInDays").getter(getter(CrlConfiguration::expirationInDays))
            .setter(setter(Builder::expirationInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationInDays").build()).build();

    private static final SdkField<String> CUSTOM_CNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomCname").getter(getter(CrlConfiguration::customCname)).setter(setter(Builder::customCname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomCname").build()).build();

    private static final SdkField<String> S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3BucketName").getter(getter(CrlConfiguration::s3BucketName)).setter(setter(Builder::s3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3BucketName").build()).build();

    private static final SdkField<String> S3_OBJECT_ACL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3ObjectAcl").getter(getter(CrlConfiguration::s3ObjectAclAsString)).setter(setter(Builder::s3ObjectAcl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3ObjectAcl").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD,
            EXPIRATION_IN_DAYS_FIELD, CUSTOM_CNAME_FIELD, S3_BUCKET_NAME_FIELD, S3_OBJECT_ACL_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final Integer expirationInDays;

    private final String customCname;

    private final String s3BucketName;

    private final String s3ObjectAcl;

    private CrlConfiguration(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.expirationInDays = builder.expirationInDays;
        this.customCname = builder.customCname;
        this.s3BucketName = builder.s3BucketName;
        this.s3ObjectAcl = builder.s3ObjectAcl;
    }

    /**
     * <p>
     * Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this value to
     * enable certificate revocation for a new CA when you call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
     * >UpdateCertificateAuthority</a> action.
     * </p>
     * 
     * @return Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this
     *         value to enable certificate revocation for a new CA when you call the <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     *         >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
     *         >UpdateCertificateAuthority</a> action.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Validity period of the CRL in days.
     * </p>
     * 
     * @return Validity period of the CRL in days.
     */
    public final Integer expirationInDays() {
        return expirationInDays;
    }

    /**
     * <p>
     * Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an alias for
     * the CRL distribution point. Use this value if you don't want the name of your S3 bucket to be public.
     * </p>
     * 
     * @return Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an
     *         alias for the CRL distribution point. Use this value if you don't want the name of your S3 bucket to be
     *         public.
     */
    public final String customCname() {
        return customCname;
    }

    /**
     * <p>
     * Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b> argument,
     * the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of the issued certificate.
     * You can change the name of your bucket by calling the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
     * >UpdateCertificateAuthority</a> operation. You must specify a <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-policies">bucket policy</a> that
     * allows ACM Private CA to write the CRL to your bucket.
     * </p>
     * 
     * @return Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b>
     *         argument, the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of the
     *         issued certificate. You can change the name of your bucket by calling the <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
     *         >UpdateCertificateAuthority</a> operation. You must specify a <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-policies">bucket
     *         policy</a> that allows ACM Private CA to write the CRL to your bucket.
     */
    public final String s3BucketName() {
        return s3BucketName;
    }

    /**
     * <p>
     * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you choose
     * PUBLIC_READ, the CRL will be accessible over the public internet. If you choose BUCKET_OWNER_FULL_CONTROL, only
     * the owner of the CRL S3 bucket can access the CRL, and your PKI clients may need an alternative method of access.
     * </p>
     * <p>
     * If no value is specified, the default is <code>PUBLIC_READ</code>.
     * </p>
     * <p>
     * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the Block
     * Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
     * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
     * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access to the
     * S3 bucket</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3ObjectAcl} will
     * return {@link S3ObjectAcl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3ObjectAclAsString}.
     * </p>
     * 
     * @return Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
     *         you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
     *         BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients
     *         may need an alternative method of access. </p>
     *         <p>
     *         If no value is specified, the default is <code>PUBLIC_READ</code>.
     *         </p>
     *         <p>
     *         <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled
     *         the Block Public Access (BPA) feature in your S3 account, then you must specify the value of this
     *         parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have
     *         disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
     *         <code>PUBLIC_READ</code> as the value.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
     *         access to the S3 bucket</a>.
     * @see S3ObjectAcl
     */
    public final S3ObjectAcl s3ObjectAcl() {
        return S3ObjectAcl.fromValue(s3ObjectAcl);
    }

    /**
     * <p>
     * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you choose
     * PUBLIC_READ, the CRL will be accessible over the public internet. If you choose BUCKET_OWNER_FULL_CONTROL, only
     * the owner of the CRL S3 bucket can access the CRL, and your PKI clients may need an alternative method of access.
     * </p>
     * <p>
     * If no value is specified, the default is <code>PUBLIC_READ</code>.
     * </p>
     * <p>
     * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the Block
     * Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
     * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
     * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access to the
     * S3 bucket</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3ObjectAcl} will
     * return {@link S3ObjectAcl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3ObjectAclAsString}.
     * </p>
     * 
     * @return Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
     *         you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
     *         BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients
     *         may need an alternative method of access. </p>
     *         <p>
     *         If no value is specified, the default is <code>PUBLIC_READ</code>.
     *         </p>
     *         <p>
     *         <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled
     *         the Block Public Access (BPA) feature in your S3 account, then you must specify the value of this
     *         parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have
     *         disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
     *         <code>PUBLIC_READ</code> as the value.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
     *         access to the S3 bucket</a>.
     * @see S3ObjectAcl
     */
    public final String s3ObjectAclAsString() {
        return s3ObjectAcl;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(expirationInDays());
        hashCode = 31 * hashCode + Objects.hashCode(customCname());
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(s3ObjectAclAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CrlConfiguration)) {
            return false;
        }
        CrlConfiguration other = (CrlConfiguration) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(expirationInDays(), other.expirationInDays())
                && Objects.equals(customCname(), other.customCname()) && Objects.equals(s3BucketName(), other.s3BucketName())
                && Objects.equals(s3ObjectAclAsString(), other.s3ObjectAclAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CrlConfiguration").add("Enabled", enabled()).add("ExpirationInDays", expirationInDays())
                .add("CustomCname", customCname()).add("S3BucketName", s3BucketName()).add("S3ObjectAcl", s3ObjectAclAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "ExpirationInDays":
            return Optional.ofNullable(clazz.cast(expirationInDays()));
        case "CustomCname":
            return Optional.ofNullable(clazz.cast(customCname()));
        case "S3BucketName":
            return Optional.ofNullable(clazz.cast(s3BucketName()));
        case "S3ObjectAcl":
            return Optional.ofNullable(clazz.cast(s3ObjectAclAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CrlConfiguration, T> g) {
        return obj -> g.apply((CrlConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CrlConfiguration> {
        /**
         * <p>
         * Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this value
         * to enable certificate revocation for a new CA when you call the <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
         * >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
         * >UpdateCertificateAuthority</a> action.
         * </p>
         * 
         * @param enabled
         *        Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this
         *        value to enable certificate revocation for a new CA when you call the <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
         *        >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
         *        >UpdateCertificateAuthority</a> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Validity period of the CRL in days.
         * </p>
         * 
         * @param expirationInDays
         *        Validity period of the CRL in days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationInDays(Integer expirationInDays);

        /**
         * <p>
         * Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an alias
         * for the CRL distribution point. Use this value if you don't want the name of your S3 bucket to be public.
         * </p>
         * 
         * @param customCname
         *        Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an
         *        alias for the CRL distribution point. Use this value if you don't want the name of your S3 bucket to
         *        be public.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customCname(String customCname);

        /**
         * <p>
         * Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b>
         * argument, the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of the
         * issued certificate. You can change the name of your bucket by calling the <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
         * >UpdateCertificateAuthority</a> operation. You must specify a <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-policies">bucket policy</a>
         * that allows ACM Private CA to write the CRL to your bucket.
         * </p>
         * 
         * @param s3BucketName
         *        Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b>
         *        argument, the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of
         *        the issued certificate. You can change the name of your bucket by calling the <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
         *        >UpdateCertificateAuthority</a> operation. You must specify a <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-policies">bucket
         *        policy</a> that allows ACM Private CA to write the CRL to your bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketName(String s3BucketName);

        /**
         * <p>
         * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you
         * choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         * BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients may
         * need an alternative method of access.
         * </p>
         * <p>
         * If no value is specified, the default is <code>PUBLIC_READ</code>.
         * </p>
         * <p>
         * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the
         * Block Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
         * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
         * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access to
         * the S3 bucket</a>.
         * </p>
         * 
         * @param s3ObjectAcl
         *        Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
         *        you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         *        BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI
         *        clients may need an alternative method of access. </p>
         *        <p>
         *        If no value is specified, the default is <code>PUBLIC_READ</code>.
         *        </p>
         *        <p>
         *        <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have
         *        enabled the Block Public Access (BPA) feature in your S3 account, then you must specify the value of
         *        this parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you
         *        have disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
         *        <code>PUBLIC_READ</code> as the value.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
         *        access to the S3 bucket</a>.
         * @see S3ObjectAcl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectAcl
         */
        Builder s3ObjectAcl(String s3ObjectAcl);

        /**
         * <p>
         * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you
         * choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         * BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients may
         * need an alternative method of access.
         * </p>
         * <p>
         * If no value is specified, the default is <code>PUBLIC_READ</code>.
         * </p>
         * <p>
         * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the
         * Block Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
         * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
         * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access to
         * the S3 bucket</a>.
         * </p>
         * 
         * @param s3ObjectAcl
         *        Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
         *        you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         *        BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI
         *        clients may need an alternative method of access. </p>
         *        <p>
         *        If no value is specified, the default is <code>PUBLIC_READ</code>.
         *        </p>
         *        <p>
         *        <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have
         *        enabled the Block Public Access (BPA) feature in your S3 account, then you must specify the value of
         *        this parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you
         *        have disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
         *        <code>PUBLIC_READ</code> as the value.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
         *        access to the S3 bucket</a>.
         * @see S3ObjectAcl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectAcl
         */
        Builder s3ObjectAcl(S3ObjectAcl s3ObjectAcl);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private Integer expirationInDays;

        private String customCname;

        private String s3BucketName;

        private String s3ObjectAcl;

        private BuilderImpl() {
        }

        private BuilderImpl(CrlConfiguration model) {
            enabled(model.enabled);
            expirationInDays(model.expirationInDays);
            customCname(model.customCname);
            s3BucketName(model.s3BucketName);
            s3ObjectAcl(model.s3ObjectAcl);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Integer getExpirationInDays() {
            return expirationInDays;
        }

        public final void setExpirationInDays(Integer expirationInDays) {
            this.expirationInDays = expirationInDays;
        }

        @Override
        public final Builder expirationInDays(Integer expirationInDays) {
            this.expirationInDays = expirationInDays;
            return this;
        }

        public final String getCustomCname() {
            return customCname;
        }

        public final void setCustomCname(String customCname) {
            this.customCname = customCname;
        }

        @Override
        public final Builder customCname(String customCname) {
            this.customCname = customCname;
            return this;
        }

        public final String getS3BucketName() {
            return s3BucketName;
        }

        public final void setS3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
        }

        @Override
        public final Builder s3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
            return this;
        }

        public final String getS3ObjectAcl() {
            return s3ObjectAcl;
        }

        public final void setS3ObjectAcl(String s3ObjectAcl) {
            this.s3ObjectAcl = s3ObjectAcl;
        }

        @Override
        public final Builder s3ObjectAcl(String s3ObjectAcl) {
            this.s3ObjectAcl = s3ObjectAcl;
            return this;
        }

        @Override
        public final Builder s3ObjectAcl(S3ObjectAcl s3ObjectAcl) {
            this.s3ObjectAcl(s3ObjectAcl == null ? null : s3ObjectAcl.toString());
            return this;
        }

        @Override
        public CrlConfiguration build() {
            return new CrlConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
