/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class IssueCertificateRequest extends AcmPcaRequest implements
        ToCopyableBuilder<IssueCertificateRequest.Builder, IssueCertificateRequest> {
    private static final SdkField<ApiPassthrough> API_PASSTHROUGH_FIELD = SdkField
            .<ApiPassthrough> builder(MarshallingType.SDK_POJO).memberName("ApiPassthrough")
            .getter(getter(IssueCertificateRequest::apiPassthrough)).setter(setter(Builder::apiPassthrough))
            .constructor(ApiPassthrough::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApiPassthrough").build()).build();

    private static final SdkField<String> CERTIFICATE_AUTHORITY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CertificateAuthorityArn").getter(getter(IssueCertificateRequest::certificateAuthorityArn))
            .setter(setter(Builder::certificateAuthorityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateAuthorityArn").build())
            .build();

    private static final SdkField<SdkBytes> CSR_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES).memberName("Csr")
            .getter(getter(IssueCertificateRequest::csr)).setter(setter(Builder::csr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Csr").build()).build();

    private static final SdkField<String> SIGNING_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningAlgorithm").getter(getter(IssueCertificateRequest::signingAlgorithmAsString))
            .setter(setter(Builder::signingAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningAlgorithm").build()).build();

    private static final SdkField<String> TEMPLATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateArn").getter(getter(IssueCertificateRequest::templateArn)).setter(setter(Builder::templateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateArn").build()).build();

    private static final SdkField<Validity> VALIDITY_FIELD = SdkField.<Validity> builder(MarshallingType.SDK_POJO)
            .memberName("Validity").getter(getter(IssueCertificateRequest::validity)).setter(setter(Builder::validity))
            .constructor(Validity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Validity").build()).build();

    private static final SdkField<Validity> VALIDITY_NOT_BEFORE_FIELD = SdkField.<Validity> builder(MarshallingType.SDK_POJO)
            .memberName("ValidityNotBefore").getter(getter(IssueCertificateRequest::validityNotBefore))
            .setter(setter(Builder::validityNotBefore)).constructor(Validity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidityNotBefore").build()).build();

    private static final SdkField<String> IDEMPOTENCY_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdempotencyToken").getter(getter(IssueCertificateRequest::idempotencyToken))
            .setter(setter(Builder::idempotencyToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdempotencyToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(API_PASSTHROUGH_FIELD,
            CERTIFICATE_AUTHORITY_ARN_FIELD, CSR_FIELD, SIGNING_ALGORITHM_FIELD, TEMPLATE_ARN_FIELD, VALIDITY_FIELD,
            VALIDITY_NOT_BEFORE_FIELD, IDEMPOTENCY_TOKEN_FIELD));

    private final ApiPassthrough apiPassthrough;

    private final String certificateAuthorityArn;

    private final SdkBytes csr;

    private final String signingAlgorithm;

    private final String templateArn;

    private final Validity validity;

    private final Validity validityNotBefore;

    private final String idempotencyToken;

    private IssueCertificateRequest(BuilderImpl builder) {
        super(builder);
        this.apiPassthrough = builder.apiPassthrough;
        this.certificateAuthorityArn = builder.certificateAuthorityArn;
        this.csr = builder.csr;
        this.signingAlgorithm = builder.signingAlgorithm;
        this.templateArn = builder.templateArn;
        this.validity = builder.validity;
        this.validityNotBefore = builder.validityNotBefore;
        this.idempotencyToken = builder.idempotencyToken;
    }

    /**
     * <p>
     * Specifies X.509 certificate information to be included in the issued certificate. An <code>APIPassthrough</code>
     * or <code>APICSRPassthrough</code> template variant must be selected, or else this parameter is ignored. For more
     * information about using these templates, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
     * Templates</a>.
     * </p>
     * <p>
     * If conflicting or duplicate certificate information is supplied during certificate issuance, ACM Private CA
     * applies <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html#template-order-of-operations"
     * >order of operation rules</a> to determine what information is used.
     * </p>
     * 
     * @return Specifies X.509 certificate information to be included in the issued certificate. An
     *         <code>APIPassthrough</code> or <code>APICSRPassthrough</code> template variant must be selected, or else
     *         this parameter is ignored. For more information about using these templates, see <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
     *         Templates</a>.</p>
     *         <p>
     *         If conflicting or duplicate certificate information is supplied during certificate issuance, ACM Private
     *         CA applies <a href=
     *         "https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html#template-order-of-operations"
     *         >order of operation rules</a> to determine what information is used.
     */
    public final ApiPassthrough apiPassthrough() {
        return apiPassthrough;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that was returned when you called <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a>. This must be of the form:
     * </p>
     * <p>
     * <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i> </code>
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that was returned when you called <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     *         >CreateCertificateAuthority</a>. This must be of the form:</p>
     *         <p>
     *         <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i> </code>
     */
    public final String certificateAuthorityArn() {
        return certificateAuthorityArn;
    }

    /**
     * <p>
     * The certificate signing request (CSR) for the certificate you want to issue. As an example, you can use the
     * following OpenSSL command to create the CSR and a 2048 bit RSA private key.
     * </p>
     * <p>
     * <code>openssl req -new -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
     * </p>
     * <p>
     * If you have a configuration file, you can then use the following OpenSSL command. The <code>usr_cert</code> block
     * in the configuration file contains your X509 version 3 extensions.
     * </p>
     * <p>
     * <code>openssl req -new -config openssl_rsa.cnf -extensions usr_cert -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
     * </p>
     * <p>
     * Note: A CSR must provide either a <i>subject name</i> or a <i>subject alternative name</i> or the request will be
     * rejected.
     * </p>
     * 
     * @return The certificate signing request (CSR) for the certificate you want to issue. As an example, you can use
     *         the following OpenSSL command to create the CSR and a 2048 bit RSA private key. </p>
     *         <p>
     *         <code>openssl req -new -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
     *         </p>
     *         <p>
     *         If you have a configuration file, you can then use the following OpenSSL command. The
     *         <code>usr_cert</code> block in the configuration file contains your X509 version 3 extensions.
     *         </p>
     *         <p>
     *         <code>openssl req -new -config openssl_rsa.cnf -extensions usr_cert -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
     *         </p>
     *         <p>
     *         Note: A CSR must provide either a <i>subject name</i> or a <i>subject alternative name</i> or the request
     *         will be rejected.
     */
    public final SdkBytes csr() {
        return csr;
    }

    /**
     * <p>
     * The name of the algorithm that will be used to sign the certificate to be issued.
     * </p>
     * <p>
     * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a CSR in the
     * <code>CreateCertificateAuthority</code> action.
     * </p>
     * <note>
     * <p>
     * The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's secret key.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return The name of the algorithm that will be used to sign the certificate to be issued. </p>
     *         <p>
     *         This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a CSR
     *         in the <code>CreateCertificateAuthority</code> action.
     *         </p>
     *         <note>
     *         <p>
     *         The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's secret
     *         key.
     *         </p>
     * @see SigningAlgorithm
     */
    public final SigningAlgorithm signingAlgorithm() {
        return SigningAlgorithm.fromValue(signingAlgorithm);
    }

    /**
     * <p>
     * The name of the algorithm that will be used to sign the certificate to be issued.
     * </p>
     * <p>
     * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a CSR in the
     * <code>CreateCertificateAuthority</code> action.
     * </p>
     * <note>
     * <p>
     * The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's secret key.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return The name of the algorithm that will be used to sign the certificate to be issued. </p>
     *         <p>
     *         This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a CSR
     *         in the <code>CreateCertificateAuthority</code> action.
     *         </p>
     *         <note>
     *         <p>
     *         The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's secret
     *         key.
     *         </p>
     * @see SigningAlgorithm
     */
    public final String signingAlgorithmAsString() {
        return signingAlgorithm;
    }

    /**
     * <p>
     * Specifies a custom configuration template to use when issuing a certificate. If this parameter is not provided,
     * ACM Private CA defaults to the <code>EndEntityCertificate/V1</code> template. For CA certificates, you should
     * choose the shortest path length that meets your needs. The path length is indicated by the PathLen<i>N</i>
     * portion of the ARN, where <i>N</i> is the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaTerms.html#terms-cadepth">CA depth</a>.
     * </p>
     * <p>
     * Note: The CA depth configured on a subordinate CA certificate must not exceed the limit set by its parents in the
     * CA hierarchy.
     * </p>
     * <p>
     * For a list of <code>TemplateArn</code> values supported by ACM Private CA, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
     * Templates</a>.
     * </p>
     * 
     * @return Specifies a custom configuration template to use when issuing a certificate. If this parameter is not
     *         provided, ACM Private CA defaults to the <code>EndEntityCertificate/V1</code> template. For CA
     *         certificates, you should choose the shortest path length that meets your needs. The path length is
     *         indicated by the PathLen<i>N</i> portion of the ARN, where <i>N</i> is the <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaTerms.html#terms-cadepth">CA depth</a>.</p>
     *         <p>
     *         Note: The CA depth configured on a subordinate CA certificate must not exceed the limit set by its
     *         parents in the CA hierarchy.
     *         </p>
     *         <p>
     *         For a list of <code>TemplateArn</code> values supported by ACM Private CA, see <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
     *         Templates</a>.
     */
    public final String templateArn() {
        return templateArn;
    }

    /**
     * <p>
     * Information describing the end of the validity period of the certificate. This parameter sets the “Not After”
     * date for the certificate.
     * </p>
     * <p>
     * Certificate validity is the period of time during which a certificate is valid. Validity can be expressed as an
     * explicit date and time when the certificate expires, or as a span of time after issuance, stated in days, months,
     * or years. For more information, see <a
     * href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC 5280.
     * </p>
     * <p>
     * This value is unaffected when <code>ValidityNotBefore</code> is also specified. For example, if
     * <code>Validity</code> is set to 20 days in the future, the certificate will expire 20 days from issuance time
     * regardless of the <code>ValidityNotBefore</code> value.
     * </p>
     * <p>
     * The end of the validity period configured on a certificate must not exceed the limit set on its parents in the CA
     * hierarchy.
     * </p>
     * 
     * @return Information describing the end of the validity period of the certificate. This parameter sets the “Not
     *         After” date for the certificate.</p>
     *         <p>
     *         Certificate validity is the period of time during which a certificate is valid. Validity can be expressed
     *         as an explicit date and time when the certificate expires, or as a span of time after issuance, stated in
     *         days, months, or years. For more information, see <a
     *         href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC 5280.
     *         </p>
     *         <p>
     *         This value is unaffected when <code>ValidityNotBefore</code> is also specified. For example, if
     *         <code>Validity</code> is set to 20 days in the future, the certificate will expire 20 days from issuance
     *         time regardless of the <code>ValidityNotBefore</code> value.
     *         </p>
     *         <p>
     *         The end of the validity period configured on a certificate must not exceed the limit set on its parents
     *         in the CA hierarchy.
     */
    public final Validity validity() {
        return validity;
    }

    /**
     * <p>
     * Information describing the start of the validity period of the certificate. This parameter sets the “Not Before"
     * date for the certificate.
     * </p>
     * <p>
     * By default, when issuing a certificate, ACM Private CA sets the "Not Before" date to the issuance time minus 60
     * minutes. This compensates for clock inconsistencies across computer systems. The <code>ValidityNotBefore</code>
     * parameter can be used to customize the “Not Before” value.
     * </p>
     * <p>
     * Unlike the <code>Validity</code> parameter, the <code>ValidityNotBefore</code> parameter is optional.
     * </p>
     * <p>
     * The <code>ValidityNotBefore</code> value is expressed as an explicit date and time, using the
     * <code>Validity</code> type value <code>ABSOLUTE</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_Validity.html">Validity</a> in this API
     * reference and <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC 5280.
     * </p>
     * 
     * @return Information describing the start of the validity period of the certificate. This parameter sets the “Not
     *         Before" date for the certificate.</p>
     *         <p>
     *         By default, when issuing a certificate, ACM Private CA sets the "Not Before" date to the issuance time
     *         minus 60 minutes. This compensates for clock inconsistencies across computer systems. The
     *         <code>ValidityNotBefore</code> parameter can be used to customize the “Not Before” value.
     *         </p>
     *         <p>
     *         Unlike the <code>Validity</code> parameter, the <code>ValidityNotBefore</code> parameter is optional.
     *         </p>
     *         <p>
     *         The <code>ValidityNotBefore</code> value is expressed as an explicit date and time, using the
     *         <code>Validity</code> type value <code>ABSOLUTE</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_Validity.html">Validity</a> in this API
     *         reference and <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC
     *         5280.
     */
    public final Validity validityNotBefore() {
        return validityNotBefore;
    }

    /**
     * <p>
     * Alphanumeric string that can be used to distinguish between calls to the <b>IssueCertificate</b> action.
     * Idempotency tokens for <b>IssueCertificate</b> time out after one minute. Therefore, if you call
     * <b>IssueCertificate</b> multiple times with the same idempotency token within one minute, ACM Private CA
     * recognizes that you are requesting only one certificate and will issue only one. If you change the idempotency
     * token for each call, PCA recognizes that you are requesting multiple certificates.
     * </p>
     * 
     * @return Alphanumeric string that can be used to distinguish between calls to the <b>IssueCertificate</b> action.
     *         Idempotency tokens for <b>IssueCertificate</b> time out after one minute. Therefore, if you call
     *         <b>IssueCertificate</b> multiple times with the same idempotency token within one minute, ACM Private CA
     *         recognizes that you are requesting only one certificate and will issue only one. If you change the
     *         idempotency token for each call, PCA recognizes that you are requesting multiple certificates.
     */
    public final String idempotencyToken() {
        return idempotencyToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(apiPassthrough());
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityArn());
        hashCode = 31 * hashCode + Objects.hashCode(csr());
        hashCode = 31 * hashCode + Objects.hashCode(signingAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(templateArn());
        hashCode = 31 * hashCode + Objects.hashCode(validity());
        hashCode = 31 * hashCode + Objects.hashCode(validityNotBefore());
        hashCode = 31 * hashCode + Objects.hashCode(idempotencyToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IssueCertificateRequest)) {
            return false;
        }
        IssueCertificateRequest other = (IssueCertificateRequest) obj;
        return Objects.equals(apiPassthrough(), other.apiPassthrough())
                && Objects.equals(certificateAuthorityArn(), other.certificateAuthorityArn())
                && Objects.equals(csr(), other.csr())
                && Objects.equals(signingAlgorithmAsString(), other.signingAlgorithmAsString())
                && Objects.equals(templateArn(), other.templateArn()) && Objects.equals(validity(), other.validity())
                && Objects.equals(validityNotBefore(), other.validityNotBefore())
                && Objects.equals(idempotencyToken(), other.idempotencyToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IssueCertificateRequest").add("ApiPassthrough", apiPassthrough())
                .add("CertificateAuthorityArn", certificateAuthorityArn()).add("Csr", csr())
                .add("SigningAlgorithm", signingAlgorithmAsString()).add("TemplateArn", templateArn())
                .add("Validity", validity()).add("ValidityNotBefore", validityNotBefore())
                .add("IdempotencyToken", idempotencyToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApiPassthrough":
            return Optional.ofNullable(clazz.cast(apiPassthrough()));
        case "CertificateAuthorityArn":
            return Optional.ofNullable(clazz.cast(certificateAuthorityArn()));
        case "Csr":
            return Optional.ofNullable(clazz.cast(csr()));
        case "SigningAlgorithm":
            return Optional.ofNullable(clazz.cast(signingAlgorithmAsString()));
        case "TemplateArn":
            return Optional.ofNullable(clazz.cast(templateArn()));
        case "Validity":
            return Optional.ofNullable(clazz.cast(validity()));
        case "ValidityNotBefore":
            return Optional.ofNullable(clazz.cast(validityNotBefore()));
        case "IdempotencyToken":
            return Optional.ofNullable(clazz.cast(idempotencyToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IssueCertificateRequest, T> g) {
        return obj -> g.apply((IssueCertificateRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends AcmPcaRequest.Builder, SdkPojo, CopyableBuilder<Builder, IssueCertificateRequest> {
        /**
         * <p>
         * Specifies X.509 certificate information to be included in the issued certificate. An
         * <code>APIPassthrough</code> or <code>APICSRPassthrough</code> template variant must be selected, or else this
         * parameter is ignored. For more information about using these templates, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
         * Templates</a>.
         * </p>
         * <p>
         * If conflicting or duplicate certificate information is supplied during certificate issuance, ACM Private CA
         * applies <a href=
         * "https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html#template-order-of-operations">order
         * of operation rules</a> to determine what information is used.
         * </p>
         * 
         * @param apiPassthrough
         *        Specifies X.509 certificate information to be included in the issued certificate. An
         *        <code>APIPassthrough</code> or <code>APICSRPassthrough</code> template variant must be selected, or
         *        else this parameter is ignored. For more information about using these templates, see <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding
         *        Certificate Templates</a>.</p>
         *        <p>
         *        If conflicting or duplicate certificate information is supplied during certificate issuance, ACM
         *        Private CA applies <a href=
         *        "https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html#template-order-of-operations"
         *        >order of operation rules</a> to determine what information is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiPassthrough(ApiPassthrough apiPassthrough);

        /**
         * <p>
         * Specifies X.509 certificate information to be included in the issued certificate. An
         * <code>APIPassthrough</code> or <code>APICSRPassthrough</code> template variant must be selected, or else this
         * parameter is ignored. For more information about using these templates, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
         * Templates</a>.
         * </p>
         * <p>
         * If conflicting or duplicate certificate information is supplied during certificate issuance, ACM Private CA
         * applies <a href=
         * "https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html#template-order-of-operations">order
         * of operation rules</a> to determine what information is used.
         * </p>
         * This is a convenience method that creates an instance of the {@link ApiPassthrough.Builder} avoiding the need
         * to create one manually via {@link ApiPassthrough#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ApiPassthrough.Builder#build()} is called immediately and its
         * result is passed to {@link #apiPassthrough(ApiPassthrough)}.
         * 
         * @param apiPassthrough
         *        a consumer that will call methods on {@link ApiPassthrough.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #apiPassthrough(ApiPassthrough)
         */
        default Builder apiPassthrough(Consumer<ApiPassthrough.Builder> apiPassthrough) {
            return apiPassthrough(ApiPassthrough.builder().applyMutation(apiPassthrough).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) that was returned when you called <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
         * >CreateCertificateAuthority</a>. This must be of the form:
         * </p>
         * <p>
         * <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i> </code>
         * </p>
         * 
         * @param certificateAuthorityArn
         *        The Amazon Resource Name (ARN) that was returned when you called <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
         *        >CreateCertificateAuthority</a>. This must be of the form:</p>
         *        <p>
         *        <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i> </code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityArn(String certificateAuthorityArn);

        /**
         * <p>
         * The certificate signing request (CSR) for the certificate you want to issue. As an example, you can use the
         * following OpenSSL command to create the CSR and a 2048 bit RSA private key.
         * </p>
         * <p>
         * <code>openssl req -new -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
         * </p>
         * <p>
         * If you have a configuration file, you can then use the following OpenSSL command. The <code>usr_cert</code>
         * block in the configuration file contains your X509 version 3 extensions.
         * </p>
         * <p>
         * <code>openssl req -new -config openssl_rsa.cnf -extensions usr_cert -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
         * </p>
         * <p>
         * Note: A CSR must provide either a <i>subject name</i> or a <i>subject alternative name</i> or the request
         * will be rejected.
         * </p>
         * 
         * @param csr
         *        The certificate signing request (CSR) for the certificate you want to issue. As an example, you can
         *        use the following OpenSSL command to create the CSR and a 2048 bit RSA private key. </p>
         *        <p>
         *        <code>openssl req -new -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
         *        </p>
         *        <p>
         *        If you have a configuration file, you can then use the following OpenSSL command. The
         *        <code>usr_cert</code> block in the configuration file contains your X509 version 3 extensions.
         *        </p>
         *        <p>
         *        <code>openssl req -new -config openssl_rsa.cnf -extensions usr_cert -newkey rsa:2048 -days 365 -keyout private/test_cert_priv_key.pem -out csr/test_cert_.csr</code>
         *        </p>
         *        <p>
         *        Note: A CSR must provide either a <i>subject name</i> or a <i>subject alternative name</i> or the
         *        request will be rejected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder csr(SdkBytes csr);

        /**
         * <p>
         * The name of the algorithm that will be used to sign the certificate to be issued.
         * </p>
         * <p>
         * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a CSR in
         * the <code>CreateCertificateAuthority</code> action.
         * </p>
         * <note>
         * <p>
         * The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's secret key.
         * </p>
         * </note>
         * 
         * @param signingAlgorithm
         *        The name of the algorithm that will be used to sign the certificate to be issued. </p>
         *        <p>
         *        This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a
         *        CSR in the <code>CreateCertificateAuthority</code> action.
         *        </p>
         *        <note>
         *        <p>
         *        The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's
         *        secret key.
         *        </p>
         * @see SigningAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlgorithm
         */
        Builder signingAlgorithm(String signingAlgorithm);

        /**
         * <p>
         * The name of the algorithm that will be used to sign the certificate to be issued.
         * </p>
         * <p>
         * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a CSR in
         * the <code>CreateCertificateAuthority</code> action.
         * </p>
         * <note>
         * <p>
         * The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's secret key.
         * </p>
         * </note>
         * 
         * @param signingAlgorithm
         *        The name of the algorithm that will be used to sign the certificate to be issued. </p>
         *        <p>
         *        This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign a
         *        CSR in the <code>CreateCertificateAuthority</code> action.
         *        </p>
         *        <note>
         *        <p>
         *        The specified signing algorithm family (RSA or ECDSA) much match the algorithm family of the CA's
         *        secret key.
         *        </p>
         * @see SigningAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlgorithm
         */
        Builder signingAlgorithm(SigningAlgorithm signingAlgorithm);

        /**
         * <p>
         * Specifies a custom configuration template to use when issuing a certificate. If this parameter is not
         * provided, ACM Private CA defaults to the <code>EndEntityCertificate/V1</code> template. For CA certificates,
         * you should choose the shortest path length that meets your needs. The path length is indicated by the
         * PathLen<i>N</i> portion of the ARN, where <i>N</i> is the <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaTerms.html#terms-cadepth">CA depth</a>.
         * </p>
         * <p>
         * Note: The CA depth configured on a subordinate CA certificate must not exceed the limit set by its parents in
         * the CA hierarchy.
         * </p>
         * <p>
         * For a list of <code>TemplateArn</code> values supported by ACM Private CA, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding Certificate
         * Templates</a>.
         * </p>
         * 
         * @param templateArn
         *        Specifies a custom configuration template to use when issuing a certificate. If this parameter is not
         *        provided, ACM Private CA defaults to the <code>EndEntityCertificate/V1</code> template. For CA
         *        certificates, you should choose the shortest path length that meets your needs. The path length is
         *        indicated by the PathLen<i>N</i> portion of the ARN, where <i>N</i> is the <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaTerms.html#terms-cadepth">CA
         *        depth</a>.</p>
         *        <p>
         *        Note: The CA depth configured on a subordinate CA certificate must not exceed the limit set by its
         *        parents in the CA hierarchy.
         *        </p>
         *        <p>
         *        For a list of <code>TemplateArn</code> values supported by ACM Private CA, see <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/userguide/UsingTemplates.html">Understanding
         *        Certificate Templates</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateArn(String templateArn);

        /**
         * <p>
         * Information describing the end of the validity period of the certificate. This parameter sets the “Not After”
         * date for the certificate.
         * </p>
         * <p>
         * Certificate validity is the period of time during which a certificate is valid. Validity can be expressed as
         * an explicit date and time when the certificate expires, or as a span of time after issuance, stated in days,
         * months, or years. For more information, see <a
         * href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC 5280.
         * </p>
         * <p>
         * This value is unaffected when <code>ValidityNotBefore</code> is also specified. For example, if
         * <code>Validity</code> is set to 20 days in the future, the certificate will expire 20 days from issuance time
         * regardless of the <code>ValidityNotBefore</code> value.
         * </p>
         * <p>
         * The end of the validity period configured on a certificate must not exceed the limit set on its parents in
         * the CA hierarchy.
         * </p>
         * 
         * @param validity
         *        Information describing the end of the validity period of the certificate. This parameter sets the “Not
         *        After” date for the certificate.</p>
         *        <p>
         *        Certificate validity is the period of time during which a certificate is valid. Validity can be
         *        expressed as an explicit date and time when the certificate expires, or as a span of time after
         *        issuance, stated in days, months, or years. For more information, see <a
         *        href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC 5280.
         *        </p>
         *        <p>
         *        This value is unaffected when <code>ValidityNotBefore</code> is also specified. For example, if
         *        <code>Validity</code> is set to 20 days in the future, the certificate will expire 20 days from
         *        issuance time regardless of the <code>ValidityNotBefore</code> value.
         *        </p>
         *        <p>
         *        The end of the validity period configured on a certificate must not exceed the limit set on its
         *        parents in the CA hierarchy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validity(Validity validity);

        /**
         * <p>
         * Information describing the end of the validity period of the certificate. This parameter sets the “Not After”
         * date for the certificate.
         * </p>
         * <p>
         * Certificate validity is the period of time during which a certificate is valid. Validity can be expressed as
         * an explicit date and time when the certificate expires, or as a span of time after issuance, stated in days,
         * months, or years. For more information, see <a
         * href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC 5280.
         * </p>
         * <p>
         * This value is unaffected when <code>ValidityNotBefore</code> is also specified. For example, if
         * <code>Validity</code> is set to 20 days in the future, the certificate will expire 20 days from issuance time
         * regardless of the <code>ValidityNotBefore</code> value.
         * </p>
         * <p>
         * The end of the validity period configured on a certificate must not exceed the limit set on its parents in
         * the CA hierarchy.
         * </p>
         * This is a convenience method that creates an instance of the {@link Validity.Builder} avoiding the need to
         * create one manually via {@link Validity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Validity.Builder#build()} is called immediately and its result is
         * passed to {@link #validity(Validity)}.
         * 
         * @param validity
         *        a consumer that will call methods on {@link Validity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validity(Validity)
         */
        default Builder validity(Consumer<Validity.Builder> validity) {
            return validity(Validity.builder().applyMutation(validity).build());
        }

        /**
         * <p>
         * Information describing the start of the validity period of the certificate. This parameter sets the “Not
         * Before" date for the certificate.
         * </p>
         * <p>
         * By default, when issuing a certificate, ACM Private CA sets the "Not Before" date to the issuance time minus
         * 60 minutes. This compensates for clock inconsistencies across computer systems. The
         * <code>ValidityNotBefore</code> parameter can be used to customize the “Not Before” value.
         * </p>
         * <p>
         * Unlike the <code>Validity</code> parameter, the <code>ValidityNotBefore</code> parameter is optional.
         * </p>
         * <p>
         * The <code>ValidityNotBefore</code> value is expressed as an explicit date and time, using the
         * <code>Validity</code> type value <code>ABSOLUTE</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_Validity.html">Validity</a> in this API
         * reference and <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC
         * 5280.
         * </p>
         * 
         * @param validityNotBefore
         *        Information describing the start of the validity period of the certificate. This parameter sets the
         *        “Not Before" date for the certificate.</p>
         *        <p>
         *        By default, when issuing a certificate, ACM Private CA sets the "Not Before" date to the issuance time
         *        minus 60 minutes. This compensates for clock inconsistencies across computer systems. The
         *        <code>ValidityNotBefore</code> parameter can be used to customize the “Not Before” value.
         *        </p>
         *        <p>
         *        Unlike the <code>Validity</code> parameter, the <code>ValidityNotBefore</code> parameter is optional.
         *        </p>
         *        <p>
         *        The <code>ValidityNotBefore</code> value is expressed as an explicit date and time, using the
         *        <code>Validity</code> type value <code>ABSOLUTE</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_Validity.html">Validity</a> in this
         *        API reference and <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a>
         *        in RFC 5280.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validityNotBefore(Validity validityNotBefore);

        /**
         * <p>
         * Information describing the start of the validity period of the certificate. This parameter sets the “Not
         * Before" date for the certificate.
         * </p>
         * <p>
         * By default, when issuing a certificate, ACM Private CA sets the "Not Before" date to the issuance time minus
         * 60 minutes. This compensates for clock inconsistencies across computer systems. The
         * <code>ValidityNotBefore</code> parameter can be used to customize the “Not Before” value.
         * </p>
         * <p>
         * Unlike the <code>Validity</code> parameter, the <code>ValidityNotBefore</code> parameter is optional.
         * </p>
         * <p>
         * The <code>ValidityNotBefore</code> value is expressed as an explicit date and time, using the
         * <code>Validity</code> type value <code>ABSOLUTE</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_Validity.html">Validity</a> in this API
         * reference and <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.1.2.5">Validity</a> in RFC
         * 5280.
         * </p>
         * This is a convenience method that creates an instance of the {@link Validity.Builder} avoiding the need to
         * create one manually via {@link Validity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Validity.Builder#build()} is called immediately and its result is
         * passed to {@link #validityNotBefore(Validity)}.
         * 
         * @param validityNotBefore
         *        a consumer that will call methods on {@link Validity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validityNotBefore(Validity)
         */
        default Builder validityNotBefore(Consumer<Validity.Builder> validityNotBefore) {
            return validityNotBefore(Validity.builder().applyMutation(validityNotBefore).build());
        }

        /**
         * <p>
         * Alphanumeric string that can be used to distinguish between calls to the <b>IssueCertificate</b> action.
         * Idempotency tokens for <b>IssueCertificate</b> time out after one minute. Therefore, if you call
         * <b>IssueCertificate</b> multiple times with the same idempotency token within one minute, ACM Private CA
         * recognizes that you are requesting only one certificate and will issue only one. If you change the
         * idempotency token for each call, PCA recognizes that you are requesting multiple certificates.
         * </p>
         * 
         * @param idempotencyToken
         *        Alphanumeric string that can be used to distinguish between calls to the <b>IssueCertificate</b>
         *        action. Idempotency tokens for <b>IssueCertificate</b> time out after one minute. Therefore, if you
         *        call <b>IssueCertificate</b> multiple times with the same idempotency token within one minute, ACM
         *        Private CA recognizes that you are requesting only one certificate and will issue only one. If you
         *        change the idempotency token for each call, PCA recognizes that you are requesting multiple
         *        certificates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idempotencyToken(String idempotencyToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends AcmPcaRequest.BuilderImpl implements Builder {
        private ApiPassthrough apiPassthrough;

        private String certificateAuthorityArn;

        private SdkBytes csr;

        private String signingAlgorithm;

        private String templateArn;

        private Validity validity;

        private Validity validityNotBefore;

        private String idempotencyToken;

        private BuilderImpl() {
        }

        private BuilderImpl(IssueCertificateRequest model) {
            super(model);
            apiPassthrough(model.apiPassthrough);
            certificateAuthorityArn(model.certificateAuthorityArn);
            csr(model.csr);
            signingAlgorithm(model.signingAlgorithm);
            templateArn(model.templateArn);
            validity(model.validity);
            validityNotBefore(model.validityNotBefore);
            idempotencyToken(model.idempotencyToken);
        }

        public final ApiPassthrough.Builder getApiPassthrough() {
            return apiPassthrough != null ? apiPassthrough.toBuilder() : null;
        }

        public final void setApiPassthrough(ApiPassthrough.BuilderImpl apiPassthrough) {
            this.apiPassthrough = apiPassthrough != null ? apiPassthrough.build() : null;
        }

        @Override
        public final Builder apiPassthrough(ApiPassthrough apiPassthrough) {
            this.apiPassthrough = apiPassthrough;
            return this;
        }

        public final String getCertificateAuthorityArn() {
            return certificateAuthorityArn;
        }

        public final void setCertificateAuthorityArn(String certificateAuthorityArn) {
            this.certificateAuthorityArn = certificateAuthorityArn;
        }

        @Override
        public final Builder certificateAuthorityArn(String certificateAuthorityArn) {
            this.certificateAuthorityArn = certificateAuthorityArn;
            return this;
        }

        public final ByteBuffer getCsr() {
            return csr == null ? null : csr.asByteBuffer();
        }

        public final void setCsr(ByteBuffer csr) {
            csr(csr == null ? null : SdkBytes.fromByteBuffer(csr));
        }

        @Override
        public final Builder csr(SdkBytes csr) {
            this.csr = csr;
            return this;
        }

        public final String getSigningAlgorithm() {
            return signingAlgorithm;
        }

        public final void setSigningAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
        }

        @Override
        public final Builder signingAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
            return this;
        }

        @Override
        public final Builder signingAlgorithm(SigningAlgorithm signingAlgorithm) {
            this.signingAlgorithm(signingAlgorithm == null ? null : signingAlgorithm.toString());
            return this;
        }

        public final String getTemplateArn() {
            return templateArn;
        }

        public final void setTemplateArn(String templateArn) {
            this.templateArn = templateArn;
        }

        @Override
        public final Builder templateArn(String templateArn) {
            this.templateArn = templateArn;
            return this;
        }

        public final Validity.Builder getValidity() {
            return validity != null ? validity.toBuilder() : null;
        }

        public final void setValidity(Validity.BuilderImpl validity) {
            this.validity = validity != null ? validity.build() : null;
        }

        @Override
        public final Builder validity(Validity validity) {
            this.validity = validity;
            return this;
        }

        public final Validity.Builder getValidityNotBefore() {
            return validityNotBefore != null ? validityNotBefore.toBuilder() : null;
        }

        public final void setValidityNotBefore(Validity.BuilderImpl validityNotBefore) {
            this.validityNotBefore = validityNotBefore != null ? validityNotBefore.build() : null;
        }

        @Override
        public final Builder validityNotBefore(Validity validityNotBefore) {
            this.validityNotBefore = validityNotBefore;
            return this;
        }

        public final String getIdempotencyToken() {
            return idempotencyToken;
        }

        public final void setIdempotencyToken(String idempotencyToken) {
            this.idempotencyToken = idempotencyToken;
        }

        @Override
        public final Builder idempotencyToken(String idempotencyToken) {
            this.idempotencyToken = idempotencyToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public IssueCertificateRequest build() {
            return new IssueCertificateRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
