/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration information for your private certificate authority (CA). This includes information about the
 * class of public key algorithm and the key pair that your private CA creates when it issues a certificate. It also
 * includes the signature algorithm that it uses when issuing certificates, and its X.500 distinguished name. You must
 * specify this information when you call the <a
 * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html"
 * >CreateCertificateAuthority</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CertificateAuthorityConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CertificateAuthorityConfiguration.Builder, CertificateAuthorityConfiguration> {
    private static final SdkField<String> KEY_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyAlgorithm").getter(getter(CertificateAuthorityConfiguration::keyAlgorithmAsString))
            .setter(setter(Builder::keyAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyAlgorithm").build()).build();

    private static final SdkField<String> SIGNING_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningAlgorithm").getter(getter(CertificateAuthorityConfiguration::signingAlgorithmAsString))
            .setter(setter(Builder::signingAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningAlgorithm").build()).build();

    private static final SdkField<ASN1Subject> SUBJECT_FIELD = SdkField.<ASN1Subject> builder(MarshallingType.SDK_POJO)
            .memberName("Subject").getter(getter(CertificateAuthorityConfiguration::subject)).setter(setter(Builder::subject))
            .constructor(ASN1Subject::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subject").build()).build();

    private static final SdkField<CsrExtensions> CSR_EXTENSIONS_FIELD = SdkField
            .<CsrExtensions> builder(MarshallingType.SDK_POJO).memberName("CsrExtensions")
            .getter(getter(CertificateAuthorityConfiguration::csrExtensions)).setter(setter(Builder::csrExtensions))
            .constructor(CsrExtensions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CsrExtensions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_ALGORITHM_FIELD,
            SIGNING_ALGORITHM_FIELD, SUBJECT_FIELD, CSR_EXTENSIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String keyAlgorithm;

    private final String signingAlgorithm;

    private final ASN1Subject subject;

    private final CsrExtensions csrExtensions;

    private CertificateAuthorityConfiguration(BuilderImpl builder) {
        this.keyAlgorithm = builder.keyAlgorithm;
        this.signingAlgorithm = builder.signingAlgorithm;
        this.subject = builder.subject;
        this.csrExtensions = builder.csrExtensions;
    }

    /**
     * <p>
     * Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it issues a
     * certificate. When you create a subordinate CA, you must use a key algorithm supported by the parent CA.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyAlgorithm} will
     * return {@link KeyAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyAlgorithmAsString}.
     * </p>
     * 
     * @return Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it issues a
     *         certificate. When you create a subordinate CA, you must use a key algorithm supported by the parent CA.
     * @see KeyAlgorithm
     */
    public final KeyAlgorithm keyAlgorithm() {
        return KeyAlgorithm.fromValue(keyAlgorithm);
    }

    /**
     * <p>
     * Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it issues a
     * certificate. When you create a subordinate CA, you must use a key algorithm supported by the parent CA.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyAlgorithm} will
     * return {@link KeyAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyAlgorithmAsString}.
     * </p>
     * 
     * @return Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it issues a
     *         certificate. When you create a subordinate CA, you must use a key algorithm supported by the parent CA.
     * @see KeyAlgorithm
     */
    public final String keyAlgorithmAsString() {
        return keyAlgorithm;
    }

    /**
     * <p>
     * Name of the algorithm your private CA uses to sign certificate requests.
     * </p>
     * <p>
     * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign certificates
     * when they are issued.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return Name of the algorithm your private CA uses to sign certificate requests.</p>
     *         <p>
     *         This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign
     *         certificates when they are issued.
     * @see SigningAlgorithm
     */
    public final SigningAlgorithm signingAlgorithm() {
        return SigningAlgorithm.fromValue(signingAlgorithm);
    }

    /**
     * <p>
     * Name of the algorithm your private CA uses to sign certificate requests.
     * </p>
     * <p>
     * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign certificates
     * when they are issued.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return Name of the algorithm your private CA uses to sign certificate requests.</p>
     *         <p>
     *         This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign
     *         certificates when they are issued.
     * @see SigningAlgorithm
     */
    public final String signingAlgorithmAsString() {
        return signingAlgorithm;
    }

    /**
     * <p>
     * Structure that contains X.500 distinguished name information for your private CA.
     * </p>
     * 
     * @return Structure that contains X.500 distinguished name information for your private CA.
     */
    public final ASN1Subject subject() {
        return subject;
    }

    /**
     * <p>
     * Specifies information to be added to the extension section of the certificate signing request (CSR).
     * </p>
     * 
     * @return Specifies information to be added to the extension section of the certificate signing request (CSR).
     */
    public final CsrExtensions csrExtensions() {
        return csrExtensions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(signingAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subject());
        hashCode = 31 * hashCode + Objects.hashCode(csrExtensions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CertificateAuthorityConfiguration)) {
            return false;
        }
        CertificateAuthorityConfiguration other = (CertificateAuthorityConfiguration) obj;
        return Objects.equals(keyAlgorithmAsString(), other.keyAlgorithmAsString())
                && Objects.equals(signingAlgorithmAsString(), other.signingAlgorithmAsString())
                && Objects.equals(subject(), other.subject()) && Objects.equals(csrExtensions(), other.csrExtensions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CertificateAuthorityConfiguration").add("KeyAlgorithm", keyAlgorithmAsString())
                .add("SigningAlgorithm", signingAlgorithmAsString()).add("Subject", subject())
                .add("CsrExtensions", csrExtensions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyAlgorithm":
            return Optional.ofNullable(clazz.cast(keyAlgorithmAsString()));
        case "SigningAlgorithm":
            return Optional.ofNullable(clazz.cast(signingAlgorithmAsString()));
        case "Subject":
            return Optional.ofNullable(clazz.cast(subject()));
        case "CsrExtensions":
            return Optional.ofNullable(clazz.cast(csrExtensions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("KeyAlgorithm", KEY_ALGORITHM_FIELD);
        map.put("SigningAlgorithm", SIGNING_ALGORITHM_FIELD);
        map.put("Subject", SUBJECT_FIELD);
        map.put("CsrExtensions", CSR_EXTENSIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CertificateAuthorityConfiguration, T> g) {
        return obj -> g.apply((CertificateAuthorityConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CertificateAuthorityConfiguration> {
        /**
         * <p>
         * Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it issues a
         * certificate. When you create a subordinate CA, you must use a key algorithm supported by the parent CA.
         * </p>
         * 
         * @param keyAlgorithm
         *        Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it
         *        issues a certificate. When you create a subordinate CA, you must use a key algorithm supported by the
         *        parent CA.
         * @see KeyAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyAlgorithm
         */
        Builder keyAlgorithm(String keyAlgorithm);

        /**
         * <p>
         * Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it issues a
         * certificate. When you create a subordinate CA, you must use a key algorithm supported by the parent CA.
         * </p>
         * 
         * @param keyAlgorithm
         *        Type of the public key algorithm and size, in bits, of the key pair that your CA creates when it
         *        issues a certificate. When you create a subordinate CA, you must use a key algorithm supported by the
         *        parent CA.
         * @see KeyAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyAlgorithm
         */
        Builder keyAlgorithm(KeyAlgorithm keyAlgorithm);

        /**
         * <p>
         * Name of the algorithm your private CA uses to sign certificate requests.
         * </p>
         * <p>
         * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign
         * certificates when they are issued.
         * </p>
         * 
         * @param signingAlgorithm
         *        Name of the algorithm your private CA uses to sign certificate requests.</p>
         *        <p>
         *        This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign
         *        certificates when they are issued.
         * @see SigningAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlgorithm
         */
        Builder signingAlgorithm(String signingAlgorithm);

        /**
         * <p>
         * Name of the algorithm your private CA uses to sign certificate requests.
         * </p>
         * <p>
         * This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign
         * certificates when they are issued.
         * </p>
         * 
         * @param signingAlgorithm
         *        Name of the algorithm your private CA uses to sign certificate requests.</p>
         *        <p>
         *        This parameter should not be confused with the <code>SigningAlgorithm</code> parameter used to sign
         *        certificates when they are issued.
         * @see SigningAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlgorithm
         */
        Builder signingAlgorithm(SigningAlgorithm signingAlgorithm);

        /**
         * <p>
         * Structure that contains X.500 distinguished name information for your private CA.
         * </p>
         * 
         * @param subject
         *        Structure that contains X.500 distinguished name information for your private CA.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subject(ASN1Subject subject);

        /**
         * <p>
         * Structure that contains X.500 distinguished name information for your private CA.
         * </p>
         * This is a convenience method that creates an instance of the {@link ASN1Subject.Builder} avoiding the need to
         * create one manually via {@link ASN1Subject#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ASN1Subject.Builder#build()} is called immediately and its result
         * is passed to {@link #subject(ASN1Subject)}.
         * 
         * @param subject
         *        a consumer that will call methods on {@link ASN1Subject.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subject(ASN1Subject)
         */
        default Builder subject(Consumer<ASN1Subject.Builder> subject) {
            return subject(ASN1Subject.builder().applyMutation(subject).build());
        }

        /**
         * <p>
         * Specifies information to be added to the extension section of the certificate signing request (CSR).
         * </p>
         * 
         * @param csrExtensions
         *        Specifies information to be added to the extension section of the certificate signing request (CSR).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder csrExtensions(CsrExtensions csrExtensions);

        /**
         * <p>
         * Specifies information to be added to the extension section of the certificate signing request (CSR).
         * </p>
         * This is a convenience method that creates an instance of the {@link CsrExtensions.Builder} avoiding the need
         * to create one manually via {@link CsrExtensions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CsrExtensions.Builder#build()} is called immediately and its
         * result is passed to {@link #csrExtensions(CsrExtensions)}.
         * 
         * @param csrExtensions
         *        a consumer that will call methods on {@link CsrExtensions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #csrExtensions(CsrExtensions)
         */
        default Builder csrExtensions(Consumer<CsrExtensions.Builder> csrExtensions) {
            return csrExtensions(CsrExtensions.builder().applyMutation(csrExtensions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String keyAlgorithm;

        private String signingAlgorithm;

        private ASN1Subject subject;

        private CsrExtensions csrExtensions;

        private BuilderImpl() {
        }

        private BuilderImpl(CertificateAuthorityConfiguration model) {
            keyAlgorithm(model.keyAlgorithm);
            signingAlgorithm(model.signingAlgorithm);
            subject(model.subject);
            csrExtensions(model.csrExtensions);
        }

        public final String getKeyAlgorithm() {
            return keyAlgorithm;
        }

        public final void setKeyAlgorithm(String keyAlgorithm) {
            this.keyAlgorithm = keyAlgorithm;
        }

        @Override
        public final Builder keyAlgorithm(String keyAlgorithm) {
            this.keyAlgorithm = keyAlgorithm;
            return this;
        }

        @Override
        public final Builder keyAlgorithm(KeyAlgorithm keyAlgorithm) {
            this.keyAlgorithm(keyAlgorithm == null ? null : keyAlgorithm.toString());
            return this;
        }

        public final String getSigningAlgorithm() {
            return signingAlgorithm;
        }

        public final void setSigningAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
        }

        @Override
        public final Builder signingAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
            return this;
        }

        @Override
        public final Builder signingAlgorithm(SigningAlgorithm signingAlgorithm) {
            this.signingAlgorithm(signingAlgorithm == null ? null : signingAlgorithm.toString());
            return this;
        }

        public final ASN1Subject.Builder getSubject() {
            return subject != null ? subject.toBuilder() : null;
        }

        public final void setSubject(ASN1Subject.BuilderImpl subject) {
            this.subject = subject != null ? subject.build() : null;
        }

        @Override
        public final Builder subject(ASN1Subject subject) {
            this.subject = subject;
            return this;
        }

        public final CsrExtensions.Builder getCsrExtensions() {
            return csrExtensions != null ? csrExtensions.toBuilder() : null;
        }

        public final void setCsrExtensions(CsrExtensions.BuilderImpl csrExtensions) {
            this.csrExtensions = csrExtensions != null ? csrExtensions.build() : null;
        }

        @Override
        public final Builder csrExtensions(CsrExtensions csrExtensions) {
            this.csrExtensions = csrExtensions;
            return this;
        }

        @Override
        public CertificateAuthorityConfiguration build() {
            return new CertificateAuthorityConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
