/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.acmpca.AcmPcaClient;
import software.amazon.awssdk.services.acmpca.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.acmpca.model.AcmPcaRequest;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityAuditReportRequest;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityAuditReportResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCsrRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCsrResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateResponse;
import software.amazon.awssdk.services.acmpca.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAcmPcaWaiter implements AcmPcaWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final AcmPcaClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetCertificateAuthorityCsrResponse> certificateAuthorityCSRCreatedWaiter;

    private final Waiter<GetCertificateResponse> certificateIssuedWaiter;

    private final Waiter<DescribeCertificateAuthorityAuditReportResponse> auditReportCreatedWaiter;

    private DefaultAcmPcaWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AcmPcaClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.certificateAuthorityCSRCreatedWaiter = Waiter.builder(GetCertificateAuthorityCsrResponse.class)
                .acceptors(certificateAuthorityCSRCreatedWaiterAcceptors())
                .overrideConfiguration(certificateAuthorityCSRCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.certificateIssuedWaiter = Waiter.builder(GetCertificateResponse.class).acceptors(certificateIssuedWaiterAcceptors())
                .overrideConfiguration(certificateIssuedWaiterConfig(builder.overrideConfiguration)).build();
        this.auditReportCreatedWaiter = Waiter.builder(DescribeCertificateAuthorityAuditReportResponse.class)
                .acceptors(auditReportCreatedWaiterAcceptors())
                .overrideConfiguration(auditReportCreatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeCertificateAuthorityAuditReportResponse> waitUntilAuditReportCreated(
            DescribeCertificateAuthorityAuditReportRequest describeCertificateAuthorityAuditReportRequest) {
        return auditReportCreatedWaiter.run(() -> client
                .describeCertificateAuthorityAuditReport(applyWaitersUserAgent(describeCertificateAuthorityAuditReportRequest)));
    }

    @Override
    public WaiterResponse<DescribeCertificateAuthorityAuditReportResponse> waitUntilAuditReportCreated(
            DescribeCertificateAuthorityAuditReportRequest describeCertificateAuthorityAuditReportRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return auditReportCreatedWaiter.run(() -> client
                .describeCertificateAuthorityAuditReport(applyWaitersUserAgent(describeCertificateAuthorityAuditReportRequest)),
                auditReportCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetCertificateAuthorityCsrResponse> waitUntilCertificateAuthorityCSRCreated(
            GetCertificateAuthorityCsrRequest getCertificateAuthorityCsrRequest) {
        return certificateAuthorityCSRCreatedWaiter.run(() -> client
                .getCertificateAuthorityCsr(applyWaitersUserAgent(getCertificateAuthorityCsrRequest)));
    }

    @Override
    public WaiterResponse<GetCertificateAuthorityCsrResponse> waitUntilCertificateAuthorityCSRCreated(
            GetCertificateAuthorityCsrRequest getCertificateAuthorityCsrRequest, WaiterOverrideConfiguration overrideConfig) {
        return certificateAuthorityCSRCreatedWaiter.run(
                () -> client.getCertificateAuthorityCsr(applyWaitersUserAgent(getCertificateAuthorityCsrRequest)),
                certificateAuthorityCSRCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetCertificateResponse> waitUntilCertificateIssued(GetCertificateRequest getCertificateRequest) {
        return certificateIssuedWaiter.run(() -> client.getCertificate(applyWaitersUserAgent(getCertificateRequest)));
    }

    @Override
    public WaiterResponse<GetCertificateResponse> waitUntilCertificateIssued(GetCertificateRequest getCertificateRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return certificateIssuedWaiter.run(() -> client.getCertificate(applyWaitersUserAgent(getCertificateRequest)),
                certificateIssuedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetCertificateAuthorityCsrResponse>> certificateAuthorityCSRCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetCertificateAuthorityCsrResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "RequestInProgressException")));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "AccessDeniedException"),
                "A waiter acceptor was matched on error condition (AccessDeniedException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetCertificateResponse>> certificateIssuedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetCertificateResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "RequestInProgressException")));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "AccessDeniedException"),
                "A waiter acceptor was matched on error condition (AccessDeniedException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCertificateAuthorityAuditReportResponse>> auditReportCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCertificateAuthorityAuditReportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AuditReportStatus").value(), "SUCCESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("AuditReportStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (AuditReportStatus=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "AccessDeniedException"),
                "A waiter acceptor was matched on error condition (AccessDeniedException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration certificateAuthorityCSRCreatedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration certificateIssuedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration auditReportCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AcmPcaWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AcmPcaRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AcmPcaWaiter.Builder {
        private AcmPcaClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public AcmPcaWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AcmPcaWaiter.Builder client(AcmPcaClient client) {
            this.client = client;
            return this;
        }

        public AcmPcaWaiter build() {
            return new DefaultAcmPcaWaiter(this);
        }
    }
}
