/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about your private certificate authority (CA). Your private CA can issue and revoke X.509
 * digital certificates. Digital certificates verify that the entity named in the certificate <b>Subject</b> field owns
 * or controls the public key contained in the <b>Subject Public Key Info</b> field. Call the <a
 * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html"
 * >CreateCertificateAuthority</a> action to create your private CA. You must then call the <a
 * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_GetCertificateAuthorityCertificate.html"
 * >GetCertificateAuthorityCertificate</a> action to retrieve a private CA certificate signing request (CSR). Sign the
 * CSR with your Amazon Web Services Private CA-hosted or on-premises root or subordinate CA certificate. Call the <a
 * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html">
 * ImportCertificateAuthorityCertificate</a> action to import the signed certificate into Certificate Manager (ACM).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CertificateAuthority implements SdkPojo, Serializable,
        ToCopyableBuilder<CertificateAuthority.Builder, CertificateAuthority> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(CertificateAuthority::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerAccount").getter(getter(CertificateAuthority::ownerAccount)).setter(setter(Builder::ownerAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerAccount").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(CertificateAuthority::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> LAST_STATE_CHANGE_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastStateChangeAt").getter(getter(CertificateAuthority::lastStateChangeAt))
            .setter(setter(Builder::lastStateChangeAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastStateChangeAt").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(CertificateAuthority::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> SERIAL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Serial")
            .getter(getter(CertificateAuthority::serial)).setter(setter(Builder::serial))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Serial").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(CertificateAuthority::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> NOT_BEFORE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("NotBefore").getter(getter(CertificateAuthority::notBefore)).setter(setter(Builder::notBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotBefore").build()).build();

    private static final SdkField<Instant> NOT_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("NotAfter").getter(getter(CertificateAuthority::notAfter)).setter(setter(Builder::notAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotAfter").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(CertificateAuthority::failureReasonAsString))
            .setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<CertificateAuthorityConfiguration> CERTIFICATE_AUTHORITY_CONFIGURATION_FIELD = SdkField
            .<CertificateAuthorityConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("CertificateAuthorityConfiguration")
            .getter(getter(CertificateAuthority::certificateAuthorityConfiguration))
            .setter(setter(Builder::certificateAuthorityConfiguration))
            .constructor(CertificateAuthorityConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateAuthorityConfiguration")
                    .build()).build();

    private static final SdkField<RevocationConfiguration> REVOCATION_CONFIGURATION_FIELD = SdkField
            .<RevocationConfiguration> builder(MarshallingType.SDK_POJO).memberName("RevocationConfiguration")
            .getter(getter(CertificateAuthority::revocationConfiguration)).setter(setter(Builder::revocationConfiguration))
            .constructor(RevocationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RevocationConfiguration").build())
            .build();

    private static final SdkField<Instant> RESTORABLE_UNTIL_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RestorableUntil").getter(getter(CertificateAuthority::restorableUntil))
            .setter(setter(Builder::restorableUntil))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestorableUntil").build()).build();

    private static final SdkField<String> KEY_STORAGE_SECURITY_STANDARD_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyStorageSecurityStandard")
            .getter(getter(CertificateAuthority::keyStorageSecurityStandardAsString))
            .setter(setter(Builder::keyStorageSecurityStandard))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyStorageSecurityStandard").build())
            .build();

    private static final SdkField<String> USAGE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UsageMode").getter(getter(CertificateAuthority::usageModeAsString)).setter(setter(Builder::usageMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageMode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            OWNER_ACCOUNT_FIELD, CREATED_AT_FIELD, LAST_STATE_CHANGE_AT_FIELD, TYPE_FIELD, SERIAL_FIELD, STATUS_FIELD,
            NOT_BEFORE_FIELD, NOT_AFTER_FIELD, FAILURE_REASON_FIELD, CERTIFICATE_AUTHORITY_CONFIGURATION_FIELD,
            REVOCATION_CONFIGURATION_FIELD, RESTORABLE_UNTIL_FIELD, KEY_STORAGE_SECURITY_STANDARD_FIELD, USAGE_MODE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String ownerAccount;

    private final Instant createdAt;

    private final Instant lastStateChangeAt;

    private final String type;

    private final String serial;

    private final String status;

    private final Instant notBefore;

    private final Instant notAfter;

    private final String failureReason;

    private final CertificateAuthorityConfiguration certificateAuthorityConfiguration;

    private final RevocationConfiguration revocationConfiguration;

    private final Instant restorableUntil;

    private final String keyStorageSecurityStandard;

    private final String usageMode;

    private CertificateAuthority(BuilderImpl builder) {
        this.arn = builder.arn;
        this.ownerAccount = builder.ownerAccount;
        this.createdAt = builder.createdAt;
        this.lastStateChangeAt = builder.lastStateChangeAt;
        this.type = builder.type;
        this.serial = builder.serial;
        this.status = builder.status;
        this.notBefore = builder.notBefore;
        this.notAfter = builder.notAfter;
        this.failureReason = builder.failureReason;
        this.certificateAuthorityConfiguration = builder.certificateAuthorityConfiguration;
        this.revocationConfiguration = builder.revocationConfiguration;
        this.restorableUntil = builder.restorableUntil;
        this.keyStorageSecurityStandard = builder.keyStorageSecurityStandard;
        this.usageMode = builder.usageMode;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) for your private certificate authority (CA). The format is
     * <code> <i>12345678-1234-1234-1234-123456789012</i> </code>.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) for your private certificate authority (CA). The format is
     *         <code> <i>12345678-1234-1234-1234-123456789012</i> </code>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that owns the certificate authority.
     * </p>
     * 
     * @return The Amazon Web Services account ID that owns the certificate authority.
     */
    public final String ownerAccount() {
        return ownerAccount;
    }

    /**
     * <p>
     * Date and time at which your private CA was created.
     * </p>
     * 
     * @return Date and time at which your private CA was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * Date and time at which your private CA was last updated.
     * </p>
     * 
     * @return Date and time at which your private CA was last updated.
     */
    public final Instant lastStateChangeAt() {
        return lastStateChangeAt;
    }

    /**
     * <p>
     * Type of your private CA.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CertificateAuthorityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Type of your private CA.
     * @see CertificateAuthorityType
     */
    public final CertificateAuthorityType type() {
        return CertificateAuthorityType.fromValue(type);
    }

    /**
     * <p>
     * Type of your private CA.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CertificateAuthorityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Type of your private CA.
     * @see CertificateAuthorityType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Serial number of your private CA.
     * </p>
     * 
     * @return Serial number of your private CA.
     */
    public final String serial() {
        return serial;
    }

    /**
     * <p>
     * Status of your private CA.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CertificateAuthorityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Status of your private CA.
     * @see CertificateAuthorityStatus
     */
    public final CertificateAuthorityStatus status() {
        return CertificateAuthorityStatus.fromValue(status);
    }

    /**
     * <p>
     * Status of your private CA.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CertificateAuthorityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Status of your private CA.
     * @see CertificateAuthorityStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Date and time before which your private CA certificate is not valid.
     * </p>
     * 
     * @return Date and time before which your private CA certificate is not valid.
     */
    public final Instant notBefore() {
        return notBefore;
    }

    /**
     * <p>
     * Date and time after which your private CA certificate is not valid.
     * </p>
     * 
     * @return Date and time after which your private CA certificate is not valid.
     */
    public final Instant notAfter() {
        return notAfter;
    }

    /**
     * <p>
     * Reason the request to create your private CA failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link FailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #failureReasonAsString}.
     * </p>
     * 
     * @return Reason the request to create your private CA failed.
     * @see FailureReason
     */
    public final FailureReason failureReason() {
        return FailureReason.fromValue(failureReason);
    }

    /**
     * <p>
     * Reason the request to create your private CA failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link FailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #failureReasonAsString}.
     * </p>
     * 
     * @return Reason the request to create your private CA failed.
     * @see FailureReason
     */
    public final String failureReasonAsString() {
        return failureReason;
    }

    /**
     * <p>
     * Your private CA configuration.
     * </p>
     * 
     * @return Your private CA configuration.
     */
    public final CertificateAuthorityConfiguration certificateAuthorityConfiguration() {
        return certificateAuthorityConfiguration;
    }

    /**
     * <p>
     * Information about the Online Certificate Status Protocol (OCSP) configuration or certificate revocation list
     * (CRL) created and maintained by your private CA.
     * </p>
     * 
     * @return Information about the Online Certificate Status Protocol (OCSP) configuration or certificate revocation
     *         list (CRL) created and maintained by your private CA.
     */
    public final RevocationConfiguration revocationConfiguration() {
        return revocationConfiguration;
    }

    /**
     * <p>
     * The period during which a deleted CA can be restored. For more information, see the
     * <code>PermanentDeletionTimeInDays</code> parameter of the <a
     * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeleteCertificateAuthorityRequest.html"
     * >DeleteCertificateAuthorityRequest</a> action.
     * </p>
     * 
     * @return The period during which a deleted CA can be restored. For more information, see the
     *         <code>PermanentDeletionTimeInDays</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeleteCertificateAuthorityRequest.html"
     *         >DeleteCertificateAuthorityRequest</a> action.
     */
    public final Instant restorableUntil() {
        return restorableUntil;
    }

    /**
     * <p>
     * Defines a cryptographic key management compliance standard for handling and protecting CA keys.
     * </p>
     * <p>
     * Default: FIPS_140_2_LEVEL_3_OR_HIGHER
     * </p>
     * <note>
     * <p>
     * Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China regions using
     * hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
     * </p>
     * <p>
     * For information about security standard support in different Amazon Web Services Regions, see <a
     * href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys">Storage and
     * security compliance of Amazon Web Services Private CA private keys</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyStorageSecurityStandard} will return {@link KeyStorageSecurityStandard#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #keyStorageSecurityStandardAsString}.
     * </p>
     * 
     * @return Defines a cryptographic key management compliance standard for handling and protecting CA keys.</p>
     *         <p>
     *         Default: FIPS_140_2_LEVEL_3_OR_HIGHER
     *         </p>
     *         <note>
     *         <p>
     *         Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China
     *         regions using hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
     *         </p>
     *         <p>
     *         For information about security standard support in different Amazon Web Services Regions, see <a
     *         href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys">Storage
     *         and security compliance of Amazon Web Services Private CA private keys</a>.
     *         </p>
     * @see KeyStorageSecurityStandard
     */
    public final KeyStorageSecurityStandard keyStorageSecurityStandard() {
        return KeyStorageSecurityStandard.fromValue(keyStorageSecurityStandard);
    }

    /**
     * <p>
     * Defines a cryptographic key management compliance standard for handling and protecting CA keys.
     * </p>
     * <p>
     * Default: FIPS_140_2_LEVEL_3_OR_HIGHER
     * </p>
     * <note>
     * <p>
     * Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China regions using
     * hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
     * </p>
     * <p>
     * For information about security standard support in different Amazon Web Services Regions, see <a
     * href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys">Storage and
     * security compliance of Amazon Web Services Private CA private keys</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyStorageSecurityStandard} will return {@link KeyStorageSecurityStandard#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #keyStorageSecurityStandardAsString}.
     * </p>
     * 
     * @return Defines a cryptographic key management compliance standard for handling and protecting CA keys.</p>
     *         <p>
     *         Default: FIPS_140_2_LEVEL_3_OR_HIGHER
     *         </p>
     *         <note>
     *         <p>
     *         Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China
     *         regions using hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
     *         </p>
     *         <p>
     *         For information about security standard support in different Amazon Web Services Regions, see <a
     *         href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys">Storage
     *         and security compliance of Amazon Web Services Private CA private keys</a>.
     *         </p>
     * @see KeyStorageSecurityStandard
     */
    public final String keyStorageSecurityStandardAsString() {
        return keyStorageSecurityStandard;
    }

    /**
     * <p>
     * Specifies whether the CA issues general-purpose certificates that typically require a revocation mechanism, or
     * short-lived certificates that may optionally omit revocation because they expire quickly. Short-lived certificate
     * validity is limited to seven days.
     * </p>
     * <p>
     * The default value is GENERAL_PURPOSE.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #usageMode} will
     * return {@link CertificateAuthorityUsageMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #usageModeAsString}.
     * </p>
     * 
     * @return Specifies whether the CA issues general-purpose certificates that typically require a revocation
     *         mechanism, or short-lived certificates that may optionally omit revocation because they expire quickly.
     *         Short-lived certificate validity is limited to seven days.</p>
     *         <p>
     *         The default value is GENERAL_PURPOSE.
     * @see CertificateAuthorityUsageMode
     */
    public final CertificateAuthorityUsageMode usageMode() {
        return CertificateAuthorityUsageMode.fromValue(usageMode);
    }

    /**
     * <p>
     * Specifies whether the CA issues general-purpose certificates that typically require a revocation mechanism, or
     * short-lived certificates that may optionally omit revocation because they expire quickly. Short-lived certificate
     * validity is limited to seven days.
     * </p>
     * <p>
     * The default value is GENERAL_PURPOSE.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #usageMode} will
     * return {@link CertificateAuthorityUsageMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #usageModeAsString}.
     * </p>
     * 
     * @return Specifies whether the CA issues general-purpose certificates that typically require a revocation
     *         mechanism, or short-lived certificates that may optionally omit revocation because they expire quickly.
     *         Short-lived certificate validity is limited to seven days.</p>
     *         <p>
     *         The default value is GENERAL_PURPOSE.
     * @see CertificateAuthorityUsageMode
     */
    public final String usageModeAsString() {
        return usageMode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccount());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastStateChangeAt());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serial());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(notBefore());
        hashCode = 31 * hashCode + Objects.hashCode(notAfter());
        hashCode = 31 * hashCode + Objects.hashCode(failureReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(revocationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(restorableUntil());
        hashCode = 31 * hashCode + Objects.hashCode(keyStorageSecurityStandardAsString());
        hashCode = 31 * hashCode + Objects.hashCode(usageModeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CertificateAuthority)) {
            return false;
        }
        CertificateAuthority other = (CertificateAuthority) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(ownerAccount(), other.ownerAccount())
                && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastStateChangeAt(), other.lastStateChangeAt())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(serial(), other.serial())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(notBefore(), other.notBefore())
                && Objects.equals(notAfter(), other.notAfter())
                && Objects.equals(failureReasonAsString(), other.failureReasonAsString())
                && Objects.equals(certificateAuthorityConfiguration(), other.certificateAuthorityConfiguration())
                && Objects.equals(revocationConfiguration(), other.revocationConfiguration())
                && Objects.equals(restorableUntil(), other.restorableUntil())
                && Objects.equals(keyStorageSecurityStandardAsString(), other.keyStorageSecurityStandardAsString())
                && Objects.equals(usageModeAsString(), other.usageModeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CertificateAuthority").add("Arn", arn()).add("OwnerAccount", ownerAccount())
                .add("CreatedAt", createdAt()).add("LastStateChangeAt", lastStateChangeAt()).add("Type", typeAsString())
                .add("Serial", serial()).add("Status", statusAsString()).add("NotBefore", notBefore())
                .add("NotAfter", notAfter()).add("FailureReason", failureReasonAsString())
                .add("CertificateAuthorityConfiguration", certificateAuthorityConfiguration())
                .add("RevocationConfiguration", revocationConfiguration()).add("RestorableUntil", restorableUntil())
                .add("KeyStorageSecurityStandard", keyStorageSecurityStandardAsString()).add("UsageMode", usageModeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "OwnerAccount":
            return Optional.ofNullable(clazz.cast(ownerAccount()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "LastStateChangeAt":
            return Optional.ofNullable(clazz.cast(lastStateChangeAt()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Serial":
            return Optional.ofNullable(clazz.cast(serial()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "NotBefore":
            return Optional.ofNullable(clazz.cast(notBefore()));
        case "NotAfter":
            return Optional.ofNullable(clazz.cast(notAfter()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReasonAsString()));
        case "CertificateAuthorityConfiguration":
            return Optional.ofNullable(clazz.cast(certificateAuthorityConfiguration()));
        case "RevocationConfiguration":
            return Optional.ofNullable(clazz.cast(revocationConfiguration()));
        case "RestorableUntil":
            return Optional.ofNullable(clazz.cast(restorableUntil()));
        case "KeyStorageSecurityStandard":
            return Optional.ofNullable(clazz.cast(keyStorageSecurityStandardAsString()));
        case "UsageMode":
            return Optional.ofNullable(clazz.cast(usageModeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Arn", ARN_FIELD);
        map.put("OwnerAccount", OWNER_ACCOUNT_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("LastStateChangeAt", LAST_STATE_CHANGE_AT_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("Serial", SERIAL_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("NotBefore", NOT_BEFORE_FIELD);
        map.put("NotAfter", NOT_AFTER_FIELD);
        map.put("FailureReason", FAILURE_REASON_FIELD);
        map.put("CertificateAuthorityConfiguration", CERTIFICATE_AUTHORITY_CONFIGURATION_FIELD);
        map.put("RevocationConfiguration", REVOCATION_CONFIGURATION_FIELD);
        map.put("RestorableUntil", RESTORABLE_UNTIL_FIELD);
        map.put("KeyStorageSecurityStandard", KEY_STORAGE_SECURITY_STANDARD_FIELD);
        map.put("UsageMode", USAGE_MODE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CertificateAuthority, T> g) {
        return obj -> g.apply((CertificateAuthority) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CertificateAuthority> {
        /**
         * <p>
         * Amazon Resource Name (ARN) for your private certificate authority (CA). The format is
         * <code> <i>12345678-1234-1234-1234-123456789012</i> </code>.
         * </p>
         * 
         * @param arn
         *        Amazon Resource Name (ARN) for your private certificate authority (CA). The format is
         *        <code> <i>12345678-1234-1234-1234-123456789012</i> </code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Amazon Web Services account ID that owns the certificate authority.
         * </p>
         * 
         * @param ownerAccount
         *        The Amazon Web Services account ID that owns the certificate authority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccount(String ownerAccount);

        /**
         * <p>
         * Date and time at which your private CA was created.
         * </p>
         * 
         * @param createdAt
         *        Date and time at which your private CA was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * Date and time at which your private CA was last updated.
         * </p>
         * 
         * @param lastStateChangeAt
         *        Date and time at which your private CA was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastStateChangeAt(Instant lastStateChangeAt);

        /**
         * <p>
         * Type of your private CA.
         * </p>
         * 
         * @param type
         *        Type of your private CA.
         * @see CertificateAuthorityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateAuthorityType
         */
        Builder type(String type);

        /**
         * <p>
         * Type of your private CA.
         * </p>
         * 
         * @param type
         *        Type of your private CA.
         * @see CertificateAuthorityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateAuthorityType
         */
        Builder type(CertificateAuthorityType type);

        /**
         * <p>
         * Serial number of your private CA.
         * </p>
         * 
         * @param serial
         *        Serial number of your private CA.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serial(String serial);

        /**
         * <p>
         * Status of your private CA.
         * </p>
         * 
         * @param status
         *        Status of your private CA.
         * @see CertificateAuthorityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateAuthorityStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Status of your private CA.
         * </p>
         * 
         * @param status
         *        Status of your private CA.
         * @see CertificateAuthorityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateAuthorityStatus
         */
        Builder status(CertificateAuthorityStatus status);

        /**
         * <p>
         * Date and time before which your private CA certificate is not valid.
         * </p>
         * 
         * @param notBefore
         *        Date and time before which your private CA certificate is not valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notBefore(Instant notBefore);

        /**
         * <p>
         * Date and time after which your private CA certificate is not valid.
         * </p>
         * 
         * @param notAfter
         *        Date and time after which your private CA certificate is not valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notAfter(Instant notAfter);

        /**
         * <p>
         * Reason the request to create your private CA failed.
         * </p>
         * 
         * @param failureReason
         *        Reason the request to create your private CA failed.
         * @see FailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FailureReason
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Reason the request to create your private CA failed.
         * </p>
         * 
         * @param failureReason
         *        Reason the request to create your private CA failed.
         * @see FailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FailureReason
         */
        Builder failureReason(FailureReason failureReason);

        /**
         * <p>
         * Your private CA configuration.
         * </p>
         * 
         * @param certificateAuthorityConfiguration
         *        Your private CA configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityConfiguration(CertificateAuthorityConfiguration certificateAuthorityConfiguration);

        /**
         * <p>
         * Your private CA configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CertificateAuthorityConfiguration.Builder} avoiding the need to create one manually via
         * {@link CertificateAuthorityConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CertificateAuthorityConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #certificateAuthorityConfiguration(CertificateAuthorityConfiguration)}.
         * 
         * @param certificateAuthorityConfiguration
         *        a consumer that will call methods on {@link CertificateAuthorityConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateAuthorityConfiguration(CertificateAuthorityConfiguration)
         */
        default Builder certificateAuthorityConfiguration(
                Consumer<CertificateAuthorityConfiguration.Builder> certificateAuthorityConfiguration) {
            return certificateAuthorityConfiguration(CertificateAuthorityConfiguration.builder()
                    .applyMutation(certificateAuthorityConfiguration).build());
        }

        /**
         * <p>
         * Information about the Online Certificate Status Protocol (OCSP) configuration or certificate revocation list
         * (CRL) created and maintained by your private CA.
         * </p>
         * 
         * @param revocationConfiguration
         *        Information about the Online Certificate Status Protocol (OCSP) configuration or certificate
         *        revocation list (CRL) created and maintained by your private CA.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revocationConfiguration(RevocationConfiguration revocationConfiguration);

        /**
         * <p>
         * Information about the Online Certificate Status Protocol (OCSP) configuration or certificate revocation list
         * (CRL) created and maintained by your private CA.
         * </p>
         * This is a convenience method that creates an instance of the {@link RevocationConfiguration.Builder} avoiding
         * the need to create one manually via {@link RevocationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RevocationConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #revocationConfiguration(RevocationConfiguration)}.
         * 
         * @param revocationConfiguration
         *        a consumer that will call methods on {@link RevocationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #revocationConfiguration(RevocationConfiguration)
         */
        default Builder revocationConfiguration(Consumer<RevocationConfiguration.Builder> revocationConfiguration) {
            return revocationConfiguration(RevocationConfiguration.builder().applyMutation(revocationConfiguration).build());
        }

        /**
         * <p>
         * The period during which a deleted CA can be restored. For more information, see the
         * <code>PermanentDeletionTimeInDays</code> parameter of the <a
         * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeleteCertificateAuthorityRequest.html"
         * >DeleteCertificateAuthorityRequest</a> action.
         * </p>
         * 
         * @param restorableUntil
         *        The period during which a deleted CA can be restored. For more information, see the
         *        <code>PermanentDeletionTimeInDays</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeleteCertificateAuthorityRequest.html"
         *        >DeleteCertificateAuthorityRequest</a> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restorableUntil(Instant restorableUntil);

        /**
         * <p>
         * Defines a cryptographic key management compliance standard for handling and protecting CA keys.
         * </p>
         * <p>
         * Default: FIPS_140_2_LEVEL_3_OR_HIGHER
         * </p>
         * <note>
         * <p>
         * Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China regions
         * using hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
         * </p>
         * <p>
         * For information about security standard support in different Amazon Web Services Regions, see <a
         * href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys">Storage and
         * security compliance of Amazon Web Services Private CA private keys</a>.
         * </p>
         * </note>
         * 
         * @param keyStorageSecurityStandard
         *        Defines a cryptographic key management compliance standard for handling and protecting CA keys.</p>
         *        <p>
         *        Default: FIPS_140_2_LEVEL_3_OR_HIGHER
         *        </p>
         *        <note>
         *        <p>
         *        Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China
         *        regions using hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
         *        </p>
         *        <p>
         *        For information about security standard support in different Amazon Web Services Regions, see <a
         *        href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys"
         *        >Storage and security compliance of Amazon Web Services Private CA private keys</a>.
         *        </p>
         * @see KeyStorageSecurityStandard
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyStorageSecurityStandard
         */
        Builder keyStorageSecurityStandard(String keyStorageSecurityStandard);

        /**
         * <p>
         * Defines a cryptographic key management compliance standard for handling and protecting CA keys.
         * </p>
         * <p>
         * Default: FIPS_140_2_LEVEL_3_OR_HIGHER
         * </p>
         * <note>
         * <p>
         * Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China regions
         * using hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
         * </p>
         * <p>
         * For information about security standard support in different Amazon Web Services Regions, see <a
         * href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys">Storage and
         * security compliance of Amazon Web Services Private CA private keys</a>.
         * </p>
         * </note>
         * 
         * @param keyStorageSecurityStandard
         *        Defines a cryptographic key management compliance standard for handling and protecting CA keys.</p>
         *        <p>
         *        Default: FIPS_140_2_LEVEL_3_OR_HIGHER
         *        </p>
         *        <note>
         *        <p>
         *        Starting January 26, 2023, Amazon Web Services Private CA protects all CA private keys in non-China
         *        regions using hardware security modules (HSMs) that comply with FIPS PUB 140-2 Level 3.
         *        </p>
         *        <p>
         *        For information about security standard support in different Amazon Web Services Regions, see <a
         *        href="https://docs.aws.amazon.com/privateca/latest/userguide/data-protection.html#private-keys"
         *        >Storage and security compliance of Amazon Web Services Private CA private keys</a>.
         *        </p>
         * @see KeyStorageSecurityStandard
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyStorageSecurityStandard
         */
        Builder keyStorageSecurityStandard(KeyStorageSecurityStandard keyStorageSecurityStandard);

        /**
         * <p>
         * Specifies whether the CA issues general-purpose certificates that typically require a revocation mechanism,
         * or short-lived certificates that may optionally omit revocation because they expire quickly. Short-lived
         * certificate validity is limited to seven days.
         * </p>
         * <p>
         * The default value is GENERAL_PURPOSE.
         * </p>
         * 
         * @param usageMode
         *        Specifies whether the CA issues general-purpose certificates that typically require a revocation
         *        mechanism, or short-lived certificates that may optionally omit revocation because they expire
         *        quickly. Short-lived certificate validity is limited to seven days.</p>
         *        <p>
         *        The default value is GENERAL_PURPOSE.
         * @see CertificateAuthorityUsageMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateAuthorityUsageMode
         */
        Builder usageMode(String usageMode);

        /**
         * <p>
         * Specifies whether the CA issues general-purpose certificates that typically require a revocation mechanism,
         * or short-lived certificates that may optionally omit revocation because they expire quickly. Short-lived
         * certificate validity is limited to seven days.
         * </p>
         * <p>
         * The default value is GENERAL_PURPOSE.
         * </p>
         * 
         * @param usageMode
         *        Specifies whether the CA issues general-purpose certificates that typically require a revocation
         *        mechanism, or short-lived certificates that may optionally omit revocation because they expire
         *        quickly. Short-lived certificate validity is limited to seven days.</p>
         *        <p>
         *        The default value is GENERAL_PURPOSE.
         * @see CertificateAuthorityUsageMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateAuthorityUsageMode
         */
        Builder usageMode(CertificateAuthorityUsageMode usageMode);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String ownerAccount;

        private Instant createdAt;

        private Instant lastStateChangeAt;

        private String type;

        private String serial;

        private String status;

        private Instant notBefore;

        private Instant notAfter;

        private String failureReason;

        private CertificateAuthorityConfiguration certificateAuthorityConfiguration;

        private RevocationConfiguration revocationConfiguration;

        private Instant restorableUntil;

        private String keyStorageSecurityStandard;

        private String usageMode;

        private BuilderImpl() {
        }

        private BuilderImpl(CertificateAuthority model) {
            arn(model.arn);
            ownerAccount(model.ownerAccount);
            createdAt(model.createdAt);
            lastStateChangeAt(model.lastStateChangeAt);
            type(model.type);
            serial(model.serial);
            status(model.status);
            notBefore(model.notBefore);
            notAfter(model.notAfter);
            failureReason(model.failureReason);
            certificateAuthorityConfiguration(model.certificateAuthorityConfiguration);
            revocationConfiguration(model.revocationConfiguration);
            restorableUntil(model.restorableUntil);
            keyStorageSecurityStandard(model.keyStorageSecurityStandard);
            usageMode(model.usageMode);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getOwnerAccount() {
            return ownerAccount;
        }

        public final void setOwnerAccount(String ownerAccount) {
            this.ownerAccount = ownerAccount;
        }

        @Override
        public final Builder ownerAccount(String ownerAccount) {
            this.ownerAccount = ownerAccount;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getLastStateChangeAt() {
            return lastStateChangeAt;
        }

        public final void setLastStateChangeAt(Instant lastStateChangeAt) {
            this.lastStateChangeAt = lastStateChangeAt;
        }

        @Override
        public final Builder lastStateChangeAt(Instant lastStateChangeAt) {
            this.lastStateChangeAt = lastStateChangeAt;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CertificateAuthorityType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getSerial() {
            return serial;
        }

        public final void setSerial(String serial) {
            this.serial = serial;
        }

        @Override
        public final Builder serial(String serial) {
            this.serial = serial;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CertificateAuthorityStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getNotBefore() {
            return notBefore;
        }

        public final void setNotBefore(Instant notBefore) {
            this.notBefore = notBefore;
        }

        @Override
        public final Builder notBefore(Instant notBefore) {
            this.notBefore = notBefore;
            return this;
        }

        public final Instant getNotAfter() {
            return notAfter;
        }

        public final void setNotAfter(Instant notAfter) {
            this.notAfter = notAfter;
        }

        @Override
        public final Builder notAfter(Instant notAfter) {
            this.notAfter = notAfter;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        @Override
        public final Builder failureReason(FailureReason failureReason) {
            this.failureReason(failureReason == null ? null : failureReason.toString());
            return this;
        }

        public final CertificateAuthorityConfiguration.Builder getCertificateAuthorityConfiguration() {
            return certificateAuthorityConfiguration != null ? certificateAuthorityConfiguration.toBuilder() : null;
        }

        public final void setCertificateAuthorityConfiguration(
                CertificateAuthorityConfiguration.BuilderImpl certificateAuthorityConfiguration) {
            this.certificateAuthorityConfiguration = certificateAuthorityConfiguration != null ? certificateAuthorityConfiguration
                    .build() : null;
        }

        @Override
        public final Builder certificateAuthorityConfiguration(CertificateAuthorityConfiguration certificateAuthorityConfiguration) {
            this.certificateAuthorityConfiguration = certificateAuthorityConfiguration;
            return this;
        }

        public final RevocationConfiguration.Builder getRevocationConfiguration() {
            return revocationConfiguration != null ? revocationConfiguration.toBuilder() : null;
        }

        public final void setRevocationConfiguration(RevocationConfiguration.BuilderImpl revocationConfiguration) {
            this.revocationConfiguration = revocationConfiguration != null ? revocationConfiguration.build() : null;
        }

        @Override
        public final Builder revocationConfiguration(RevocationConfiguration revocationConfiguration) {
            this.revocationConfiguration = revocationConfiguration;
            return this;
        }

        public final Instant getRestorableUntil() {
            return restorableUntil;
        }

        public final void setRestorableUntil(Instant restorableUntil) {
            this.restorableUntil = restorableUntil;
        }

        @Override
        public final Builder restorableUntil(Instant restorableUntil) {
            this.restorableUntil = restorableUntil;
            return this;
        }

        public final String getKeyStorageSecurityStandard() {
            return keyStorageSecurityStandard;
        }

        public final void setKeyStorageSecurityStandard(String keyStorageSecurityStandard) {
            this.keyStorageSecurityStandard = keyStorageSecurityStandard;
        }

        @Override
        public final Builder keyStorageSecurityStandard(String keyStorageSecurityStandard) {
            this.keyStorageSecurityStandard = keyStorageSecurityStandard;
            return this;
        }

        @Override
        public final Builder keyStorageSecurityStandard(KeyStorageSecurityStandard keyStorageSecurityStandard) {
            this.keyStorageSecurityStandard(keyStorageSecurityStandard == null ? null : keyStorageSecurityStandard.toString());
            return this;
        }

        public final String getUsageMode() {
            return usageMode;
        }

        public final void setUsageMode(String usageMode) {
            this.usageMode = usageMode;
        }

        @Override
        public final Builder usageMode(String usageMode) {
            this.usageMode = usageMode;
            return this;
        }

        @Override
        public final Builder usageMode(CertificateAuthorityUsageMode usageMode) {
            this.usageMode(usageMode == null ? null : usageMode.toString());
            return this;
        }

        @Override
        public CertificateAuthority build() {
            return new CertificateAuthority(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
