/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.aiops;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.aiops.internal.AiOpsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.aiops.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.aiops.model.AccessDeniedException;
import software.amazon.awssdk.services.aiops.model.AiOpsException;
import software.amazon.awssdk.services.aiops.model.ConflictException;
import software.amazon.awssdk.services.aiops.model.CreateInvestigationGroupRequest;
import software.amazon.awssdk.services.aiops.model.CreateInvestigationGroupResponse;
import software.amazon.awssdk.services.aiops.model.DeleteInvestigationGroupPolicyRequest;
import software.amazon.awssdk.services.aiops.model.DeleteInvestigationGroupPolicyResponse;
import software.amazon.awssdk.services.aiops.model.DeleteInvestigationGroupRequest;
import software.amazon.awssdk.services.aiops.model.DeleteInvestigationGroupResponse;
import software.amazon.awssdk.services.aiops.model.ForbiddenException;
import software.amazon.awssdk.services.aiops.model.GetInvestigationGroupPolicyRequest;
import software.amazon.awssdk.services.aiops.model.GetInvestigationGroupPolicyResponse;
import software.amazon.awssdk.services.aiops.model.GetInvestigationGroupRequest;
import software.amazon.awssdk.services.aiops.model.GetInvestigationGroupResponse;
import software.amazon.awssdk.services.aiops.model.InternalServerException;
import software.amazon.awssdk.services.aiops.model.ListInvestigationGroupsRequest;
import software.amazon.awssdk.services.aiops.model.ListInvestigationGroupsResponse;
import software.amazon.awssdk.services.aiops.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.aiops.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.aiops.model.PutInvestigationGroupPolicyRequest;
import software.amazon.awssdk.services.aiops.model.PutInvestigationGroupPolicyResponse;
import software.amazon.awssdk.services.aiops.model.ResourceNotFoundException;
import software.amazon.awssdk.services.aiops.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.aiops.model.TagResourceRequest;
import software.amazon.awssdk.services.aiops.model.TagResourceResponse;
import software.amazon.awssdk.services.aiops.model.ThrottlingException;
import software.amazon.awssdk.services.aiops.model.UntagResourceRequest;
import software.amazon.awssdk.services.aiops.model.UntagResourceResponse;
import software.amazon.awssdk.services.aiops.model.UpdateInvestigationGroupRequest;
import software.amazon.awssdk.services.aiops.model.UpdateInvestigationGroupResponse;
import software.amazon.awssdk.services.aiops.model.ValidationException;
import software.amazon.awssdk.services.aiops.transform.CreateInvestigationGroupRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.DeleteInvestigationGroupPolicyRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.DeleteInvestigationGroupRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.GetInvestigationGroupPolicyRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.GetInvestigationGroupRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.ListInvestigationGroupsRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.PutInvestigationGroupPolicyRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.aiops.transform.UpdateInvestigationGroupRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AiOpsClient}.
 *
 * @see AiOpsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAiOpsClient implements AiOpsClient {
    private static final Logger log = Logger.loggerFor(DefaultAiOpsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAiOpsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "AIOps" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an <i>investigation group</i> in your account. Creating an investigation group is a one-time setup task
     * for each Region in your account. It is a necessary task to be able to perform investigations.
     * </p>
     * <p>
     * Settings in the investigation group help you centrally manage the common properties of your investigations, such
     * as the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Who can access the investigations
     * </p>
     * </li>
     * <li>
     * <p>
     * Whether investigation data is encrypted with a customer managed Key Management Service key.
     * </p>
     * </li>
     * <li>
     * <p>
     * How long investigations and their data are retained by default.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Currently, you can have one investigation group in each Region in your account. Each investigation in a Region is
     * a part of the investigation group in that Region
     * </p>
     * <p>
     * To create an investigation group and set up CloudWatch investigations, you must be signed in to an IAM principal
     * that has either the <code>AIOpsConsoleAdminPolicy</code> or the <code>AdministratorAccess</code> IAM policy
     * attached, or to an account that has similar permissions.
     * </p>
     * <important>
     * <p>
     * You can configure CloudWatch alarms to start investigations and add events to investigations. If you create your
     * investigation group with <code>CreateInvestigationGroup</code> and you want to enable alarms to do this, you must
     * use <code>PutInvestigationGroupPolicy</code> to create a resource policy that grants this permission to
     * CloudWatch alarms.
     * </p>
     * <p>
     * For more information about configuring CloudWatch alarms, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html">Using Amazon
     * CloudWatch alarms</a>
     * </p>
     * </important>
     *
     * @param createInvestigationGroupRequest
     * @return Result of the CreateInvestigationGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.CreateInvestigationGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/CreateInvestigationGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateInvestigationGroupResponse createInvestigationGroup(
            CreateInvestigationGroupRequest createInvestigationGroupRequest) throws AccessDeniedException, ForbiddenException,
            ResourceNotFoundException, ThrottlingException, ServiceQuotaExceededException, ValidationException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInvestigationGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateInvestigationGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInvestigationGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInvestigationGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInvestigationGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateInvestigationGroupRequest, CreateInvestigationGroupResponse>()
                            .withOperationName("CreateInvestigationGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createInvestigationGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateInvestigationGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified investigation group from your account. You can currently have one investigation group per
     * Region in your account. After you delete an investigation group, you can later create a new investigation group
     * in the same Region.
     * </p>
     *
     * @param deleteInvestigationGroupRequest
     * @return Result of the DeleteInvestigationGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.DeleteInvestigationGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/DeleteInvestigationGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteInvestigationGroupResponse deleteInvestigationGroup(
            DeleteInvestigationGroupRequest deleteInvestigationGroupRequest) throws AccessDeniedException, ForbiddenException,
            ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInvestigationGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteInvestigationGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInvestigationGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInvestigationGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInvestigationGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteInvestigationGroupRequest, DeleteInvestigationGroupResponse>()
                            .withOperationName("DeleteInvestigationGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteInvestigationGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteInvestigationGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the IAM resource policy from being associated with the investigation group that you specify.
     * </p>
     *
     * @param deleteInvestigationGroupPolicyRequest
     * @return Result of the DeleteInvestigationGroupPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.DeleteInvestigationGroupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/DeleteInvestigationGroupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteInvestigationGroupPolicyResponse deleteInvestigationGroupPolicy(
            DeleteInvestigationGroupPolicyRequest deleteInvestigationGroupPolicyRequest) throws AccessDeniedException,
            ForbiddenException, ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInvestigationGroupPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteInvestigationGroupPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInvestigationGroupPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteInvestigationGroupPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInvestigationGroupPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteInvestigationGroupPolicyRequest, DeleteInvestigationGroupPolicyResponse>()
                            .withOperationName("DeleteInvestigationGroupPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteInvestigationGroupPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteInvestigationGroupPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the configuration information for the specified investigation group.
     * </p>
     *
     * @param getInvestigationGroupRequest
     * @return Result of the GetInvestigationGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.GetInvestigationGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/GetInvestigationGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetInvestigationGroupResponse getInvestigationGroup(GetInvestigationGroupRequest getInvestigationGroupRequest)
            throws AccessDeniedException, ForbiddenException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInvestigationGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInvestigationGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInvestigationGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvestigationGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvestigationGroup");

            return clientHandler.execute(new ClientExecutionParams<GetInvestigationGroupRequest, GetInvestigationGroupResponse>()
                    .withOperationName("GetInvestigationGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getInvestigationGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInvestigationGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the JSON of the IAM resource policy associated with the specified investigation group in a string. For
     * example,
     * <code>{\"Version\":\"2012-10-17\",\"Statement\":[{\"Effect\":\"Allow\",\"Principal\":{\"Service\":\"aiops.alarms.cloudwatch.amazonaws.com\"},\"Action\":[\"aiops:CreateInvestigation\",\"aiops:CreateInvestigationEvent\"],\"Resource\":\"*\",\"Condition\":{\"StringEquals\":{\"aws:SourceAccount\":\"111122223333\"},\"ArnLike\":{\"aws:SourceArn\":\"arn:aws:cloudwatch:us-east-1:111122223333:alarm:*\"}}}]}</code>
     * .
     * </p>
     *
     * @param getInvestigationGroupPolicyRequest
     * @return Result of the GetInvestigationGroupPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.GetInvestigationGroupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/GetInvestigationGroupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInvestigationGroupPolicyResponse getInvestigationGroupPolicy(
            GetInvestigationGroupPolicyRequest getInvestigationGroupPolicyRequest) throws AccessDeniedException,
            ForbiddenException, ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInvestigationGroupPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInvestigationGroupPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInvestigationGroupPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvestigationGroupPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvestigationGroupPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetInvestigationGroupPolicyRequest, GetInvestigationGroupPolicyResponse>()
                            .withOperationName("GetInvestigationGroupPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getInvestigationGroupPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetInvestigationGroupPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the ARN and name of each investigation group in the account.
     * </p>
     *
     * @param listInvestigationGroupsRequest
     * @return Result of the ListInvestigationGroups operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.ListInvestigationGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/ListInvestigationGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInvestigationGroupsResponse listInvestigationGroups(ListInvestigationGroupsRequest listInvestigationGroupsRequest)
            throws AccessDeniedException, ForbiddenException, ThrottlingException, ResourceNotFoundException,
            ValidationException, InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvestigationGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInvestigationGroupsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvestigationGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvestigationGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvestigationGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListInvestigationGroupsRequest, ListInvestigationGroupsResponse>()
                            .withOperationName("ListInvestigationGroups").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listInvestigationGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListInvestigationGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch investigations resource. Currently, investigation groups support
     * tagging.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AccessDeniedException, ForbiddenException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an IAM resource policy and assigns it to the specified investigation group.
     * </p>
     * <p>
     * If you create your investigation group with <code>CreateInvestigationGroup</code> and you want to enable
     * CloudWatch alarms to create investigations and add events to investigations, you must use this operation to
     * create a policy similar to this example.
     * </p>
     * <p>
     * <code> { "Version": "2008-10-17", "Statement": [ { "Effect": "Allow", "Principal": { "Service": "aiops.alarms.cloudwatch.amazonaws.com" }, "Action": [ "aiops:CreateInvestigation", "aiops:CreateInvestigationEvent" ], "Resource": "*", "Condition": { "StringEquals": { "aws:SourceAccount": "account-id" }, "ArnLike": { "aws:SourceArn": "arn:aws:cloudwatch:region:account-id:alarm:*" } } } ] } </code>
     * </p>
     *
     * @param putInvestigationGroupPolicyRequest
     * @return Result of the PutInvestigationGroupPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.PutInvestigationGroupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/PutInvestigationGroupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutInvestigationGroupPolicyResponse putInvestigationGroupPolicy(
            PutInvestigationGroupPolicyRequest putInvestigationGroupPolicyRequest) throws AccessDeniedException,
            ForbiddenException, ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutInvestigationGroupPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutInvestigationGroupPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putInvestigationGroupPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInvestigationGroupPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInvestigationGroupPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PutInvestigationGroupPolicyRequest, PutInvestigationGroupPolicyResponse>()
                            .withOperationName("PutInvestigationGroupPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putInvestigationGroupPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutInvestigationGroupPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AccessDeniedException,
            ForbiddenException, ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AccessDeniedException,
            ForbiddenException, ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of the specified investigation group.
     * </p>
     *
     * @param updateInvestigationGroupRequest
     * @return Result of the UpdateInvestigationGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ForbiddenException
     *         Access id denied for this operation, or this operation is not valid for the specified resource.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits. You can try again later.
     * @throws ValidationException
     *         This operation or its parameters aren't formatted correctly.
     * @throws InternalServerException
     *         An internal server error occurred. You can try again later.
     * @throws ConflictException
     *         This operation couldn't be completed because of a conflict in resource states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AiOpsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AiOpsClient.UpdateInvestigationGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/aiops-2018-05-10/UpdateInvestigationGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateInvestigationGroupResponse updateInvestigationGroup(
            UpdateInvestigationGroupRequest updateInvestigationGroupRequest) throws AccessDeniedException, ForbiddenException,
            ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, AiOpsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInvestigationGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateInvestigationGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInvestigationGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInvestigationGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AIOps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInvestigationGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateInvestigationGroupRequest, UpdateInvestigationGroupResponse>()
                            .withOperationName("UpdateInvestigationGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateInvestigationGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateInvestigationGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        AiOpsServiceClientConfigurationBuilder serviceConfigBuilder = new AiOpsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(AiOpsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final AiOpsServiceClientConfiguration serviceClientConfiguration() {
        return new AiOpsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
