/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.alexaforbusiness.paginators;

import java.util.Collections;
import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.services.alexaforbusiness.AlexaForBusinessClient;
import software.amazon.awssdk.services.alexaforbusiness.model.SearchSkillGroupsRequest;
import software.amazon.awssdk.services.alexaforbusiness.model.SearchSkillGroupsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.alexaforbusiness.AlexaForBusinessClient#searchSkillGroupsPaginator(software.amazon.awssdk.services.alexaforbusiness.model.SearchSkillGroupsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.alexaforbusiness.model.SearchSkillGroupsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.alexaforbusiness.paginators.SearchSkillGroupsIterable responses = client.searchSkillGroupsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.alexaforbusiness.paginators.SearchSkillGroupsIterable responses = client
 *             .searchSkillGroupsPaginator(request);
 *     for (software.amazon.awssdk.services.alexaforbusiness.model.SearchSkillGroupsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.alexaforbusiness.paginators.SearchSkillGroupsIterable responses = client.searchSkillGroupsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #searchSkillGroups(software.amazon.awssdk.services.alexaforbusiness.model.SearchSkillGroupsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SearchSkillGroupsIterable implements SdkIterable<SearchSkillGroupsResponse> {
    private final AlexaForBusinessClient client;

    private final SearchSkillGroupsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public SearchSkillGroupsIterable(AlexaForBusinessClient client, SearchSkillGroupsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new SearchSkillGroupsResponseFetcher();
    }

    @Override
    public Iterator<SearchSkillGroupsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link SearchSkillGroupsIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final SearchSkillGroupsIterable resume(SearchSkillGroupsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new SearchSkillGroupsIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new SearchSkillGroupsIterable(client, firstRequest) {
            @Override
            public Iterator<SearchSkillGroupsResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class SearchSkillGroupsResponseFetcher implements SyncPageFetcher<SearchSkillGroupsResponse> {
        @Override
        public boolean hasNextPage(SearchSkillGroupsResponse previousPage) {
            return previousPage.nextToken() != null && !SdkAutoConstructList.class.isInstance(previousPage.nextToken())
                    && !SdkAutoConstructMap.class.isInstance(previousPage.nextToken());
        }

        @Override
        public SearchSkillGroupsResponse nextPage(SearchSkillGroupsResponse previousPage) {
            if (previousPage == null) {
                return client.searchSkillGroups(firstRequest);
            }
            return client.searchSkillGroups(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
