/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amp;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amp.model.AccessDeniedException;
import software.amazon.awssdk.services.amp.model.AmpException;
import software.amazon.awssdk.services.amp.model.AmpRequest;
import software.amazon.awssdk.services.amp.model.ConflictException;
import software.amazon.awssdk.services.amp.model.CreateWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.CreateWorkspaceResponse;
import software.amazon.awssdk.services.amp.model.DeleteWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DeleteWorkspaceResponse;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.services.amp.model.InternalServerException;
import software.amazon.awssdk.services.amp.model.ListWorkspacesRequest;
import software.amazon.awssdk.services.amp.model.ListWorkspacesResponse;
import software.amazon.awssdk.services.amp.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amp.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.amp.model.ThrottlingException;
import software.amazon.awssdk.services.amp.model.UpdateWorkspaceAliasRequest;
import software.amazon.awssdk.services.amp.model.UpdateWorkspaceAliasResponse;
import software.amazon.awssdk.services.amp.model.ValidationException;
import software.amazon.awssdk.services.amp.paginators.ListWorkspacesIterable;
import software.amazon.awssdk.services.amp.transform.CreateWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DeleteWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DescribeWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.ListWorkspacesRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.UpdateWorkspaceAliasRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AmpClient}.
 *
 * @see AmpClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmpClient implements AmpClient {
    private static final Logger log = Logger.loggerFor(DefaultAmpClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmpClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new AMP workspace.
     * </p>
     *
     * @param createWorkspaceRequest
     *        Represents the input of a CreateWorkspace operation.
     * @return Result of the CreateWorkspace operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmpException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmpClient.CreateWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/CreateWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWorkspaceResponse createWorkspace(CreateWorkspaceRequest createWorkspaceRequest) throws ThrottlingException,
            ConflictException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, AmpException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspace");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkspaceRequest, CreateWorkspaceResponse>()
                    .withOperationName("CreateWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an AMP workspace.
     * </p>
     *
     * @param deleteWorkspaceRequest
     *        Represents the input of a DeleteWorkspace operation.
     * @return Result of the DeleteWorkspace operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmpException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmpClient.DeleteWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DeleteWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteWorkspaceResponse deleteWorkspace(DeleteWorkspaceRequest deleteWorkspaceRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, AmpException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspace");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkspaceRequest, DeleteWorkspaceResponse>()
                    .withOperationName("DeleteWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing AMP workspace.
     * </p>
     *
     * @param describeWorkspaceRequest
     *        Represents the input of a DescribeWorkspace operation.
     * @return Result of the DescribeWorkspace operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmpException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmpClient.DescribeWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DescribeWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeWorkspaceResponse describeWorkspace(DescribeWorkspaceRequest describeWorkspaceRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, AmpException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspace");

            return clientHandler.execute(new ClientExecutionParams<DescribeWorkspaceRequest, DescribeWorkspaceResponse>()
                    .withOperationName("DescribeWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all AMP workspaces, including workspaces being created or deleted.
     * </p>
     *
     * @param listWorkspacesRequest
     *        Represents the input of a ListWorkspaces operation.
     * @return Result of the ListWorkspaces operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmpException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmpClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListWorkspacesResponse listWorkspaces(ListWorkspacesRequest listWorkspacesRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            AmpException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkspacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkspacesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaces");

            return clientHandler.execute(new ClientExecutionParams<ListWorkspacesRequest, ListWorkspacesResponse>()
                    .withOperationName("ListWorkspaces").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWorkspacesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkspacesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all AMP workspaces, including workspaces being created or deleted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkspaces(software.amazon.awssdk.services.amp.model.ListWorkspacesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amp.paginators.ListWorkspacesIterable responses = client.listWorkspacesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.amp.paginators.ListWorkspacesIterable responses = client.listWorkspacesPaginator(request);
     *     for (software.amazon.awssdk.services.amp.model.ListWorkspacesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amp.paginators.ListWorkspacesIterable responses = client.listWorkspacesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkspaces(software.amazon.awssdk.services.amp.model.ListWorkspacesRequest)} operation.</b>
     * </p>
     *
     * @param listWorkspacesRequest
     *        Represents the input of a ListWorkspaces operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmpException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmpClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListWorkspacesIterable listWorkspacesPaginator(ListWorkspacesRequest listWorkspacesRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, AmpException {
        return new ListWorkspacesIterable(this, applyPaginatorUserAgent(listWorkspacesRequest));
    }

    /**
     * <p>
     * Updates an AMP workspace alias.
     * </p>
     *
     * @param updateWorkspaceAliasRequest
     *        Represents the input of an UpdateWorkspaceAlias operation.
     * @return Result of the UpdateWorkspaceAlias operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmpException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmpClient.UpdateWorkspaceAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/UpdateWorkspaceAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateWorkspaceAliasResponse updateWorkspaceAlias(UpdateWorkspaceAliasRequest updateWorkspaceAliasRequest)
            throws ThrottlingException, ConflictException, ValidationException, ResourceNotFoundException, AccessDeniedException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, AmpException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkspaceAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateWorkspaceAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkspaceAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspaceAlias");

            return clientHandler.execute(new ClientExecutionParams<UpdateWorkspaceAliasRequest, UpdateWorkspaceAliasResponse>()
                    .withOperationName("UpdateWorkspaceAlias").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateWorkspaceAliasRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWorkspaceAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmpException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends AmpRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
