/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amp;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amp.model.AccessDeniedException;
import software.amazon.awssdk.services.amp.model.AmpException;
import software.amazon.awssdk.services.amp.model.AmpRequest;
import software.amazon.awssdk.services.amp.model.ConflictException;
import software.amazon.awssdk.services.amp.model.CreateAlertManagerDefinitionRequest;
import software.amazon.awssdk.services.amp.model.CreateAlertManagerDefinitionResponse;
import software.amazon.awssdk.services.amp.model.CreateRuleGroupsNamespaceRequest;
import software.amazon.awssdk.services.amp.model.CreateRuleGroupsNamespaceResponse;
import software.amazon.awssdk.services.amp.model.CreateWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.CreateWorkspaceResponse;
import software.amazon.awssdk.services.amp.model.DeleteAlertManagerDefinitionRequest;
import software.amazon.awssdk.services.amp.model.DeleteAlertManagerDefinitionResponse;
import software.amazon.awssdk.services.amp.model.DeleteRuleGroupsNamespaceRequest;
import software.amazon.awssdk.services.amp.model.DeleteRuleGroupsNamespaceResponse;
import software.amazon.awssdk.services.amp.model.DeleteWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DeleteWorkspaceResponse;
import software.amazon.awssdk.services.amp.model.DescribeAlertManagerDefinitionRequest;
import software.amazon.awssdk.services.amp.model.DescribeAlertManagerDefinitionResponse;
import software.amazon.awssdk.services.amp.model.DescribeRuleGroupsNamespaceRequest;
import software.amazon.awssdk.services.amp.model.DescribeRuleGroupsNamespaceResponse;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.services.amp.model.InternalServerException;
import software.amazon.awssdk.services.amp.model.ListRuleGroupsNamespacesRequest;
import software.amazon.awssdk.services.amp.model.ListRuleGroupsNamespacesResponse;
import software.amazon.awssdk.services.amp.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.amp.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.amp.model.ListWorkspacesRequest;
import software.amazon.awssdk.services.amp.model.ListWorkspacesResponse;
import software.amazon.awssdk.services.amp.model.PutAlertManagerDefinitionRequest;
import software.amazon.awssdk.services.amp.model.PutAlertManagerDefinitionResponse;
import software.amazon.awssdk.services.amp.model.PutRuleGroupsNamespaceRequest;
import software.amazon.awssdk.services.amp.model.PutRuleGroupsNamespaceResponse;
import software.amazon.awssdk.services.amp.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amp.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.amp.model.TagResourceRequest;
import software.amazon.awssdk.services.amp.model.TagResourceResponse;
import software.amazon.awssdk.services.amp.model.ThrottlingException;
import software.amazon.awssdk.services.amp.model.UntagResourceRequest;
import software.amazon.awssdk.services.amp.model.UntagResourceResponse;
import software.amazon.awssdk.services.amp.model.UpdateWorkspaceAliasRequest;
import software.amazon.awssdk.services.amp.model.UpdateWorkspaceAliasResponse;
import software.amazon.awssdk.services.amp.model.ValidationException;
import software.amazon.awssdk.services.amp.paginators.ListRuleGroupsNamespacesPublisher;
import software.amazon.awssdk.services.amp.paginators.ListWorkspacesPublisher;
import software.amazon.awssdk.services.amp.transform.CreateAlertManagerDefinitionRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.CreateRuleGroupsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.CreateWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DeleteAlertManagerDefinitionRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DeleteRuleGroupsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DeleteWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DescribeAlertManagerDefinitionRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DescribeRuleGroupsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.DescribeWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.ListRuleGroupsNamespacesRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.ListWorkspacesRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.PutAlertManagerDefinitionRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.PutRuleGroupsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.amp.transform.UpdateWorkspaceAliasRequestMarshaller;
import software.amazon.awssdk.services.amp.waiters.AmpAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AmpAsyncClient}.
 *
 * @see AmpAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmpAsyncClient implements AmpAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAmpAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultAmpAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Create an alert manager definition.
     * </p>
     *
     * @param createAlertManagerDefinitionRequest
     *        Represents the input of a CreateAlertManagerDefinition operation.
     * @return A Java Future containing the result of the CreateAlertManagerDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.CreateAlertManagerDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/CreateAlertManagerDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAlertManagerDefinitionResponse> createAlertManagerDefinition(
            CreateAlertManagerDefinitionRequest createAlertManagerDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAlertManagerDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlertManagerDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAlertManagerDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAlertManagerDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAlertManagerDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAlertManagerDefinitionRequest, CreateAlertManagerDefinitionResponse>()
                            .withOperationName("CreateAlertManagerDefinition")
                            .withMarshaller(new CreateAlertManagerDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAlertManagerDefinitionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAlertManagerDefinitionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateAlertManagerDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a rule group namespace.
     * </p>
     *
     * @param createRuleGroupsNamespaceRequest
     *        Represents the input of a CreateRuleGroupsNamespace operation.
     * @return A Java Future containing the result of the CreateRuleGroupsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.CreateRuleGroupsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/CreateRuleGroupsNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRuleGroupsNamespaceResponse> createRuleGroupsNamespace(
            CreateRuleGroupsNamespaceRequest createRuleGroupsNamespaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRuleGroupsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRuleGroupsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRuleGroupsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRuleGroupsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRuleGroupsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRuleGroupsNamespaceRequest, CreateRuleGroupsNamespaceResponse>()
                            .withOperationName("CreateRuleGroupsNamespace")
                            .withMarshaller(new CreateRuleGroupsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRuleGroupsNamespaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createRuleGroupsNamespaceRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateRuleGroupsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new AMP workspace.
     * </p>
     *
     * @param createWorkspaceRequest
     *        Represents the input of a CreateWorkspace operation.
     * @return A Java Future containing the result of the CreateWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.CreateWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/CreateWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkspaceResponse> createWorkspace(CreateWorkspaceRequest createWorkspaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkspaceRequest, CreateWorkspaceResponse>()
                            .withOperationName("CreateWorkspace")
                            .withMarshaller(new CreateWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWorkspaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createWorkspaceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an alert manager definition.
     * </p>
     *
     * @param deleteAlertManagerDefinitionRequest
     *        Represents the input of a DeleteAlertManagerDefinition operation.
     * @return A Java Future containing the result of the DeleteAlertManagerDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.DeleteAlertManagerDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DeleteAlertManagerDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAlertManagerDefinitionResponse> deleteAlertManagerDefinition(
            DeleteAlertManagerDefinitionRequest deleteAlertManagerDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlertManagerDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlertManagerDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAlertManagerDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAlertManagerDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAlertManagerDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAlertManagerDefinitionRequest, DeleteAlertManagerDefinitionResponse>()
                            .withOperationName("DeleteAlertManagerDefinition")
                            .withMarshaller(new DeleteAlertManagerDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAlertManagerDefinitionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAlertManagerDefinitionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteAlertManagerDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a rule groups namespace.
     * </p>
     *
     * @param deleteRuleGroupsNamespaceRequest
     *        Represents the input of a DeleteRuleGroupsNamespace operation.
     * @return A Java Future containing the result of the DeleteRuleGroupsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.DeleteRuleGroupsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DeleteRuleGroupsNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRuleGroupsNamespaceResponse> deleteRuleGroupsNamespace(
            DeleteRuleGroupsNamespaceRequest deleteRuleGroupsNamespaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRuleGroupsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRuleGroupsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRuleGroupsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRuleGroupsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRuleGroupsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRuleGroupsNamespaceRequest, DeleteRuleGroupsNamespaceResponse>()
                            .withOperationName("DeleteRuleGroupsNamespace")
                            .withMarshaller(new DeleteRuleGroupsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRuleGroupsNamespaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteRuleGroupsNamespaceRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteRuleGroupsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AMP workspace.
     * </p>
     *
     * @param deleteWorkspaceRequest
     *        Represents the input of a DeleteWorkspace operation.
     * @return A Java Future containing the result of the DeleteWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.DeleteWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DeleteWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkspaceResponse> deleteWorkspace(DeleteWorkspaceRequest deleteWorkspaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkspaceRequest, DeleteWorkspaceResponse>()
                            .withOperationName("DeleteWorkspace")
                            .withMarshaller(new DeleteWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWorkspaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteWorkspaceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an alert manager definition.
     * </p>
     *
     * @param describeAlertManagerDefinitionRequest
     *        Represents the input of a DescribeAlertManagerDefinition operation.
     * @return A Java Future containing the result of the DescribeAlertManagerDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.DescribeAlertManagerDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DescribeAlertManagerDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAlertManagerDefinitionResponse> describeAlertManagerDefinition(
            DescribeAlertManagerDefinitionRequest describeAlertManagerDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAlertManagerDefinitionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlertManagerDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAlertManagerDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAlertManagerDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAlertManagerDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAlertManagerDefinitionRequest, DescribeAlertManagerDefinitionResponse>()
                            .withOperationName("DescribeAlertManagerDefinition")
                            .withMarshaller(new DescribeAlertManagerDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAlertManagerDefinitionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeAlertManagerDefinitionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeAlertManagerDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe a rule groups namespace.
     * </p>
     *
     * @param describeRuleGroupsNamespaceRequest
     *        Represents the input of a DescribeRuleGroupsNamespace operation.
     * @return A Java Future containing the result of the DescribeRuleGroupsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.DescribeRuleGroupsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DescribeRuleGroupsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRuleGroupsNamespaceResponse> describeRuleGroupsNamespace(
            DescribeRuleGroupsNamespaceRequest describeRuleGroupsNamespaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRuleGroupsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRuleGroupsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRuleGroupsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRuleGroupsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRuleGroupsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRuleGroupsNamespaceRequest, DescribeRuleGroupsNamespaceResponse>()
                            .withOperationName("DescribeRuleGroupsNamespace")
                            .withMarshaller(new DescribeRuleGroupsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRuleGroupsNamespaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeRuleGroupsNamespaceRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeRuleGroupsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing AMP workspace.
     * </p>
     *
     * @param describeWorkspaceRequest
     *        Represents the input of a DescribeWorkspace operation.
     * @return A Java Future containing the result of the DescribeWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.DescribeWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/DescribeWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWorkspaceResponse> describeWorkspace(DescribeWorkspaceRequest describeWorkspaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkspaceRequest, DescribeWorkspaceResponse>()
                            .withOperationName("DescribeWorkspace")
                            .withMarshaller(new DescribeWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeWorkspaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeWorkspaceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists rule groups namespaces.
     * </p>
     *
     * @param listRuleGroupsNamespacesRequest
     *        Represents the input of a ListRuleGroupsNamespaces operation.
     * @return A Java Future containing the result of the ListRuleGroupsNamespaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.ListRuleGroupsNamespaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListRuleGroupsNamespaces" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRuleGroupsNamespacesResponse> listRuleGroupsNamespaces(
            ListRuleGroupsNamespacesRequest listRuleGroupsNamespacesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRuleGroupsNamespacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRuleGroupsNamespaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRuleGroupsNamespacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRuleGroupsNamespacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRuleGroupsNamespacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRuleGroupsNamespacesRequest, ListRuleGroupsNamespacesResponse>()
                            .withOperationName("ListRuleGroupsNamespaces")
                            .withMarshaller(new ListRuleGroupsNamespacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRuleGroupsNamespacesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRuleGroupsNamespacesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListRuleGroupsNamespacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists rule groups namespaces.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRuleGroupsNamespaces(software.amazon.awssdk.services.amp.model.ListRuleGroupsNamespacesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amp.paginators.ListRuleGroupsNamespacesPublisher publisher = client.listRuleGroupsNamespacesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amp.paginators.ListRuleGroupsNamespacesPublisher publisher = client.listRuleGroupsNamespacesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.amp.model.ListRuleGroupsNamespacesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.amp.model.ListRuleGroupsNamespacesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRuleGroupsNamespaces(software.amazon.awssdk.services.amp.model.ListRuleGroupsNamespacesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRuleGroupsNamespacesRequest
     *        Represents the input of a ListRuleGroupsNamespaces operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.ListRuleGroupsNamespaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListRuleGroupsNamespaces" target="_top">AWS
     *      API Documentation</a>
     */
    public ListRuleGroupsNamespacesPublisher listRuleGroupsNamespacesPaginator(
            ListRuleGroupsNamespacesRequest listRuleGroupsNamespacesRequest) {
        return new ListRuleGroupsNamespacesPublisher(this, applyPaginatorUserAgent(listRuleGroupsNamespacesRequest));
    }

    /**
     * <p>
     * Lists the tags you have assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all AMP workspaces, including workspaces being created or deleted.
     * </p>
     *
     * @param listWorkspacesRequest
     *        Represents the input of a ListWorkspaces operation.
     * @return A Java Future containing the result of the ListWorkspaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkspacesResponse> listWorkspaces(ListWorkspacesRequest listWorkspacesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkspacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkspacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkspacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkspacesRequest, ListWorkspacesResponse>()
                            .withOperationName("ListWorkspaces")
                            .withMarshaller(new ListWorkspacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkspacesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listWorkspacesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListWorkspacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all AMP workspaces, including workspaces being created or deleted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkspaces(software.amazon.awssdk.services.amp.model.ListWorkspacesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amp.paginators.ListWorkspacesPublisher publisher = client.listWorkspacesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amp.paginators.ListWorkspacesPublisher publisher = client.listWorkspacesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.amp.model.ListWorkspacesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.amp.model.ListWorkspacesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkspaces(software.amazon.awssdk.services.amp.model.ListWorkspacesRequest)} operation.</b>
     * </p>
     *
     * @param listWorkspacesRequest
     *        Represents the input of a ListWorkspaces operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    public ListWorkspacesPublisher listWorkspacesPaginator(ListWorkspacesRequest listWorkspacesRequest) {
        return new ListWorkspacesPublisher(this, applyPaginatorUserAgent(listWorkspacesRequest));
    }

    /**
     * <p>
     * Update an alert manager definition.
     * </p>
     *
     * @param putAlertManagerDefinitionRequest
     *        Represents the input of a PutAlertManagerDefinition operation.
     * @return A Java Future containing the result of the PutAlertManagerDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.PutAlertManagerDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/PutAlertManagerDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAlertManagerDefinitionResponse> putAlertManagerDefinition(
            PutAlertManagerDefinitionRequest putAlertManagerDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAlertManagerDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAlertManagerDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAlertManagerDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAlertManagerDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAlertManagerDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAlertManagerDefinitionRequest, PutAlertManagerDefinitionResponse>()
                            .withOperationName("PutAlertManagerDefinition")
                            .withMarshaller(new PutAlertManagerDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putAlertManagerDefinitionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putAlertManagerDefinitionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<PutAlertManagerDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a rule groups namespace.
     * </p>
     *
     * @param putRuleGroupsNamespaceRequest
     *        Represents the input of a PutRuleGroupsNamespace operation.
     * @return A Java Future containing the result of the PutRuleGroupsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.PutRuleGroupsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/PutRuleGroupsNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRuleGroupsNamespaceResponse> putRuleGroupsNamespace(
            PutRuleGroupsNamespaceRequest putRuleGroupsNamespaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRuleGroupsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRuleGroupsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRuleGroupsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRuleGroupsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRuleGroupsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRuleGroupsNamespaceRequest, PutRuleGroupsNamespaceResponse>()
                            .withOperationName("PutRuleGroupsNamespace")
                            .withMarshaller(new PutRuleGroupsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putRuleGroupsNamespaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putRuleGroupsNamespaceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutRuleGroupsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates tags for the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an AMP workspace alias.
     * </p>
     *
     * @param updateWorkspaceAliasRequest
     *        Represents the input of an UpdateWorkspaceAlias operation.
     * @return A Java Future containing the result of the UpdateWorkspaceAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmpException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample AmpAsyncClient.UpdateWorkspaceAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/UpdateWorkspaceAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkspaceAliasResponse> updateWorkspaceAlias(
            UpdateWorkspaceAliasRequest updateWorkspaceAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkspaceAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "amp");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspaceAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkspaceAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkspaceAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkspaceAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkspaceAliasRequest, UpdateWorkspaceAliasResponse>()
                            .withOperationName("UpdateWorkspaceAlias")
                            .withMarshaller(new UpdateWorkspaceAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWorkspaceAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateWorkspaceAliasRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateWorkspaceAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmpException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends AmpRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public AmpAsyncWaiter waiter() {
        return AmpAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }
}
