/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amp.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.amp.AmpAsyncClient;
import software.amazon.awssdk.services.amp.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.amp.model.AmpRequest;
import software.amazon.awssdk.services.amp.model.DescribeScraperRequest;
import software.amazon.awssdk.services.amp.model.DescribeScraperResponse;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.services.amp.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAmpAsyncWaiter implements AmpAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final AmpAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeScraperResponse> scraperActiveWaiter;

    private final AsyncWaiter<DescribeScraperResponse> scraperDeletedWaiter;

    private final AsyncWaiter<DescribeWorkspaceResponse> workspaceActiveWaiter;

    private final AsyncWaiter<DescribeWorkspaceResponse> workspaceDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultAmpAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AmpAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.scraperActiveWaiter = AsyncWaiter.builder(DescribeScraperResponse.class).acceptors(scraperActiveWaiterAcceptors())
                .overrideConfiguration(scraperActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.scraperDeletedWaiter = AsyncWaiter.builder(DescribeScraperResponse.class).acceptors(scraperDeletedWaiterAcceptors())
                .overrideConfiguration(scraperDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.workspaceActiveWaiter = AsyncWaiter.builder(DescribeWorkspaceResponse.class)
                .acceptors(workspaceActiveWaiterAcceptors())
                .overrideConfiguration(workspaceActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.workspaceDeletedWaiter = AsyncWaiter.builder(DescribeWorkspaceResponse.class)
                .acceptors(workspaceDeletedWaiterAcceptors())
                .overrideConfiguration(workspaceDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperActive(
            DescribeScraperRequest describeScraperRequest) {
        return scraperActiveWaiter.runAsync(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperActive(
            DescribeScraperRequest describeScraperRequest, WaiterOverrideConfiguration overrideConfig) {
        return scraperActiveWaiter.runAsync(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)),
                scraperActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperDeleted(
            DescribeScraperRequest describeScraperRequest) {
        return scraperDeletedWaiter.runAsync(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperDeleted(
            DescribeScraperRequest describeScraperRequest, WaiterOverrideConfiguration overrideConfig) {
        return scraperDeletedWaiter.runAsync(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)),
                scraperDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceActive(
            DescribeWorkspaceRequest describeWorkspaceRequest) {
        return workspaceActiveWaiter.runAsync(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceActive(
            DescribeWorkspaceRequest describeWorkspaceRequest, WaiterOverrideConfiguration overrideConfig) {
        return workspaceActiveWaiter.runAsync(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)),
                workspaceActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceDeleted(
            DescribeWorkspaceRequest describeWorkspaceRequest) {
        return workspaceDeletedWaiter.runAsync(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceDeleted(
            DescribeWorkspaceRequest describeWorkspaceRequest, WaiterOverrideConfiguration overrideConfig) {
        return workspaceDeletedWaiter.runAsync(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)),
                workspaceDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeScraperResponse>> scraperActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeScraperResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("scraper").field("status").field("statusCode").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("scraper").field("status").field("statusCode").value(), "CREATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (scraper.status.statusCode=CREATION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeScraperResponse>> scraperDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeScraperResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("scraper").field("status").field("statusCode").value(), "DELETION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (scraper.status.statusCode=DELETION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeWorkspaceResponse>> workspaceActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeWorkspaceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "UPDATING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "CREATING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeWorkspaceResponse>> workspaceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeWorkspaceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "DELETING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration scraperActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration scraperDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration workspaceActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration workspaceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AmpAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AmpRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AmpAsyncWaiter.Builder {
        private AmpAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public AmpAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public AmpAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AmpAsyncWaiter.Builder client(AmpAsyncClient client) {
            this.client = client;
            return this;
        }

        public AmpAsyncWaiter build() {
            return new DefaultAmpAsyncWaiter(this);
        }
    }
}
